package org.eclipse.hyades.execution.local.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.net.SocketException;

import org.eclipse.hyades.execution.core.file.IFileLocater;
import org.eclipse.hyades.execution.core.file.IFileLocation;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.core.impl.ExecutionComponentImpl;
import org.eclipse.hyades.internal.execution.file.FileServiceConstants;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.ConnectionImpl;
import org.eclipse.hyades.internal.execution.local.control.SecureConnectionImpl;

import org.eclipse.hyades.internal.execution.local.common.ManageFileCommand;
import org.eclipse.hyades.internal.execution.local.common.Message;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * @author rduggan
 *
 */
public class FileManagerImpl extends ExecutionComponentImpl implements IFileManager {
	
	private IFileLocater locator;
	private Connection connection;
	
	public FileManagerImpl(Connection connection) {
		locator=new FileLocaterImpl(connection);
		this.connection=connection;
	}
	
	public void getFile(String localName, String remoteName) throws IOException{
		/* Locate the remote file */
		IFileLocation location=locator.getFile(remoteName);
		
		/* We may have to create the directory if it does not exist */
		String directoryName = null;

		File file=new File(localName);
		
		/* We try to get the directory where the file has to be written */
		directoryName = file.getParent();
		File dirFile = new File(directoryName);
		
		/* We have to create the directory only if it does not exist */
		if (!dirFile.exists())
			dirFile.mkdirs();	/* we will have to create parent directories as well */
		
		/* Create the local file.  Should we check if it exists? */
		file.createNewFile();
		
		/* Connect to the remote server and retrieve the file*/
		if(connection instanceof SecureConnectionImpl) {
			SecureResourceConnection resourceConnection=new SecureResourceConnection();
			
			try {
				resourceConnection.connect(connection.getNode(), location.getPort());
			}
			catch(AgentControllerUnavailableException e) {
				/* We can ignore this as the same node object is being used
				 * for this connection as the originalconnection to the server.  The
				 * node object has all the proper credential info in its cache
				 */
			}
		}
		else if(connection instanceof ConnectionImpl) {
			ResourceConnection resourceConnection=new ResourceConnection();
			try {
				resourceConnection.setParams(remoteName, ManageFileCommand.GET);
				resourceConnection.setInputStream(null);
				resourceConnection.setOutputStream(new FileOutputStream(file));
				resourceConnection.connect(connection.getNode(), location.getPort());
			}
			catch(AgentControllerUnavailableException e) {
				/* We can ignore this as the same node object is being used
				 * for this connection as the originalconnection to the server.  The
				 * node object has all the proper credential info in its cache
				 */
			}
			catch (FileNotFoundException e1){
				/* This is not expected to happen, since we have created
				 * the file ourselves. But if this does happen for some reason, we should not continue.
				 */
				throw e1;	/* The consumer has to catch this */
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.file.IFileManager#deleteFile(java.lang.String)
	 */
	public void deleteFile(String absFileName) throws IOException {
		locator.deleteFile(absFileName);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.file.IFileManager#putFile(java.lang.String, java.lang.String)
	 */
	public void putFile(String localName, String remoteName) throws IOException {
		IFileLocation location=locator.putFile(remoteName);
		
		File file=new File(localName);
		
		/* Connect to the remote server and send the file*/
		if(connection instanceof SecureConnectionImpl) {
			SecureResourceConnection resourceConnection=new SecureResourceConnection();
			
			try {
				resourceConnection.connect(connection.getNode(), location.getPort());
			}
			catch(AgentControllerUnavailableException e) {
				/* We can ignore this as the same node object is being used
				 * for this connection as the originalconnection to the server.  The
				 * node object has all the proper credential info in its cache
				 */
			}
		}
		else if(connection instanceof ConnectionImpl) {
			ResourceConnection resourceConnection=new ResourceConnection();
			try {
				/* We have to tell the file server the file name and the operation to perform */
				resourceConnection.setParams(remoteName, ManageFileCommand.PUT);
				
				/* The socket's input/output stream is not visible here, since the socket is created
				 * later in the connect method. So if we want to set the input/output stream of the
				 * socket, we set null here, and the socketreader thread checks if a stream is null, and 
				 * sets it to the sockets corresponding stream 
				 */
				try	{
					resourceConnection.setInputStream(new FileInputStream(file));
					resourceConnection.setOutputStream(null);
				}
				catch (FileNotFoundException e)	{
					
					/* If the file we are trying to open does not exist, we cannot proceed.
					 * Therefore, we just return.
					 */
					throw e;
					//return;
				}
				
				/* We connect to the file server, and begin the file transfer */
				resourceConnection.connect(connection.getNode(), location.getPort());
			}
			catch(AgentControllerUnavailableException e) {
				/* We can ignore this as the same node object is being used
				 * for this connection as the originalconnection to the server.  The
				 * node object has all the proper credential info in its cache
				 */
			}
		}
	}
	
	class ResourceConnection extends ConnectionImpl {
		
		private FileOutputStream  stream;
		private InputStream inStream;
		private OutputStream outStream;
		private String fileName;
		private long operation;
		
		protected void init() {
			/* Set the timeout on the socket reader to be 1 second */
			try {
				_socket.setSoTimeout(1000);
				_socket.setTcpNoDelay(true);
			}
			catch(SocketException e) {
				/* We can ignore this */
			}
			
			/* Currently, the implementation has changed from asynchronous file transfer, to
			 * synchronous file transfer. This is because, clients which have to process the file
			 * after it is transferred completely, are not notified when the file transfer is complete.
			 * When the implementation changes back to asynchronous, we just uncomment this call 
			 */
			
			//new Thread(new SocketReader()).start();
			
			/* For now, we do a synchronous file transfer */
			(new SocketReader()).run();
		}
		
		public void setStream(FileOutputStream  stream) {
			this.stream=stream;
		}
		
		/* We set the current input stream here */
		public void setInputStream (InputStream in)	{
			this.inStream = in;
		}
		
		/* We set the current output stream here */
		public void setOutputStream (OutputStream out)	{
			this.outStream = out;
		}
		
		/* We set the file name and operation, to send to the file server */
		public void setParams(String fileName, long operation)	{
			this.fileName = fileName;
			this.operation = operation;
		}
		
		class SocketReader implements Runnable, FileServiceConstants, org.eclipse.hyades.internal.execution.local.common.Constants {
			private boolean _isComplete = false;
			
			private byte[] buffer=new byte[RA_BUFFER_SIZE];
			private byte[] tempFileName=fileName.getBytes();
			
			private int byteCount = 0;
			
			public void run() {
				try {
					
					/* We first send the file name and operation type over the socket to the file server */
					
					OutputStream os = _socket.getOutputStream();
					long len = (int)fileName.length();
					int offset = 0;
					
					offset = Message.writeRALongToBuffer(buffer, 0, operation);
					offset = Message.writeRALongToBuffer(buffer, offset, len);
					System.arraycopy((Object)tempFileName, 0, (Object)buffer, offset, fileName.length());
					os.write(buffer);
					
				} catch (IOException e1) {
				}
				
				/* 
				 * This is where we check to see if we should set the current input/output stream to the
				 * socket's input/output stream. TODO: Check this again.
				 */ 
				if (inStream == null) {
					try {
						inStream =_socket.getInputStream();
					} catch (IOException e2) {
					}
				}
					
				if (outStream == null) {
					try {
						outStream = _socket.getOutputStream();
					} catch (IOException e3) {
					}
				}
	
			/* Run forever */
				
			outer:	
				while(!_isComplete) {
					try {
						/* Get the InputStream for this socket so we can read some data */
						//inStream = _socket.getInputStream();
						int bytesRead=inStream.read(buffer);
						
						if(bytesRead==-1) {
							break;
						}
						
						byteCount += bytesRead;
						outStream.write(buffer, 0, bytesRead);
					}
					catch(InterruptedIOException e) {
						/* This happens as the read has timed out.  Just continue as the socket is still valid.
						 * 
						 */
					}
					catch(SocketException e) {
						/* this is thrown when the RAC cuts our connection because we must support security or if the
						 * host has been denied access
						 */
						
						break;	
					}
					catch(IOException e) {
						
						break;
					}	
				} /* end of outer while */
				/* The server is now stopping */
				disconnect();
				
				try {
					outStream.close();
					inStream.close();
				}
				catch(IOException e) {
					/* Do we need to handle this? */
				}
			}
		}
	}
	
	class SecureResourceConnection extends SecureConnectionImpl {
		protected void init() {
		}
	}
}
