/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.loaders.internal;

import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.loaders.trace.TraceUtils;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.uml2sd.trace.TraceSDPlugin;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria;

/**
 * Utils
 */
public class TraceInteractionUtils {
	
	/**
	 * This method searches for the reference of the TRCProcess 
	 * selected in the profiling monitor view
	 */
	static public void searchTRCProcesses(EObject object, TraceProcesses list, IProgressMonitor monitor) {
		
		TRCAgent agent;
		
		if (monitor.isCanceled()) {
			return;
		}
		if (object instanceof TRCMonitor) {
			TRCMonitor m = (TRCMonitor)object;
			if (m.getNodes() == null ||
				m.getNodes().isEmpty()) {
				return;
			}
			for (Iterator i = m.getNodes().iterator(); i.hasNext() && !monitor.isCanceled(); ) {
				searchTRCProcesses((EObject)i.next(), list, monitor);
			}
		} else if (object instanceof TRCNode) {
			TRCNode n = (TRCNode)object;
			if (n.getProcessProxies() == null ||
				n.getProcessProxies().isEmpty()) {
				return;
			}
			for (Iterator i = n.getProcessProxies().iterator(); i.hasNext() && !monitor.isCanceled(); ) {
				searchTRCProcesses((EObject)i.next(), list, monitor);
			}
		} else if (object instanceof TRCProcessProxy) {
			TRCProcessProxy pp = (TRCProcessProxy)object;
			if (pp.getAgentProxies() == null ||
				pp.getAgentProxies().isEmpty()) {
				return;
			}
			for (Iterator i = pp.getAgentProxies().iterator(); i.hasNext() && !monitor.isCanceled(); ) {
				searchTRCProcesses((EObject)i.next(), list, monitor);
			}
		} else if (object instanceof TRCAgentProxy &&
				   ((TRCAgentProxy) object).getType().equals(HyadesConstants.PROFILE_AGENT_TYPE)) {
			agent = ((TRCAgentProxy) object).getAgent();
			TRCProcess process = TraceUtils.getProcess(agent);
			if (process != null &&
				process.getInvocations() != null &&
				process.getInvocations().size() > 0) {
				list.add(new TraceProcess(process));
			}
		}
	}

	/**
	 * @param methodInvocation
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCFullMethodInvocation methodInvocation) {
		return getAbsoluteEntryTime(methodInvocation.getProcess().getAgent())+
			   methodInvocation.getEntryTime();
	}

	/**
	 * @param agent
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCAgent agent) {
		double delta = agent.getAgentProxy().getProcessProxy().getNode().getDeltaTime()/1E6;
		return delta+agent.getStartTime();
	}

	/**
	 * @param process
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCProcess process) {
		return getAbsoluteEntryTime(process.getAgent())+
			   process.getStartTime();
	}

	/**
	 * @param thread
	 * @return
	 */
	public static double getAbsoluteEntryTime(TRCThread thread) {
		return getAbsoluteEntryTime(thread.getProcess().getAgent())+
			   thread.getStartTime();
	}

	/**
	 * @param methodInvocation
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCFullMethodInvocation methodInvocation) {
		double start = getAbsoluteEntryTime(methodInvocation.getProcess().getAgent());
		if (methodInvocation.getExitTime() != 0) {
			return start+methodInvocation.getExitTime();
		}
		if (methodInvocation.getInvokes() == null || methodInvocation.getInvokes().size() == 0) {
			return getAbsoluteExitTime(methodInvocation.getThread());
		}
		TRCFullMethodInvocation lastCalled =
			(TRCFullMethodInvocation)methodInvocation.getInvokes().get(methodInvocation.getInvokes().size()-1);
		if (lastCalled.getExitTime() == 0) {
			return getAbsoluteExitTime(lastCalled);
		}
		return start+lastCalled.getExitTime();
	}

	/**
	 * @param agent
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCAgent agent) {
		double delta = agent.getAgentProxy().getProcessProxy().getNode().getDeltaTime()/1E6;
		return delta+agent.getStopTime();
	}

	/**
	 * @param process
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCProcess process) {
		if (process.getStopTime() == 0) {
			return getAbsoluteExitTime(process.getAgent());
		}
		return getAbsoluteEntryTime(process.getAgent())+
			   process.getStopTime();
	}

	/**
	 * @param thread
	 * @return
	 */
	public static double getAbsoluteExitTime(TRCThread thread) {
		if (thread.getStopTime() == 0) {
			return getAbsoluteExitTime(thread.getProcess().getAgent());
		}
		return getAbsoluteEntryTime(thread.getProcess().getAgent())+
			   thread.getStopTime();
	}

	public static boolean matchCharacter(char c1, char c2, boolean isCaseSensitive) {
//		if (TraceSDPlugin.debugMatch) {
//			TraceSDPlugin.debugTraceMatch("Comparing "+c1+" to "+c2);
//		}
		return (c1 == c2) ||
			    (!isCaseSensitive &&	Character.toLowerCase(c1) == Character.toLowerCase(c2));
	}
	
	private static boolean trailingStars(String s, int l) {
		int i;
		for (i = l; i < s.length() && s.charAt(i) == '*'; i++);
		return i == s.length();
	}
			
	public static boolean matchCriteria(String s, Criteria criteria) {
		String e = criteria.getExpression();
		if (TraceSDPlugin.debugMatch) {
			TraceSDPlugin.debugTraceMatch("Comparing "+s+" to "+e); //$NON-NLS-1$ //$NON-NLS-2$
		}
		int i, j;
		for (i = 0, j = 0; i < s.length() && j < e.length(); i++) {
			char sc = s.charAt(i), ec = e.charAt(j);
			if (TraceSDPlugin.debugMatch) {
				TraceSDPlugin.debugTraceMatch("s["+i+"]="+sc+" e["+j+"]="+ec); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			}
			if (ec == '*') {
				if (j+1 == e.length()) {
					// trailing * -> everything matches
					if (TraceSDPlugin.debugMatch) {
						TraceSDPlugin.debugTraceMatch("#1a true"); //$NON-NLS-1$
					}
					return true;
				}
				int k;
				for (k = 1; j+k < e.length(); k++) {
					if (e.charAt(j+k) == '*' || e.charAt(j+k) == '?') {
						j += k;
						i += k - 2;
						break; // another wildcard -> next loop
					}
					if (!matchCharacter(s.charAt(i+k-1), e.charAt(j+k), criteria.isCaseSenstiveSelected())) {
						i += k - 1;
						break; // i will be incremented at the beginning of the next loop
					}
				}
				if (i+k-1 == s.length() && j+k == e.length()) {
					if (TraceSDPlugin.debugMatch) {
						TraceSDPlugin.debugTraceMatch("#1b true"); //$NON-NLS-1$
					}
					return true;
				}
			} else if (ec == '?') {
				if (i+1 < s.length() && j+1 < e.length()) {
					j++;
				} else {
					// trailing ? -> one char matches
					if (TraceSDPlugin.debugMatch) {
						TraceSDPlugin.debugTraceMatch(i+1 == s.length() && trailingStars(e, j+1) ?"#3 true":"#3 false"); //$NON-NLS-1$ //$NON-NLS-2$
					}
					return i+1 == s.length() && trailingStars(e, j+1);
				}
			} else if (!matchCharacter(sc, ec, criteria.isCaseSenstiveSelected())) {
				if (TraceSDPlugin.debugMatch) {
					TraceSDPlugin.debugTraceMatch("#4 false"); //$NON-NLS-1$
				}
				return false;
			} else {
				j++;
			}
		}
		if (TraceSDPlugin.debugMatch) {
			TraceSDPlugin.debugTraceMatch(i == s.length() && (j == e.length() || (j+1 == e.length() && (e.charAt(j) == '*' || e.charAt(j) == '?')))?"#5 true":"#5 false"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return i == s.length() && trailingStars(e, j);
	}

//	private static void testIt(String s, String e, boolean c, boolean ev) {
//		Criteria cr = new Criteria();
//		cr.setCaseSenstiveSelected(c); cr.setExpression(e);
//		boolean res = matchCriteria(s, cr);
//		if (res == ev) {
//			System.out.println("OK : "+s+" "+e+" ["+c+"]="+res+" (expecting "+ev+")");
//		} else {
//			System.out.println(" KO: "+s+" "+e+" ["+c+"]="+res+" (expecting "+ev+")");
//		}
//	}
//
//	public static void main(String args[]) {
//		TraceSDPlugin.debugMatch = true;
//		testIt("ThreadRun:1234", "Th*Run", false, false);
//		testIt("ThreadRun:1234", "Th*Run:123?", false, true);
//		testIt("ThreadRun:1234", "Th*Run?123?", false, true);
//		testIt("ThreadRun:1234", "Th*Run:1?3??", false, false);
//		testIt("ThreadRun:1234", "Th*Run:1?3?*", false, true);
//		testIt("ThreadRun", "Th*Run", true, true);
//		testIt("ThreadRun", "Th*Run", false, true);
//		testIt("ThreadRun", "ThreadRun*****************", true, true);
//		testIt("ThreadRun", "ThreadRu*********?*******", true, true);
//		testIt("ThreadRun", "ThreadRun*********?*******", true, false);
//		testIt("ThrennnadRun", "*n", false, true);
//		testIt("ThrennnadRun", "*N", true, false);
//		testIt("", "*", true, true);
//		testIt("TTT", "*", true, true);
//		testIt("TxxxT", "T???T", true, true);
//		testIt("TxxxT", "T?*?T", true, true); // This one does not work
//		testIt("TxxxT", "T??T", true, false);
//		testIt("T", "?", true, true);
//		testIt("TT", "*T", true, true);
//		testIt("TT", "?T", true, true);
//		testIt("TT", "T*", true, true);
//		testIt("TT", "T?", true, true);
//		testIt("TT", "**", true, true);
//		testIt("TT", "??", true, true);
//	}
			
	
}
