/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.recorder.local;

import java.io.IOException;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.util.Enumeration;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.execution.core.DaemonConnectException;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IRemoteHyadesComponent;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.local.ExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub;
import org.eclipse.hyades.execution.local.JavaProcessExecutorStub;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.execution.local.SessionStub;
import org.eclipse.hyades.execution.recorder.RecorderPlugin;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.recorder.Recorder;
import org.eclipse.hyades.internal.execution.recorder.local.appadapters.IRecorderApplicationAdapter;
import org.eclipse.hyades.internal.execution.recorder.local.appadapters.RecorderAppAdapterFactory;
import org.eclipse.hyades.internal.execution.recorder.local.appadapters.RecorderApplicationAdapterException;
import org.eclipse.hyades.internal.execution.recorder.ui.dialogs.RecorderProgressDialog;
import org.eclipse.hyades.internal.execution.recorder.ui.views.RecorderControlView;
import org.eclipse.hyades.internal.execution.testgen.TestGenerator;
import org.eclipse.hyades.internal.execution.testgen.TestGeneratorFactory;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.widgets.Display;

/**
 * This object is responsible for initiating the recorder session.  It constructs 
 * all necessary object, create a RAC Session, and initiates.  It also stay in scope
 * with sub-classes to shut down all machinery after the recording is complete.
 * @author Ernest Jessee
 */
public class RecorderClient 
{
	/**
	 * This is our handle to the remote Java process.  It is our abstraction of the Java Process 
	 */	
	private JavaProcessExecutorStub executor;

	/**
	 * The the object used to setup the execution environment which will be used 
	 * on the RAC to run the recorder
	 */
	private IExecutionEnvironment exeEnvironment = null;

	/**
	 * The factory which manages the instances of RAC execution infrastructure
	 * objects.
	 */
	private IExecutionComponentFactory factory=null;

	/**
	 * The session in which the recorder resides and performs it's duties.
	 */
	private ISession session=null;

	/**
	 * this object is the a reference to the execution object which runs on
	 * the RAC and facilitates communication between the remote process and Eclipse.
	 * This could be called an agent.
	 */
	private IRemoteHyadesComponent remoteHyadesComponent=null;

	/**
	 * This is the object that actually runs on the RAC. Although this object does not contain the 
	 * desired functionality, it knows how to start the java object that does contain the desired
	 * functionality.
	 */
	private IExecutableObject executableObject=null;

	/**
	 * The recorder that this client will spin up
	 */
	private Recorder theRecorder;
	
	/**
	 * The Agent object used to send control messages from eclipse to the Recorder as it 
	 * runs on the RAC
	 */
	private Agent controlAgent = null;
	
	/**
	 * This object is constructed with an IRecorderDataProcessor and contains methods
	 * to facilitate data routing as it comes from the RecorderAgent.
	 * This object is registered with the execution infrastructure as the dataprocessor 
	 * rather than the IRecorderDataProcessor
	 */
	private RecorderDataProcessorHelper dataProcessorHelper;

	private Display display = Display.getCurrent();

	/**
	 * Dialog indicating initialization progress
	 */
	private RecorderProgressDialog progressDlg = null;
	private int ticks = 0;
	final int RECORDER_MAX_TICKS = 15;
	
	public IRecorderApplicationAdapter appAdapter = null;	
	
	/**
	 * object which watches the automatically started application for shutdown and performs cleanup.
	 */
	private RecorderStopper recorderStopper = null;
	
	/** 
	 * This thread object is started by the RecorderClient and waits for a complete message to come from the RAC 
	 * upon receiving this message, the scripts are generated if a script generator was specified.
	 * @author Ernest Jessee
	 */
	private class FinishCleanupAndGenTestThread extends Thread
	{
		
		/**
		 * the constructor
		 */
		public FinishCleanupAndGenTestThread()
		{
			setName("Recording completion listener");			 //$NON-NLS-1$
		}
		
		/**
		 * the main run method of the thread.
		 */
		public void run()
		{				
			
			//wait for the dataProcessor to get a complete call
			while(!dataProcessorHelper.isRecordingComplete())
			{
				delay(200);
			}		
			try  {
				stopApplication();
			} catch (Exception e) {}
			
			//since we have the completion notification, we'll de-activate this recorder
			theRecorder.setActive(false);
			
			//and tell the user that the recorder is done.
			RecorderControlView.getInstance().addMessage(RecorderPlugin.getResourceString("RecorderClient.RECORDING_COMPLETE_MESSAGE")); //$NON-NLS-1$
			//now we need tdo seeif the user wants to generate the suite.  If he does, there will be a scriptgenID in the recorder object.
			//if he does not, the scriptgenid wil be -1.
		
			
			// First, write all Unwritten "CLOSE"   mdd
			
			//if we don't have a -1, then we'll kick off scriptgen
			if(!theRecorder.getScriptgenID().equalsIgnoreCase("-1")) //$NON-NLS-1$
			{
				//Testgen will cause TestHttp to be initialized.
				//if we testgen in a worker thread, when TestHttp is initialized, it will have no
				//concept of the current display (which it needs to load images)
				//Thus, we'll run this asynchronously in the UI thread
				display.asyncExec( new Runnable() {					
					public void run() {
							String scriptGenID = theRecorder.getScriptgenID();
							IConfigurationElement scriptGenElement = null;										
							String recordingPath = theRecorder.getRecordingPath();
							RecorderControlView.getInstance().addMessage(RecorderPlugin.getResourceString("RecorderClient.Test_generation_started_with_generator___4")+scriptGenID); //$NON-NLS-1$
							scriptGenElement=TestGeneratorFactory.getInstance().getGeneratorConfigElement(scriptGenID);
								
							try
							{
								TestGenerator generator = (TestGenerator)scriptGenElement.createExecutableExtension("class"); //$NON-NLS-1$
								
								String baseFilePath = null;
								if(recordingPath.endsWith(".rec")) //$NON-NLS-1$
								{
									baseFilePath = recordingPath.substring(0,recordingPath.length()-3);
								}
								else
								{
									String[] messageChildren = {recordingPath,".rec"}; //$NON-NLS-1$
									String formatString =RecorderPlugin.getResourceString("RecorderClient.RECORDING_PATH_NOT_CORRECT_FORMAT_STRING") ; //$NON-NLS-1$
									sendStatusMessageToControlView(MessageFormat.format(formatString,messageChildren));
									return;
								}
								
								IFile traceFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(baseFilePath+"recmodel")); //$NON-NLS-1$
								if(traceFile.exists())
								{
									RecorderControlView.getInstance().addMessage(RecorderPlugin.getResourceString("RecorderClient.TEST_TO_BE_GENERATED")+theRecorder.getTestPath()); //$NON-NLS-1$
									generator.initialize(traceFile,theRecorder.getTestPath());									
									generator.run();
									if (generator.isSuccess())
										RecorderControlView.getInstance().addMessage(RecorderPlugin.getResourceString("RecorderClient.TEST_GENERATION_COMPLETE")); //$NON-NLS-1$
									else
										RecorderControlView.getInstance().addMessage(RecorderPlugin.getResourceString("RecorderClient.TEST_GENERATION_ERRORS")); //$NON-NLS-1$
								}
								else
									RecorderControlView.getInstance().addMessage(RecorderPlugin.getResourceString("RecorderClient.UNABLE_TO_GENERATE_NO_TRACE_FILE")); //$NON-NLS-1$
							}
							catch (CoreException e)
							{
								e.printStackTrace();
							} 
					}
				});
			}
			updateViewStatus(RecorderPlugin.getResourceString("RecorderClient.STATUS_STOPPED"));
			session.release();
			
			
		
			
		}

		private void delay(int miliseconds)
		{
			try
			{
				Thread.sleep(miliseconds);
			}
			catch (InterruptedException e)
			{
				e.printStackTrace();
			}
		}

	}
	
	
	/**
	 * This object is an agent listener to intercept messages coming across the data channel from the remote side
	 * @author Ernest Jessee
	 */
	private class RecorderAgentListener implements AgentListener
	{
			
		public void agentActive(Agent agent)
		{
		}

		
		public void agentInactive(Agent agent)
		{

		}

		
		public void error(Agent agent, String errorId, String errorMessage)
		{
//			System.out.println("received Error: "+errorMessage);
		}

		
		public void handleCommand(Agent agent, CommandElement command)
		{
			if(command instanceof CustomCommand)
			{
				String errorMessage=((CustomCommand)command).getData();
				System.out.println("recieving message: "+errorMessage); //$NON-NLS-1$
			}	

		}

	}

	/**
	 * An object to contain control messages sent to the remote side
	 * @author Ernest Jessee
	 */
	private class ControlMessage implements IControlMessage
	{
		
		private String messageData;

		public ControlMessage(String message)
		{
			messageData = message;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.hyades.execution.core.IControlMessage#getMessageData()
		 */
		public String getMessageData()
		{
			return messageData;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.hyades.execution.core.IControlMessage#setMessageData(java.lang.String)
		 */
		public void setMessageData(String data)
		{
			messageData = data;

		}

	}
	
	
	
	/**
	 * This is where the RecorderClient actually instantiates all the objects it needs and initiates a recording on the RAC
	 */
	public void run(final Recorder recorder) throws IllegalAccessException, InstantiationException, IOException 
	{					
		//this work is done in a secondary thread so the wizard will go away.  If we don't do it this way, the wizard 
		//stays in a funky state for several seconds.
		new Thread()
		{				
			public void run()
			{
				try 
				{								
					theRecorder = recorder;
					initProgressDialog();
					
					updateViewStatus(RecorderPlugin.getResourceString("RecorderClient.STATUS_INIT"));
				
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.RECORDING_STARTED_STATUS_MESSAGE")); //$NON-NLS-1$
					
					session = createSession();								
	
					//get our reference to the ExecutionComponentFactory
					factory = ExecutionComponentFactoryImpl.getInstance(session);
		
					//create an execution environment -- in this case, a java environment.
					//this is our sandbox
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.SETTING_UP_JAVA_ENVIRONMENT_STATUS_MESSAGE")); //$NON-NLS-1$
					exeEnvironment = setupJavaEnvironment();

					//add this environment to our RAC session
					session.addChild(exeEnvironment);
		
					//create an executor.
					//this is our hands
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.CONFIGURING_EXECUTOR_STATUS_MESSAGE")); //$NON-NLS-1$
					executor = setupExecutor();
		
					//setup an agent
					//this is our ears
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.CONFIGURING_CONTROL_CHANNEL_COMMUNICATION_STATUS_MESSAGE")); //$NON-NLS-1$
					remoteHyadesComponent = setupAgent();
					dataProcessorHelper = new RecorderDataProcessorHelper(theRecorder.getDataProcessor());
					remoteHyadesComponent.startMonitoring(dataProcessorHelper);
		
					//setup our executable object
					//this is the instructions for our hand (our behaviour)
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.SENDING_EXECUTABLE_OBJECT_STATUS_MESSAGE")); //$NON-NLS-1$
					executableObject = setupHarnessAgentExecutableObject();
					
					//add our executor to our environment.
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.SENDING_EXECUTOR_STATUS_MESSAGE")); //$NON-NLS-1$
					exeEnvironment.addChild(executor);
		
					//initialize the recorder's data processer
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.INITIALIZING_DATA_PROCESSOR_STATUS_MESSAGE")); //$NON-NLS-1$
					theRecorder.getDataProcessor().initialize();
		
					//send our executor to the RAC	
					executor.setExecutableObject(executableObject);
		
					//add our agent to the executor (point our ears in the right direction
					executor.addChild(remoteHyadesComponent);
				
					//and launch it.
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.LAUNCHING_EXECUTOR_STATUS_MESSAGE")); //$NON-NLS-1$
					executor.launch();
					
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.EXECUTOR_LAUNCH_SUCCESSFUL_STATUS_MESSAGE")); //$NON-NLS-1$
																					
					//now let's tell the agent controller to load our agent and get to work.
					startRecording();
					updateViewStatus(RecorderPlugin.getResourceString("RecorderClient.STATUS_RUNNING"));
					new FinishCleanupAndGenTestThread().start();
		
				}
				catch(Exception e) 
				{					
					sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.START_RECORDING_ABORTED_DUE_TO_EXCEPTION_STATUS_MESSAGE")+e.getClass().getName()); //$NON-NLS-1$
					endProgressDialog();
					updateViewStatus(RecorderPlugin.getResourceString("RecorderClient.STATUS_STOPPED"));
					String cause = e.getMessage();
					if (cause == null)
						cause = e.getClass().getName();
					
					//handle particular errors & give more detailed error messages
					if (e instanceof AgentControllerUnavailableException)
					{
						String[] strs = new String[2];
						strs[0] = RecorderPlugin.getDefault().getRAServerHost();
						strs[1] = RecorderPlugin.getDefault().getRAServerPort();
						cause = RecorderPlugin.getDefault().getString("RecorderClient.RAC_UNAVAILABLE_MESSAGE", strs);
						cause += "\r\n";
						cause += RecorderPlugin.getResourceString("RecorderClient.RAC_UNAVAILABLE_MESSAGE_DESC1");
						sendStatusMessageToControlView(RecorderPlugin.getDefault().getString("RecorderClient.RAC_UNAVAILABLE_MESSAGE", strs)); //$NON-NLS-1$
					}
					
					RecorderPlugin.reportExceptionToUser(e,RecorderPlugin.getResourceString("RecorderClient.START_RECORDING_ABORTED"),cause,RecorderPlugin.getResourceString("RecorderClient.RECORDER_ERROR")); //$NON-NLS-1$ //$NON-NLS-2$
					
					
					//TODO: Logging
					if(session!=null)
						session.release();
					
					theRecorder.setActive(false);
					
				}
				 
			}
			
		}.start();
			
	}


	/**
	 * A method for sending a message to the Recorder control view.  Also updates progress dialog
	 * @param String message
	 */
	private void sendStatusMessageToControlView(final String message)
	{
		final RecorderControlView controlView = RecorderControlView.getInstance();
		controlView.addMessage(message);
		tickProgressDialog(message);
	}
	

	/**
	 * This private method is called by run() after all necessary objects are instantiated.  It sends all necessary control messages
	 * to start recording on the remote side.
	 */
	private void startRecording() throws InstantiationException, IllegalAccessException, ClassNotFoundException, RecorderApplicationAdapterException
	{
		String appAdapterAgentClasspath = null;
		String appAdapterInitString = "-1";
		
		/* jn - initializing application will now be done locally */
		String commandMessage = "Init "+theRecorder.getConfigParams(); //$NON-NLS-1$
		sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.SENDING_COMMAND_STATUS_MESSAGE")+commandMessage); //$NON-NLS-1$		
		sendMessage(commandMessage);
		
		
		commandMessage = "StartRecording"; //$NON-NLS-1$
		sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.SENDING_COMMAND_STATUS_MESSAGE")+commandMessage); //$NON-NLS-1$
		sendMessage(commandMessage);		
		
		String appAdapter = theRecorder.getApplicationAdapterID();
		if (!(appAdapter.equalsIgnoreCase("-1")))
		{			
			appAdapterAgentClasspath = RecorderAppAdapterFactory.getInstance().getAgentClasspath(theRecorder.getApplicationAdapterID());
			appAdapterInitString = RecorderAppAdapterFactory.getInstance().getClient(theRecorder.getApplicationAdapterID()).getInitString();
		}
		if(appAdapterAgentClasspath!=null && appAdapterAgentClasspath!="-1") //$NON-NLS-1$
		{			
			commandMessage = "StartApp "+appAdapterAgentClasspath+";"+appAdapterInitString; //$NON-NLS-1$ //$NON-NLS-2$
			sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.SENDING_COMMAND_STATUS_MESSAGE")+commandMessage); //$NON-NLS-1$
			//sendMessage(commandMessage);
			startApplication(appAdapterAgentClasspath, appAdapterInitString);
		}
		endProgressDialog();
	}
	
	/**
	 * Method used to stop recording from the client side. 
	 */
	public void stopRecording()
	{
		//sendMessage("StopApp"); //$NON-NLS-1$
		try 
		{
			stopApplication();			
		}
		catch(Exception e) 
		{					
			sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.STOP_RECORDING_ABORTED_DUE_TO_EXCEPTION_STATUS_MESSAGE")+e.getClass().getName()); //$NON-NLS-1$
			endProgressDialog();
			RecorderPlugin.reportExceptionToUser(e,RecorderPlugin.getResourceString("RecorderClient.STOP_RECORDING_ABORTED"),e.getClass().getName(),RecorderPlugin.getResourceString("RecorderClient.RECORDER_ERROR")); //$NON-NLS-1$ //$NON-NLS-2$
			if(session!=null)
				session.release();
					
			theRecorder.setActive(false);					
		}
		
		sendMessage("Stop"); //$NON-NLS-1$
				
	}
	


	/**
	 *sends a message across the data channel
	 */
	private void sendMessage(String message)
	{
		try {
			Thread.sleep(500);
		}
		catch (Exception e) {}
		//TODO: solve circular stop issue that requires this sleep. 
		
		if(controlAgent == null)
			initControlAgent();	
		
		CustomCommand command=new CustomCommand();
		command.setData(message);

		try
		{
			controlAgent.invokeCustomCommand(command);	
		}
		catch (InactiveAgentException e1)
		{				
			e1.printStackTrace();
		}
	}


	/**
	 *This method initiates the control agent reference that will be used to send control messages across the control channel to the remote
	 *side of the RAC.  To accomplish this initialization, it prompts the executor for all process that are currently active.  IT then
	 *must look at each until it finds a process whose ID matches our processes ID.  
	 *This should only be done once and the reference is reused after that.
	 */
	private void initControlAgent()
	{
		String pid=executor.getPid();
		//String pid = executor.getUniqueId().toString();
		int trialCount=100;
		
		try 
		{
			boolean found=false;
			Node node=((SessionStub)executor.getSessionContext()).getAgent().getProcess().getNode();
	
			while(!found && trialCount>0) 
			{
				Enumeration processes=node.listProcesses();
		
				while(processes.hasMoreElements()) 
				{
					Process proc=(Process)processes.nextElement();
					if(proc.getProcessId().equals(pid)) 
					{
						found=true;
						synchronized(proc) 
						{
							Enumeration agents=proc.getAgentsByType("tester"); //$NON-NLS-1$
							while(agents.hasMoreElements()) 
							{
								controlAgent = (Agent)agents.nextElement();
								try 
								{
									controlAgent.addAgentListener(new RecorderAgentListener());																				
									controlAgent.attach();
									break;								
								}
								catch(InactiveAgentException e) 
								{									
								}
							
							}
						}
					}						
					trialCount--;
				}
			}
		
		}
		catch(InactiveProcessException e) 
		{
			
		}
		catch(NotConnectedException e) {
			
		}
	}


	/**
	 * adds the appropriate Executable Object triple to the factory and then pulls an instantiation from the factory.
	 */
	private IExecutableObject setupHarnessAgentExecutableObject()throws ClassNotFoundException
	{
		factory.addExecutionComponent("JAVA_EXECUTABLE","org.eclipse.hyades.execution.core.impl.JavaProcessExecutableObjectImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("JAVA_EXECUTABLE", "org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("JAVA_EXECUTABLE", "org.eclipse.hyades.execution.remote.JavaProcessExecutableObjectSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
					
		IExecutableObject execObject = (IExecutableObject)executor.getCompatibleExecutableObject("JAVA_EXECUTABLE"); //$NON-NLS-1$
		
		theRecorder.getExecutableObjectAdapter().setupExecutableObject((JavaProcessExecutableObjectStub) execObject);
		
		return execObject;
	}

	

	/**
	 * adds the approprate remote execution component triple to the factory and pulls and instantiation from the factory.
	 */
	private IRemoteHyadesComponent setupAgent()	throws ClassNotFoundException
	{
		factory.addExecutionComponent("AGENT", "org.eclipse.hyades.execution.core.impl.JavaTaskRemoteHyadesComponentImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("AGENT", "org.eclipse.hyades.execution.local.JavaTaskRemoteHyadesComponentStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("AGENT", "org.eclipse.hyades.execution.remote.JavaTaskRemoteHyadesComponentSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
		return (IRemoteHyadesComponent)factory.createExecutionComponentByType("AGENT"); //$NON-NLS-1$
		
	}

	/**
	 * adds the appropriate executor triple to the factory and pulls and instantiation from the factory.
	 */
	private JavaProcessExecutorStub setupExecutor()	throws ClassNotFoundException
	{
		
		factory.addExecutionComponent("EXECUTOR", "org.eclipse.hyades.execution.core.impl.ProcessExecutorImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("EXECUTOR", "org.eclipse.hyades.execution.local.JavaProcessExecutorStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("EXECUTOR", "org.eclipse.hyades.execution.remote.JavaProcessExecutorSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
		
		/* create our executor */
		return (JavaProcessExecutorStub)factory.createExecutionComponentByType("EXECUTOR"); //$NON-NLS-1$
	}

	/**
	 * method which connects us to the RAC and returns our session
	 * @return ISession
	 */
	private ISession createSession() throws UnknownHostException, UnknownDaemonException, DaemonConnectException 
	{
		final String raserverHost = RecorderPlugin.getDefault().getRAServerHost();
		final String raserverPort = RecorderPlugin.getDefault().getRAServerPort();
		sendStatusMessageToControlView(RecorderPlugin.getResourceString("RecorderClient.CONNECTING_TO_RASERVER_STATUS_MESSAGE")+raserverHost+RecorderPlugin.getResourceString("RecorderClient.PORT_LABEL")+raserverPort); //$NON-NLS-1$ //$NON-NLS-2$
		
		NodeImpl node = new NodeImpl(raserverHost);
		
		ISession session = null;
		
		session=node.connect(raserverPort, null);	
		
		return session;
	}


	
	/**
	 * adds the appropriate environment triple to the factory and pulls and instantation from the factory.
	 * @return IExecutionEnvironment
	 */
	private IExecutionEnvironment setupJavaEnvironment() throws ClassNotFoundException, IOException
	{
		IExecutionEnvironment environment = null;
		factory.addExecutionComponent("ENVIRONMENT", "org.eclipse.hyades.execution.core.impl.ExecutionEnvironmentImpl"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addStub("ENVIRONMENT", "org.eclipse.hyades.execution.local.ExecutionEnvironmentStub"); //$NON-NLS-1$ //$NON-NLS-2$
		factory.addSkeleton("ENVIRONMENT", "org.eclipse.hyades.execution.remote.ExecutionEnvironmentSkeleton"); //$NON-NLS-1$ //$NON-NLS-2$
				
		environment =(IExecutionEnvironment)factory.createExecutionComponentByType("ENVIRONMENT"); //$NON-NLS-1$
				
		theRecorder.getAgentEnvironmentAdapter().setupExecutionEnvironment(environment);
				
		return environment;
		
	}

	private void initProgressDialog() throws Exception
	{
		final RecorderClient client = this;
		display.syncExec( new Runnable() {
		
		public void run() {
			progressDlg = new RecorderProgressDialog(client);
			progressDlg.setBlockOnOpen(false);
			progressDlg.open();
			progressDlg.setMaxProgress(RECORDER_MAX_TICKS);
		}
		});

	}

	private void tickProgressDialog(final String msg)
	{
		if (progressDlg != null)
		{
			display.asyncExec( new Runnable() {
				
			public void run() {
				progressDlg.setProgress(++ticks);  //on a 100 scale
				progressDlg.setMessage(msg);			
			}
			});
		}
	}
	
	private void endProgressDialog()
	{
		if (progressDlg != null)
		{
			display.syncExec( new Runnable() {
				
			public void run() {
				try {
					progressDlg.setProgress(RECORDER_MAX_TICKS);
					progressDlg.complete();
				}
				catch (SWTException e) //if dialog is already closed
				{ 
				}
			}
			});
		}
		
	}
	
	private void startApplication(String adapterClasspath, String adapterInitString) throws InstantiationException, IllegalAccessException, ClassNotFoundException, RecorderApplicationAdapterException 
	{
		appAdapter = (IRecorderApplicationAdapter)Class.forName(adapterClasspath).newInstance();

		appAdapter.init(adapterInitString);
		
		java.lang.Process process = appAdapter.start();

		
		recorderStopper = new RecorderStopper(process,this);
		recorderStopper.start();
	
		
	}
	
	private void stopApplication() throws RecorderApplicationAdapterException
	{
		if(appAdapter!=null )
		{
			//we want to handle this correctly, so we need to interrupt the process stopper.
			if (recorderStopper.isActive())
				recorderStopper.interrupt();			
			appAdapter.stop();
			appAdapter.cleanup();
		}
	}
	
	private void updateViewStatus(final String status)
	{
		if (display != null)
		display.asyncExec( new Runnable() {
			public void run() {
				RecorderControlView.getInstance().setStatus(status);
			}
		});
	}
	/**
	 * @return Returns the progressDlg.
	 */
	public RecorderProgressDialog getProgressDlg() {
		return progressDlg;
	}
	/**
	 * @param progressDlg The progressDlg to set.
	 */
	public void setProgressDlg(RecorderProgressDialog progressDlg) {
		this.progressDlg = progressDlg;
	}
}
