/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.view;


import java.text.MessageFormat;
import org.eclipse.hyades.uml2sd.ui.SDViewerPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.AccessibleControlAdapter;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.AccessibleControlEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;



/**
 * This class is used to reproduce the same tooltip behavior on Windows and Linux
 * when the mouse move hover the time compression bar
 * Used to display enlapsed time using a tooltip.
 * The tooltip is composed of 2 parts, the text value and below, a scale to visually
 * locate the value in a time range (usually the min an max enlapsed time in the whole diagram)
 * @author sveyrier
 */
public class DrawableToolTip implements PaintListener {

	/**
	 * The parent control where the tooltip must be drawn
	 */
	private Composite parent = null;
	/**
	 * The tooltip shell
	 */
	private Shell toolTipShell = null;
	/**
	 * Time range datas
	 */
	private double min,max,currentValue;
	
	private static int H_MARGIN = 10;
	private static int V_MARGIN = 10;
	private static int TEXT_SCALE_MARGIN = 20;
	private static int SCALE_LENGTH = 100;
	/**
	 * Unit name
	 */
	private String unitName = null;
	
	/**
	 * The color array used to repesent the 10 time range slices
	 */
	private Color[] col;

	public DrawableToolTip(Composite _parent)
	{
		parent = _parent;
		toolTipShell = new Shell(parent.getShell(),SWT.ON_TOP);
		toolTipShell.setLayout(new RowLayout());
		toolTipShell.setBackground(Display.getDefault().getSystemColor(SWT.COLOR_INFO_BACKGROUND));
		toolTipShell.addPaintListener(this);
		toolTipShell.setSize(200,50);
		toolTipShell.getAccessible().addAccessibleControlListener( new AccessibleControlAdapter()
				{
					public void getRole(AccessibleControlEvent e)
					{
						e.detail=ACC.ROLE_TOOLTIP;
					}
					public void getValue(AccessibleControlEvent e) 
					{
						if ((unitName==null)||(unitName.equals(""))) //$NON-NLS-1$
						{	
							Object [] args = {new Double(currentValue)};
							String msg=MessageFormat.format(SDViewerPlugin.getResourceString("POPUP_FORMAT_NO_UNIT"),args);	 //$NON-NLS-1$
							e.result=msg;
						}
						else
						{
							Object [] args = {new Double(currentValue), unitName};
							String msg=MessageFormat.format(SDViewerPlugin.getResourceString("POPUP_FORMAT"),args);	//$NON-NLS-1$
							e.result=msg;
						}
					}
					public void getState(AccessibleControlEvent e)
					{
						e.detail=ACC.STATE_FOCUSED;
					}
				}
				);
		
		col = new Color[10];
		col[0] = new Color(Display.getDefault(),255,229,229);
		col[1] = new Color(Display.getDefault(),255,204,204);
		col[2] = new Color(Display.getDefault(),255,178,178);
		col[3] = new Color(Display.getDefault(),255,153,153);
		col[4] = new Color(Display.getDefault(),255,127,127);
		col[5] = new Color(Display.getDefault(),255,102,102);
		col[6] = new Color(Display.getDefault(),255,76,76);
		col[7] = new Color(Display.getDefault(),255,51,51);
		col[8] = new Color(Display.getDefault(),255,25,25);
		col[9] = new Color(Display.getDefault(),255,0,0);
	}
	
	/**
	 * Display the tooltip using the given time range(min,max) the current value and the time unit
	 * The tooltip will stay on screen until it is told otherwise
	 * 
	 * @param _value the current in the scale
	 * @param _min the scale min
	 * @param _max the scale max
	 * @param unit the scale unit
	 */
	public void showToolTip(double _value, double _min, double _max, String unit)
	{
		min = _min;
		max = _max;
		currentValue = _value;
		unitName =unit;
		
		int w = toolTipShell.getBounds().width;
		int h = toolTipShell.getBounds().height;
		Point hr = Display.getDefault().getCursorLocation();
		toolTipShell.setBounds(hr.x,hr.y+26,w,h);
		toolTipShell.setVisible(true);
		toolTipShell.getAccessible().setFocus(parent.handle);
	}
	
	/**
	 * Hide the tooltip
	 */
	public void hideToolTip()
	{
		toolTipShell.setVisible(false);
	}
	
	/**
	 * Draw the tooltip text on the control widget when a paint event is received
	 */
	public void paintControl(PaintEvent event)
	{
		Point size = event.gc.textExtent(Double.toString(currentValue));		
		if (size.x<SCALE_LENGTH)
			size.x = SCALE_LENGTH;
		if ((unitName==null)||(unitName.equals(""))) //$NON-NLS-1$
		{	
			Object [] args = {new Double(currentValue)};
			String msg=MessageFormat.format(SDViewerPlugin.getResourceString("POPUP_FORMAT_NO_UNIT"),args);	 //$NON-NLS-1$
			event.gc.drawText(msg,H_MARGIN,V_MARGIN,true); 
		}
		else
		{
			Object [] args = {new Double(currentValue), unitName};
			String msg=MessageFormat.format(SDViewerPlugin.getResourceString("POPUP_FORMAT"),args);	//$NON-NLS-1$
			event.gc.drawText(msg,H_MARGIN,V_MARGIN,true);
		}
		event.gc.drawLine(H_MARGIN,V_MARGIN+TEXT_SCALE_MARGIN+size.y,H_MARGIN+SCALE_LENGTH,
					V_MARGIN+TEXT_SCALE_MARGIN+size.y);
					
		int step = SCALE_LENGTH/10;
		double gr = (max - min)/10;
		double delta = Math.abs(currentValue + Math.abs(min));
		int colIndex = 0;
		if (delta <= Math.abs(max + Math.abs(min)))
		{
			colIndex = Math.round((float)(delta/gr));
			for (int i=0;i<=10;i++)
			{
				if (i<10)
					event.gc.setBackground(col[i]);
				if ((i<colIndex)&&(i<10))
					event.gc.fillRectangle(H_MARGIN+i*step,V_MARGIN+TEXT_SCALE_MARGIN+size.y-5,step,11);
				if (i==0)
					event.gc.drawText(SDViewerPlugin.getResourceString("MIN_LABEL"), //$NON-NLS-1$
							H_MARGIN,size.y+2*V_MARGIN+TEXT_SCALE_MARGIN,true);
				if (i==0)
				{
					int len  = event.gc.textExtent(SDViewerPlugin.getResourceString("MAX_LABEL")).x; //$NON-NLS-1$
					event.gc.drawText(SDViewerPlugin.getResourceString("MAX_LABEL"), //$NON-NLS-1$
						H_MARGIN+SCALE_LENGTH-len+1,size.y+2*V_MARGIN+TEXT_SCALE_MARGIN,true);
				}
				int lineWidth = 10;
				if ((i==0)||(i==10))
					lineWidth = 14;
				event.gc.drawLine(H_MARGIN+i*step,V_MARGIN+TEXT_SCALE_MARGIN+size.y-lineWidth/2,
								H_MARGIN+i*step,V_MARGIN+TEXT_SCALE_MARGIN+size.y+lineWidth/2);
			}
		}
		toolTipShell.setSize(size.x+2*H_MARGIN+2,2*size.y+3*V_MARGIN+TEXT_SCALE_MARGIN);
	}
	
	/**
	 * Dispose the system resource used by this kind of toolTips
	 * (a colors array essentialy)
	 *
	 */
	public void dispose()
	{
		for (int i=0; i<col.length;i++)
			col[i].dispose();	
	}
}
