package org.eclipse.hyades.logging.events.cbe;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Event factory home for retrieving event factories.
 * <p>
 * <code>EventFactoryHome</code> instances provide <code>EventFactory</code> resolution,
 * configuration updates based on a specific type of configuration template 
 * for the <code>EventFactory</code> instance and releases <code>EventFactory</code>
 * instances.
 * <p>
 * This abstract class also provides permission-based security for all event factory home 
 * operations.  Permission-based security is achieved by restricting access
 * to event factory home operations to callers whom have been explicitly granted
 * <code>EventPermission("factory")</code> permission.
 * <p>
 *
 * @author Paul E Slauenwhite
 * @version 1.0.1
 * @since 1.0.1 
 * @see org.eclipse.hyades.logging.events.cbe.EventFactory
 * @see org.eclipse.hyades.logging.events.cbe.util.EventPermission
 */
public interface EventFactoryHome {

    /**
     * Retrieves an instance of the named <code>EventFactory</code>.
     * <p>
     * An instance of a named <code>EventFactory</code> is created if
     * no named instance current exists or a named instance
     * is released. 
     * <p>
     * Once a named <code>EventFactory</code> is created, the instance
     * is cached for future calls to retrieve the same 
     * named <code>EventFactory</code>.
     * <p>
     * The name of the <code>EventFactory</code> uniquely identifies an
     * instance of an <code>EventFactory</code>.  All subsequent calls will
     * return the same instance of the named <code>EventFactory</code>.  
     * <p>
     * <code>EventFactory</code> names are hierarchal represented using the 
     * standard Java dot-delimited name-space naming conventions.  
     * <p>
     *  
     * @param factoryName The name of the <code>EventFactory</code>.
     * @return The retrieved or newly created <code>EventFactory</code>.
     * @throws SecurityException If the <code>SecurityManager</code> exists and if the caller does not have permission (e.g. <code>EventPermission("factory")</code>).
     */
    public EventFactory getEventFactory(String factoryName) throws SecurityException;

    /**
     * Updates the configuration on the instance of the named <code>EventFactory</code>.
     * <p>
     * If no named <code>EventFactory</code> instance exists, no update occurs. 
     * <p>
     * The name of the <code>EventFactory</code> uniquely identifies an
     * instance of an <code>EventFactory</code>.  All calls will
     * update the configuration on the same named instance 
     * of the <code>EventFactory</code>.
     * <p>
     * <code>EventFactory</code> names are hierarchal represented using the 
     * standard Java dot-delimited name-space naming conventions.  
     * <p>
     * 
     * @param factoryName The name of the <code>EventFactory</code>.
     * @throws SecurityException If the <code>SecurityManager</code> exists and if the caller does not have permission (e.g. <code>EventPermission("factory")</code>).
     */
    public void updateEventFactory(String factoryName) throws SecurityException;

    /**
     * Releases the reference to the instance of the named <code>EventFactory</code>.
     * <p>
     * If no named <code>EventFactory</code> instance exists, no release occurs. 
     * <p>
     * The name of the <code>EventFactory</code> uniquely identifies an
     * instance of an <code>EventFactory</code>.  
     * <p>
     * <code>EventFactory</code> names are hierarchal represented using the 
     * standard Java dot-delimited name-space naming conventions.  
     * <p>
     * 
     * @param factoryName The name of the <code>EventFactory</code>.
     * @throws SecurityException If the <code>SecurityManager</code> exists and if the caller does not have permission (e.g. <code>EventPermission("factory")</code>).
     */
    public void releaseEventFactory(String factoryName) throws SecurityException;

    /**
     * Checks the system's <code>SecurityManager</code> for the context of the currently 
     * executing thread to see if the caller has permission to create, retrieve, update 
     * and release Event Factory instances from the Event Factory Home.  The caller has 
     * permission if the <code>EventPermission("factory")</code> permission has been 
     * explicitly granted to the caller.
     * <p>  
     * If the caller has been granted permission the method quietly returns.  
     * Otherwise, the method throws a <code>SecurityException</code>.  
     * <p>
     *
     * @throws SecurityException If the <code>SecurityManager</code> exists and if the caller does not have permission (e.g. <code>EventPermission("factory")</code>).
     */
    public void checkAccess() throws SecurityException;
}
