package org.eclipse.hyades.logging.events.cbe.impl;

import java.security.Permission;
import java.util.Hashtable;

import org.eclipse.hyades.logging.events.cbe.ContentHandler;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventFactoryHome;
import org.eclipse.hyades.logging.events.cbe.util.EventPermission;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * An abstract event factory home implementation for working with event factories.
 * <p>
 * This abstract class provides <code>EventFactory</code> resolution, persistence of 
 * retrieved <code>EventFactory</code> instances, possible configuration updates based 
 * on a specific type of configuration template for the <code>EventFactory</code> 
 * instance and releasing <code>EventFactory</code> instances.
 * <p>
 * This abstract class also provides permission-based security for all event factory home 
 * operations.  Permission-based security is achieved by restricting access
 * to event factory home operations to callers whom have been explicitly granted
 * <code>EventPermission("factory")</code> permission.
 * <p>
 * Concrete implementations of this abstract class implement the 
 * <code>createContentHandler(String factoryName)</code> and 
 * <code>resolveContentHandler()</code> APIs to typically provide a <code>ContentHandler</code>
 * implementation tightly coupled to the type of the concrete implementation of 
 * this abstract class.  Concrete implementations that do not require a 
 * <code>ContentHandler</code> nor configuration template, simply return <code>null</code>
 * for each API.
 * <p>
 * For example, a concrete implementation of this abstract class may provide the 
 * an implementation of the <code>createContentHandler(String factoryName)</code>
 * API that creates a specialized <code>ContentHandler</code> pre-configured with 
 * an event template resolved from a local event property file.  The concrete 
 * implementation of this abstract class would also provide an implementation 
 * of the <code>resolveContentHandler()</code> API that retrieves a new instance 
 * of the specialized <code>ContentHandler</code> used in the implementation of 
 * the <code>createContentHandler(String factoryName)</code> API. 
 * <p>
 * It is assumed that implementations of the <code>createContentHandler(String factoryName)</code>
 * API would reference the implementation of the <code>resolveContentHandler()</code> API in the
 * same concrete implementation of this abstract class when creating the returned <code>ContentHandler</code>.
 * <p>
 *
 * @author Paul E Slauenwhite
 * @version 1.0.1
 * @since 1.0.1 
 * @see org.eclipse.hyades.logging.events.cbe.EventFactoryHome
 * @see org.eclipse.hyades.logging.events.cbe.util.EventPermission
 */
public abstract class AbstractEventFactoryHome implements EventFactoryHome {

    private Hashtable eventFactories = new Hashtable();

    private Permission eventPermission = new EventPermission("factory");

    /**
     * Creates an instance of the <code>ContentHandler</code> implementation which 
     * is tightly coupled to the type of the concrete implementation of 
     * this abstract class.
     * <p> 
     * The returned instance of the <code>ContentHandler</code> implementation is 
     * pre-configured with a event properties resolved from a source tightly coupled 
     * to the type of the concrete implementation of this abstract class.
     * <p>
     * It is assumed that implementations of this method would reference the implementation of the 
     * <code>resolveContentHandler()</code> method in the same concrete implementation of this 
     * abstract class when creating the returned <code>ContentHandler</code>.
     * <p>
     * 
     * @param factoryName The name of the event factory associated with the returned <code>ContentHandler</code> implementation.
     * @return An instance of the <code>ContentHandler</code> pre-configured with a event properties.
     */
    public abstract ContentHandler createContentHandler(String factoryName);

    /**
     * Resolves an instance of the <code>ContentHandler</code> implementation which 
     * is tightly coupled to the type of the concrete implementation of 
     * this abstract class.
     * <p>
     * It is assumed that implementations of the <code>createContentHandler(String factoryName)</code> 
     * method would reference the implementation of this method in the same concrete implementation of this 
     * abstract class when creating the returned <code>ContentHandler</code>.
     * <p>
     * 
     * @return An instance of the <code>ContentHandler</code> tightly coupled with this type of this event factory home implementation.
     */
    public abstract ContentHandler resolveContentHandler();

    /**
     * @see org.eclipse.hyades.logging.events.cbe.EventFactoryHome#checkAccess()
     */
    public void checkAccess() throws SecurityException {

        SecurityManager securityManager = System.getSecurityManager();

        if (securityManager != null) {
            securityManager.checkPermission(eventPermission);
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.cbe.EventFactoryHome#getEventFactory(java.lang.String)
     */
    public EventFactory getEventFactory(String factoryName) throws SecurityException {

        checkAccess();

        if (eventFactories.containsKey(factoryName)) {
            return (((EventFactory) (eventFactories.get(factoryName))));
        }

        EventFactory eventFactory = new EventFactoryImpl();
        
        ContentHandler contentHandler = createContentHandler(factoryName);
        
        if(contentHandler != null){
            eventFactory.setContentHandler(contentHandler);
        }
        
        eventFactories.put(factoryName, eventFactory);

        return eventFactory;
    }

    /**
     * @see org.eclipse.hyades.logging.events.cbe.EventFactoryHome#updateEventFactory(java.lang.String)
     */
    public void updateEventFactory(String factoryName) throws SecurityException {

        checkAccess();

        if (eventFactories.containsKey(factoryName)) {
            ((EventFactory) (eventFactories.get(factoryName))).setContentHandler(createContentHandler(factoryName));
        }
    }

    /**
     * @see org.eclipse.hyades.logging.events.cbe.EventFactoryHome#releaseEventFactory(java.lang.String)
     */
    public void releaseEventFactory(String factoryName) throws SecurityException {

        checkAccess();

        if (eventFactories.containsKey(factoryName)) {
            eventFactories.remove(factoryName);
        }
    }
}