package org.eclipse.hyades.logging.events.cbe.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * The <code>SAXEventHandler</code> class defines a utility object used for 
 * de-serializing <code>CommonBaseEvent</code> XML documents and fragments 
 * using a Simple API for XML (SAX) parser.  
 * <p>
 * 
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 * @see org.xml.sax.helpers.DefaultHandler
 */
public final class SAXEventHandler extends DefaultHandler {

    /**
     * Situation type needs special handling during parsing. 
     */
    private final static String SITUATION_TYPE_CLASS = EventPackage.eINSTANCE.getSituationType().getName();

    /** 
     * CommonBaseEvent needs special handling during parsing. 
     */
    private final static String COMMON_BASE_EVENT_CLASS = EventPackage.eINSTANCE.getCommonBaseEvent().getName();

    /** 
     * AssociationEngine needs special handling during parsing. 
     */
    private final static String ASSOCIATION_ENGINE_CLASS = EventPackage.eINSTANCE.getAssociationEngine().getName();

    /**
     * Map of element names and respective structural feature.
     * <p>
     * The key is a <code>String</code> object representing the element name
     * and the value is a <code>EClass</code> object required to create the
     * Java instance through the <code>EventFactory</code> object.
     */
    private HashMap elementMap = new HashMap();

    /**
     * Map of xml element names and its EMF structural features.
     * <p>
     * The key is a <code>String</code> object representing the element name
     * and the value is another <code>HashMap</code> representing the mapping
     * between the feature name and the feature itself.
     * <p>
     * For this map, the key is a <code>String</code> object representing the
     * feature name and the value is a <code>EStructuralFeature</code> object
     * required to create the Java instance through the
     * <code>EventFactory</code> object.
     */
    private HashMap featureMap = new HashMap();

    /** 
     * Current document position. 
     */
    private int column = 0;

    private int line = 0;

    /**
     * Indicates the indentation level of elements when the parser is going
     * through an "any" element.
     */
    private int parsingAny = -1;

    private boolean nextElementIsAnAny = false;

    /**
     * Indicates the first character for the element being parsed.
     * <P>
     * Required to backtrack the parsing of "any" elements.
     */
    private int anyStartCharacter = -1;

    private int startCharacterCopy = -1;

    private int lengthCharacterCopy = -1;

    /** Final product for this parser */
    private ArrayList events = null;

    private ArrayList associationEngines = null;

    /** Stack of objects already parsed through the XML elements. */
    private List stack = null;

    /**
     *  No argument constructor.
     */
    public SAXEventHandler() {
        init();
    }

    public void init(){
    
        reset();
        associationEngines = new ArrayList();
        events = new ArrayList();
    }
    
    private void reset(){
                  
        column = 0;

        line = 0;

        parsingAny = -1;

        nextElementIsAnAny = false;

        anyStartCharacter = -1;

        startCharacterCopy = -1;

        lengthCharacterCopy = -1;

        stack = new ArrayList();        
        
        List eventPckContents = EventPackage.eINSTANCE.eContents();
        Iterator i = eventPckContents.iterator();

        while (i.hasNext()) {
            EClass c = (EClass) i.next();
            elementMap.put(c.getName(), c);

            List features = c.getEAllStructuralFeatures();

            for (int j = 0; j < features.size(); j++) {
                EStructuralFeature r = (EStructuralFeature) features.get(j);

                HashMap featureSubMap = (HashMap) featureMap.get(c.getName());

                if (featureSubMap == null) {
                    featureSubMap = new HashMap();
                    featureMap.put(c.getName(), featureSubMap);
                }

                featureSubMap.put(r.getName(), r);
            }// for features
        }// while package items
    }
    
    /**
     * Returns the list of <code>CommonBaseEvent</code> objects created when 
     * de-serializing a Common Base Event XML document 
     * or fragment.
     * <p>
     *  
     * @return List of <code>CommonBaseEvent</code> objects created when de-serializing a Common Base Event XML document or fragment.
     */
    public CommonBaseEvent[] getCommonBaseEvents() {
        
        if((events != null) && (events.size() > 0)){
            return (((CommonBaseEvent[])(events.toArray(new CommonBaseEvent[events.size()]))));
        }
        
        return null;
    }

    /**
     * Returns the list of <code>AssociationEngine</code> objects created when 
     * de-serializing a Common Base Event XML document 
     * or fragment.
     * <p>
     *  
     * @return List of <code>AssociationEngine</code> objects created when de-serializing a Common Base Event XML document or fragment.
     */
    public AssociationEngine[] getAssociationEngines() {
        
        if((associationEngines != null) && (associationEngines.size() > 0)){
            return (((AssociationEngine[])(associationEngines.toArray(new AssociationEngine[associationEngines.size()]))));
        }
        
        return null;
    }

    /**
     * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
     */
    public void setDocumentLocator(Locator loc) {
        column = loc.getColumnNumber();
        line = loc.getLineNumber();
        super.setDocumentLocator(loc);
    }

    /**
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char ch[], int start, int length) throws SAXException {
        startCharacterCopy = start;
        lengthCharacterCopy = length;

        if (nextElementIsAnAny) {
            nextElementIsAnAny = false;
            CommonBaseEvent cbe = (CommonBaseEvent) stack.get(stack.size() - 1);
            String st = new String(ch, anyStartCharacter, start - anyStartCharacter);
            cbe.addAny(st.trim());
        }

        // Checking whether these are the values for an extended
        // data element list.
        if (stack.size() > 2) {
            Object top = stack.get(stack.size() - 1);

            if (top instanceof List) {
                String value = new String(ch, start, length);
                List l = (List) top;
                l.add(value);
            }
            if (top instanceof EStructuralFeature) {
                String value = new String(ch, start, length);
                EObject container = (EObject) stack.get(stack.size() - 2);
                EStructuralFeature feature = (EStructuralFeature) stack.get(stack.size() - 1);
                container.eSet(feature, value);
            }
        }

        super.characters(ch, start, length);
    }

    /**
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String qName) throws SAXException {

        super.endElement(uri, localName, qName);

	    if (parsingAny >= 0) {
	        parsingAny--;
	        if (parsingAny < 0) {
	            nextElementIsAnAny = true;
	        }
	    } 
	    else {
	        String elementName = localName;
            
            if((elementName == null) || (elementName.trim().length() == 0)){
                
                int lastColonIndex = -1;
                
                if((qName != null) && ((lastColonIndex = qName.lastIndexOf(':')) != -1)){
                    elementName = qName.substring(lastColonIndex + 1);                    
                }
                else{
                    elementName = qName;
                }
            }
            
	        if((!elementName.equals(ASSOCIATION_ENGINE_CLASS)) && (!elementName.equalsIgnoreCase("CommonBaseEvents")) && (!elementName.equalsIgnoreCase("TemplateEvent"))){
	    	         
		        Object o = stack.remove(stack.size() - 1);
		
		        if (o instanceof CommonBaseEvent) {
		                
		            events.add((CommonBaseEvent) o);
		
		            reset();
		        }
	        }
	    }
    }

    /**
     * @see org.xml.sax.ErrorHandler#error(org.xml.sax.SAXParseException)
     */
    public void error(SAXParseException arg0) throws SAXException {
        super.error(arg0);
    }

    /**
     * @see org.xml.sax.ErrorHandler#fatalError(org.xml.sax.SAXParseException)
     */
    public void fatalError(SAXParseException arg0) throws SAXException {
        super.fatalError(arg0);
    }

    /**
     * @see org.xml.sax.ContentHandler#ignorableWhitespace(char[], int, int)
     */
    public void ignorableWhitespace(char arg0[], int arg1, int arg2) throws SAXException {
        super.ignorableWhitespace(arg0, arg1, arg2);
    }

    /**
     * @see org.xml.sax.DTDHandler#notationDecl(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void notationDecl(String arg0, String arg1, String arg2) throws SAXException {
        super.notationDecl(arg0, arg1, arg2);
    }

    /**
     * @see org.xml.sax.ContentHandler#processingInstruction(java.lang.String,
     *      java.lang.String)
     */
    public void processingInstruction(String arg0, String arg1) throws SAXException {
        super.processingInstruction(arg0, arg1);
    }

    /**
     * @see org.xml.sax.EntityResolver#resolveEntity(java.lang.String,
     *      java.lang.String)
     */
    public InputSource resolveEntity(String publicId, String systemId) throws IOException, SAXException {       
        return super.resolveEntity(publicId, systemId);
    }

    /**
     * @see org.xml.sax.ContentHandler#skippedEntity(java.lang.String)
     */
    public void skippedEntity(String name) throws SAXException {        
        super.skippedEntity(name);
    }

    /**
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String,
     *      java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {

        super.startElement(uri, localName, qName, attributes);

        String elementName = localName;
        
        if((elementName == null) || (elementName.trim().length() == 0)){
            
            int lastColonIndex = -1;
            
            if((qName != null) && ((lastColonIndex = qName.lastIndexOf(':')) != -1)){
                elementName = qName.substring(lastColonIndex + 1);                    
            }
            else{
                elementName = qName;
            }
        }
        
        if(!elementName.equalsIgnoreCase("CommonBaseEvents") && (!elementName.equalsIgnoreCase("TemplateEvent"))){

	        // If we are inside an "any" block, let's just ignore
	        // everything in it, "endElement" will catch everything.
	        if (parsingAny >= 0) {
	            parsingAny++;
	        }
	
	        // Find class name and instantiate object for this XML element
	        EObject emfObj = null;
	        EClass emfObjClass = null;
	
	        if (stack.isEmpty()) {

	            //If the stack is empty, this better be a common base event
	            if(elementName.equals(ASSOCIATION_ENGINE_CLASS)){
	                
	                AssociationEngine associationEngine = EventFactory.eINSTANCE.createAssociationEngine();
	                
	                //Set the engine's id property with a new GUID:
	                associationEngine.setId(Guid.generate());

	                String attributeName = null;
	                
	                for (int counter = 0; counter < attributes.getLength(); counter++) {
                        
		                attributeName = attributes.getLocalName(counter);
		                
		                if((attributeName == null) || (attributeName.trim().length() == 0)){
		                    
		                    int lastColonIndex = -1;
		                    String attrbuteQName = attributes.getQName(counter);
		                    
		                    if((attrbuteQName != null) && ((lastColonIndex = attrbuteQName.lastIndexOf(':')) != -1)){
		                        attributeName = attrbuteQName.substring(lastColonIndex + 1);                    
		                    }
		                    else{
		                        attributeName = attrbuteQName;
		                    }
		                }
		                
		                if(attributeName.equals("id")){
			                associationEngine.setId(attributes.getValue(counter));		                    
		                }
		                else if(attributeName.equals("name")){
			                associationEngine.setName(attributes.getValue(counter));		                    
		                }
		                else if(attributeName.equals("type")){
			                associationEngine.setType(attributes.getValue(counter));		                    
		                }	                   
                    }
	                
	                associationEngines.add(associationEngine);
	                
	                return;
	            }
	            else if (!elementName.equals(COMMON_BASE_EVENT_CLASS)) {
	                throw new SAXException("Expected a '" + COMMON_BASE_EVENT_CLASS + "' element, but found '" + elementName + "' at line " + line + ", column " + column);
	            }
	            	            
	            emfObjClass = (EClass) elementMap.get(elementName);
	            emfObj = EventFactory.eINSTANCE.create(emfObjClass);
	            stack.add(emfObj);
	        } 
	        else {
	            EObject container = (EObject) stack.get(stack.size() - 1);
	
	            HashMap featureSubMap = (HashMap) featureMap.get(container.eClass().getName());
	            EStructuralFeature objStructuralFeature = (EStructuralFeature) featureSubMap.get(elementName);
	
	            // The main reason for not finding something in the feature map
	            // is the "any" element.
	            if (objStructuralFeature == null) {
	                if (container.eClass().getName().equals(COMMON_BASE_EVENT_CLASS)) {
	                    parsingAny = 0;
	                    anyStartCharacter = startCharacterCopy + lengthCharacterCopy;
	                } 
	                else {
	                    throw new SAXException("Unexpected element: '" + elementName + "' at line " + line + ", column " + column);
	                }
	            } 
	            else {
	                // Special case for the abstract "SituationType" class in
	                // the CBE specification
	                String className = objStructuralFeature.getEType().getName();
	
	                if (className.equals(SITUATION_TYPE_CLASS)) {
	                    Situation s = (Situation) container;
	                    className = s.getCategoryName();
	                }
	
	                // Create instance
	                if (className.equals("EString")) {
	                    if (objStructuralFeature.isMany()) {
	                        List l = (List) container.eGet(objStructuralFeature);
	                        stack.add(l);
	                    } 
	                    else {
	                        // This trick will allow the 'characters'
	                        // method to find out it has to set this
	                        // structural feature with the next string
	                        // it reads
	                        stack.add(objStructuralFeature);
	                    }
	                } 
	                else {
	                    
	                    emfObjClass = (EClass) elementMap.get(className);
	                    emfObj = EventFactory.eINSTANCE.create(emfObjClass);
	
	                    // Setting the object inside the container object
	                    if (objStructuralFeature.isMany()) {
	                        List l = (List) container.eGet(objStructuralFeature);
	                        l.add(emfObj);
	                    } 
	                    else {
	                        container.eSet(objStructuralFeature, emfObj);
	                    }
	
	                    stack.add(emfObj);
	                } // if class name is String
	            } // if any
	        }
	
	        // if stack is not empty
	        // Fill instance attributes
	        if (emfObj != null) {
	            int attrSize = attributes.getLength();
	
	            for (int i = 0; i < attrSize; i++) {
	                // Find structural feature
	                String attributeName = attributes.getLocalName(i);
	                
	                if((attributeName == null) || (attributeName.trim().length() == 0)){
	                    
	                    int lastColonIndex = -1;
	                    String attrbuteQName = attributes.getQName(i);
	                    
	                    if((attrbuteQName != null) && ((lastColonIndex = attrbuteQName.lastIndexOf(':')) != -1)){
	                        attributeName = attrbuteQName.substring(lastColonIndex + 1);                    
	                    }
	                    else{
	                        attributeName = attrbuteQName;
	                    }
	                }
	                
	                HashMap featureSubMap = (HashMap) featureMap.get(emfObjClass.getName());
	                EAttribute f = (EAttribute) featureSubMap.get(attributeName);
	                // The attribute could not be in the featureSubMap because
	                // it is an xmlns declaration or some other xml directive.
	                // This is not the place to do true CBE XML schema
	                // validation.
	                if (f != null) {
	                    String attributeValue = attributes.getValue(i);
	                    String attributeTypeName = f.getEAttributeType().getName();
	                    if (attributeTypeName.equals("ELong")) {
	                        Long longAttributeValue = Long.valueOf(attributeValue);
	                        emfObj.eSet(f, longAttributeValue);
	                    } 
	                    else if (attributeTypeName.equals("EShort")) {
	                        Short shortAttributeValue = Short.valueOf(attributeValue);
	                        emfObj.eSet(f, shortAttributeValue);
	                    } 
	                    else {
	                        emfObj.eSet(f, attributeValue);
	                    }
	                }
	            }// for attr
	        }// if emfObj
        }
    }

    /**
     * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String,
     *      java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri) throws SAXException {
        super.startPrefixMapping(prefix, uri);
    }

    /**
     * @see org.xml.sax.DTDHandler#unparsedEntityDecl(java.lang.String,
     *      java.lang.String, java.lang.String, java.lang.String)
     */
    public void unparsedEntityDecl(String arg0, String arg1, String arg2, String arg3) throws SAXException {
        super.unparsedEntityDecl(arg0, arg1, arg2, arg3);
    }

    /**
     * @see org.xml.sax.ErrorHandler#warning(org.xml.sax.SAXParseException)
     */
    public void warning(SAXParseException exc) throws SAXException {
        super.warning(exc);
    }
}

