package org.eclipse.hyades.logging.events.cbe.util.tests;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.FormattingException;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.tests.Util;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Tests the EventFormatter class.
 * <p>
 *
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1 
 * @see org.eclipse.hyades.logging.events.cbe.util.EventFormatter
 */
public class EventFormatterTest extends TestCase {

    /**
     * Constructor for EventFormatterTest.
     * @param arg0
     */
    public EventFormatterTest(String testcase) {
        super(testcase);
    }

    public void testInvalidXMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"> <CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsed Time=\"0\" extensionName=\"Name\" globalInstanceId=\"i0000000000000000000000000000040\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </context DataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>       <children name=\"\" type=\"noValue\">          <values>values</values>       </children>      </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents> <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/><sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\"><situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation>	    <a:a xmlns:a=\"http://bar\" />    </CommonBaseEvent></CommonBaseEvents>";

        try {            
            EventFormatter.eventsFromCanonicalXMLDoc(eventXML);
            assertTrue(false);
        } 
        catch (FormattingException f) {    
        }        
    }

    public void testInvalidXMLFragmentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"> <CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsed Time=\"0\" extensionName=\"Name\" globalInstanceId=\"i0000000000000000000000000000040\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </context DataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>       <children name=\"\" type=\"noValue\">          <values>values</values>       </children>      </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents> <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/><sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\"><situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation>	    <a:a xmlns:a=\"http://bar\" />    </CommonBaseEvent></CommonBaseEvents>";

        try {            
            EventFormatter.eventFromCanonicalXML(eventXML);
            assertTrue(false);
        } 
        catch (FormattingException f) {    
        }        
    }

    public void testAlternateNameSpacesXMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"ASCII\"?><cbe:CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><cbe:AssociationEngine cbe:id=\"i0000000000000000000000000000044\" cbe:name=\"MyCorrelationEngine\" cbe:type=\"Correlate\"/><cbe:CommonBaseEvent cbe:creationTime=\"2001-12-31T12:00:00\" cbe:extensionName=\"Name\" cbe:globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <cbe:contextDataElements name=\"myContext\" type=\"String\">    <cbe:contextValue>contextValue</cbe:contextValue>  </cbe:contextDataElements>  <cbe:extendedDataElements name=\"ede1\" type=\"noValue\">    <cbe:values>values</cbe:values>    <cbe:children name=\"chname1\" type=\"noValue\">      <cbe:values>values</cbe:values>    </cbe:children>  </cbe:extendedDataElements>  <cbe:extendedDataElements name=\"ede2\" type=\"noValue\">    <cbe:values>values</cbe:values>    <cbe:children name=\"chname2\" type=\"noValue\">      <cbe:values>values</cbe:values>    </cbe:children>  </cbe:extendedDataElements>  <cbe:reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <cbe:sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <cbe:msgDataElement msgLocale=\"EN\">    <cbe:msgCatalogTokens value=\"value\"/>    <cbe:msgId>msgId</cbe:msgId>    <cbe:msgIdType>Name</cbe:msgIdType>    <cbe:msgCatalogId>msgCatalogId</cbe:msgCatalogId>    <cbe:msgCatalogType>msgCatalogType</cbe:msgCatalogType>    <cbe:msgCatalog>msgCatalog</cbe:msgCatalog>  </cbe:msgDataElement>  <cbe:situation categoryName=\"StartSituation\">    <cbe:situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </cbe:situation></cbe:CommonBaseEvent></cbe:CommonBaseEvents>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));       

            assertEquals(event1, event2);   

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);
            
            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {    
            assertTrue(t.toString(),false);
        }        
    }

    public void testXMLDocumentDeserializationWithSpaces() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\"    xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"    xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"    ><AssociationEngine id=\"i0000000000000000000000000000044\"    name=\"MyCorrelationEngine\"    type=\"Correlate\"/><CommonBaseEvent    creationTime=\"2001-12-31T12:00:00\"    extensionName=\"Name\"    globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"    >  <contextDataElements name=\"myContext\"    type=\"String\"    >    <contextValue    >contextValue</contextValue    >  </contextDataElements    >  <extendedDataElements name=\"ede1\"    type=\"noValue\"    >    <values       >values</values      >    <children name=\"chname1\"    type=\"noValue\"    >      <values    >values</values    >    </children    >  </extendedDataElements    >  <extendedDataElements name=\"ede2\"    type=\"noValue\"    >    <values    >values</values    >    <children name=\"chname2\"    type=\"noValue\"    >      <values    >values</values    >    </children    >  </extendedDataElements    >  <reporterComponentId application=\"appl1\"    componentType=\"compType1\"    component=\"comp1\"    componentIdType=\"compIdType1\"    executionEnvironment=\"exc1\"    instanceId=\"id1\"    location=\"loc1\"    locationType=\"Hostname\"    processId=\"pr1\"    subComponent=\"sub1\"    threadId=\"th1\"/>  <sourceComponentId application=\"app2\"    componentType=\"compType2\"    component=\"comp2\"    componentIdType=\"compIdType2\"    executionEnvironment=\"exc2\"    instanceId=\"id2\"    location=\"loc2\"    locationType=\"SNA\"    processId=\"pr2\"    subComponent=\"sub2\"    threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\"    >    <msgCatalogTokens value=\"value\"/>    <msgId    >msgId</msgId    >    <msgIdType    >Name</msgIdType    >    <msgCatalogId    >msgCatalogId</msgCatalogId    >    <msgCatalogType    >msgCatalogType</msgCatalogType    >    <msgCatalog    >msgCatalog</msgCatalog    >  </msgDataElement    >  <situation categoryName=\"StartSituation\"    >    <situationType reasoningScope=\"EXTERNAL\"    successDisposition=\"SUCCESSFUL\"    situationQualifier=\"START COMPLETED\"    xsi:type=\"StartSituation\"    />  </situation    ></CommonBaseEvent    ></CommonBaseEvents    >";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));       

            assertEquals(event1, event2);   

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);
            
            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {  
            assertTrue(t.toString(),false);
        }        
    }

    public void testXMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));       

            assertEquals(event1, event2);   
            
            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);
            
            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testXMLDocumentWithMultipleEventsDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><AssociationEngine id=\"i0000000000000000000000000000045\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><AssociationEngine id=\"i0000000000000000000000000000046\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A2\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A3\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        try {
            
            CommonBaseEvent[] events1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);
            CommonBaseEvent[] events2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);

            assertTrue(events1.length == events2.length);
            
            for (int counter = 0; counter < events1.length; counter++) {

                assertEquals(events1[counter].getExtendedDataElements("root"), events2[counter].getExtendedDataElements("root"));       

                assertEquals(events1[counter], events2[counter]);   

                assertTrue(events1[counter].getGlobalInstanceId().startsWith("CEE736B610030D11D88000A27990EF8D1A"));
                assertTrue(events2[counter].getGlobalInstanceId().startsWith("CEE736B610030D11D88000A27990EF8D1A"));
            }

            
            AssociationEngine[] engines1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);
            AssociationEngine[] engines2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);

            assertTrue(engines1.length == engines2.length);
            
            for (int counter = 0; counter < engines1.length; counter++) {

	            assertEquals(engines1[counter], engines2[counter]);
	            
                assertTrue(engines1[counter].getId().startsWith("i000000000000000000000000000004"));
                assertTrue(engines2[counter].getId().startsWith("i000000000000000000000000000004"));
            }
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testCBE101Sample1XMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"> <AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </contextDataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>        <children name=\"\" type=\"noValue\">          <values>values</values>       </children>       </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">       <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">     	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />     </situation>	       <a:a xmlns:a=\"http://bar\" />   </CommonBaseEvent></CommonBaseEvents>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1, event2);   
            
            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);
            
            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testCBE101Sample2XMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <a:a xmlns:a=\"http://bar\" > 	     <expression logicalOperator=\"AND\"><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></preCondition><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></postCondition></expression></a:a></CommonBaseEvent></CommonBaseEvents>";
        
        try {
            
            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1, event2);   
            
            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);
            
            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testXMLFragmentDeserialization() {

        String eventXML =
            "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"LOCAL INSTANCE ID :-) :-)\" msg=\"msg 2 &lt;&gt;&amp;;&quot;&apos; :-)\" elapsedTime=\"123\" priority=\"3\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"5\" version=\"1.0.1\"><contextDataElements name=\"ettk id\" type=\"anIDTYPE\"><contextId> FIRST</contextId></contextDataElements><contextDataElements name=\"&lt;ettk&amp;value&gt;\" type=\"anVALUETYPE\"><contextValue>SECOND</contextValue></contextDataElements><extendedDataElements name=\"root\" type=\"long\"><children name=\"Child2\" type=\"long\"><children name=\"Child31\" type=\"long\"><children name=\"Child41\" type=\"hexBinary\"><hexValue>818283</hexValue></children></children><children name=\"Child32\" type=\"long\"><children name=\"Child42\" type=\"stringArray\"><values>one</values><values>two</values><values>three</values></children></children><children name=\"Child33\" type=\"long\"><children name=\"Child43\" type=\"longArray\"><values>4</values><values>5</values></children></children></children></extendedDataElements><associatedEvents resolvedEvents=\"cbe1 cbe2\"><associationEngine>AE id</associationEngine></associatedEvents><sourceComponentId application=\"ettkAPP\"/><msgDataElement msgLocale=\"msgLocale\"><msgCatalogTokens value=\"msgCatalogToken1\"/><msgCatalogTokens value=\"msgCatalogToken2\"/><msgId>msgId</msgId><msgIdType>msgIdType</msgIdType><msgCatalogId>msgCatalogId</msgCatalogId><msgCatalogType>msgCatalogType</msgCatalogType><msgCatalog>MsgCatalog</msgCatalog></msgDataElement></CommonBaseEvent>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventFromCanonicalXML(eventXML);
            CommonBaseEvent event2 = EventFormatter.eventFromCanonicalXML(eventXML);

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));       
            
            assertEquals(event1, event2);   

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testXMLFragmentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {
            
            eventUnformatted = EventFormatter.eventFromCanonicalXML(eventXMLUnformatted);
            eventFormatted = EventFormatter.eventFromCanonicalXML(eventXMLFormatted);
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {           
            assertTrue(t.toString(),false);
        }        
    
        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testXMLDocumentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {
            
            eventUnformatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLUnformatted)[0];
            eventFormatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLFormatted)[0];
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
        
        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }
    
    public void testEventFactoryXMLTemplateDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><TemplateEvent version=\"1.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"templateEvent.xsd\"><CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent></TemplateEvent>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getSituation(), event2.getSituation());
            
            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testEventFactoryXMLTemplateFragmentDeserialization() {

        String eventXML =
            "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventFromCanonicalXML(eventXML);
            CommonBaseEvent event2 = EventFormatter.eventFromCanonicalXML(eventXML);

            assertEquals(event1.getSituation(), event2.getSituation());
                        
            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testEventFactoryXMLTemplateFragmentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL","LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {
            
            eventUnformatted = EventFormatter.eventFromCanonicalXML(eventXMLUnformatted);
            eventFormatted = EventFormatter.eventFromCanonicalXML(eventXMLFormatted);
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testEventFactoryXMLTemplateDocumentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL","LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {
            
            eventUnformatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLUnformatted)[0];
            eventFormatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLFormatted)[0];
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
        
        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }
 
    public void testExtendedDataElementTypesInXMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
            assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
            assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

            assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
            assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

            assertEquals(event1, event2);
            
            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);
            
            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        
    }

    public void testExtendedDataElementTypesInXMLFragmentDeserialization() {

        String eventXML =
            "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent>";

        try {
            
            CommonBaseEvent event1 = EventFormatter.eventFromCanonicalXML(eventXML);
            CommonBaseEvent event2 = EventFormatter.eventFromCanonicalXML(eventXML);

            assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
            assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
            assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

            assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
            assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

            assertEquals(event1, event2);
            
            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } 
        catch(AssertionFailedError a){
            throw(a);
        }
        catch (Throwable t) {
            assertTrue(t.toString(),false);
        }        

    }
    
    public void testDeprecatedXMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));       
        
        assertEquals(event1, event2);
        
        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);
        
        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
    }

    public void testDeprecatedCBE101Sample1XMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </contextDataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>        <children name=\"\" type=\"noValue\">          <values>values</values>       </children>       </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">       <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">     	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />     </situation>	       <a:a xmlns:a=\"http://bar\" />   </CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1, event2);
        
        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);
        
        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
   }

    public void testDeprecatedCBE101Sample2XMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <a:a xmlns:a=\"http://bar\" > 	     <expression logicalOperator=\"AND\"><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></preCondition><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></postCondition></expression></a:a></CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);
        
        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
    }

    public void testDeprecatedXMLFragmentDeserialization() {

        String eventXML =
            "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"LOCAL INSTANCE ID :-) :-)\" msg=\"msg 2 &lt;&gt;&amp;;&quot;&apos; :-)\" elapsedTime=\"123\" priority=\"3\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"5\" version=\"1.0.1\"><contextDataElements name=\"ettk id\" type=\"anIDTYPE\"><contextId> FIRST</contextId></contextDataElements><contextDataElements name=\"&lt;ettk&amp;value&gt;\" type=\"anVALUETYPE\"><contextValue>SECOND</contextValue></contextDataElements><extendedDataElements name=\"root\" type=\"long\"><children name=\"Child2\" type=\"long\"><children name=\"Child31\" type=\"long\"><children name=\"Child41\" type=\"hexBinary\"><hexValue>818283</hexValue></children></children><children name=\"Child32\" type=\"long\"><children name=\"Child42\" type=\"stringArray\"><values>one</values><values>two</values><values>three</values></children></children><children name=\"Child33\" type=\"long\"><children name=\"Child43\" type=\"longArray\"><values>4</values><values>5</values></children></children></children></extendedDataElements><associatedEvents resolvedEvents=\"cbe1 cbe2\"><associationEngine>AE id</associationEngine></associatedEvents><sourceComponentId application=\"ettkAPP\"/><msgDataElement msgLocale=\"msgLocale\"><msgCatalogTokens value=\"msgCatalogToken1\"/><msgCatalogTokens value=\"msgCatalogToken2\"/><msgId>msgId</msgId><msgIdType>msgIdType</msgIdType><msgCatalogId>msgCatalogId</msgCatalogId><msgCatalogType>msgCatalogType</msgCatalogType><msgCatalog>MsgCatalog</msgCatalog></msgDataElement></CommonBaseEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(event1, eventXML);
        EventFormatter.fromCanonicalXMLString(event2, eventXML);

        assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));

        assertEquals(event1, event2);
        
        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }

    public void testDeprecatedXMLFragmentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLString(eventFormatted, eventXMLFormatted);

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testDeprecatedXMLDocumentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLDocString(eventFormatted, eventXMLFormatted);
        
        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }
    
    public void testDeprecatedEventFactoryXMLTemplateDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?><TemplateEvent version=\"1.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"templateEvent.xsd\"><CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent></TemplateEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1.getSituation(), event2.getSituation());

        assertEquals(event1, event2);
        
        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }

    public void testDeprecatedEventFactoryXMLTemplateFragmentDeserialization() {

        String eventXML =
            "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(event1, eventXML);
        EventFormatter.fromCanonicalXMLString(event2, eventXML);

        assertEquals(event1.getSituation(), event2.getSituation());

        assertEquals(event1, event2);
        
        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }

    public void testDeprecatedEventFactoryXMLTemplateFragmentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL","LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLString(eventFormatted, eventXMLFormatted);

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testDeprecatedEventFactoryXMLTemplateDocumentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL","LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLDocString(eventFormatted, eventXMLFormatted);
        
        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }
 
    public void testDeprecatedExtendedDataElementTypesInXMLDocumentDeserialization() {

        String eventXML =
            "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
        assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
        assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

        assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
        assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

        assertEquals(event1, event2);
        
        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);
        
        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
    }

    public void testDeprecatedExtendedDataElementTypesInXMLFragmentDeserialization() {

        String eventXML =
            "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(event1, eventXML);
        EventFormatter.fromCanonicalXMLString(event2, eventXML);

        assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
        assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
        assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

        assertEquals(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
        assertEquals(((ExtendedDataElement)(((ExtendedDataElement)(event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

        assertEquals(event1, event2);
        
        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }
}