package org.eclipse.hyades.internal.collection.framework;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;

import org.eclipse.hyades.internal.execution.file.FileServiceConstants;
import org.eclipse.hyades.execution.security.IClientHandler;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.common.ManageFileCommand;
import org.eclipse.hyades.internal.execution.local.common.Message;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * @author Giridhar.S
 *
 * This is the class which is resposible for actual file transfer. It 
 * contains socket reading and writing methods, and methods for doing
 * server side file transfer operations.
 * 
 */
public class FileClientHandlerImpl extends Thread implements IClientHandler, FileServiceConstants, Constants {

	/* The socket which we recieve */
	private Socket _socket;
	
	/*  The associated input and output streams */
	private InputStream _instrm;
	private OutputStream _outstrm;

	/* The data buffer */
	private byte[] _data;
	
	/* The various counters */
	private int byteCount ,count ;
	
	/**
	 *  Constructor
	 * @param s - Socket passed by the connection handler, for performing file transfer.
	 */
	public FileClientHandlerImpl(Socket s) {
		_socket = s;
		
		try {
			_instrm = s.getInputStream();
			_outstrm = s.getOutputStream();
		} catch (IOException e) {
		}
	}
	
	/**
	 *  Method to get the raw IP address of the client machine.
	 */
	public byte[] getRemoteAddress() {
		return _socket.getInetAddress().getAddress();
	}

	/**
	 * Method to read bytes off the socket input stream
	 * @param b - byte buffer to fill up
	 * @param off - starting offset
	 * @param len - length of the data to be read
	 * @return int - number of bytes read 
	 */
	public int read(byte[] b, int off, int len) {
		int bytesRead;
		
		try {
			bytesRead = _instrm.read(b, off,len);
		}
		catch (IOException ioe) {
			bytesRead = -1;
		}
		
		return bytesRead;
	}

	/**
	 * Method to write bytes into the socket output stream
	 * @param b - byte buffer to read from
	 * @param off - starting offset
	 * @param len - length of the data to be written
	 * @return int - 0 - success; -1 - failed 
	 */
	
	public int write(byte[] b, int off, int len) {
		try {
			_outstrm.write(b, off, len);
		}
		catch (IOException ioe) {
			return -1;
		}
		
		return 0;
		
	}
	
	/**
	 * A method to close the connection with the client
	 */
	public void closeConnection() {
		try {
			_instrm.close();
			_outstrm.close();
			
			_socket.close();
		}
		catch (IOException ioe) {
		}
	}
	
	/**
	 * Method to write bytes into the socket output stream
	 * @param b - byte buffer to read from
	 * @return int - 0 - success; -1 - failed 
	 */
	
	
	public int write (byte[] b)	{
		try {
			_outstrm.write(b);
		}
		catch (IOException ioe) {
			return RA_IO_ERROR;
		}
		
		return 0;
	}
	
	/**
	 * Method to read bytes off the socket input stream
	 * @param b - byte buffer to fill up
	 * @return int - number of bytes read 
	 */
	
	
	public int read (byte[] b)	{
		int bytesRead;
		try {
			bytesRead = _instrm.read(b);
		}
		catch (IOException ioe) {
			bytesRead = -1;
		}
		
		return bytesRead;
	}
	
	/**
	 * The get file method which transfers a file from the RAC machine to the client machine.
	 * @param fileName
	 * @return int - number of bytes transferred / error status
	 */
	
	private int getFile (String fileName){
		
		FileInputStream _fis = null;
		
		try {
			_data = new byte[RA_BUFFER_SIZE];
			_fis = new FileInputStream(fileName);
		} catch (FileNotFoundException e) {
			closeConnection();
			return RA_FILE_NOT_FOUND_ERROR;
		}
		
		try {
			/* Read the file bits and send them over the socket */
			while((count=_fis.read(_data)) != -1)	{
				write(_data, 0, count);
				byteCount += count;
			}
			_fis.close();
			closeConnection();
		} catch (IOException e1) {
			closeConnection();
			return RA_IO_ERROR;
		}
		
		return 0;
	}
	
	/**
	 * A method to recieve a file from the client machine and put it on the RAC machine.
	 * @param destFileName - name of the file on the RAC machine.
	 * @return int - number of bytes read / error status
	 */
	private int putFile (String destFileName){
		
		/* We may have to create the directory if it does not exist */
		String directoryName = null;
		
		/* We have to parse the file name differently for Windows and Unix */
		//if (System.getProperty("os.name", "null").startsWith("Windows"))	{
			/* For Windows */
		//	directoryName = destFileName.substring(0, destFileName.lastIndexOf("\\"));
		//}
		//else	{
		//	directoryName = destFileName.substring(0, destFileName.lastIndexOf("/"));
		//}
		// The above block is changed to the following 2 lines to fix problems with hard coded file separater.
		// bjiang 4/16/2004
		File destFile = new File(destFileName);
		directoryName = destFile.getParent();		
		
		File dirFile = new File(directoryName);
		if (!dirFile.exists())
			dirFile.mkdirs();		

		File _file = new File (destFileName);
		FileOutputStream _fos=null;
		
		try {
			_file.createNewFile();
			_fos=new FileOutputStream(_file);
		} catch (FileNotFoundException e) {
			closeConnection();	// We close the connection with the client and return 
			return RA_FILE_NOT_FOUND_ERROR;
		} catch (IOException e2) {
			closeConnection();	// We close the connection with the client and return
			return RA_FILE_NOT_FOUND_ERROR;
		}
		
		/* We read the data which the client sends */
		try {
			_data = new byte[RA_BUFFER_SIZE];
			_socket.setSoTimeout(3000); /* A fairly large time out */
				while ((count = read(_data)) != -1)	{
					_fos.write(_data, 0, count);
					byteCount += count;
				}
			_fos.flush();
			_fos.close();
		} catch (IOException e1) {
			return RA_IO_ERROR;
		}finally {
			closeConnection();
		}
		
		
		return 0;
	}
	
	/**
	 * Method to delete a file. This never gets called since RAC does the file deletion in native code.
	 * @param fileName
	 * @return 0
	 */
	
	private int deleteFile (String fileName){
		return 0;
	}
	
	public void run()	{
		
		_data = new byte[RA_BUFFER_SIZE];
		int offset = 0, result = -1;
		
		/* Read the request information that the client sends */
		/* We first read the bytes from the input stream */
		int count = read(_data);

		/* First 4 bytes will be the type of the operation */
		long operation = Message.readRALongFromBuffer(_data, offset);
		offset += sizeofLong;

		/* The next 4 bytes will be the length of the file name coming next */
		long len = Message.readRALongFromBuffer(_data, offset);
		offset += sizeofLong;

		/* The actual file name */
		String fileName = new String (_data, offset, (int)len);
		
		switch ((int)operation)	{
		
		case (int) ManageFileCommand.GET:		/* Send the file to client workbench */
			result = getFile (fileName);
			break;
			
		case (int) ManageFileCommand.PUT:		/* Recieve a file from client workbench */
			result = putFile (fileName);
			break;
			
		case (int) ManageFileCommand.DELETE:	/* We are never going to come here, since RAC handles this */
			result = deleteFile(fileName);
			break;
		/* We have to just swallow the return status, because we cannot return it */	
		default:
		}
	}
}
