/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.extractors;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.util.Messages;

/**
 *
 * This is an extractor implementation that uses regular expressions 
 * to extract records.  The JDK 1.4 java.util.regex library is used to
 * process the regular expressions.
 * 
 */
public class RegularExpressionExtractor extends SimpleExtractor {

	private Pattern compiledStartPattern;
	private Pattern compiledEndPattern;
	
	public void update() throws AdapterInvalidConfig {
		super.update();
	
		/* If we have a start pattern then compile it now */
		if(getStartPattern()!=null && !getStartPattern().equals("")) {
			try {
				compiledStartPattern = Pattern.compile(getStartPattern(), Pattern.MULTILINE);
			}
			catch (PatternSyntaxException e) {
				throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_Regular_Expression_Extractor_Start_Pattern_Invalid_ERROR_",e.getMessage()));
			}
		}
	
		/* If we have a end pattern then compile it now */
		if(getEndPattern()!=null && !getEndPattern().equals("")) {
			try {
				compiledEndPattern = Pattern.compile(getEndPattern(), Pattern.MULTILINE);
			}
			catch (PatternSyntaxException e) {
				throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_Regular_Expression_Extractor_End_Pattern_Invalid_ERROR_",e.getMessage()));
			}
		}
	}
	
	/**
	 * Search for the pattern in the specified search string.  If the pattern is located in the 
	 * search string then the position variable is loaded with the start of the pattern and the 
	 * end of the pattern.
	 * 
	 * @param pos
	 * @param pattern
	 * @param searchString
	 * @param startOffset
	 */
	protected void search(StringPosition position, String pattern, String searchString, int startOffset) {
		boolean found = false;
		
		Pattern expression=null;
		
		/* Which pattern are we searhing for */
		if(pattern==getStartPattern()) {
			expression=compiledStartPattern;
		}
		else {
			expression=compiledEndPattern;
		}
		
		/* Get the input string to search in */
		String input = searchString.substring(startOffset);

		/* Get the matcher and try to find the pattern */
		Matcher matcher = expression.matcher(input);
		
		found = matcher.find();
		
		if (found) {

			/* For special characters such as '^' and '$' the pattern may have length 0 */
			if (matcher.start() == matcher.end()) {
				int matchOffset = startOffset + matcher.start();

				position.setStartPosition(matchOffset);
				position.setEndPosition(matchOffset);				
			}
			else {
				position.setStartPosition(startOffset + matcher.start());
				position.setEndPosition(startOffset + matcher.end()-1);
			}
			position.setValid(true);
		}

	}

}

