/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.eclipse.hyades.logging.core.XmlUtility;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * This class is used to migrate pre 1.3 Hyades adapter configuration files 
 * to 1.3 Hyades adapter configuration files
 */
public class AdapterMigrator {
	
	// Indicator whether migration was necessary and occurred
	private boolean migrated = false;

	public static void main(String[] args) {
		AdapterMigrator migrator = new AdapterMigrator();
		
		if ((args != null) && (args.length >= 1)) {

			String originalFileName = args[0];
			String newFileName;
			
			if (args.length >= 2)
				newFileName = args[1];
			else
				newFileName = originalFileName;
			try {
				migrator.migrate(originalFileName, newFileName);
				
				if (!migrator.isMigrated()) {
					System.out.println(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_Migration_Required_INFO_", originalFileName));
				}
			}
			catch (AdapterMigrationException e) {
				System.err.println(e.getLocalizedMessage());
			}
		}
		else {
			System.out.println(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_Specified_ERROR_"));
		}
	}

	/**
	 * Migrate an adapter configuration file, overwriting the original file.
	 * @param originalFile
	 * @throws AdapterMigrationException
	 */
	public boolean migrate(String originalFile) throws AdapterMigrationException {
		return migrate(originalFile, originalFile);
	}	
	
	/**
	 * Migrate an adapter configuration file to a new file.  The original file
	 * will not be modified.
	 * @param originalFile
	 * @param newFile
	 * @return boolean - migration occurred indicator
	 * @throws AdapterMigrationException
	 */
	public boolean migrate(String originalFile, String newFile) throws AdapterMigrationException {
		// get the configuration from the original adapter file
		InputStream inStream = null;
		String fileName = originalFile;
		migrated = false;
		
		/* Get input stream for config file relative to the current directory if an
		 * absolute path is not specified
		 */
		try {
			String fileSeparator = System.getProperty("file.separator");
			// If we have the wrong file separator in the config file path then
			if (fileName.indexOf('/') != -1 && fileSeparator != "/") {
				// Replace file separators
				fileName = fileName.replace('/', fileSeparator.charAt(0));
			}
			// Remove ./
			if (fileName.startsWith("." + fileSeparator)) {
				fileName = fileName.substring(2);
			}
			inStream = new FileInputStream(fileName);
		} catch (FileNotFoundException exc) {
			throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_ERROR_", originalFile));
		} catch (SecurityException exc) {
			throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", originalFile));
		}
		
		/* Parse the config file */
		Document doc = null;
		
		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(inStream);
		} catch (SAXException e) {
			throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", originalFile),e);
		} catch (IOException e) {
			throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", originalFile),e);
		} catch (ParserConfigurationException e) {
			throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", originalFile),e);
		} catch (Exception e) {
			throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Parse_ERROR_", originalFile),e);
		}

		NodeList nodeList;		

		// migrate the sensor configuration
		
		// Get the sensor config - there should only be one sensor
		nodeList = doc.getElementsByTagName(Messages.getString("HyadesGASensorTagName"));
		int nodeCount = nodeList.getLength();
		Element oldElement;
		Element newElement;
		IComponentMigrator migrator;
		migrator = new SensorMigrator();
		for (int i = 0; i < nodeCount; ++i)
		{
			oldElement = (Element) nodeList.item(i);
			newElement = migrator.migrate(oldElement);
			// Only replace the sensor child if we have a new one
			if (newElement != null) {
				oldElement.getParentNode().replaceChild((Node)newElement, (Node)oldElement);
				migrated = true;
			}
		}
		
		// migrate the outputter configuration
		
		// Get the outputter configurations
		nodeList = doc.getElementsByTagName(Messages.getString("HyadesGAOutputterTagName"));
		nodeCount = nodeList.getLength();
		migrator = new OutputterMigrator();
		for (int i = 0; i < nodeCount; ++i)
		{
			oldElement = (Element) nodeList.item(i);
			newElement = migrator.migrate(oldElement);
			// Only replace the outputter child if we have a new one
			if (newElement != null) {
				oldElement.getParentNode().replaceChild((Node)newElement, (Node)oldElement);
				migrated = true;
			}
		}

		// If migration was performed Then update the adapter attributes and save the file
		if (isMigrated()) {
			// update the adapter attributes - add the ProcessUnit namespace attribute

			nodeList = doc.getElementsByTagName(Messages.getString("HyadesGAAdapterElementTagName"));
		
			if (nodeList.getLength() == 0) {
				throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Missing_Element_ERROR_", originalFile, Messages.getString("HyadesGAAdapterElementTagName"))); 
			}
				
			Element adapterElement = (Element)nodeList.item(0);
			
			if (!adapterElement.hasAttribute(Messages.getString("HyadesGAAdapterProcessUnitNamespaceAttributeTagName"))) {
				adapterElement.setAttribute(Messages.getString("HyadesGAAdapterProcessUnitNamespaceAttributeTagName"),Messages.getString("HyadesGAAdapterProcessUnitNamespaceAttributeValue"));
			}		
			
			//Write out the configuration to the new adapter file:
			try {
			    XmlUtility.serialize(doc, new File(newFile));	
			} 
			catch (TransformerException t) {
			    throw new AdapterMigrationException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Save_ERROR_", newFile),t);			    
			}
		}
		
		return isMigrated();
	}
	
	/**
	 * @return
	 */
	public boolean isMigrated() {
		return migrated;
	}

}
