package org.eclipse.hyades.logging.core.tests;

import java.util.ArrayList;
import java.util.StringTokenizer;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.hyades.logging.core.Guid;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * All JUnit test cases for the <code>org.eclipse.hyades.logging.core.Guid</code> class.
 * <p>
 * 
 * @author Paul E. Slauenwhite
 * @version May 11, 2004
 * @since May 11, 2004
 */
public class GuidTest extends TestCase {
    
    private TestingUtilities utilities = null;
    
    /**
     * Constructor for GuidTest.
     * <p>
     * 
     * @param name The name of the test case.
     */
    public GuidTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {

        super.setUp();

        utilities = TestingUtilities.getInstance();
    }
    
    /**
     * Test for duplicate Guids in a single threaded environment.
     */
    public void testDuplicateGuidsSingleThread() {
        
        try{

            final int TOTAL_GUIDS = 10000;

            ArrayList guids = new ArrayList(TOTAL_GUIDS);

	        for (int counter = 0; counter < TOTAL_GUIDS; counter++) {
	            guids.add(Guid.generate());            
	        }
	
	        assertEquals(guids.size(),TOTAL_GUIDS);
	        
	        for (int counter = 0; counter < TOTAL_GUIDS; counter++) {
	            assertFalse("Duplicate GUID '" + guids.get(counter) + "' found!",guids.subList((counter + 1),TOTAL_GUIDS).contains(guids.get(counter)));
	        } 
	    }
	    catch(AssertionFailedError a){
	        throw(a);
	    }
	    catch (Throwable t) {
	        assertTrue(t.toString(),false);
	    }
    }
    
    /**
     * Test for duplicate Guids in a multiple threaded environment.
     */
    public void testDuplicateGuidsMultipleThread() {

        try{        

            final int TOTAL_GUIDS_PER_THREAD = 1000;
	        final int TOTAL_THREADS = 10;
	        final int TOTAL_GUIDS = (TOTAL_GUIDS_PER_THREAD * TOTAL_THREADS);
	
	        final ArrayList GUIDS = new ArrayList(TOTAL_GUIDS);
	
	        for (int counter = 0; counter < TOTAL_THREADS; counter++) {
	            
	            new Thread() {
	
	                public void run() {
	
	                    for (int counter = 0; counter < TOTAL_GUIDS_PER_THREAD; counter++) {
	                        GUIDS.add(Guid.generate());
	                    }
	                }
	            }
	            .start();
	        }
	        
	        while(GUIDS.size() < TOTAL_GUIDS){
	
	            try {
	                Thread.sleep(1000);
	            } 
	            catch (Exception e) {
	                assertTrue(e.toString(),false);
	            } 
	        }
	
	        for (int counter = 0; counter < TOTAL_GUIDS; counter++) {
	            assertFalse("Duplicate GUID '" + GUIDS.get(counter) + "' found!",GUIDS.subList((counter + 1),TOTAL_GUIDS).contains(GUIDS.get(counter)));
	        } 
	    }
	    catch(AssertionFailedError a){
	        throw(a);
	    }
	    catch (Throwable t) {
	        assertTrue(t.toString(),false);
	    }
}
    
    public void testDuplicateGuidsMultipleProcess() {

        try{     

	        final int TOTAL_GUIDS_PER_PROCESS = 1000;
	        final int TOTAL_PROCESSES = 10;
	        final int TOTAL_GUIDS = (TOTAL_GUIDS_PER_PROCESS * TOTAL_PROCESSES);
	
	        //Lauch the test class:
	        final String[] command = { "java", "-classpath", utilities.getClassLocation(GuidTestProcessClass.class), GuidTestProcessClass.class.getName(), String.valueOf(TOTAL_GUIDS_PER_PROCESS)};
	        	    
	        final ArrayList PROCESS_OUTPUT = new ArrayList(TOTAL_PROCESSES);
	            
	        for (int counter = 0; counter < TOTAL_PROCESSES; counter++) {
	                
	            new Thread() {
	
	                public void run() {   
	                        
	                    try {
	                        PROCESS_OUTPUT.add(utilities.executeProcess(command));
	                    } 
	                    catch (Exception e) {
	                        //Ignore.
	                    }
	                }
	            }
	            .start();
	        }
	        
	        while(PROCESS_OUTPUT.size() < TOTAL_PROCESSES){
	        	
	            try {
	                Thread.sleep(1000);
	            } 
	            catch (Exception e) {
	                assertTrue(e.toString(),false);
	            } 
	        }
	        
	        ArrayList guids = new ArrayList(TOTAL_GUIDS);
	        StringTokenizer tokens = null;
	        
	        for (int counter = 0; counter < TOTAL_PROCESSES; counter++) {
	            
	            tokens = new StringTokenizer(((String)(PROCESS_OUTPUT.get(counter))));
	            
	            while(tokens.hasMoreTokens()){
	                guids.add(tokens.nextToken().trim());
	            }                
            }
	        
	        assertEquals(guids.size(),TOTAL_GUIDS);

	        for (int counter = 0; counter < TOTAL_GUIDS; counter++) {
	            assertFalse("Duplicate GUID '" + guids.get(counter) + "' found!",guids.subList((counter + 1),TOTAL_GUIDS).contains(guids.get(counter)));
	        } 
	   }
       catch(AssertionFailedError a){
           throw(a);
       }
       catch (Throwable t) {
           assertTrue(t.toString(),false);
       }
    }
}
