/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.internal.ui;

import java.util.*;

import org.eclipse.debug.core.*;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.internal.ui.DebugUIPlugin;
import org.eclipse.debug.internal.ui.views.console.ProcessConsole;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.models.hierarchy.impl.*;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.*;
import org.eclipse.jface.util.ListenerList;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.*;
import org.eclipse.ui.console.*;

/**
 * @author popescu
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public class TraceUIManager
							implements ISelectionChangedListener

{

    private static TraceUIManager traceUIManager;
    
	protected ResourceSet resourceSet;
	protected HierarchyFactory factory;

	protected HashMap fSelectionModel = new HashMap();
	protected ProfileEvent fProfileEvent = new ProfileEvent();
	protected ViewSelectionChangedEvent pdViewEvent = new ViewSelectionChangedEvent();
	protected ProfileEvent updateModelEvent =
		new ProfileEvent(ProfileEvent.UPDATE_MODEL);
	protected ProfileEvent refreshViewsEvent =
		new ProfileEvent(ProfileEvent.REFRESH_VIEWS);
	
	private boolean isLinkingEnabled = false;

    private TraceUIManager()
    {
    }
    
	/**
	 * The collection of profile event listeners.
	 */
	protected ListenerList fProfileEventListeners = new ListenerList(20);

	/**
	 * The collection of problem determination view event listeners.
	 */
	protected ListenerList fPDViewEventListeners = new ListenerList(20);

	/**
	 * The desktop parts that contain the selections
	 */
	protected List fSelectionParts = new ArrayList(2);

	/**
	 * The selection providers for the trace UI
	 */
	protected List fSelectionProviders = new ArrayList(2);

	/**
	 * The list of selection listeners for the trace UI
	 */
	protected ListenerList fListeners = new ListenerList(2);

    /**
     * list of delete listeners
     */
	protected ArrayList fDeleteListener = new ArrayList(1);

    /**
     * 
     * @return traceUIManager singleton
     */
    public static TraceUIManager getTraceUIManager()
    {
    	if(traceUIManager == null)
    	  traceUIManager = new TraceUIManager();
    	   
    	return traceUIManager;   
    }
    

	/**
	 * Adds an <code>ISelectionListener</code> to the trace selection manager.
	 */
	public void addSelectionListener(ISelectionListener l) {
		fListeners.add(l);
	}

	/**
	 * Remove an <code>ISelectionListener</code> to the trace selection manager.
	 */
	public void removeSelectionListener(ISelectionListener l) {
		fListeners.remove(l);
	}

	/**
	 * Adds the selection provider for the debug UI.
	 */
	public void addSelectionProvider(
		ISelectionProvider provider,
		IWorkbenchPart part) {
		fSelectionProviders.add(provider);
		fSelectionParts.add(part);
		provider.addSelectionChangedListener(this);
	}


	/**
	 * @param process
	 */
 public void deregisterLaunchProcess(IProcess iprocess) {

 }
 
	/**
	 * Remove the console for this process
	 * @param process
	 */
public void deleteLaunch(IProcess iprocess) {
	if (iprocess instanceof ProcessAdapter) {
	 ProcessAdapter adapter = (ProcessAdapter)iprocess;
	 ILaunch launch = adapter.getLaunch();
	 if (launch != null) {
	 	DebugPlugin.getDefault().getLaunchManager().removeLaunch(launch);
	 }
	}
}

	/**
	 * Insert the method's description here.
	 * Creation date: (12/28/2000 1:05:55 PM)
	 * @return com.ibm.etools.perftrace.PerftraceFactory
	 */
	public HierarchyFactory getPerftraceFactory() {
		return factory;
	} 	/**
	 * 
	 * @return ResourceSet
	 */
	public ResourceSet getResourceSet() {
		
		if(resourceSet == null)
		{			
			HierarchyPackageImpl.init();
			factory = new HierarchyFactoryImpl();

			// init resourceSet
			resourceSet = HierarchyResourceSetImpl.getInstance();			
		}
		
		return resourceSet;
		
	}
	
	public void registerLaunchProcess(final IProcess iprocess)
    {
	  if (iprocess instanceof ProcessAdapter) {
	  	ProcessAdapter processAdapter = (ProcessAdapter)iprocess;
	  	ILaunch launch = processAdapter.getLaunch();
	  	if (launch != null) {
		  	launch.addProcess(processAdapter);
	  	}
	
	  	Display.getDefault().syncExec(new Runnable() {
			public void run() {
			  	IConsole console = getConsole(iprocess);
			  	showConsole(console);
			}
	  	});
	  }
    }

	/**
	 * Removes the selection provider from the debug UI.
	 */
	public void removeSelectionProvider(
		ISelectionProvider provider,
		IWorkbenchPart part) {
		fSelectionProviders.remove(provider);
		fSelectionParts.remove(part);
		provider.removeSelectionChangedListener(this);
		selectionChanged(null);
	}
	
	/**
	 * Selection has changed in the debug selection provider.
	 * Notify the listeners.
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		Object[] copiedListeners = fListeners.getListeners();
		ISelection selection = null;
		ISelectionProvider provider = null;
		IWorkbenchPart part = null;
		if (event != null) {
			selection = event.getSelection();
			provider = (ISelectionProvider) event.getSource();
			int index = fSelectionProviders.indexOf(provider);
			if (index == -1) {
				return;
			}
			part = (IWorkbenchPart) fSelectionParts.get(index);
			selectionChanged(part, selection);
		}else{
			
			return;
		}
		for (int i = 0; i < copiedListeners.length; i++) {
			ISelectionListener l = ((ISelectionListener) copiedListeners[i]);
//			if(l!=this)
				l.selectionChanged(
				part,
				selection);
		}

	}
 
 /**
  * Returns the console for the given process, or <code>null</code> if none.
  * 
  * @param process
  * @return the console for the given process, or <code>null</code> if none
  */
 private IConsole getConsole(IProcess process) {
	  IConsoleManager manager = ConsolePlugin.getDefault().getConsoleManager(); 
	  IConsole[] consoles = manager.getConsoles();
	  for (int i = 0; i < consoles.length; i++) {
	   IConsole console = consoles[i];
	   if (console instanceof ProcessConsole) {
	    ProcessConsole pc = (ProcessConsole)console;
	    if (pc.getProcess().equals(process)) {
	     return pc;
	    }
	   }
	  }
	  return null;
 }
 
 private TRCProcessProxy getProcessFromInput(Object input)
 {
	  TRCProcessProxy processInput = null;
	  if (input instanceof TRCProcessProxy)
	   processInput= (TRCProcessProxy) input;
	  else if(input instanceof TRCAgentProxy)
	   processInput = ((TRCAgentProxy)input).getProcessProxy();
	
	  return processInput;
 }
 
 public void selectionChanged(IWorkbenchPart part, ISelection sel){
  if (sel != null && !sel.isEmpty())
  {
   IStructuredSelection selection = (IStructuredSelection)sel;
   IProcess process = ProcessMap.get(getProcessFromInput(selection.getFirstElement()));
   final IConsole console = getConsole(process);
   
   showConsole(console);
  }
 }
 

 /**
 * @param console
 */
private void showConsole(final IConsole console) {
	IWorkbenchWindow window = DebugUIPlugin.getActiveWorkbenchWindow();
	   if (window != null) {
	    IWorkbenchPage page = window.getActivePage();
	    if (page != null) {
	     IViewPart consoleView = page.findView(IConsoleConstants.ID_CONSOLE_VIEW);
	     if (consoleView != null) {
	     		//display (ie. update the content) the console, if it exists
	     		//but do not bring it to the front
		     if (consoleView instanceof IConsoleView) {
		      ((IConsoleView) consoleView).display(console);
		     }
	     }
	    }
	   }
}


	/**
	 * Dispose all resources on this plugin shut down
	 */
	public void dispose() {
		
		
		fListeners.clear();
		fDeleteListener.clear();
		fSelectionParts.clear();
		fSelectionProviders.clear();
		resourceSet = null;
		factory = null;
		
		fPDViewEventListeners.clear();
		fDeleteListener.clear();
		fProfileEventListeners.clear();
		fSelectionModel.clear();
		fProfileEvent = null;
		pdViewEvent = null;
		refreshViewsEvent = null;
		updateModelEvent = null;		
		
		traceUIManager=null;
	}
	
	/**
	 * Adds the given listener to the collection of registered profile
	 * event listeners. Has no effect if an identical listener is already
	 * registered.
	 *
	 * @param listener the listener to add
	 */
	public void addProfileEventListener(IProfileEventListener listener) {
		fProfileEventListeners.add(listener);
	}
	
	/**
	 * Removes the given listener from the collection of registered profile
	 * event listeners. Has no effect if an identical listener is not already
	 * registered.
	 *
	 * @param listener the listener to remove
	 */
	public void removeProfileEventListener(IProfileEventListener listener) {
		fProfileEventListeners.remove(listener);
	}

	
	/**
	 * Insert the method's description here.
	 * Creation date: (5/25/2001 3:24:32 PM)
	 * @param listener com.ibm.etools.pd.core.IDeleteListener
	 */
	public void addDeleteListener(IDeleteListener listener) {
		fDeleteListener.add(listener);
	}


	/**
	 * Adds the given listener to the collection of registered pd view
	 * event listeners. Has no effect if an identical listener is already
	 * registered.
	 *
	 * @param listener the listener to add
	 */
	public void addViewSelectionChangedListener(IViewSelectionChangedListener listener) {
		fPDViewEventListeners.add(listener);
	}


	/**
	 * 
	 * @param mofObject
	 * @param model
	 */
	public void addSelectionModel(EObject mofObject, ITraceSelection model) {
		fSelectionModel.put(mofObject, model);
	}


	/**
	 * Insert the method's description here.
	 * Creation date: (5/25/2001 3:25:42 PM)
	 * @return java.util.ArrayList
	 */
	public ArrayList getDeleteListeners() {
		return fDeleteListener;
	}

	public ViewSelectionChangedEvent getViewSelectionChangedEvent() {
		return pdViewEvent;
	}


	public ProfileEvent getProfileEvent() {
		return fProfileEvent;
	}


	/**
	 * @return ProfileEvent
	 */
	public ProfileEvent getRefreshViewEvent(Object source) {
		refreshViewsEvent.setSource(source);
		return refreshViewsEvent;
	}


	/**
	 * 
	 * @param mofObject
	 * @return SelectionModel
	 */
	public ITraceSelection getSelectionModel(EObject mofObject) {

		Object model = fSelectionModel.get(mofObject);
		if (model == null) {
			model = new TraceSelection();
			fSelectionModel.put(mofObject, model);
		}

		return (ITraceSelection) model;
	}


	/**
	 * @return ProfileEvent
	 */
	public ProfileEvent getUpdateModelEvent(Object source) {
		updateModelEvent.setSource(source);
		return updateModelEvent;
	}


	/**
	 * 
	 * @param event
	 */
	public void notifyViewSelectionChangedListener(ViewSelectionChangedEvent event) {
		Object[] listeners = fPDViewEventListeners.getListeners();
		for (int idx = 0; idx < listeners.length; idx++) {
			((IViewSelectionChangedListener) listeners[idx]).handleViewSelectionChangedEvent(event);
		}
	}


	/**
	 * 
	 * @param event
	 */
	public void notifyProfileEventListener(ProfileEvent event) {
		Object[] listeners = fProfileEventListeners.getListeners();
		for (int idx = 0; idx < listeners.length; idx++) {
			((IProfileEventListener) listeners[idx]).handleProfileEvent(event);
		}
	}


	/**
	 * Insert the method's description here.
	 * Creation date: (10/03/2002 3:24:32 PM)
	 * @param listener com.ibm.etools.pd.core.IDeleteListener
	 */
	public void removeDeleteListener(IDeleteListener listener) {
		fDeleteListener.remove(listener);
	}


	/**
	 * Removes the given listener from the collection of registered pd view
	 * event listeners. Has no effect if an identical listener is not already
	 * registered.
	 *
	 * @param listener the listener to remove
	 */
	public void removeViewSelectionChangedListener(IViewSelectionChangedListener listener) {
		fPDViewEventListeners.remove(listener);
	}


	/**
	 * 
	 * @param mofObject
	 */
	public void removeSelectionModel(EObject mofObject) {
		fSelectionModel.remove(mofObject);
	}
	
	

}
