/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.util.*;

import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.internal.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.*;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.eclipse.hyades.ui.internal.util.*;

/*
* CONTEXT_ID pfpp0001 for Trace Filters table in Profile Filters ui page
*/
/*
* CONTEXT_ID pfpp0002 for New button in Profile Filters ui page
*/
/*
* CONTEXT_ID pfpp0003 for Edit button in Profile Filters ui page
*/
/*
* CONTEXT_ID pfpp0004 for Delete button in Profile Filters ui page
*/
/*
* CONTEXT_ID pfpp0005 for Up button in Profile Filters ui page
*/
/*
* CONTEXT_ID pfpp0006 for Down button in Profile Filters ui page
*/
/*
* CONTEXT_ID pfpp0007 for Package entry field in Profile Filters add/edit dialog page
*/
/*
* CONTEXT_ID pfpp0008 for Method Name entry field in Profile Filters for Package entry field in Profile Filters add/edit dialog page
*/
/*
* CONTEXT_ID pfpp0009 for Rule combobox in Profile Filters for Package entry field in Profile Filters add/edit dialog page
*/

/*
* CONTEXT_ID pfps0001 for Trace Filters Set table in Profile Filters ui page
*/

/*
* CONTEXT_ID pfps0002 for New Set button in Profile Filters ui page
*/
/*
* CONTEXT_ID pfps0003 for Edit Set button in Profile Filters ui page
*/
/*
* CONTEXT_ID pfps0004 for Delete Set button in Profile Filters ui page
*/

public class TraceProfileFiltersUI extends AbstractChangeable implements SelectionListener, ICheckStateListener {
	private Composite _result;

	protected FilterList _filterList;
	protected FilterSetList _filterSetList;

	private Button _upBtn;
	private Button _downBtn;

	private Button _addBtn;
	private Button _editBtn;
	private Button _removeBtn;

	protected Table _table;
	protected CheckboxTableViewer _tableViewer;
	private static String[] tableColumnProperties =
		{
			FilterTableElement.P_TEXT,
			FilterTableElement.P_METHOD,
			FilterTableElement.P_VISIBILITY };

	private Table _filterSetTable;
	private CheckboxTableViewer _filterSetTableViewer;
	private Button _addSetBtn;
	private Button _editSetBtn;
	private Button _removeSetBtn;

	private Label _contentLabel;
	private FilterSetElement _oldSelection;
	
	private String activeFS;

	private boolean agentTerminated = false;
	private static final String OPEN_FILTERS_TAG = "<filters>";
	private static final String CLOSE_FILTERS_TAG  = "</filters>";
	
	private static final String OPEN_FILTER_TAG = "<filter";
	private static final String CLOSE_FILTER_TAG  = "</filter>";
	
	private static final String FILTER_NAME_ATTRIBUTE="name";
	private static final String FILTER_ENABLED_ATTRIBUTE="enabled";
	
	private static final String OPEN_CONTENTS_TAG = "<contents>";
	private static final String CLOSE_CONTENTS_TAG = "</contents>";
	
	private static final String OPEN_CONTENT_TAG = "<content";
	private static final String CLOSE_CONTENT_TAG = "/>";
	
	private static final String CONTENT_TEXT_ATTRIBUTE="text";
	private static final String CONTENT_METHOD_ATTRIBUTE="method";
	private static final String CONTENT_VISIBILITY_ATTRIBUTE="visibility";


	public class FilterSetLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {

		public FilterSetLabelProvider() {
			super();
		}
		public Image getColumnImage(Object element, int columnIndex) {
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
		public Image getColumnImage(
			Viewer viewer,
			Object element,
			int columnIndex) {
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
		public String getColumnText(Object element, int columnIndex) {
			return ((FilterSetElement) element).getName();
		}
		public String getColumnText(
			Viewer viewer,
			Object element,
			int columnIndex) {
			return ((FilterSetElement) element).getName();
		}
	}

	class FilterSetContentProvider implements IStructuredContentProvider {
		private CheckboxTableViewer fViewer;
		protected FilterSetElement fChangedElement;

		/**
		 * The constructor.
		 */
		public FilterSetContentProvider(CheckboxTableViewer tableViewer) {
			fViewer = tableViewer;
		}
		/**
		 * The visual part that is using this content provider is about
		 * to be disposed. Deallocate all allocated SWT resources.
		 */
		public void dispose() {
		}
		/**
		 * 
		 */
		public void elementChanged(
			FilterSetElement changedElement,
			String changedProperty) {
			fChangedElement = changedElement;
			fViewer.getControl().getDisplay().asyncExec(new Runnable() {
				public void run() {

					fViewer.refresh(fChangedElement);
				}
			});

		}
		/**
		 * Returns all the markers that should be shown for
		 * the current settings.
		 */

		public Object[] getElements(Object parent) {
			return ((FilterSetList) parent).elements();
		}
		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
			fViewer = (CheckboxTableViewer) viewer;

			// Disconnect from old input.
			if (oldInput != null) {
				if (oldInput instanceof FilterList) {
					((FilterSetList) oldInput).removeElementChangedListener(
						this);

				}
			}

			// Connect to new input.
			if (newInput != null) {
				if (newInput instanceof FilterList)
					((FilterSetList) newInput).addElementChangedListener(this);
			}

		}
		public boolean isDeleted(Object element) {
			return false;
		}
	}

	public class FilterSetList implements IBasicPropertyConstants {
		ArrayList filters = new ArrayList();
		TableViewer fViewer;

		private ArrayList listeners;

		public FilterSetList(TableViewer viewer) {
			fViewer = viewer;
		}
		public void add(FilterSetElement element) {
			add(element, -1);
		}

		public void add(FilterSetElement element, int index) {
			if (index < 0)
				filters.add(element);
			else
				filters.add(index, element);
		}
		/**
		 * 
		 */
		public void addElementChangedListener(FilterSetContentProvider newListener) {

			if (newListener == null)
				return;
			if (listeners == null)
				listeners = new ArrayList();
			if (!listeners.contains(newListener))
				listeners.add(newListener);

		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/23/2000 11:34:27 AM)
		 * @return java.util.Enumeration
		 */
		public Object[] elements() {
			return filters.toArray();
		}
		/**
		 * 
		 */
		public void fireElementChanged(
			FilterSetElement changedElement,
			String changedProperty) {
			// forward change to content providers.
			if (listeners != null) {
				Iterator e = listeners.iterator();
				while (e.hasNext()) {
					FilterSetContentProvider l =
						(FilterSetContentProvider) e.next();
					l.elementChanged(changedElement, changedProperty);
				}
			}
		}
		public Object getElementAt(int index) {
			return filters.get(index);
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (11/13/2000 12:53:52 PM)
		 * @return java.util.Vector
		 */
		public ArrayList getFilters() {
			return (ArrayList) filters.clone();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (09/18/2000 4:02:44 PM)
		 * @return int
		 */
		public int getSize() {
			return filters.size();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/22/2000 4:53:57 PM)
		 */
		public void removeAll() {
			filters.clear();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/22/2000 3:48:34 PM)
		 * @param idx int
		 */
		public Object removeElementAt(int idx) {
			Object element = filters.get(idx);

			if (element == null)
				return null;

			filters.remove(element);

			return element;
		}
		/**
		 * Removes a listener for selection changes.
		 */
		public void removeElementChangedListener(FilterSetContentProvider listener) {

			if (listeners != null)
				listeners.remove(listener);
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/23/2000 11:34:27 AM)
		 * @return java.util.Enumeration
		 */
		public Object[] uiElements() {
			return filters.toArray();
		}
	}

	class EditDialog extends Dialog implements ModifyListener {
		private String _class = "mypackage.*";
		private String _method;
		private String _rule;
		private Text _classText;
		private Text _methodText;
		private Combo _ruleText;
		private String _title;

		public EditDialog(
			Shell shell,
			String title,
			String clas,
			String method,
			String rule) {
			super(shell);

			if (clas != null)
				_class = clas;
			_method = method;
			_rule = rule;
			_title = title;
		}

		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(_title);
		}

		public void clas(String clas) {
			_class = clas;
		}

		public String clas() {
			return _class;
		}

		public void method(String method) {
			_method = method;
		}

		public String method() {
			return _method;
		}

		public void rule(String rule) {
			_rule = rule;
		}

		public String rule() {
			return _rule;
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			Label label;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			data.widthHint = 400;
			result.setLayoutData(data);

			label = new Label(result, SWT.NULL);
			label.setText(UIPlugin.getResourceString("HEADER_FILTER"));
			_classText = new Text(result, SWT.BORDER);
			_classText.setLayoutData(GridUtil.createHorizontalFill());

			label = new Label(result, SWT.NULL);
			label.setText(UIPlugin.getResourceString("HEADER_METHOD"));
			_methodText = new Text(result, SWT.BORDER);
			_methodText.setLayoutData(GridUtil.createHorizontalFill());

			label = new Label(result, SWT.NULL);
			label.setText(UIPlugin.getResourceString("HEADER_VISIBILITY"));
			_ruleText = new Combo(result, SWT.BORDER | SWT.READ_ONLY);
			_ruleText.setLayoutData(GridUtil.createHorizontalFill());

			_ruleText.add(UIPlugin.getResourceString("STR_EXCLUDE"));
			_ruleText.add(UIPlugin.getResourceString("STR_INCLUDE"));

			_classText.addModifyListener(this);
			_methodText.addModifyListener(this);
			_ruleText.select(1);

			if (_class != null)
				_classText.setText(_class);
			if (_method != null)
				_methodText.setText(_method);
			if (_rule != null)
				_ruleText.setText(_rule);

			_classText.setFocus();
			_classText.selectAll();
/*
			org.eclipse.ui.help.WorkbenchHelp.setHelp(
				_classText,
				UIPlugin.getPluginId() + ".pfpp0007");
			org.eclipse.ui.help.WorkbenchHelp.setHelp(
				_methodText,
				UIPlugin.getPluginId() + ".pfpp0008");
			org.eclipse.ui.help.WorkbenchHelp.setHelp(
				_ruleText,
				UIPlugin.getPluginId() + ".pfpp0009");
*/
			return result;
		}

		public void modifyText(ModifyEvent e) {
			if (e.widget == _classText || e.widget == _methodText) {
				if (getButton(IDialogConstants.OK_ID) != null)
					getButton(IDialogConstants.OK_ID).setEnabled(
						_classText.getText().trim() != ""
							|| _methodText.getText().trim() != "");
			}
		}

		protected void okPressed() {

			_class = _classText.getText().trim();
			_method = _methodText.getText().trim();
			_rule = _ruleText.getText();
			super.okPressed();
		}

	}

	/*
	 * 
	 */
	class EditWSetDialog extends Dialog implements ModifyListener {
		private String _name;
		private Text _nameText;
		private String _title;

		public EditWSetDialog(Shell shell, String title, String name) {
			super(shell);

			_name = name;
			_title = title;
		}

		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(_title);
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			Label label;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			data.widthHint = 400;
			result.setLayoutData(data);

			label = new Label(result, SWT.NULL);
			label.setText(UIPlugin.getResourceString("STR_FILTER_SET_NAME"));
			_nameText = new Text(result, SWT.BORDER);
			_nameText.setLayoutData(GridUtil.createHorizontalFill());

			_nameText.addModifyListener(this);

			if (_name != null)
				_nameText.setText(_name);
/*
			org.eclipse.ui.help.WorkbenchHelp.setHelp(
				_nameText,
				UIPlugin.getPluginId() + ".pfpp0007");
*/
			return result;
		}

		public void modifyText(ModifyEvent e) {
			if (e.widget == _nameText) {
				if (getButton(IDialogConstants.OK_ID) != null)
					getButton(IDialogConstants.OK_ID).setEnabled(
						_nameText.getText().trim() != "");
			}
		}

		protected void okPressed() {

			_name = _nameText.getText().trim();
			super.okPressed();
		}

	}

	public TraceProfileFiltersUI() {
		super();
	}

	private void createColumns() {
		TableLayout layout = new TableLayout();
		_table.setLayout(layout);
		_table.setHeaderVisible(true);

		String headers[] =
			{
				UIPlugin.getResourceString("HEADER_FILTER"),
				UIPlugin.getResourceString("HEADER_METHOD"),
				UIPlugin.getResourceString("HEADER_VISIBILITY")};
		ColumnLayoutData layouts[] =
			{
				new ColumnPixelData(120, true),
				new ColumnPixelData(110, true),
				new ColumnPixelData(70, true)};

		final TableColumn columns[] = new TableColumn[headers.length];
		int ncolumns = 0;

		for (int i = 0; i < headers.length; i++) {
			layout.addColumnData(layouts[i]);
			TableColumn tc = new TableColumn(_table, SWT.NONE, i);
			tc.setResizable(layouts[i].resizable);
			tc.setText(headers[i]);
			columns[i] = tc;
		}
	}
	public Composite createControl(Composite parent) {
		GridLayout layout;

		_result = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 0;
		_result.setLayout(layout);
		GridData data = GridUtil.createFill();
		_result.setLayoutData(data);

		createFilterSet(_result);
		createFilterContents(_result);

		Label label = new Label(_result, SWT.NONE);
		label.setText(UIPlugin.getResourceString("STR_FILTER_NOTE"));

		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);

		return _result;
	}

	/*
	 * 
	 */
	private void createFilterContents(Composite parent) {
		GridData data;
		GridLayout layout;

		Label label = new Label(parent, SWT.NULL);
		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);

		_contentLabel = new Label(parent, SWT.NULL);
		_contentLabel.setText(
			UIPlugin.getResourceString("STR_FILTER_CONTENTS"));
		data = new GridData();
		data.horizontalSpan = 2;
		_contentLabel.setLayoutData(data);

		Composite tableGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 180;
		tableGroup.setLayoutData(data);
		initializeTable(tableGroup);

		Composite btnGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addBtn = new Button(addGroup, SWT.PUSH);
		_addBtn.setText(UIPlugin.getResourceString("ADD_FILTER_TEXT1"));
		_addBtn.setToolTipText(
			UIPlugin.getResourceString("ADD_FILTER_TOOLTIP_TEXT"));
		_addBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn = new Button(addGroup, SWT.PUSH);
		_editBtn.setText(UIPlugin.getResourceString("STR_EDIT_FILTER_BTN1"));
		_editBtn.setToolTipText(
			UIPlugin.getResourceString("EDIT_FILTER_TOOLTIP_TEXT"));
		_editBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeBtn = new Button(addGroup, SWT.PUSH);
		_removeBtn.setText(UIPlugin.getResourceString("REMOVE_TEXT1"));
		_removeBtn.setToolTipText(
			UIPlugin.getResourceString("REMOVE_TOOLTIP_TEXT"));
		_removeBtn.setLayoutData(GridUtil.createHorizontalFill());

		Composite moveGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		moveGroup.setLayout(layout);
		moveGroup.setLayoutData(GridUtil.createHorizontalFill());

		_upBtn = new Button(moveGroup, SWT.PUSH);
		_upBtn.setText(UIPlugin.getResourceString("MOVEUP_TEXT"));
		_upBtn.setToolTipText(
			UIPlugin.getResourceString("MOVEUP_TOOLTIP_TEXT"));
		_upBtn.setLayoutData(GridUtil.createHorizontalFill());

		_downBtn = new Button(moveGroup, SWT.PUSH);
		_downBtn.setText(UIPlugin.getResourceString("MOVEDOWN_TEXT"));
		_downBtn.setToolTipText(
			UIPlugin.getResourceString("MOVEDOWN_TOOLTIP_TEXT"));
		_downBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn.setEnabled(false);
		_removeBtn.setEnabled(false);
		_upBtn.setEnabled(false);
		_downBtn.setEnabled(false);

		//add listeners
		_removeBtn.addSelectionListener(this);
		_upBtn.addSelectionListener(this);
		_downBtn.addSelectionListener(this);
		_addBtn.addSelectionListener(this);
		_editBtn.addSelectionListener(this);
		_table.addSelectionListener(this);

		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_table,
			UIPlugin.getPluginId() + ".pfpp0002");
			/*
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_addBtn,
			UIPlugin.getPluginId() + ".pfpp0002");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_editBtn,
			UIPlugin.getPluginId() + ".pfpp0003");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_removeBtn,
			UIPlugin.getPluginId() + ".pfpp0004");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_upBtn,
			UIPlugin.getPluginId() + ".pfpp0005");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_downBtn,
			UIPlugin.getPluginId() + ".pfpp0006");
*/
	}

	/*
	 * 
	 */
	private void createFilterSet(Composite parent) {
		GridData data;
		GridLayout layout;

		Label label = new Label(parent, SWT.NULL);
		label.setText(UIPlugin.getResourceString("STR_SELECT_FILTER_SET"));
		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);

		Composite tableGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 100;

		tableGroup.setLayoutData(data);
		_filterSetTable =
			new Table(
				tableGroup,
				SWT.V_SCROLL
					| SWT.BORDER
					| SWT.SINGLE
					| SWT.FULL_SELECTION
					| SWT.CHECK);
		_filterSetTable.setLinesVisible(false);
		_filterSetTable.setLayoutData(GridUtil.createFill());

		_filterSetTableViewer = new CheckboxTableViewer(_filterSetTable);
		_filterSetTableViewer.setContentProvider(
			new FilterSetContentProvider(_filterSetTableViewer));
		_filterSetTableViewer.setLabelProvider(new FilterSetLabelProvider());

		_filterSetList = new FilterSetList(_filterSetTableViewer);
		_filterSetTableViewer.setInput(_filterSetList);
		_filterSetTableViewer.addCheckStateListener(this);

		TableLayout layoutt = new TableLayout();
		_filterSetTable.setLayout(layoutt);

		ColumnLayoutData l = new ColumnPixelData(420, true);
		layoutt.addColumnData(l);

		TableColumn tc = new TableColumn(_filterSetTable, SWT.NONE, 0);
		tc.setResizable(l.resizable);

		Composite btnGroup = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addSetBtn = new Button(addGroup, SWT.PUSH);
		_addSetBtn.setText(UIPlugin.getResourceString("ADD_FILTER_TEXT"));
		_addSetBtn.setToolTipText(
			UIPlugin.getResourceString("ADD_FILTER_TOOLTIP_TEXT"));
		_addSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editSetBtn = new Button(addGroup, SWT.PUSH);
		_editSetBtn.setText(UIPlugin.getResourceString("STR_EDIT_SET_BTN"));
		_editSetBtn.setToolTipText(
			UIPlugin.getResourceString("EDIT_SET_TOOLTIP_TEXT"));
		_editSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeSetBtn = new Button(addGroup, SWT.PUSH);
		_removeSetBtn.setText(UIPlugin.getResourceString("REMOVE_TEXT"));
		_removeSetBtn.setToolTipText(
			UIPlugin.getResourceString("REMOVE_TOOLTIP_TEXT"));
		_removeSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editSetBtn.setEnabled(false);
		_removeSetBtn.setEnabled(false);

		//add listeners
		_removeSetBtn.addSelectionListener(this);
		_addSetBtn.addSelectionListener(this);
		_editSetBtn.addSelectionListener(this);
		_filterSetTable.addSelectionListener(this);

		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_filterSetTable,
			UIPlugin.getPluginId() + ".pfps0001");
			/*
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_addSetBtn,
			UIPlugin.getPluginId() + ".pfps0002");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_editSetBtn,
			UIPlugin.getPluginId() + ".pfps0003");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_removeSetBtn,
			UIPlugin.getPluginId() + ".pfps0004");
*/
	}

	/**
	 * createPartControl method comment.
	 */
	private void createTable(Composite parent) {
		_table =
			new Table(
				parent,
				SWT.H_SCROLL
					| SWT.V_SCROLL
					| SWT.BORDER
					| SWT.MULTI
					| SWT.FULL_SELECTION);
		_table.setLinesVisible(true);
		_table.setLayoutData(GridUtil.createFill());

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/22/2000 10:16:27 AM)
	 */
	public void enable(boolean enable) {
		_addSetBtn.setEnabled(enable);
		_removeSetBtn.setEnabled(enable);
		_editSetBtn.setEnabled(enable);

		_filterSetTable.setEnabled(enable);
		_filterSetTableViewer.getTable().setEnabled(enable);

		_addBtn.setEnabled(enable);
		_removeBtn.setEnabled(enable);
		_editBtn.setEnabled(enable);
		_upBtn.setEnabled(enable);
		_downBtn.setEnabled(enable);

		if (!enable) {
			_tableViewer.setCellEditors(null);
			_tableViewer.setCellModifier(null);
		}

	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/22/2000 10:16:27 AM)
	 */
	protected void enableButtons() {
		if (isAgentTerminated()) {
			enable(false);
			return;
		}

		int selection = _table.getSelectionIndex();
		int selectionCount = _table.getSelectionCount();

		_removeBtn.setEnabled(_filterList.getSize() > 0 && selection != -1);
		_editBtn.setEnabled(_filterList.getSize() > 0 && selection != -1);

		_upBtn.setEnabled(selection > 0 && selectionCount == 1);
		_downBtn.setEnabled(
			selection >= 0
				&& selection < _table.getItemCount() - 1
				&& selectionCount == 1);

		selection = _filterSetTable.getSelectionIndex();

		_removeSetBtn.setEnabled(
			_filterSetList.getSize() > 1 && selection != -1);
		_editSetBtn.setEnabled(_filterSetList.getSize() > 0 && selection != -1);

	}
	public ArrayList getFilterSet() {
		if (_oldSelection != null) {
			_oldSelection.setChildren(
				(ArrayList) _filterList.getFilters().clone());
		}

		validateFilterSet(_filterSetList);
		return _filterSetList.getFilters();
	}
	
	/**
	 * Example:
	 * <filter name = enabled = >
	 * 	<contents>
	 * 		<content text = method = visibility = />
	 * 		....
	 * </contents>
	 * ....
	 * </filter>
	 * 
	 */
	public static String getFilterSet(ArrayList list) {
		StringBuffer temp = new StringBuffer();
		for (int i = 0; i < list.size(); i++)
		{
			temp.append(OPEN_FILTER_TAG);
			FilterSetElement elem = (FilterSetElement)list.get(i);
			temp.append(" " + FILTER_NAME_ATTRIBUTE + " = '" + elem.getName() + "'>");
			
			ArrayList children = elem.getChildren();
			temp.append(OPEN_CONTENTS_TAG);
			for (int j = 0; j < children.size(); j++)
			{
				temp.append(OPEN_CONTENT_TAG);
				FilterTableElement childElem = (FilterTableElement)children.get(j);
				temp.append(" " + CONTENT_TEXT_ATTRIBUTE + " = '" + childElem.getText() + "'");
				temp.append(" " + CONTENT_METHOD_ATTRIBUTE + " = '" + childElem.getMethod() + "'");
				temp.append(" " + CONTENT_VISIBILITY_ATTRIBUTE + " = '" + childElem.getVisibility() + "'");
				temp.append(CLOSE_CONTENT_TAG);
			}

			temp.append(CLOSE_CONTENTS_TAG);
			temp.append(CLOSE_FILTER_TAG);
			
			
		}
		return temp.toString();
	}

	/**
	 * Validates the filter set; add an EXCLUDE * filter is the 
	 * filter set contains only INCLUDE filter 
	 * @param filterSet
	 */
	private void validateFilterSet(FilterSetList filterSet) {
		boolean hasIncludeFilterOnly = true;
		ArrayList filters = filterSet.getFilters();

		for (int idx = 0; idx < filters.size(); idx++) {
			FilterSetElement set = (FilterSetElement) filters.get(idx);
			ArrayList elements = set.getChildren();
			boolean addExcludeFilter = true;
			for (int i = 0; i < elements.size(); i++) {
				FilterTableElement elem = (FilterTableElement) elements.get(i);
				if (elem
					.getVisibility()
					.equals(UIPlugin.getResourceString("STR_EXCLUDE"))) {
					addExcludeFilter = false;
					break; //the filter set is valid
				} else if (elem.getText().equals("*")) {
					addExcludeFilter = false;
					break; //the filter set is valid
				}
			}

			if (elements.size() > 0 && addExcludeFilter) {
				//the set has only include filters, add exclude * to exclude any other data
				elements.add(
					new FilterTableElement(
						"*",
						"*",
						UIPlugin.getResourceString("STR_EXCLUDE")));
				set.setChildren(elements);
			}
		}

	}


	/**
	 * Insert the method's description here.
	 * Creation date: (11/13/2000 1:19:41 PM)
	 * @return boolean
	 */
	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:12:31 PM)
	 */
	protected void initializeTable(Composite parent) {
		createTable(parent);
		_tableViewer = new CheckboxTableViewer(_table);
		createColumns();

		_tableViewer.setContentProvider(
			new FilterListContentProvider(_tableViewer));
		_tableViewer.setLabelProvider(new FilterListLabelProvider());

		_filterList = new FilterList(_tableViewer);
		_tableViewer.setInput(_filterList);

		CellEditor editors[] = new CellEditor[_table.getColumnCount()];

		editors[0] = new TextCellEditor(_table);
		editors[1] = new TextCellEditor(_table);
		editors[2] =
			new ComboBoxCellEditor(
				_table,
				FilterTableElement.getVisibilityStates());

		_tableViewer.setCellEditors(editors);

		FilterElementModifier modifier =
			new FilterElementModifier(_tableViewer);
		_tableViewer.setCellModifier(modifier);
		_tableViewer.setColumnProperties(tableColumnProperties);
	}
	
	public static ArrayList getFiltersSet(String filters){
		Element prefElement = XMLUtil.loadDom(filters, "filters");
		return getFiltersSet(prefElement);
	}
	
	public static ArrayList getFiltersSet(Element filters)
	{
		//expects the filters element
		ArrayList list = new ArrayList();
		
		//list of filter elements
		NodeList nodeList = filters.getChildNodes();
		ArrayList children;
		
		for (int i = 0; i < nodeList.getLength(); i++)
		{
			Element elem = (Element)nodeList.item(i);
			FilterSetElement fse = new FilterSetElement(elem.getAttribute("name"));
			
			//set children
			children = getFilter(elem);
			fse.setChildren(children);

			list.add(fse);			
		}
		return list;
	}

	private void setXMLFilterSet(Element filters)
	{
		//expects the filters element
		//which contains the nodes
			//filter
				//contents
		ArrayList list = getFiltersSet(filters);
		for (int idx = 0; idx < list.size(); idx++)
			addFilterSet((FilterSetElement) list.get(idx));
		
		selectTableItem(_filterSetTable, 0);
		
		
	
		
	}
	
	public void setFilterSet(String filters)
	{
		Element prefElement = XMLUtil.loadDom(filters, "filters");
		setXMLFilterSet(prefElement);
	}
	
	public void setFilterSet(Element filters)
	{
		setXMLFilterSet(filters);
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:12:31 PM)
	 */
	private void addFilterSet(String name, ArrayList children) {
		FilterSetElement elem = new FilterSetElement(name);
		addFilterSet(elem);
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:12:31 PM)
	 */
	public void addFilterSet(FilterSetElement elem) {
		_filterSetList.add(elem);
		_filterSetTableViewer.refresh();
		_filterSetTableViewer.setChecked(elem, false);
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (4/30/2001 1:12:16 PM)
	 */
	public void reset() {
		activeFS = null;
		_oldSelection = null;

		_table.removeAll();
		_filterList.removeAll();
		_filterSetList.removeAll();
		_filterSetTableViewer.refresh();
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/22/2000 10:27:08 AM)
	 * @param selection int
	 */
	protected void selectTableItem(Table table, int selection) {
		table.setFocus();
		table.setSelection(selection);
		table.showSelection();

		if (table == _filterSetTable) {
			showFiltersForSelection(
				(FilterSetElement) _filterSetList.getElementAt(selection));
		}
	}

	/*
	 * 
	 */
	private void showFiltersForSelection(FilterSetElement elem) {
		if (_oldSelection != null) {
			_oldSelection.setChildren(
				(ArrayList) _filterList.getFilters().clone());
		}

		_table.setRedraw(false);

		_filterList.removeAll();
		ArrayList list = elem.getChildren();
		for (int idx = 0; idx < list.size(); idx++) {
			FilterTableElement filter = (FilterTableElement) list.get(idx);
			filter.setParent(_filterList);
			_filterList.add(filter, _filterList.getSize());
			_tableViewer.refresh();
		}

		_tableViewer.refresh();
		_oldSelection = elem;

		_table.setRedraw(true);
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:49:33 PM)
	 * @param event com.ibm.swt.events.SelectionEvent
	 */
	public void widgetDefaultSelected(SelectionEvent event) {
	}
	/**
	 * Insert the method's description here.
	 * Creation date: (08/21/2000 7:49:33 PM)
	 * @param event com.ibm.swt.events.SelectionEvent
	 */
	public void widgetSelected(SelectionEvent event) {
		//  System.out.println("widgetSelected " + event);
		if (event.widget == _addSetBtn) {
			showWSetDialog(UIPlugin.getResourceString("STR_ADD_SET_DLG"), null);
			enableButtons();
			return;

		} else if (event.widget == _editSetBtn) {
			int idx = _filterSetTable.getSelectionIndex();
			if (idx != -1) {
				showWSetDialog(
					UIPlugin.getResourceString("STR_EDIT_SET_DLG"),
					_filterSetTable.getItem(idx).getText());
			}

			return;
		} else if (event.widget == _removeSetBtn) {
			int selection = _filterSetTable.getSelectionIndex();
			if (selection != -1) {
				boolean select = false;
				if (_filterSetTable.getItem(selection).getChecked())
					select = true;

				_filterSetList.removeElementAt(selection);
				if (select) {
					FilterSetElement elem =
						(FilterSetElement) _filterSetList.getElementAt(0);
					activeFS = elem.getName();
					_filterSetTableViewer.setChecked(elem, true);
				}
				_filterSetTableViewer.refresh();

				selectTableItem(_filterSetTable, 0);
			}

			enableButtons();
			return;
		} else if (event.widget == _addBtn) {
			showEditDialog(
				UIPlugin.getResourceString("STR_ADD_FILTER_DLG"),
				null,
				"*",
				null);
			enableButtons();
		} else if (event.widget == _editBtn) {
			int idx = _table.getSelectionIndex();
			if (idx != -1) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(idx);

				String filter = sel.getText();
				String method = sel.getMethod();
				String vis = sel.getVisibility();

				showEditDialog(
					UIPlugin.getResourceString("STR_EDIT_FILTER_DLG"),
					filter,
					method,
					vis);
			}
		} else if (event.widget == _table) {
			enableButtons();
			return;
		} else if (event.widget == _filterSetTable) {
			int idx = _filterSetTable.getSelectionIndex();

			if (idx != -1) {

				// Set corresponding checkbox.
				FilterSetElement elem1 =
					(FilterSetElement) _filterSetList.getElementAt(idx);
				showFiltersForSelection(elem1);
				_filterSetTableViewer.setChecked(elem1, true);
				activeFS = elem1.getName();

				// Clear all other checkboxes.
				for (int kdx = 0; kdx < _filterSetList.getSize(); kdx++) {
					if (kdx != idx) {
						FilterSetElement elem2 =
							(FilterSetElement) _filterSetList.getElementAt(kdx);
						_filterSetTableViewer.setChecked(elem2, false);
					}
				}
				_filterSetTableViewer.refresh();

			}

			enableButtons();
			return;
		} else if (event.widget == _removeBtn) {
			int selection = _table.getSelectionIndex();

			_filterList.removeElements(_table.getSelectionIndices());

			_tableViewer.refresh();

			int size = _filterList.getSize();
			if (size > 0) {
				if (selection > 0 && selection < size) {
					selectTableItem(_table, selection - 1);
				} else {
					selectTableItem(_table, 0);
				}
			}
			enableButtons();
		} else if (event.widget == _upBtn) {
			_tableViewer.refresh();

			int selection = _table.getSelectionIndex();
			if (selection > 0) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(selection);
				FilterTableElement previous =
					(FilterTableElement) _filterList.getElementAt(
						selection - 1);

				String filter = sel.getText();
				String method = sel.getMethod();
				String vis = sel.getVisibility();

				sel.setText(previous.getText());
				sel.setMethod(previous.getMethod());
				previous.setText(filter);
				previous.setMethod(method);

				sel.setVisibility(previous.getVisibility());
				previous.setVisibility(vis);

				selectTableItem(_table, selection - 1);

			}

			_tableViewer.refresh();
			enableButtons();
		} else if (event.widget == _downBtn) {
			_tableViewer.refresh();

			int selection = _table.getSelectionIndex();
			if (selection < _table.getItemCount() - 1) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(selection);
				FilterTableElement next =
					(FilterTableElement) _filterList.getElementAt(
						selection + 1);

				String filter = sel.getText();
				String method = sel.getMethod();
				String vis = sel.getVisibility();

				sel.setText(next.getText());
				sel.setMethod(next.getMethod());
				next.setText(filter);
				next.setMethod(method);
				sel.setVisibility(next.getVisibility());
				next.setVisibility(vis);

				selectTableItem(_table, selection + 1);

			}

			_tableViewer.refresh();
			enableButtons();
		}
	}

	private void showEditDialog(
		String title,
		String clas,
		String method,
		String rule) {

		EditDialog dialog =
			new EditDialog(_result.getShell(), title, clas, method, rule);
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {

			if (clas == null) {

				//add filter
				FilterTableElement elem =
					new FilterTableElement(
						dialog.clas(),
						dialog.method(),
						dialog.rule());

				if (dialog
					.rule()
					.equals(UIPlugin.getResourceString("STR_EXCLUDE")))
					_filterList.add(elem, _filterList.getSize());
				else
					_filterList.add(elem, 0);

				elem.setParent(_filterList);

				_tableViewer.refresh();

				if (dialog
					.rule()
					.equals(UIPlugin.getResourceString("STR_EXCLUDE")))
					selectTableItem(_table, _filterList.getSize() - 1);
				else
					selectTableItem(_table, 0);
			} else {
				int idx = _table.getSelectionIndex();
				if (idx != -1) {

					FilterTableElement sel =
						(FilterTableElement) _filterList.getElementAt(idx);

					sel.setText(dialog.clas());
					sel.setMethod(dialog.method());
					sel.setVisibility(dialog.rule());

					_tableViewer.refresh();

				}

			}
		}

	}

	private void showWSetDialog(String title, String name) {

		EditWSetDialog dialog =
			new EditWSetDialog(_result.getShell(), title, name);
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {

			if (name == null) {

				for (int idx = 0; idx < _filterSetList.getSize(); idx++) {
					FilterSetElement el =
						(FilterSetElement) _filterSetList.getElementAt(idx);
					if (el.getName().equals(dialog._name)) {
						selectTableItem(_filterSetTable, idx);
						return;
					}

				}
				
				FilterSetElement elem =
					new FilterSetElement(dialog._name);
				String filters = TraceConstants.DEFAULT_FILTER_SET;
				Element prefElement = XMLUtil.loadDom(filters, "filters");
				ArrayList list = getFiltersSet(prefElement);
				if (list.size() > 0) {
					FilterSetElement set = (FilterSetElement) list.get(0);
					elem.setChildren((ArrayList) set.getChildren().clone());
				}
				addFilterSet(elem);
				selectTableItem(
					_filterSetTable,
					0);
			} else {
				int idx = _filterSetTable.getSelectionIndex();
				if (idx != -1)
					(
						(FilterSetElement) _filterSetList.getElementAt(
							idx)).setName(
						dialog._name);
				_filterSetTableViewer.refresh(
					((FilterSetElement) _filterSetList.getElementAt(idx)));
			}
		}

	}
	
	private static ArrayList getFilter(Element elem)
	{
		ArrayList list = new ArrayList();
		Element contents, temp;
		//contents tag
		contents = (Element)elem.getFirstChild();
		
		//not parse though content tags
		NodeList contentList = contents.getChildNodes();
		for (int i = 0; i < contentList.getLength(); i++)
		{
			FilterTableElement fte;
			temp = (Element)contentList.item(i);
			if ((temp.getAttribute(CONTENT_VISIBILITY_ATTRIBUTE).equals("0")) || (temp.getAttribute("visibility").equals("EXCLUDE"))) fte = new FilterTableElement(temp.getAttribute("text"), temp.getAttribute("method"), "EXCLUDE");
			else fte = new FilterTableElement(temp.getAttribute(CONTENT_TEXT_ATTRIBUTE), temp.getAttribute(CONTENT_METHOD_ATTRIBUTE), "INCLUDE");
			list.add(fte);
		}
		return list;
	}

	private static void getFilter(FilterSetElement elem, String filterList) {

		ArrayList list = new ArrayList();

		int idx = filterList.indexOf("#");
		while (idx != -1) {
			String filter = filterList.substring(0, idx);
			filterList = filterList.substring(idx + 1);

			idx = filter.indexOf("!");

			String text = filter.substring(0, idx);
			filter = filter.substring(idx + 1);

			idx = filter.indexOf("!");
			String method = filter.substring(0, idx);
			filter = filter.substring(idx + 1);

			String include =
				filter.equals("0")
					? UIPlugin.getResourceString("STR_EXCLUDE")
					: UIPlugin.getResourceString("STR_INCLUDE");

			FilterTableElement filterElem =
				new FilterTableElement(text, method, include);
			list.add(filterElem);

			idx = filterList.indexOf("#");
		}

		if (!filterList.equals("")) {
			idx = filterList.indexOf("!");

			String text = filterList.substring(0, idx);
			filterList = filterList.substring(idx + 1);

			idx = filterList.indexOf("!");
			String method = filterList.substring(0, idx);
			filterList = filterList.substring(idx + 1);

			String include =
				filterList.equals("0")
					? UIPlugin.getResourceString("STR_EXCLUDE")
					: UIPlugin.getResourceString("STR_INCLUDE");

			FilterTableElement filterElem =
				new FilterTableElement(text, method, include);
			list.add(filterElem);

		}

		elem.setChildren(list);

	}

	public void checkStateChanged(CheckStateChangedEvent event) {
		FilterSetElement elem = (FilterSetElement) event.getElement();
		//update any previous selected set  	 		
		for (int i = 0; i < _filterSetList.getSize(); i++) {
			FilterSetElement data =
				(FilterSetElement) _filterSetList.getElementAt(i);
			if (data == elem) {
				// Set corresponding selection.
				if (_filterSetTableViewer.getChecked(data))
					_filterSetTable.setSelection(i);
				continue;
			}
		}

		_filterSetTableViewer.refresh();
	}

	/**
		 * Stores the values of the controls back to the preference store.
		 */
	public static void storeValues(ArrayList items) {
		String filters = "";

		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		store.setValue(TraceConstants.FILTERS_SET_KEY, OPEN_FILTERS_TAG + TraceProfileFiltersUI.getFilterSet(items) + CLOSE_FILTERS_TAG);
	}

	/**
	 * Returns the agentTerminated.
	 * @return boolean
	 */
	public boolean isAgentTerminated() {
		return agentTerminated;
	}

	/**
	 * Sets the agentTerminated.
	 * @param agentTerminated The agentTerminated to set
	 */
	public void setAgentTerminated(boolean agentTerminated) {
		this.agentTerminated = agentTerminated;
	}

	/**
	 * removes a filter set
	 */
	public void removeFilterSet(int idx) {
		if (idx < 0 || idx > _filterSetList.getSize())
			return;
		_filterSetList.removeElementAt(idx);
	}

	/**
	 * used by ProfileOptionsPage
	 * @param elem
	 * @param children
	 */
	public void alterFilterSetChildren(
		FilterSetElement elem,
		ArrayList children) {
		elem.setChildren(children);
	}
	
	/*
	 * returns the active filter set
	 * used by ProfileFiltersTab
	 */
	public String getActiveFilterSet()
	{
		return activeFS;
	}
	
	public void setActiveFilterSet(FilterSetElement set) {
		activeFS = set.getName();
		_filterSetTableViewer.setChecked(set, true);
		_filterSetTableViewer.setSelection(new StructuredSelection(set));
		showFiltersForSelection(set);
		_filterSetTableViewer.refresh();
	}
	
	public void addFilterSets(FilterSetElement[] list)
	{
		for(int idx=0; idx<list.length; idx++)
			addFilterSet(list[idx]);
	}
}
