/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IDebugEventSetListener;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.jdt.internal.launching.LaunchingMessages;
import org.eclipse.jdt.internal.launching.LaunchingPlugin;
import org.eclipse.jdt.launching.AbstractJavaLaunchConfigurationDelegate;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.framework.msg.MessageFormat;

public class ProfileJavaAppletDelegate extends AbstractJavaLaunchConfigurationDelegate
		implements IDebugEventSetListener {
		
	/**
	 * Mapping of ILaunch objects to File objects that represent the .html file
	 * used to initiate the applet launch.  This is used to delete the .html
	 * file when the launch terminates.
	 */
	private static Map fgLaunchToFileMap = new HashMap();
	
	/**
	 * @see org.eclipse.debug.core.model.ILaunchConfigurationDelegate#launch(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String, org.eclipse.debug.core.ILaunch, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void launch(ILaunchConfiguration conf, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException {
			
		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		verifyMainTypeName(conf);

		File workingDir = verifyWorkingDirectory(conf);
		String workingDirName = workingDir.getAbsolutePath();

		TraceArguments args = new TraceArguments(conf.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_APPLETVIEWER_CLASS, IJavaLaunchConfigurationConstants.DEFAULT_APPLETVIEWER_CLASS));
		args.setClassPath(getClasspathString(conf));
		args.setVMArguments(getJavaPolicyFile(workingDir) + ' ' + getVMArguments(conf));

		// Construct the HTML file and set its name as a program argument
		File htmlFile = buildHTMLFile(conf, workingDir);
		args.setParameters(htmlFile.getName());
		args.setLocation(workingDirName);
		
		// Add a debug listener if necessary 
		if (fgLaunchToFileMap.isEmpty()) {
			DebugPlugin.getDefault().addDebugEventListener(this);
		}
		
		// Add a mapping of the launch to the html file 
		fgLaunchToFileMap.put(launch, htmlFile);
		setDefaultSourceLocator(launch, conf);

		String projectName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(TraceConstants.TRACE_PROJECT_NAME));
		String monitorName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(TraceConstants.TRACE_MONITOR_NAME));
		
		args.setHostName("localhost");
		args.setPortNumber(store.getInt(HyadesConstants.LOCALHOST_PORT));
		args.setAutoMonitoring(manager.getAutoMonitoring(conf));
		args.setEnvironmentVariable(new ArrayList());
		
		if (conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, false))
			args.setProfileFile(conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String)null));

		ArrayList filters = manager.getFilters(conf);
		Vector options = manager.getOptions(conf);

		PDCoreUtil.launchTrace(args, filters, options, projectName, monitorName, launch);
	}

	/**
	 * Returns the system property string for the policy file
	 * 
	 * @param workingDir the working directory
	 * @return system property for the policy file
	 */
	public String getJavaPolicyFile(File workingDir) {
			File file = new File(workingDir, "java.policy.applet");//$NON-NLS-1$ 
			if (!file.exists()) {
				// copy it to the working directory
				File test = LaunchingPlugin.getFileInPlugin(new Path("java.policy.applet")); //$NON-NLS-1$
				try {
					byte[] bytes = getFileByteContent(test);
					BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(file));
					outputStream.write(bytes);
					outputStream.close();
				} catch (IOException e) {
					return "";//$NON-NLS-1$
				}
			}
		return "-Djava.security.policy=java.policy.applet";//$NON-NLS-1$
	}

	/**
	 * Using the specified launch configuration, build an HTML file that specifies the
	 * applet to launch.  Return the name of the HTML file.
	 * 
	 * @param dir the directoru in which to make the file
	 */
	private File buildHTMLFile(ILaunchConfiguration configuration, File dir) {
		FileWriter writer = null;
		File tempFile = null;
		try {
			String name = getMainTypeName(configuration);
			tempFile = new File(dir, name + System.currentTimeMillis() + ".html"); //$NON-NLS-1$ //$NON-NLS-2$
			writer = new FileWriter(tempFile);
			writer.write("<html>\n"); //$NON-NLS-1$
			writer.write("<body>\n"); //$NON-NLS-1$
			writer.write("<applet code="); //$NON-NLS-1$
			writer.write(name);
			writer.write(".class "); //$NON-NLS-1$
			String appletName = configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_NAME, ""); //$NON-NLS-1$
			if (appletName.length() != 0) {
				writer.write("NAME =\"" + appletName + "\" "); //$NON-NLS-1$ //$NON-NLS-2$
			}
			writer.write("width=\""); //$NON-NLS-1$
			writer.write(Integer.toString(configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_WIDTH, 200))); //$NON-NLS-1$
			writer.write("\" height=\""); //$NON-NLS-1$
			writer.write(Integer.toString(configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_HEIGHT, 200))); //$NON-NLS-1$
			writer.write("\" >\n"); //$NON-NLS-1$
			Map parameters = configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_PARAMETERS, new HashMap());
			if (parameters.size() != 0) {
				Iterator iterator= parameters.entrySet().iterator();
				while(iterator.hasNext()) {
		 			Map.Entry next = (Map.Entry) iterator.next();
					writer.write("<param name="); //$NON-NLS-1$
					writer.write(getQuotedString((String)next.getKey()));
					writer.write(" value="); //$NON-NLS-1$
					writer.write(getQuotedString((String)next.getValue()));
					writer.write(">\n"); //$NON-NLS-1$
				}
			}
			writer.write("</applet>\n"); //$NON-NLS-1$
			writer.write("</body>\n"); //$NON-NLS-1$
			writer.write("</html>\n"); //$NON-NLS-1$
		} catch(IOException e) {
		} catch(CoreException e) {
		} finally {
			if (writer != null) {
				try {
					writer.close();
				} catch(IOException e) {
				}
			}
		}
		if (tempFile == null) {
			return null;
		}
		return tempFile;
	}
	
	private String getQuotedString(String string) {
		if (string.indexOf('"') == -1) {
			return '"' + string + '"';
		} else {
			return '\'' + string + '\'';
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IDebugEventSetListener#handleDebugEvents(org.eclipse.debug.core.DebugEvent[])
	 */
	public void handleDebugEvents(DebugEvent[] events) {
		for (int i = 0; i < events.length; i++) {
			DebugEvent event = events[i];
			Object eventSource = event.getSource();
			switch(event.getKind()) {
				
				// Delete the HTML file used for the launch
				case DebugEvent.TERMINATE :
					if (eventSource != null) {
						ILaunch launch = null;
						if (eventSource instanceof IProcess) {
							IProcess process = (IProcess) eventSource;
							launch = process.getLaunch();
						} else if (eventSource instanceof IDebugTarget) {
							IDebugTarget debugTarget = (IDebugTarget) eventSource;
							launch = debugTarget.getLaunch();
						}
						File temp = (File) fgLaunchToFileMap.get(launch);
						if (temp != null) {
							try {
								fgLaunchToFileMap.remove(launch);
								temp.delete();
							} finally {
								if (fgLaunchToFileMap.isEmpty()) {
									DebugPlugin.getDefault().removeDebugEventListener(this);
								}
							}
						}
					}
					break;
			}
		}
	}

	/**
	 * Returns the contents of the given file as a byte array.
	 * @throws IOException if a problem occured reading the file.
	 */
	protected static byte[] getFileByteContent(File file) throws IOException {
		InputStream stream = null;
		try {
			stream = new BufferedInputStream(new FileInputStream(file));
			return getInputStreamAsByteArray(stream, (int) file.length());
		} finally {
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e) {
				}
			}
		}
	}
	
	/**
	 * Returns the given input stream's contents as a byte array.
	 * If a length is specified (ie. if length != -1), only length bytes
	 * are returned. Otherwise all bytes in the stream are returned.
	 * Note this doesn't close the stream.
	 * @throws IOException if a problem occured reading the stream.
	 */
	protected static byte[] getInputStreamAsByteArray(InputStream stream, int length) throws IOException {
		byte[] contents;
		if (length == -1) {
			contents = new byte[0];
			int contentsLength = 0;
			int bytesRead = -1;
			do {
				int available = stream.available();

				// resize contents if needed
				if (contentsLength + available > contents.length) {
					System.arraycopy(
						contents,
						0,
						contents = new byte[contentsLength + available],
						0,
						contentsLength);
				}

				// read as many bytes as possible
				bytesRead = stream.read(contents, contentsLength, available);

				if (bytesRead > 0) {
					// remember length of contents
					contentsLength += bytesRead;
				}
			} while (bytesRead > 0);

			// resize contents if necessary
			if (contentsLength < contents.length) {
				System.arraycopy(
					contents,
					0,
					contents = new byte[contentsLength],
					0,
					contentsLength);
			}
		} else {
			contents = new byte[length];
			int len = 0;
			int readSize = 0;
			while ((readSize != -1) && (len != length)) {
				// See PR 1FMS89U
				// We record first the read size. In this case len is the actual read size.
				len += readSize;
				readSize = stream.read(contents, len, length - len);
			}
		}

		return contents;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jdt.launching.AbstractJavaLaunchConfigurationDelegate#verifyWorkingDirectory(org.eclipse.debug.core.ILaunchConfiguration)
	 */
	public File verifyWorkingDirectory(ILaunchConfiguration configuration) throws CoreException {
		IPath path = getWorkingDirectoryPath(configuration);
		if (path == null) {
			// default working dir for applets is the project's output directory
			String outputDir = JavaRuntime.getProjectOutputDirectory(configuration);
			if (outputDir == null) {
				// if no project attribute, default to eclipse directory
				return new File(System.getProperty("user.dir"));  //$NON-NLS-1$
			}
			IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(outputDir);
			if (resource == null || !resource.exists()) {
				//default to eclipse directory
				return new File(System.getProperty("user.dir"));  //$NON-NLS-1$
			}
			return resource.getLocation().toFile(); 
		} else {
			if (path.isAbsolute()) {
				File dir = new File(path.toOSString());
				if (dir.isDirectory()) {
					return dir;
				} else {
					abort(MessageFormat.format(LaunchingMessages.getString("AbstractJavaLaunchConfigurationDelegate.Working_directory_does_not_exist__{0}_12"), new String[] {path.toString()}), null, IJavaLaunchConfigurationConstants.ERR_WORKING_DIRECTORY_DOES_NOT_EXIST); //$NON-NLS-1$
				}
			} else {
				IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
				if (res instanceof IContainer && res.exists()) {
					return res.getLocation().toFile();
				} else {
					abort(MessageFormat.format(LaunchingMessages.getString("AbstractJavaLaunchConfigurationDelegate.Working_directory_does_not_exist__{0}_12"), new String[] {path.toString()}), null, IJavaLaunchConfigurationConstants.ERR_WORKING_DIRECTORY_DOES_NOT_EXIST); //$NON-NLS-1$
				}
			}
		}
		// cannot return null - an exception will be thrown
		return null;		
	}

	private String getClasspathString(ILaunchConfiguration conf) throws CoreException {
		String classPath = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_CLASSPATH, (String)null);

		if (classPath == null) {
			StringBuffer buf = new StringBuffer();
			String[] entries = getClasspath(conf);
			for (int i = 0; i < entries.length-1; ++i) {
				buf.append(entries[i]);
				buf.append(File.pathSeparatorChar);
			}
			buf.append(entries[entries.length-1]);
			return buf.toString();
		}
		else {
			return classPath;
		}
	}
}
