/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.extensions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.internal.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.launch.extensions.IDeploymentsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider;
import org.eclipse.hyades.test.ui.launch.extensions.IExecutionHistoryDefaultsProvider2;
import org.eclipse.hyades.test.ui.launch.extensions.IRunHandler;
import org.eclipse.hyades.test.ui.launch.extensions.ITestLaunchConfigurationValidator;

/**
 * This class is the central point for registering and getting extensions, based
 * on the type and the kind of the Launchable Element.
 * @author jcanches
 */
public class LaunchConfigurationExtensionsManager {

	private static LaunchConfigurationExtensionsManager instance;
	
	public static LaunchConfigurationExtensionsManager getInstance() {
		if (instance == null) {
			instance = new LaunchConfigurationExtensionsManager();
		}
		return instance;
	}
	
	private static final String KIND_TEST = "test"; //$NON-NLS-1$
	private static final String KIND_COMPONENT = "testComponent"; //$NON-NLS-1$
	
	private LaunchConfigurationExtensionsManager() {
		registerRunHandlers();
		registerDeploymentsProviders();
		registerExecutionHistoryDefaultsProviders();
		registerExecutionHistoryDefaults2Providers();
		registerSupportedTestTypes();
		registerLaunchConfigurationValidators();
	}
	
	private Map testExtensionMap = new HashMap();
	private Map componentExtensionMap = new HashMap();
	private Map launchConfigurationMap = new HashMap();
	
	private TestExtensionDescriptor getExtensionDescriptor(String type, String kind) {
		Assert.isNotNull(kind);
		Assert.isNotNull(type);
		
		Map map = null;
		if (kind.equals(KIND_TEST)) {
			map = testExtensionMap;
		} else if (kind.equals(KIND_COMPONENT)) {
			map = componentExtensionMap;
		} else {
			throw new IllegalArgumentException("Invalid kind");
		}
		Assert.isNotNull(map);
		
		Object o = map.get(type);
		if (o == null) {
			o = new TestExtensionDescriptor();
			map.put(type, o);
		}
		return (TestExtensionDescriptor)o;
	}
	
	private TestExtensionDescriptor getTestExtensionDescriptor(Object testElement) {
    	if (testElement != null) {
			if (testElement instanceof TPFTest) {
				TPFTest test = (TPFTest)testElement;
				if (test.getType() != null) {
					return getExtensionDescriptor(test.getType(), KIND_TEST);
				}
			} else if (testElement instanceof TPFTestComponent) {
				TPFTestComponent testComponent = (TPFTestComponent)testElement;
				if (testComponent.getType() != null) {
					return getExtensionDescriptor(testComponent.getType(), KIND_COMPONENT);
				}
			}
    	}
    	return null;
	}
	
	private class TestExtensionDescriptor {
		public IConfigurationElement runHandlerConfigElem = null;
		private IRunHandler runHandler = null;
		public List supportedLaunchModes = new ArrayList(3);
		public List availableShortcutModes = new ArrayList(3);
		public String launchConfigurationType = null;
		public IConfigurationElement execHistoryDefaultsProviderConfigElem = null;
		private IExecutionHistoryDefaultsProvider execHistoryDefaultsProvider = null;
		public IConfigurationElement deploymentsProviderConfigElem = null;
		private IDeploymentsProvider deploymentsProvider = null;
		public IConfigurationElement launchConfigurationValidatorConfigElem = null;
		private ITestLaunchConfigurationValidator launchConfigurationValidator = null;
		public IRunHandler getRunHandler() {
			if (this.runHandler == null) {
				if (this.runHandlerConfigElem != null) {
					try {
						this.runHandler = (IRunHandler)this.runHandlerConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.runHandlerConfigElem = null;
					}
				}
			}
			return this.runHandler;
		}
		public IExecutionHistoryDefaultsProvider getExecutionHistoryDefaultsProvider() {
			if (this.execHistoryDefaultsProvider == null) {
				if (this.execHistoryDefaultsProviderConfigElem != null) {
					try {
						this.execHistoryDefaultsProvider = (IExecutionHistoryDefaultsProvider)this.execHistoryDefaultsProviderConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.execHistoryDefaultsProviderConfigElem = null;
					}
				}
			}
			return this.execHistoryDefaultsProvider;
		}
		public IDeploymentsProvider getDeploymentsProvider() {
			if (this.deploymentsProvider == null) {
				if (this.deploymentsProviderConfigElem != null) {
					try {
						this.deploymentsProvider = (IDeploymentsProvider)this.deploymentsProviderConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.deploymentsProviderConfigElem = null;
					}
				}
			}
			return this.deploymentsProvider;
		}
		public ITestLaunchConfigurationValidator getLaunchConfigurationValidator() {
			if (this.launchConfigurationValidator == null) {
				if (this.launchConfigurationValidatorConfigElem != null) {
					try {
						this.launchConfigurationValidator = (ITestLaunchConfigurationValidator)this.launchConfigurationValidatorConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.launchConfigurationValidatorConfigElem = null;
					}
				}
			}
			return this.launchConfigurationValidator;
		}
		public boolean isLaunchConfigurationTypeSupported(String lcType) {
			return (this.launchConfigurationType == lcType)
				|| (lcType != null && lcType.equals(this.launchConfigurationType));
		}
	}
	
	private LaunchConfigurationExtensionDescriptor getLaunchConfigurationExtensionDescriptor(String launchConfigurationType) {
		Assert.isNotNull(launchConfigurationType);
		
		Object o = this.launchConfigurationMap.get(launchConfigurationType);
		if (o == null) {
			o = new LaunchConfigurationExtensionDescriptor();
			this.launchConfigurationMap.put(launchConfigurationType, o);
		}
		return (LaunchConfigurationExtensionDescriptor)o;
	}
	
	private class LaunchConfigurationExtensionDescriptor {
		public IConfigurationElement execHistoryDefaultsProviderConfigElem = null;
		private IExecutionHistoryDefaultsProvider2 execHistoryDefaultsProvider = null;
		public IExecutionHistoryDefaultsProvider2 getExecutionHistoryDefaultsProvider() {
			if (this.execHistoryDefaultsProvider == null) {
				if (this.execHistoryDefaultsProviderConfigElem != null) {
					try {
						this.execHistoryDefaultsProvider = (IExecutionHistoryDefaultsProvider2)this.execHistoryDefaultsProviderConfigElem.createExecutableExtension("class"); //$NON-NLS-1$
					} catch (CoreException e) {
						TestUIPlugin.logError(e);
						this.execHistoryDefaultsProviderConfigElem = null;
					}
				}
			}
			return this.execHistoryDefaultsProvider;
		}
	}
	
	private IDeploymentsProvider defaultDeploymentsProvider = new DefaultDeploymentsProvider();
	private IExecutionHistoryDefaultsProvider defaultExecutionHistoryDefaultsProvider = new DefaultExecutionHistoryDefaultsProvider();
	
	private void registerRunHandlers() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".runHandler"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] runHandlers = extPoint.getConfigurationElements();
			for (int i = 0; i < runHandlers.length; i++) {
				IConfigurationElement[] supportedTypes = runHandlers[i].getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedTypes[j].getAttribute("kind"); //$NON-NLS-1$
					try {
						TestExtensionDescriptor ed = getExtensionDescriptor(type, kind);
						ed.runHandlerConfigElem = runHandlers[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + runHandlers[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}
	
	private void registerExecutionHistoryDefaultsProviders() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".executionHistoryDefaults"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] defaultsProviders = extPoint.getConfigurationElements();
			for (int i = 0; i < defaultsProviders.length; i++) {
				IConfigurationElement[] supportedStuff = defaultsProviders[i].getChildren();
				for (int j = 0; j < supportedStuff.length; j++) {
					String type = supportedStuff[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedStuff[j].getAttribute("kind"); //$NON-NLS-1$
					try {
						TestExtensionDescriptor ed = getExtensionDescriptor(type, kind);
						ed.execHistoryDefaultsProviderConfigElem = defaultsProviders[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + defaultsProviders[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}
	
	private void registerExecutionHistoryDefaults2Providers() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".executionHistoryDefaults2"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] defaultsProviders = extPoint.getConfigurationElements();
			for (int i = 0; i < defaultsProviders.length; i++) {
				IConfigurationElement[] supportedStuff = defaultsProviders[i].getChildren();
				for (int j = 0; j < supportedStuff.length; j++) {
					String type = supportedStuff[j].getAttribute("type");
					try {
						LaunchConfigurationExtensionDescriptor ed = getLaunchConfigurationExtensionDescriptor(type);
						ed.execHistoryDefaultsProviderConfigElem = defaultsProviders[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + defaultsProviders[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}

	private void registerDeploymentsProviders() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".deploymentProvider"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] deploymentProviders = extPoint.getConfigurationElements();
			for (int i = 0; i < deploymentProviders.length; i++) {
				IConfigurationElement[] supportedTypes = deploymentProviders[i].getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedTypes[j].getAttribute("kind"); //$NON-NLS-1$
					try {
						TestExtensionDescriptor ed = getExtensionDescriptor(type, kind);
						ed.deploymentsProviderConfigElem = deploymentProviders[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + deploymentProviders[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}
	
	private void registerLaunchConfigurationValidators() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".launchConfigValidator"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] launchConfigurationValidators = extPoint.getConfigurationElements();
			for (int i = 0; i < launchConfigurationValidators.length; i++) {
				IConfigurationElement[] supportedTypes = launchConfigurationValidators[i].getChildren();
				for (int j = 0; j < supportedTypes.length; j++) {
					String type = supportedTypes[j].getAttribute("type"); //$NON-NLS-1$
					String kind = supportedTypes[j].getAttribute("kind"); //$NON-NLS-1$
					try {
						TestExtensionDescriptor ed = getExtensionDescriptor(type, kind);
						ed.launchConfigurationValidatorConfigElem = launchConfigurationValidators[i];
					} catch (Exception e) {
						TestUIPlugin.logError("Extension " + launchConfigurationValidators[i].getName() + " was ignored. See next messages for details.");
						TestUIPlugin.logError(e);
					}
				}
			}
		}
	}
	
	private void registerSupportedTestTypes() {
		IExtensionPoint extPoint = Platform.getPluginRegistry().getExtensionPoint(TestUIPlugin.getID() + ".launchableType"); //$NON-NLS-1$
		if (extPoint != null) {
			IConfigurationElement[] supportedTypes = extPoint.getConfigurationElements();
			for (int i = 0; i < supportedTypes.length; i++) {
				String type = supportedTypes[i].getAttribute("type"); //$NON-NLS-1$
				String kind = supportedTypes[i].getAttribute("kind"); //$NON-NLS-1$
				String lcType = supportedTypes[i].getAttribute("launchConfigurationType"); //$NON-NLS-1$
				try {
					TestExtensionDescriptor ed = getExtensionDescriptor(type, kind);
					ed.launchConfigurationType = lcType;
					IConfigurationElement[] supportedModes = supportedTypes[i].getChildren();
					for (int j = 0; j < supportedModes.length; j++) {
						String mode = supportedModes[j].getAttribute("mode"); //$NON-NLS-1$
						boolean shortcutAvailable = "true".equals(supportedModes[j].getAttribute("shortcutAvailable")); //$NON-NLS-1$ //$NON-NLS-2$
						ed.supportedLaunchModes.add(mode);
						if (shortcutAvailable) {
							ed.availableShortcutModes.add(mode);
						}
					}
				} catch (Exception e) {
					TestUIPlugin.logError("Extension " + supportedTypes[i].getName() + " was ignored. See next messages for details.");
					TestUIPlugin.logError(e);
				}
			}
		}
	}
	
	/**
	 * Returns a Run Handler for the specified testElement. This method may return
	 * null if the specified testElement has no Run Handler associated. In that case,
	 * no pre- or post-run handling is expected.
	 * @param testElement
	 * @return
	 */
    public IRunHandler getRunHandler(Object testElement) {
    	TestExtensionDescriptor ed = getTestExtensionDescriptor(testElement);
    	if (ed != null) {
    		return ed.getRunHandler();
    	}
    	return null;
    }

    /**
     * Return an ExecutionHistoryDefaultsProvider for the specified testElement.
     * This method never returns null.
     * @param testElement
     * @return
     */
    public IExecutionHistoryDefaultsProvider getExecutionHistoryDefaultsProvider(Object testElement) {
    	IExecutionHistoryDefaultsProvider result = null;
    	TestExtensionDescriptor ed = getTestExtensionDescriptor(testElement);
    	if (ed != null) {
	    	result = ed.getExecutionHistoryDefaultsProvider();
    	}
    	if (result == null) {
    		result = defaultExecutionHistoryDefaultsProvider;
    	}
    	return result;
    }

    /**
     * Return an ExecutionHistoryDefaultsProvider2 for the specified launchConfiguration.
     * This method may return null.
     * @param testElement
     * @return
     */
    public IExecutionHistoryDefaultsProvider2 getExecutionHistoryDefaultsProvider2(String launchConfigurationType) {
    	IExecutionHistoryDefaultsProvider2 result = null;
    	LaunchConfigurationExtensionDescriptor ed = getLaunchConfigurationExtensionDescriptor(launchConfigurationType);
    	if (ed != null) {
	    	result = ed.getExecutionHistoryDefaultsProvider();
    	}
    	return result;
    }

    /**
     * Return a DeploymentsProvider for the specified testElement.
     * This method never returns null.
     * @param testElement
     * @return
     */
    public IDeploymentsProvider getDeploymentsProvider(Object testElement) {
    	IDeploymentsProvider result = null;
    	TestExtensionDescriptor ed = getTestExtensionDescriptor(testElement);
    	if (ed != null) {
    		result = ed.getDeploymentsProvider();
    	}
    	if (result == null) {
    		result = defaultDeploymentsProvider;
    	}
    	return result;
    }
    
    /**
     * Return a TestLaunchConfigurationValidator for the specified testElement.
     * This method may return null.
     * @param testElement
     * @return
     */
    public ITestLaunchConfigurationValidator getLaunchConfigurationValidator(Object testElement) {
    	ITestLaunchConfigurationValidator result = null;
    	TestExtensionDescriptor ed = getTestExtensionDescriptor(testElement);
    	if (ed != null) {
    		result = ed.getLaunchConfigurationValidator();
    	}
    	return result;
    }
    
    /**
     * Returns whether the specified testElement is launchable by a specific LaunchConfigurationType
     * in the given mode.
     * @param testElement The testElement
     * @param mode The run mode used for launching the testElement (usually run, debug or profile)
     * @param launchConfigurationType The launchConfigurationType. Use <code>null</code> for the 
     * Hyades Basic Launch Configuration Type.
     * @return
     */
    public boolean isLaunchableElement(Object testElement, String mode, String launchConfigurationType) {
    	TestExtensionDescriptor ed = getTestExtensionDescriptor(testElement);
    	if (ed != null) {
    		return ed.isLaunchConfigurationTypeSupported(launchConfigurationType)
				&& ed.supportedLaunchModes.contains(mode);
    	}
    	return false;
    }
    
    /**
     * Returns whether a shortcut should be available for launching the specified testElement
     * in the given mode.
     * @param testElement The testElement
     * @param mode The run mode used for launching the testElement (usually run, debug or profile)
      * @return
     */
    public boolean isShortcutAvailable(Object testElement, String mode) {
    	TestExtensionDescriptor ed = getTestExtensionDescriptor(testElement);
    	if (ed != null) {
    		return ed.launchConfigurationType == null // We support shortcuts for default LC only
				&& ed.availableShortcutModes.contains(mode);
    	}
    	return false;
    }
    
	/**
	 * @return Returns the defaultExecutionHistoryDefaultsProvider.
	 */
	public IExecutionHistoryDefaultsProvider getDefaultExecutionHistoryDefaultsProvider() {
		return defaultExecutionHistoryDefaultsProvider;
	}

	/**
	 * @return Returns the defaultDeploymentsProvider.
	 */
	public IDeploymentsProvider getDefaultDeploymentsProvider() {
		return defaultDeploymentsProvider;
	}
	
}
