/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;

import org.eclipse.hyades.models.hierarchy.util.XMLUtil;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;


/**
 * @author slavescu
 */
public abstract class RegistryReader {
    //~ Static fields/initializers -----------------------------------------------------------------

    public static StandaloneConfiguration standaloneConfiguration;
    protected static int mode = 0;
    protected static final String TAG_DESCRIPTION = "description";
    protected static final String LOADER_PPID = "fragment_loader";
    protected static final String HANDLER_PPID = "fragment_handler";
    protected static final String LOADERS_FACTORY_PPID = "fragment_loaders_factory";
    protected static final String RESOURCE_LOADER_PPID = "resource_loader";
    protected static final String LOOKUP_SERVICE_PPID = "lookup_service";
    protected static final String HYADES_RESOURCE_FACTORY_PPID = "hyades_resource_factory";
    protected static final String thisPluginId = "org.eclipse.hyades.models.hierarchy";

    //~ Instance fields ----------------------------------------------------------------------------

    private List alreadyProcessedEclipseFolders;
    private String extensionPointID;

    //~ Constructors -------------------------------------------------------------------------------

    public RegistryReader(String extensionPointID) {
        this.extensionPointID = extensionPointID;
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     * @return
     */
    public static synchronized boolean isPlatformMode() {
        if (mode == 0) {
            setMode();
        }

        return mode == 1;
    }

    public IHyadesPluginClassDescriptor createPluginClassDescriptor(HyadesConfigurationElement element, String attribute) {
        if (isPlatformMode()) {
            return new PlatformPluginClassDescriptor((IConfigurationElement) element.getElement(), attribute);
        } else {
            return new StandalonePluginClassDescriptor((Node) element.getElement(), attribute);
        }
    }

    /**
     * Reads from the plugin registry and parses it.
     */
    public void readRegistry() {
        if (isPlatformMode()) {
            platformReadRegistry();
        } else {
            standaloneReadRegistry();
        }
    }

    protected void logError(HyadesConfigurationElement element, String text) {
        System.err.println("Plugin " + element.getDeclaringExtensionName() + ", extension " + element.getAttribute("class"));
        System.err.println(text);
    }

    protected void logMissingAttribute(HyadesConfigurationElement element, String attributeName) {
        logError(element, "The required attribute '" + attributeName + "' not defined");
    }

    /**
     * Implement this method to read element attributes.
     * If this element has subelements, the reader will recursively cycle through them
     * and will call this method, so don't do it here.
     */
    protected abstract boolean readElement(HyadesConfigurationElement element);

    /**
     *
     */
    private static void setMode() {
        try {
            Class resourcePlugin = Class.forName("org.eclipse.core.resources.ResourcesPlugin");
            Method getWorkspace = resourcePlugin.getMethod("getWorkspace", null);
            Object workspace = getWorkspace.invoke(null, null);
            Method getRoot = workspace.getClass().getMethod("getRoot", null);
            Object root = getRoot.invoke(workspace, null);

            mode = (root != null) ? 1 : 2;
        } catch (ClassNotFoundException e) {
            mode = 2;
        } catch (SecurityException e) {
            mode = 2;
        } catch (NoSuchMethodException e) {
            mode = 2;
        } catch (IllegalArgumentException e) {
            mode = 2;
        } catch (IllegalAccessException e) {
            mode = 2;
        } catch (InvocationTargetException e) {
            mode = 2;
        } catch (NullPointerException e) {
        	mode = 2;
        }
        
        //		ResourcesPlugin.getWorkspace().getRoot()
    }

    /**
     *
     */
    private void addPackageExtensionPoints() {
        for (Iterator iter = standaloneConfiguration.getExtensionsList().iterator();
                 iter.hasNext();) {
            HyadesConfigurationElement element = (HyadesConfigurationElement) iter.next();

            if (element.getDeclaringExtensionName().equals("org.eclipse.emf.ecore.generated_package")) {
                try {
                    String emfPackageClassName = element.getAttribute("class");

                    if (standaloneConfiguration.getEmfPackageClassesList().contains(emfPackageClassName)) {
                        continue;
                    }

                    standaloneConfiguration.getEmfPackageClassesList().add(emfPackageClassName);
                } catch (RuntimeException e) {
                    LoadersUtils.log(e);
                }
            }
        }
    }

    private void internalReadElement(HyadesConfigurationElement element) {
        boolean recognized = this.readElement(element);

        if (recognized) {
            HyadesConfigurationElement[] children = element.getChildren();

            for (int i = 0; i < children.length; ++i) {
                internalReadElement(children[i]);
            }
        } else {
            logError(element, "Error processing extension: " + element.getDeclaringExtensionName());
        }
    }

    private void platformReadRegistry() {
        IExtensionPoint point = Platform.getPluginRegistry().getExtensionPoint(thisPluginId, extensionPointID);

        if (point != null) {
            IConfigurationElement[] elements = point.getConfigurationElements();

            for (int i = 0; i < elements.length; i++) {
                internalReadElement(new HyadesConfigurationElement(elements[i]));
            }
        }
    }

    private void processEMFpackages() {
        addPackageExtensionPoints();

        for (Iterator iter = standaloneConfiguration.getEmfPackageClassesList().iterator();
                 iter.hasNext();) {
            String element = (String) iter.next();

            try {
                Class clazz = Class.forName(element);

                try {
                    Field field = clazz.getField("eINSTANCE");

                    field.get(null);
                } catch (NoSuchFieldException e1) {
                    Method method = clazz.getMethod("init", null);

                    method.invoke(null, null);
                }
            } catch (ClassNotFoundException e) {
                LoadersUtils.log(e);
            } catch (SecurityException e) {
                LoadersUtils.log(e);
            } catch (NoSuchMethodException e) {
                LoadersUtils.log(e);
            } catch (IllegalArgumentException e) {
                LoadersUtils.log(e);
            } catch (IllegalAccessException e) {
                LoadersUtils.log(e);
            } catch (InvocationTargetException e) {
                LoadersUtils.log(e);
            }
        }
    }

    private void processEclipseFolder() {
        String eclipseFolder = standaloneConfiguration.getEclipseFolder();

        processEclipseFolder(eclipseFolder);

        File linksFolder = new File(eclipseFolder + "/links");

        if (linksFolder.exists()) {
            processLinksFolder(linksFolder);
        }
    }

    /**
     *
     */
    private void processEclipseFolder(String eclipseFolder) {
        if (eclipseFolder != null) {
            if (alreadyProcessedEclipseFolders.contains(eclipseFolder)) {
                return;
            }

            alreadyProcessedEclipseFolders.add(eclipseFolder);

            File folder = new File(eclipseFolder + "/plugins");

            if (folder.exists()) {
                File[] folders = folder.listFiles();

                for (int i = folders.length - 1; i >= 0; i--) {
                    File pluginFile = new File(folders[i].getAbsolutePath() + "/plugin.xml");

                    if (pluginFile.exists()) {
                        String pluginXMLFileName = pluginFile.getAbsolutePath();

                        if (standaloneConfiguration.getPluginsXMLFiles().contains(pluginXMLFileName)) {
                            continue;
                        }

                        standaloneConfiguration.getPluginsXMLFiles().add(pluginXMLFileName);
                    }
                }
            }
        }
    }

    private void processExtensions() {
        for (Iterator iter = standaloneConfiguration.getExtensionsList().iterator();
                 iter.hasNext();) {
            HyadesConfigurationElement element = (HyadesConfigurationElement) iter.next();

            if (element.getDeclaringExtensionName().endsWith("." + extensionPointID)) {
                internalReadElement(element);
            }
        }
    }

    /**
     * @param linksFolder
     */
    private void processLinksFolder(File linksFolder) {
        File[] files = linksFolder.listFiles(new FilenameFilter() { /* (non-Javadoc)
                 * @see java.io.FilenameFilter#accept(java.io.File, java.lang.String)
                 */
                public boolean accept(File arg0, String arg1) {
                    if (arg1.endsWith(".link")) {
                        return true;
                    }

                    return false;
                }
            });

        for (int i = files.length - 1; i >= 0; i--) {
            try {
                BufferedReader bufferedReader = new BufferedReader(new FileReader(files[i]));
                String line;

                while ((line = bufferedReader.readLine()) != null) {
                    if (line.startsWith("path=")) {
                        line = line.substring("path=".length()).trim();
                        processEclipseFolder(line + "/eclipse");
                    }
                }
            } catch (FileNotFoundException e) {
                LoadersUtils.log(e);
            } catch (IOException e) {
                LoadersUtils.log(e);
            }
        }
    }

    /**
     *
     */
    private void processPluginXMLFiles() {
        List alredyProcessed = new ArrayList();

        for (Iterator iter = standaloneConfiguration.getPluginsXMLFiles().iterator();
                 iter.hasNext();) {
            try {
                String element = (String) iter.next();

                if (alredyProcessed.contains(element)) {
                    continue;
                }

                alredyProcessed.add(element);

                Document document = XMLUtil.getXmlDom(element);
                NodeList nodeList = document.getElementsByTagName("extension");

                for (int i = nodeList.getLength() - 1; i >= 0; i--) {
                    Node extension = nodeList.item(i);

                    standaloneConfiguration.getExtensionsList().add(new HyadesConfigurationElement(extension));
                }
            } catch (RuntimeException e) {
                LoadersUtils.log(e);
            }
        }
    }

    private synchronized void standaloneReadRegistry() {
        if (standaloneConfiguration != null) {
            if (!standaloneConfiguration.isAlreadyProcessed()) {
                alreadyProcessedEclipseFolders = new ArrayList();
                processEclipseFolder();
                processPluginXMLFiles();
                processEMFpackages(); // initialize EMF packages
                standaloneConfiguration.alreadyProcessed = true;
            }

            processExtensions(); // now process the extensions
        }
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    public static final class StandaloneConfiguration {
        protected List emfPackageClassesList = new ArrayList(); // list of class names for EMF package initialization 
        protected List extensionsList = new ArrayList(); // list of HyadesConfigurationElement instances for each extension point (using XML DOM and by following the extension point schema)
        protected List pluginsXMLFiles = new ArrayList(); // list of paths to plugin.xml files
        protected String eclipseFolder; // path to an Eclipse runtime folder - will support links folder also
        protected boolean alreadyProcessed = false;

        public StandaloneConfiguration() {
        	// empty method
        }

        /**
         * @return
         */
        public boolean isAlreadyProcessed() {
            return alreadyProcessed;
        }

        /**
         * @param string
         */
        public void setEclipseFolder(String string) {
            eclipseFolder = string;
        }

        /**
         * @return
         */
        public String getEclipseFolder() {
            return eclipseFolder;
        }

        /**
         * @return
         */
        public List getEmfPackageClassesList() {
            return emfPackageClassesList;
        }

        /**
         * @return
         */
        public List getExtensionsList() {
            return extensionsList;
        }

        /**
         * @return
         */
        public List getPluginsXMLFiles() {
            return pluginsXMLFiles;
        }
    }
}


final class PlatformPluginClassDescriptor implements IHyadesPluginClassDescriptor {
    //~ Instance fields ----------------------------------------------------------------------------

    protected IConfigurationElement element;
    protected String attributeName;

    //~ Constructors -------------------------------------------------------------------------------

    public PlatformPluginClassDescriptor(IConfigurationElement element, String attributeName) {
        this.element = element;
        this.attributeName = attributeName;
    }

    //~ Methods ------------------------------------------------------------------------------------

    public Object createInstance() {
        try {
            return element.createExecutableExtension(attributeName);
        } catch (CoreException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }
}


/**
 * @author slavescu
 */
class StandalonePluginClassDescriptor implements IHyadesPluginClassDescriptor {
    //~ Instance fields ----------------------------------------------------------------------------

    protected Node element;
    protected String attributeName;

    //~ Constructors -------------------------------------------------------------------------------

    public StandalonePluginClassDescriptor(Node element, String attributeName) {
        this.element = element;
        this.attributeName = attributeName;
    }

    //~ Methods ------------------------------------------------------------------------------------

    public Object createInstance() {
        Object ret = null;
        String className = getFirstRealChild().getAttributes().getNamedItem(attributeName).getNodeValue();

        try {
            Class clazz = Class.forName(className);

            ret = clazz.newInstance();
        } catch (ClassNotFoundException e) {
            LoadersUtils.log(e);
            throw new RuntimeException(e.getLocalizedMessage());
        } catch (InstantiationException e) {
            LoadersUtils.log(e);
            throw new RuntimeException(e.getLocalizedMessage());
        } catch (IllegalAccessException e) {
            LoadersUtils.log(e);
            throw new RuntimeException(e.getLocalizedMessage());
        }

        return ret;
    }

    private Node getFirstRealChild() {
        Node node = element.getFirstChild();

        while ((node != null) && (node.getAttributes() == null)) {
            node = node.getNextSibling();
        }

        return node;
    }
}


