/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.loaders.hierarchy.IgnoredXMLFragmentLoader;
import org.eclipse.hyades.models.hierarchy.CorrelationSourceInfo;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.UnresolvedCorrelation;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;
import org.xml.sax.SAXParseException;

/**
 * @author slavescu
 */
public class XMLLoader implements IXMLLoader {
	//~ Instance fields ----------------------------------------------------------------------------

	protected FileWriter file;
	protected HierarchyContext context;
	protected InvalidXMLException error;
	protected Map loaders = new HashMap();
	protected String currentAttributeName;
	protected String currentElementName;
	protected XMLFragmentHandler scanner;
	protected XMLFragmentLoader currentLoader;
	protected XMLFragmentLoader defaultLoader = new IgnoredXMLFragmentLoader();
	protected boolean startDocument;
	protected int depth = 0;
	protected int fragmentsCount;

	//~ Constructors -------------------------------------------------------------------------------

	public XMLLoader(TRCAgentProxy agentProxy) {
		makeScanner();

		TRCAgent agent = agentProxy.getAgent();

		if ((agent != null) && (agent.eResource() != null)) {
			context = LoadersUtils.locateHierarchyContext(agent);
		}

		if (context == null) {
			context = new HierarchyContext();

			if (agentProxy != null) {
				context.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
				context.setNode(agentProxy.getProcessProxy().getNode());
				context.setProcessProxy(agentProxy.getProcessProxy());
				context.setAgentProxy(agentProxy);
			}

			LookupServiceExtensions.getInstance().register(null, context);
		} else {
			if (context.getAgentProxy() == null) {
				context.setAgentProxy(agentProxy);
			}

			if (context.getProcessProxy() == null) {
				context.setProcessProxy(agentProxy.getProcessProxy());
			}

			if (context.getMonitor() == null) {
				context.setMonitor(agentProxy.getProcessProxy().getNode().getMonitor());
			}

			if (context.getNode() == null) {
				context.setNode(agentProxy.getProcessProxy().getNode());
			}
		}

		defaultLoader = getLoader("DefaultXMLFragmentLoader");
	}

	public XMLLoader(TRCAgent agent) {
		super();
		makeScanner();

		if ((agent != null)) {
			context = LoadersUtils.locateHierarchyContext(agent);
		}

		if (context == null) {
			context = new HierarchyContext();
			context.setMonitor(agent.getAgentProxy().getProcessProxy().getNode().getMonitor());
			context.setNode(agent.getAgentProxy().getProcessProxy().getNode());
			context.setProcessProxy(agent.getAgentProxy().getProcessProxy());
			context.setAgentProxy(agent.getAgentProxy());
			context.setAgent(agent);

			if (agent.eResource() == null) {
				try {
					addToResource(agent, context);
				} catch (Exception e) {
					// TODO MS - add proper handling
				}
			}

			LookupServiceExtensions.getInstance().register(null, context);
		} else {
			if (context.getAgentProxy() == null) {
				context.setAgentProxy(agent.getAgentProxy());

				if (context.getProcessProxy() == null) {
					context.setProcessProxy(agent.getAgentProxy().getProcessProxy());
				}

				if (context.getMonitor() == null) {
					context.setMonitor(agent.getAgentProxy().getProcessProxy().getNode().getMonitor());
				}

				if (context.getNode() == null) {
					context.setNode(agent.getAgentProxy().getProcessProxy().getNode());
				}
			}
		}

		defaultLoader = getLoader("DefaultXMLFragmentLoader");
	}

	public XMLLoader(TRCMonitor monitor) {
		super();

		//        if ((monitor != null) && (monitor.eResource() != null)) {
		//            context = (HierarchyContext) LookupServiceExtensions.getInstance().locate(null, HierarchyContext.class, monitor.eResource().getURI().toString());
		//        }
		//
		if (context == null) {
			context = new HierarchyContext();
			context.setMonitor(monitor);

			//            LookupServiceExtensions.getInstance().register(null, context);
		}

		makeScanner();
		defaultLoader = getLoader("DefaultXMLFragmentLoader");
	}

	//~ Methods ------------------------------------------------------------------------------------

	public void setCollectionMode(int collectionMode) {
		context.setCollectionMode(TRCCollectionMode.get(collectionMode));
	}

	public HierarchyContext getContext() {
		return context;
	}

	public int getProcessedFragments() {
		return fragmentsCount;
	}

	public void attributeName(String name) {
		if (isIgnoredElement()) {
			return;
		}

		currentAttributeName = name;
	}

	public void attributeValueCharacters(String attributeValue) {
		if (isIgnoredElement()) {
			return;
		}

		try {
			currentLoader.addAttribute(currentAttributeName, attributeValue);
		} catch (Exception e) {
			log(e);
		}
	}

	public void characters(char[] ch, int start, int length) {
		if (isIgnoredElement()) {
			return;
		}

		try {
			if (currentLoader != null) {
				currentLoader.addCharacters(ch, start, length);
			}
		} catch (Exception e) {
			log(e);
		}
	}

	public synchronized void cleanUp() {
		if (scanner == null) {
			return;
		}

		try {
			scanner.terminateParser();
		} catch (Exception e) {
			// TODO MS - add proper handling
			log(e);
		}

		scanner = null;

		if (file != null) {
			try {
				file.close();
			} catch (IOException e) {
				log(e);
			}
		}

		file = null;
		error = null;
		depth = 0;
		startDocument = true;
		fragmentsCount = 0;

		List processed = new ArrayList();

		for (Iterator iter = loaders.entrySet().iterator(); iter.hasNext();) {
			Map.Entry entry = (Map.Entry) iter.next();

			if (!processed.contains(entry.getValue())) {
				((XMLFragmentLoader) entry.getValue()).cleanUp();
				processed.add(entry.getValue());
			}
		}

		loaders.clear();

		if (getContext().getGlobalForwardReferences().size() > 0) {
			processGlobalForwardReferences();
		}
	}

	public void endDocument(Object object, int i) {
		//        cleanUp();
	}

	public void endElement(String elementName, int currentOffset) {
		currentElementName = elementName;

		try {
			if (currentLoader != null) {
				currentLoader.endChild(currentElementName);
			}

			depth--;

			if (depth == 0) {
				currentLoader.addYourselfInContext();
			}
		} catch (Exception e) {
			log(e);
		}
	}

	public void error(InvalidXMLException exception) {
		error = exception;
		error.fillInStackTrace();

		//        if(exception.getEnclosedException()!=null)
		//        	log(exception.getEnclosedException());
		//		throw error;
	}

	public void loadEvent(byte[] buffer, int length, boolean loadToModel, boolean toProfileFile) throws InvalidXMLException {
		if ((error != null) || (scanner == null)) {
			return;
		}

		if (toProfileFile) {
			writeToProfileFile(LoadersUtils.makeString(buffer, 0, length));
		} else {
			loadEvent(buffer, length, loadToModel);
		}
	}

	public void loadEvent(byte[] buffer, int length, boolean loadToModel) throws InvalidXMLException {
//		LoadersUtils.log("loadEvent(length)="+length+" error="+error+" scanner="+scanner);
		if ((error != null)) {
			if(error.getEnclosedException()!=null)
				LoadersUtils.log(error.getEnclosedException());
			return;
		}
		if(scanner==null)
			return;

		if ((getContext().getAgent() != null) && (getContext().getAgent().eResource() != null) && !getContext().getAgent().eResource().isModified()) {
			getContext().getAgent().eResource().setModified(true);
		}

		setLoadToModel(loadToModel);

		try {
//			LoadersUtils.log("loadEvent1(length)="+length);
			scanner.scanContent(buffer, 0, length);
			//LoadersUtils.log("loadEvent(byte[], int, boolean)======\n" + LoadersUtils.makeString(buffer,0,length));
			//LoadersUtils.log("loadEvent(byte[], int, boolean)******");
			
//						System.out.println(LoadersUtils.makeString(buffer,0,length));
//			            				if(!startDocument)
//			            				{
//			            					context.getAgentProxy().setProfileFile("D:/temp/log_"+(new java.util.Date()).getTime()+".xml");
//			            			    }
//			            			  writeToProfileFile(LoadersUtils.makeString(buffer,0,length)+"\n");
		} catch (Exception e) {
			if (error == null) {
				log(e);
			}
		}

		if (error != null) {
			throw error;
		}
	}

	public void loadEvent(byte[] buffer, int length) throws InvalidXMLException {
		this.loadEvent(buffer, length, true);
	}

	public void restartParser() {
		if (scanner != null) {
			try {
				scanner.terminateParser();
			} catch (Exception e) {
				// TODO MS - add proper handling
				log(e);
			}
		}

		error = null;
		startDocument = true;
		depth = 0;

		makeScanner();
	}

	public void startDocument() {
		startDocument = true;
	}

	public void startElement(String elementName, boolean hasAttributes, boolean isEmpty) {
		currentElementName = elementName;

		if (isIgnoredElement()) {
			return;
		}

		if (depth == 0) {
			currentLoader = getLoader(getName(currentElementName));
			currentLoader.initialize(getContext(), currentElementName);
			fragmentsCount++;
		} else {
			try {
				currentLoader.startChild(currentElementName);
			} catch (Exception e) {
				log(e);
			}
		}

		depth++;
	}

	protected boolean isIgnoredElement() {
		if (startDocument) {
			if (currentElementName == null) {
				return true;
			}

			if (currentElementName.equals("TRACE") || currentElementName.equals("CommonBaseEvents") || currentElementName.equals("Statistic")) {
				return true;
			} else {
				startDocument = false;
			}
		}

		return false;
	}

	protected XMLFragmentLoader getLoader(String currentElementName) {
		XMLFragmentLoader loader = (XMLFragmentLoader) loaders.get(currentElementName);

		if (loader == null) {
			loader = (XMLFragmentLoader) LoaderExtensions.getInstance().get(currentElementName);

			if (loader != null) {
				try {
					loader = (XMLFragmentLoader) loader.getClass().newInstance();
					loaders.put(currentElementName, loader);

					return loader;
				} catch (InstantiationException e) {
					// TODO MS - add proper handling
				} catch (IllegalAccessException e) {
//					 TODO MS - add proper handling
				}
			}

			loader = defaultLoader;
		}

		return loader;
	}

	protected String getName(String currentElementName) {
		int column = currentElementName.lastIndexOf(":");

		if (column != -1) {
			return currentElementName.substring(column);
		}

		return currentElementName;
	}

	protected void log(Exception e) {
		if (e instanceof SAXParseException) {
			SAXParseException spx = (SAXParseException) e;

			System.err.println(spx.getLocalizedMessage());
			System.err.println("at line: " + spx.getLineNumber());
			System.err.println("at column: " + spx.getColumnNumber());
			spx.fillInStackTrace();
			LoadersUtils.log(spx);
		} else {
			LoadersUtils.log(e);
		}
	}

	protected void makeScanner() {
		scanner = (XMLFragmentHandler) XMLFragmentHandlerExtensions.getInstance().get(XMLFragmentHandlerRegistryReader.TAG_HANDLER);

		if (scanner != null) {
			try {
				scanner = (XMLFragmentHandler) scanner.getClass().newInstance();

				scanner.setXMLLoader(this);
			} catch (Exception e) {
				log(e);
				scanner = new SAXFragmentHandler();
				scanner.setXMLLoader(this);
			}
		} else {
			scanner = new SAXFragmentHandler();
			scanner.setXMLLoader(this);
		}
	}

	protected void processGlobalForwardReferences() {
		for (Iterator iter = context.getAgent().getUnresolvedCorrelations().iterator(); iter.hasNext();) {
			UnresolvedCorrelation unresolvedCorrelation = (UnresolvedCorrelation) iter.next();
			for (Iterator iterator = unresolvedCorrelation.getSourceInfos().iterator(); iterator.hasNext();) {
				CorrelationSourceInfo correlationSourceInfo = (CorrelationSourceInfo) iterator.next();

				EObject owner = correlationSourceInfo.getOwner();
				EReference reference = correlationSourceInfo.getReference();
				EObject o = (EObject) LookupServiceExtensions.getInstance().locate(null, reference.getEReferenceType().getInstanceClass(), LoadersUtils.getLookUpKey(unresolvedCorrelation.getContextId()));

				if (o != null) {

					if (reference.isMany()) {
						((EList) owner.eGet(reference)).add(o);
					} else {
						owner.eSet(reference, o);
					}

					iterator.remove();
				}
			}
			if (unresolvedCorrelation.getSourceInfos().size() == 0) {
				iter.remove();
			}
		}
	}

	private void setLoadToModel(boolean loadToModel) {
		if (context.isLoadToModel() != loadToModel) {
			context.setLoadToModel(loadToModel);
		}
	}

	private void addToResource(TRCAgent agent, HierarchyContext context) {
		String monitorFolder = LoadersUtils.getMonitorFolder(context.getMonitor());
		Resource agDoc = null;

		if (agent.getName() == null) {
			agent.setName("unknown");
		}

		String aName =
			StringUtil.change(context.getMonitor().getName().trim(), " ", "") + "_" + context.getNode().getName().trim() + "_" + context.getProcessProxy().getPid() + "_" + context.getProcessProxy().getRuntimeId() + "_" + StringUtil.change(agent.getName().trim(), " ", "");

		String pFileName = monitorFolder + aName;

		agDoc = Resource.Factory.Registry.INSTANCE.getFactory(SaveUtil.createURI(pFileName + ".trcaxmi")).createResource(SaveUtil.createURI(pFileName + ".trcaxmi"));
		agDoc.setModified(true);

		HierarchyResourceSetImpl.getInstance().getResources().add(agDoc);
		//        SaveUtil.addDocument(agDoc);

		if (agDoc != null) {
			agDoc.getContents().add(agent);
		}
	}

	private void writeToProfileFile(String xml) {
		if ((context.getAgentProxy() != null) && (context.getAgentProxy().getProfileFile() != null)) {
			try {
				if (file == null) {
					file = new FileWriter(context.getAgentProxy().getProfileFile());
				}

				file.write(xml);
				file.flush();
			} catch (IOException e) {
				error = new InvalidXMLException(e);
				throw error;
			}
		}
	}
}
