package org.eclipse.hyades.logging.events.cbe.util.tests;

import java.text.SimpleDateFormat;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.SimpleTimeZone;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.FormattingException;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.impl.tests.Util;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Tests the EventFormatter class.
 * <p>
 * 
 * @author Paul E. Slauenwhite
 * @version June 8, 2004
 * @since April 20, 2004
 * @see org.eclipse.hyades.logging.events.cbe.util.EventFormatter
 */
public class EventFormatterTest extends TestCase {

    /**
     * Constructor for EventFormatterTest.
     * 
     * @param arg0
     */
    public EventFormatterTest(String testcase) {
        super(testcase);
    }

    public void testInvalidXMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"> <CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsed Time=\"0\" extensionName=\"Name\" globalInstanceId=\"i0000000000000000000000000000040\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </context DataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>       <children name=\"\" type=\"noValue\">          <values>values</values>       </children>      </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents> <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/><sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\"><situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation>	    <a:a xmlns:a=\"http://bar\" />    </CommonBaseEvent></CommonBaseEvents>";

        try {
            EventFormatter.eventsFromCanonicalXMLDoc(eventXML);
            assertTrue(false);
        } catch (FormattingException f) {
        }
    }

    public void testInvalidXMLFragmentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"> <CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsed Time=\"0\" extensionName=\"Name\" globalInstanceId=\"i0000000000000000000000000000040\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </context DataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>       <children name=\"\" type=\"noValue\">          <values>values</values>       </children>      </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents> <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/><sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\"><situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation>	    <a:a xmlns:a=\"http://bar\" />    </CommonBaseEvent></CommonBaseEvents>";

        try {
            EventFormatter.eventFromCanonicalXML(eventXML);
            assertTrue(false);
        } catch (FormattingException f) {
        }
    }

    public void testEventMultipleAnyElementsWithWhiteSpaceInXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <any namespace=\"##other\" processContents=\"skip\" />      <any1 abc=\"def\" />        <any2>          This is an any element</any2>  <GetLastTrade>          <symbol>DEF      </symbol>       </GetLastTrade>         <any3>     <xyz>       a=\"b c\" </xyz>     </any3></CommonBaseEvent>";

        try {

            String[] anyElements = { "<any namespace=\"##other\" processContents=\"skip\"></any>", "<any1 abc=\"def\"></any1>", "<any2>This is an any element</any2>", "<GetLastTrade><symbol>DEF</symbol></GetLastTrade>", "<any3><xyz>a=&quot;b c&quot;</xyz></any3>"};

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(eventXML);

            List any = event.getAny();

            assertNotNull(any);

            assertTrue(any.size() == 5);

            for (int counter = 0; counter < any.size(); counter++) {
                assertTrue(any.get(counter).equals(anyElements[counter]));
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testAlternateNameSpacesXMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"ASCII\"?><cbe:CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><cbe:AssociationEngine cbe:id=\"i0000000000000000000000000000044\" cbe:name=\"MyCorrelationEngine\" cbe:type=\"Correlate\"/><cbe:CommonBaseEvent cbe:creationTime=\"2001-12-31T12:00:00\" cbe:extensionName=\"Name\" cbe:globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <cbe:contextDataElements name=\"myContext\" type=\"String\">    <cbe:contextValue>contextValue</cbe:contextValue>  </cbe:contextDataElements>  <cbe:extendedDataElements name=\"ede1\" type=\"noValue\">    <cbe:values>values</cbe:values>    <cbe:children name=\"chname1\" type=\"noValue\">      <cbe:values>values</cbe:values>    </cbe:children>  </cbe:extendedDataElements>  <cbe:extendedDataElements name=\"ede2\" type=\"noValue\">    <cbe:values>values</cbe:values>    <cbe:children name=\"chname2\" type=\"noValue\">      <cbe:values>values</cbe:values>    </cbe:children>  </cbe:extendedDataElements>  <cbe:reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <cbe:sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <cbe:msgDataElement msgLocale=\"EN\">    <cbe:msgCatalogTokens value=\"value\"/>    <cbe:msgId>msgId</cbe:msgId>    <cbe:msgIdType>Name</cbe:msgIdType>    <cbe:msgCatalogId>msgCatalogId</cbe:msgCatalogId>    <cbe:msgCatalogType>msgCatalogType</cbe:msgCatalogType>    <cbe:msgCatalog>msgCatalog</cbe:msgCatalog>  </cbe:msgDataElement>  <cbe:situation categoryName=\"StartSituation\">    <cbe:situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </cbe:situation></cbe:CommonBaseEvent></cbe:CommonBaseEvents>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);

            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testXMLFragmentDeserializationWithMultipleAssociatedEvents() {

        String eventXML = "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements> <associatedEvents resolvedEvents=\"i0000000000000000000000000000040 i00000000000000000000000000011132 i0000000000000000000000000022232\"><associationEngine>i0000000000000000000000000000032</associationEngine></associatedEvents><associatedEvents resolvedEvents=\"i0000000000000000000000000000040\"><associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/></associatedEvents> <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent>";

        try {

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(eventXML);

            assertEquals(event.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            List associatedEvents = event.getAssociatedEvents();

            assertNotNull(associatedEvents);

            assertTrue(associatedEvents.size() == 2);

            AssociatedEvent associatedEvent = ((AssociatedEvent) (associatedEvents.get(0)));

            assertNotNull(associatedEvent);

            assertTrue(associatedEvent.getResolvedEvents().equals("i0000000000000000000000000000040 i00000000000000000000000000011132 i0000000000000000000000000022232"));

            assertTrue(associatedEvent.getAssociationEngine().equals("i0000000000000000000000000000032"));

            assertNull(associatedEvent.getAssociationEngineInfo());

            associatedEvent = ((AssociatedEvent) (associatedEvents.get(1)));

            assertNotNull(associatedEvent);

            assertTrue(associatedEvent.getResolvedEvents().equals("i0000000000000000000000000000040"));

            assertNull(associatedEvent.getAssociationEngine());

            AssociationEngine associationEngine = associatedEvent.getAssociationEngineInfo();

            assertNotNull(associationEngine);

            assertTrue(associationEngine.getName().equals("MyCorrelationEngine"));

            assertTrue(associationEngine.getId().equals("i0000000000000000000000000000044"));

            assertTrue(associationEngine.getType().equals("Correlate"));
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testEventAnyElementInXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <a:a xmlns:a=\"http://bar\" ><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a><a:a xmlns:a=\"http://bar\" ><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a><a:a xmlns:a=\"http://bar\" ><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a></CommonBaseEvent>";

        try {

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(eventXML);

            String anyElementXMLFragment = "<a:a xmlns:a=\"http://bar\"><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a>";
            List anyElements = event.getAny();

            assertNotNull(anyElements);
            assertTrue(anyElements.size() == 3);

            for (int counter = 0; counter < anyElements.size(); counter++) {
                assertTrue(anyElements.get(counter).equals(anyElementXMLFragment));
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testOtherSituationAnyElementInXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"OtherSituation\">    	<situationType reasoningScope=\"EXTERNAL\" xsi:type=\"OtherSituation\"><a:a xmlns:a=\"http://bar\" ><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a><a:a xmlns:a=\"http://bar\" ><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a><a:a xmlns:a=\"http://bar\" ><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a></situationType>	</situation></CommonBaseEvent>";

        try {

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(eventXML);

            String anyElementXMLFragment = "<a:a xmlns:a=\"http://bar\"><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a><a:a xmlns:a=\"http://bar\"><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a><a:a xmlns:a=\"http://bar\"><expression logicalOperator=\"AND\"><answer value=\"10\"></answer><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"equals\"><answer value=\"9\"></answer><testPropertyName><answer value=\"8\"></answer>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName><answer value=\"7\"></answer></preCondition><answer value=\"6\"></answer><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><answer value=\"5\"></answer><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;<answer value=\"4\"></answer></testPropertyName><answer value=\"3\"></answer></postCondition><answer value=\"2\"></answer></expression></a:a>";

            Situation situation = event.getSituation();

            assertNotNull(situation);

            assertTrue(situation.getCategoryName().equals("OtherSituation"));

            SituationType situationType = situation.getSituationType();

            assertNotNull(situationType);

            assertTrue(((OtherSituation) (situationType)).getAny().equals(anyElementXMLFragment));
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testXMLDocumentDeserializationWithSpaces() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\"    xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"    xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"    ><AssociationEngine id=\"i0000000000000000000000000000044\"    name=\"MyCorrelationEngine\"    type=\"Correlate\"/><CommonBaseEvent    creationTime=\"2001-12-31T12:00:00\"    extensionName=\"Name\"    globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"    >  <contextDataElements name=\"myContext\"    type=\"String\"    >    <contextValue    >contextValue</contextValue    >  </contextDataElements    >  <extendedDataElements name=\"ede1\"    type=\"noValue\"    >    <values       >values</values      >    <children name=\"chname1\"    type=\"noValue\"    >      <values    >values</values    >    </children    >  </extendedDataElements    >  <extendedDataElements name=\"ede2\"    type=\"noValue\"    >    <values    >values</values    >    <children name=\"chname2\"    type=\"noValue\"    >      <values    >values</values    >    </children    >  </extendedDataElements    >  <reporterComponentId application=\"appl1\"    componentType=\"compType1\"    component=\"comp1\"    componentIdType=\"compIdType1\"    executionEnvironment=\"exc1\"    instanceId=\"id1\"    location=\"loc1\"    locationType=\"Hostname\"    processId=\"pr1\"    subComponent=\"sub1\"    threadId=\"th1\"/>  <sourceComponentId application=\"app2\"    componentType=\"compType2\"    component=\"comp2\"    componentIdType=\"compIdType2\"    executionEnvironment=\"exc2\"    instanceId=\"id2\"    location=\"loc2\"    locationType=\"SNA\"    processId=\"pr2\"    subComponent=\"sub2\"    threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\"    >    <msgCatalogTokens value=\"value\"/>    <msgId    >msgId</msgId    >    <msgIdType    >Name</msgIdType    >    <msgCatalogId    >msgCatalogId</msgCatalogId    >    <msgCatalogType    >msgCatalogType</msgCatalogType    >    <msgCatalog    >msgCatalog</msgCatalog    >  </msgDataElement    >  <situation categoryName=\"StartSituation\"    >    <situationType reasoningScope=\"EXTERNAL\"    successDisposition=\"SUCCESSFUL\"    situationQualifier=\"START COMPLETED\"    xsi:type=\"StartSituation\"    />  </situation    ></CommonBaseEvent    ></CommonBaseEvents    >";

        try {

            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);

            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testXMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);

            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testXMLDocumentWithMultipleEventsDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><AssociationEngine id=\"i0000000000000000000000000000045\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><AssociationEngine id=\"i0000000000000000000000000000046\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A1\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A2\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A3\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        try {

            CommonBaseEvent[] events1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);
            CommonBaseEvent[] events2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);

            assertTrue(events1.length == events2.length);

            for (int counter = 0; counter < events1.length; counter++) {

                assertEquals(events1[counter].getExtendedDataElements("root"), events2[counter].getExtendedDataElements("root"));

                assertEquals(events1[counter], events2[counter]);

                assertTrue(events1[counter].getGlobalInstanceId().startsWith("CEE736B610030D11D88000A27990EF8D1A"));
                assertTrue(events2[counter].getGlobalInstanceId().startsWith("CEE736B610030D11D88000A27990EF8D1A"));
            }

            AssociationEngine[] engines1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);
            AssociationEngine[] engines2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);

            assertTrue(engines1.length == engines2.length);

            for (int counter = 0; counter < engines1.length; counter++) {

                assertEquals(engines1[counter], engines2[counter]);

                assertTrue(engines1[counter].getId().startsWith("i000000000000000000000000000004"));
                assertTrue(engines2[counter].getId().startsWith("i000000000000000000000000000004"));
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testCBE101Sample1XMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"> <AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </contextDataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>        <children name=\"\" type=\"noValue\">          <values>values</values>       </children>       </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">       <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">     	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />     </situation>	       <a:a xmlns:a=\"http://bar\" />   </CommonBaseEvent></CommonBaseEvents>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);

            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testCBE101Sample2XMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <a:a xmlns:a=\"http://bar\" > 	     <expression logicalOperator=\"AND\"><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></preCondition><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></postCondition></expression><answer value=\"0\"/></a:a></CommonBaseEvent></CommonBaseEvents>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);

            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testEventScatteredAnyElementInXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent   xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"   xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\"   creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"i0000000000000000000000000000040\"><GetLastTrade><symbol>DEF</symbol></GetLastTrade><any1 namespace=\"##other\" processContents=\"skip\" /><sourceComponentId   application=\"appl_src\"  componentType=\"compType_src\"   component=\"comp_src\"   componentIdType=\"compIdType_src\" executionEnvironment=\"exc_src\"   instanceId=\"id_src\"   location=\"loc_src\"   locationType=\"Hostname\"  processId=\"pr_src\"   subComponent=\"sub_src\"   threadId=\"th_src\"/><any2><abc>d=\"e f\"</abc></any2><situation categoryName=\"AvailableSituation\">      <situationType     reasoningScope=\"INTERNAL\" operationDisposition=\"STARTABLE\"   availabilityDisposition=\"AVAILABLE\"  processingDisposition=\"FUNCTION_PROCESS\"  xsi:type=\"AvailableSituation\" />  </situation><any namespace=\"##other\" processContents=\"skip\" /></CommonBaseEvent>";

        try {

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(eventXML);

            String[] anyElementXMLFragments = { "<GetLastTrade><symbol>DEF</symbol></GetLastTrade>", "<any1 namespace=\"##other\" processContents=\"skip\"></any1>", "<any2><abc>d=&quot;e f&quot;</abc></any2>", "<any namespace=\"##other\" processContents=\"skip\"></any>"};

            List anyElements = event.getAny();

            assertNotNull(anyElements);

            assertTrue(anyElements.size() == 4);

            for (int counter = 0; counter < anyElements.size(); counter++) {
                assertTrue(anyElements.get(counter).equals(anyElementXMLFragments[counter]));
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testCBEPropertiesAfterXMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <a:a xmlns:a=\"http://bar\" > 	     <expression logicalOperator=\"AND\"><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></preCondition><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></postCondition></expression><answer value=\"0\"/></a:a></CommonBaseEvent></CommonBaseEvents>";

        try {

            AssociationEngine[] engines = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);

            assertNotNull(engines);

            assertTrue(engines.length == 1);

            assertNotNull(engines[0]);

            assertTrue(engines[0].getId().equals("i0000000000000000000000000000044"));
            assertTrue(engines[0].getName().equals("MyCorrelationEngine"));
            assertTrue(engines[0].getType().equals("Correlate"));

            CommonBaseEvent[] events = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);

            assertNotNull(events);

            assertTrue(events.length == 1);

            assertNotNull(events[0]);

            assertTrue(events[0].isSetCreationTime());

            assertTrue(events[0].getCreationTime().equals("2001-12-31T12:00:00"));

            assertTrue(events[0].getCreationTimeAsLong() == EventHelpers.dateToLong("2001-12-31T12:00:00"));

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
            simpleDateFormat.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "UTC")));

            assertTrue(events[0].getCreationTimeAsLong() == simpleDateFormat.parse("2001-12-31T12:00:00").getTime());

            assertTrue(events[0].isSetElapsedTime());

            assertTrue(events[0].getElapsedTime() == 0);

            assertTrue(events[0].getExtensionName().equals("Name"));

            assertTrue(events[0].getGlobalInstanceId().equals("CEE736B610030D11D88000A27990EF8D1A"));

            assertTrue(events[0].getLocalInstanceId().equals(""));

            assertTrue(events[0].isSetPriority());

            assertTrue(events[0].getPriority() == 0);

            assertTrue(events[0].isSetSeverity());

            assertTrue(events[0].getSeverity() == 0);

            assertTrue(events[0].isSetRepeatCount());

            assertTrue(events[0].getRepeatCount() == 0);

            assertTrue(events[0].isSetSequenceNumber());

            assertTrue(events[0].getSequenceNumber() == 0);

            assertTrue(events[0].getVersion().equals(""));

            List contextDataElements = events[0].getContextDataElements();

            assertNotNull(contextDataElements);

            assertTrue(contextDataElements.size() == 1);

            assertNotNull(contextDataElements.get(0));

            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getName().equals(""));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getType().equals(""));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getContextValue().equals("contextValue"));
            assertNull(((ContextDataElement) (contextDataElements.get(0))).getContextId());

            List extendedDataElements = events[0].getExtendedDataElements();

            assertNotNull(extendedDataElements);

            assertTrue(extendedDataElements.size() == 1);

            assertNotNull(extendedDataElements.get(0));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getName().equals(""));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getType().equals("noValue"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().get(0).equals("values"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(0))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getChildren().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getName().equals(""));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getType().equals("noValue"));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getValues().get(0).equals("values"));
            assertNull(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getHexValue());

            List associatedEvents = events[0].getAssociatedEvents();

            assertNotNull(associatedEvents);

            assertTrue(associatedEvents.size() == 1);

            assertNotNull(associatedEvents.get(0));

            assertTrue(((AssociatedEvent) (associatedEvents.get(0))).getResolvedEvents().equals("i0000000000000000000000000000040"));
            assertTrue(((AssociatedEvent) (associatedEvents.get(0))).getAssociationEngine().equals("i0000000000000000000000000000044"));
            assertNull(((AssociatedEvent) (associatedEvents.get(0))).getAssociationEngineInfo());

            ComponentIdentification reporterComponentId = events[0].getReporterComponentId();

            assertNotNull(reporterComponentId);

            assertTrue(reporterComponentId.getApplication().equals(""));
            assertTrue(reporterComponentId.getComponentType().equals(""));
            assertTrue(reporterComponentId.getComponent().equals(""));
            assertTrue(reporterComponentId.getComponentIdType().equals(""));
            assertTrue(reporterComponentId.getExecutionEnvironment().equals(""));
            assertTrue(reporterComponentId.getInstanceId().equals(""));
            assertTrue(reporterComponentId.getLocation().equals(""));
            assertTrue(reporterComponentId.getLocationType().equals("Name"));
            assertTrue(reporterComponentId.getProcessId().equals(""));
            assertTrue(reporterComponentId.getSubComponent().equals(""));
            assertTrue(reporterComponentId.getThreadId().equals(""));

            ComponentIdentification sourceComponentId = events[0].getSourceComponentId();

            assertNotNull(sourceComponentId);

            assertTrue(sourceComponentId.getApplication().equals(""));
            assertTrue(sourceComponentId.getComponentType().equals(""));
            assertTrue(sourceComponentId.getComponent().equals(""));
            assertTrue(sourceComponentId.getComponentIdType().equals(""));
            assertTrue(sourceComponentId.getExecutionEnvironment().equals(""));
            assertTrue(sourceComponentId.getInstanceId().equals(""));
            assertTrue(sourceComponentId.getLocation().equals(""));
            assertTrue(sourceComponentId.getLocationType().equals("Name"));
            assertTrue(sourceComponentId.getProcessId().equals(""));
            assertTrue(sourceComponentId.getSubComponent().equals(""));
            assertTrue(sourceComponentId.getThreadId().equals(""));

            MsgDataElement msgDataElement = events[0].getMsgDataElement();

            assertNotNull(msgDataElement);

            assertTrue(msgDataElement.getMsgLocale().equals("EN"));
            assertTrue(msgDataElement.getMsgCatalog().equals("msgCatalog"));
            assertTrue(msgDataElement.getMsgCatalogId().equals("msgCatalogId"));
            assertTrue(msgDataElement.getMsgCatalogType().equals("msgCatalogType"));
            assertTrue(msgDataElement.getMsgId().equals("msgId"));
            assertTrue(msgDataElement.getMsgIdType().equals("Name"));

            assertNotNull(msgDataElement.getMsgCatalogTokens());
            assertTrue(msgDataElement.getMsgCatalogTokens().size() == 1);
            assertTrue(((MsgCatalogToken) (msgDataElement.getMsgCatalogTokens().get(0))).getValue().equals(""));

            Situation situation = events[0].getSituation();

            assertNotNull(situation);

            assertTrue(situation.getCategoryName().equals("StartSituation"));

            SituationType situationType = situation.getSituationType();

            assertNotNull(situationType);

            assertTrue(situationType instanceof StartSituation);

            assertTrue(((StartSituation) (situationType)).getReasoningScope().equals("EXTERNAL"));

            assertTrue(((StartSituation) (situationType)).getSituationQualifier().equals("START COMPLETED"));

            assertTrue(((StartSituation) (situationType)).getSuccessDisposition().equals("SUCCESSFUL"));

            String anyElementXMLFragment = "<a:a xmlns:a=\"http://bar\"><expression logicalOperator=\"AND\"><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"equals\"><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName></preCondition><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName></postCondition></expression><answer value=\"0\"></answer></a:a>";

            List anyElements = events[0].getAny();

            assertNotNull(anyElements);

            assertTrue(anyElements.size() == 1);

            assertTrue(anyElements.get(0).equals(anyElementXMLFragment));

            assertEquals(events[0], EventFormatter.eventFromCanonicalXML(EventFormatter.toCanonicalXMLString(events[0], false)));
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testUnOrderedCBEPropertiesAfterXMLDocumentDeserialization_1() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">        <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <sourceComponentId location=\"loc_src\" locationType=\"HID\" component=\"comp_src\" subComponent=\"sub_src\" componentIdType=\"compIdType_src\" componentType=\"compType_src\"/>                        <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>            <contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements>                 </CommonBaseEvent></CommonBaseEvents>";

        try {

            AssociationEngine[] engines = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);

            assertNotNull(engines);

            assertTrue(engines.length == 1);

            assertNotNull(engines[0]);

            assertTrue(engines[0].getId().equals("i0000000000000000000000000000044"));
            assertTrue(engines[0].getName().equals("MyCorrelationEngine"));
            assertTrue(engines[0].getType().equals("Correlate"));

            CommonBaseEvent[] events = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);

            assertNotNull(events);

            assertTrue(events.length == 1);

            assertNotNull(events[0]);

            assertTrue(events[0].isSetCreationTime());

            assertTrue(events[0].getCreationTime().equals("2001-12-31T12:00:00"));

            assertTrue(events[0].getCreationTimeAsLong() == EventHelpers.dateToLong("2001-12-31T12:00:00"));

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
            simpleDateFormat.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "UTC")));

            assertTrue(events[0].getCreationTimeAsLong() == simpleDateFormat.parse("2001-12-31T12:00:00").getTime());

            assertTrue(events[0].isSetElapsedTime());

            assertTrue(events[0].getElapsedTime() == 0);

            assertTrue(events[0].getExtensionName().equals("Name"));

            assertTrue(events[0].getGlobalInstanceId().equals("CEE736B610030D11D88000A27990EF8D1A"));

            assertTrue(events[0].getLocalInstanceId().equals(""));

            assertTrue(events[0].isSetPriority());

            assertTrue(events[0].getPriority() == 0);

            assertTrue(events[0].isSetSeverity());

            assertTrue(events[0].getSeverity() == 0);

            assertTrue(events[0].isSetRepeatCount());

            assertTrue(events[0].getRepeatCount() == 0);

            assertTrue(events[0].isSetSequenceNumber());

            assertTrue(events[0].getSequenceNumber() == 0);

            assertTrue(events[0].getVersion().equals(""));

            List contextDataElements = events[0].getContextDataElements();

            assertNotNull(contextDataElements);

            assertTrue(contextDataElements.size() == 1);

            assertNotNull(contextDataElements.get(0));

            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getName().equals(""));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getType().equals(""));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getContextValue().equals("contextValue"));
            assertNull(((ContextDataElement) (contextDataElements.get(0))).getContextId());

            List extendedDataElements = events[0].getExtendedDataElements();

            assertNotNull(extendedDataElements);

            assertTrue(extendedDataElements.size() == 1);

            assertNotNull(extendedDataElements.get(0));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getName().equals(""));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getType().equals("noValue"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().get(0).equals("values"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(0))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getChildren().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getName().equals(""));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getType().equals("noValue"));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getValues().get(0).equals("values"));
            assertNull(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getHexValue());

            List associatedEvents = events[0].getAssociatedEvents();

            assertNotNull(associatedEvents);

            assertTrue(associatedEvents.size() == 0);

            ComponentIdentification reporterComponentId = events[0].getReporterComponentId();

            assertNull(reporterComponentId);

            ComponentIdentification sourceComponentId = events[0].getSourceComponentId();

            assertNotNull(sourceComponentId);

            assertNull(sourceComponentId.getApplication());
            assertTrue(sourceComponentId.getComponentType().equals("compType_src"));
            assertTrue(sourceComponentId.getComponent().equals("comp_src"));
            assertTrue(sourceComponentId.getComponentIdType().equals("compIdType_src"));
            assertNull(sourceComponentId.getExecutionEnvironment());
            assertNull(sourceComponentId.getInstanceId());
            assertTrue(sourceComponentId.getLocation().equals("loc_src"));
            assertTrue(sourceComponentId.getLocationType().equals("HID"));
            assertNull(sourceComponentId.getProcessId());
            assertTrue(sourceComponentId.getSubComponent().equals("sub_src"));
            assertNull(sourceComponentId.getThreadId());

            MsgDataElement msgDataElement = events[0].getMsgDataElement();

            assertNotNull(msgDataElement);

            assertTrue(msgDataElement.getMsgLocale().equals("EN"));
            assertTrue(msgDataElement.getMsgCatalog().equals("msgCatalog"));
            assertTrue(msgDataElement.getMsgCatalogId().equals("msgCatalogId"));
            assertTrue(msgDataElement.getMsgCatalogType().equals("msgCatalogType"));
            assertTrue(msgDataElement.getMsgId().equals("msgId"));
            assertTrue(msgDataElement.getMsgIdType().equals("Name"));

            assertNotNull(msgDataElement.getMsgCatalogTokens());
            assertTrue(msgDataElement.getMsgCatalogTokens().size() == 1);
            assertTrue(((MsgCatalogToken) (msgDataElement.getMsgCatalogTokens().get(0))).getValue().equals(""));

            Situation situation = events[0].getSituation();

            assertNull(situation);

            List anyElements = events[0].getAny();

            assertNotNull(anyElements);

            assertTrue(anyElements.size() == 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testUnOrderedCBEPropertiesAfterXMLDocumentDeserialization_2() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">        <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <sourceComponentId location=\"loc_src\" locationType=\"HID\" component=\"comp_src\" subComponent=\"sub_src\" componentIdType=\"compIdType_src\" componentType=\"compType_src\"/>                        </CommonBaseEvent></CommonBaseEvents>";

        try {

            AssociationEngine[] engines = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);

            assertNotNull(engines);

            assertTrue(engines.length == 1);

            assertNotNull(engines[0]);

            assertTrue(engines[0].getId().equals("i0000000000000000000000000000044"));
            assertTrue(engines[0].getName().equals("MyCorrelationEngine"));
            assertTrue(engines[0].getType().equals("Correlate"));

            CommonBaseEvent[] events = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);

            assertNotNull(events);

            assertTrue(events.length == 1);

            assertNotNull(events[0]);

            assertTrue(events[0].isSetCreationTime());

            assertTrue(events[0].getCreationTime().equals("2001-12-31T12:00:00"));

            assertTrue(events[0].getCreationTimeAsLong() == EventHelpers.dateToLong("2001-12-31T12:00:00"));

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
            simpleDateFormat.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "UTC")));

            assertTrue(events[0].getCreationTimeAsLong() == simpleDateFormat.parse("2001-12-31T12:00:00").getTime());

            assertTrue(events[0].isSetElapsedTime());

            assertTrue(events[0].getElapsedTime() == 0);

            assertTrue(events[0].getExtensionName().equals("Name"));

            assertTrue(events[0].getGlobalInstanceId().equals("CEE736B610030D11D88000A27990EF8D1A"));

            assertTrue(events[0].getLocalInstanceId().equals(""));

            assertTrue(events[0].isSetPriority());

            assertTrue(events[0].getPriority() == 0);

            assertTrue(events[0].isSetSeverity());

            assertTrue(events[0].getSeverity() == 0);

            assertTrue(events[0].isSetRepeatCount());

            assertTrue(events[0].getRepeatCount() == 0);

            assertTrue(events[0].isSetSequenceNumber());

            assertTrue(events[0].getSequenceNumber() == 0);

            assertTrue(events[0].getVersion().equals(""));

            List contextDataElements = events[0].getContextDataElements();

            assertNotNull(contextDataElements);

            assertTrue(contextDataElements.size() == 0);

            List extendedDataElements = events[0].getExtendedDataElements();

            assertNotNull(extendedDataElements);

            assertTrue(extendedDataElements.size() == 0);

            List associatedEvents = events[0].getAssociatedEvents();

            assertNotNull(associatedEvents);

            assertTrue(associatedEvents.size() == 0);

            ComponentIdentification reporterComponentId = events[0].getReporterComponentId();

            assertNull(reporterComponentId);

            ComponentIdentification sourceComponentId = events[0].getSourceComponentId();

            assertNotNull(sourceComponentId);

            assertNull(sourceComponentId.getApplication());
            assertTrue(sourceComponentId.getComponentType().equals("compType_src"));
            assertTrue(sourceComponentId.getComponent().equals("comp_src"));
            assertTrue(sourceComponentId.getComponentIdType().equals("compIdType_src"));
            assertNull(sourceComponentId.getExecutionEnvironment());
            assertNull(sourceComponentId.getInstanceId());
            assertTrue(sourceComponentId.getLocation().equals("loc_src"));
            assertTrue(sourceComponentId.getLocationType().equals("HID"));
            assertNull(sourceComponentId.getProcessId());
            assertTrue(sourceComponentId.getSubComponent().equals("sub_src"));
            assertNull(sourceComponentId.getThreadId());

            MsgDataElement msgDataElement = events[0].getMsgDataElement();

            assertNull(msgDataElement);

            Situation situation = events[0].getSituation();

            assertNotNull(situation);

            assertTrue(situation.getCategoryName().equals("StartSituation"));

            SituationType situationType = situation.getSituationType();

            assertNotNull(situationType);

            assertTrue(situationType instanceof StartSituation);

            assertTrue(((StartSituation) (situationType)).getReasoningScope().equals("EXTERNAL"));

            assertTrue(((StartSituation) (situationType)).getSituationQualifier().equals("START COMPLETED"));

            assertTrue(((StartSituation) (situationType)).getSuccessDisposition().equals("SUCCESSFUL"));

            List anyElements = events[0].getAny();

            assertNotNull(anyElements);

            assertTrue(anyElements.size() == 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testUnOrderedCBEPropertiesAfterXMLDocumentDeserialization_3() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">        <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <reporterComponentId application=\"appl_rep\" componentType=\"compType_rep\" component=\"comp_rep\" componentIdType=\"compIdType_rep\" executionEnvironment=\"exc_rep\" instanceId=\"id_rep\" location=\"loc_rep\" locationType=\"Hostname\" processId=\"pr_rep\" subComponent=\"sub_rep\" threadId=\"th_rep\"/> <sourceComponentId location=\"loc_src\" locationType=\"HID\" component=\"comp_src\" subComponent=\"sub_src\" componentIdType=\"compIdType_src\" componentType=\"compType_src\"/>                <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>            <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>                    <contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements>                                         </CommonBaseEvent></CommonBaseEvents>";

        try {

            AssociationEngine[] engines = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML);

            assertNotNull(engines);

            assertTrue(engines.length == 1);

            assertNotNull(engines[0]);

            assertTrue(engines[0].getId().equals("i0000000000000000000000000000044"));
            assertTrue(engines[0].getName().equals("MyCorrelationEngine"));
            assertTrue(engines[0].getType().equals("Correlate"));

            CommonBaseEvent[] events = EventFormatter.eventsFromCanonicalXMLDoc(eventXML);

            assertNotNull(events);

            assertTrue(events.length == 1);

            assertNotNull(events[0]);

            assertTrue(events[0].isSetCreationTime());

            assertTrue(events[0].getCreationTime().equals("2001-12-31T12:00:00"));

            assertTrue(events[0].getCreationTimeAsLong() == EventHelpers.dateToLong("2001-12-31T12:00:00"));

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
            simpleDateFormat.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "UTC")));

            assertTrue(events[0].getCreationTimeAsLong() == simpleDateFormat.parse("2001-12-31T12:00:00").getTime());

            assertTrue(events[0].isSetElapsedTime());

            assertTrue(events[0].getElapsedTime() == 0);

            assertTrue(events[0].getExtensionName().equals("Name"));

            assertTrue(events[0].getGlobalInstanceId().equals("CEE736B610030D11D88000A27990EF8D1A"));

            assertTrue(events[0].getLocalInstanceId().equals(""));

            assertTrue(events[0].isSetPriority());

            assertTrue(events[0].getPriority() == 0);

            assertTrue(events[0].isSetSeverity());

            assertTrue(events[0].getSeverity() == 0);

            assertTrue(events[0].isSetRepeatCount());

            assertTrue(events[0].getRepeatCount() == 0);

            assertTrue(events[0].isSetSequenceNumber());

            assertTrue(events[0].getSequenceNumber() == 0);

            assertTrue(events[0].getVersion().equals(""));

            List contextDataElements = events[0].getContextDataElements();

            assertNotNull(contextDataElements);

            assertTrue(contextDataElements.size() == 1);

            assertNotNull(contextDataElements.get(0));

            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getName().equals(""));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getType().equals(""));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getContextValue().equals("contextValue"));
            assertNull(((ContextDataElement) (contextDataElements.get(0))).getContextId());

            List extendedDataElements = events[0].getExtendedDataElements();

            assertNotNull(extendedDataElements);

            assertTrue(extendedDataElements.size() == 1);

            assertNotNull(extendedDataElements.get(0));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getName().equals(""));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getType().equals("noValue"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().get(0).equals("values"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(0))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getChildren().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getName().equals(""));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getType().equals("noValue"));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getValues().get(0).equals("values"));
            assertNull(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(0)))).getChildren().get(0)).getHexValue());

            List associatedEvents = events[0].getAssociatedEvents();

            assertNotNull(associatedEvents);

            assertTrue(associatedEvents.size() == 0);

            ComponentIdentification reporterComponentId = events[0].getReporterComponentId();

            assertNotNull(reporterComponentId);

            assertTrue(reporterComponentId.getApplication().equals("appl_rep"));
            assertTrue(reporterComponentId.getComponentType().equals("compType_rep"));
            assertTrue(reporterComponentId.getComponent().equals("comp_rep"));
            assertTrue(reporterComponentId.getComponentIdType().equals("compIdType_rep"));
            assertTrue(reporterComponentId.getExecutionEnvironment().equals("exc_rep"));
            assertTrue(reporterComponentId.getInstanceId().equals("id_rep"));
            assertTrue(reporterComponentId.getLocation().equals("loc_rep"));
            assertTrue(reporterComponentId.getLocationType().equals("Hostname"));
            assertTrue(reporterComponentId.getProcessId().equals("pr_rep"));
            assertTrue(reporterComponentId.getSubComponent().equals("sub_rep"));
            assertTrue(reporterComponentId.getThreadId().equals("th_rep"));

            ComponentIdentification sourceComponentId = events[0].getSourceComponentId();

            assertNotNull(sourceComponentId);

            assertNull(sourceComponentId.getApplication());
            assertTrue(sourceComponentId.getComponentType().equals("compType_src"));
            assertTrue(sourceComponentId.getComponent().equals("comp_src"));
            assertTrue(sourceComponentId.getComponentIdType().equals("compIdType_src"));
            assertNull(sourceComponentId.getExecutionEnvironment());
            assertNull(sourceComponentId.getInstanceId());
            assertTrue(sourceComponentId.getLocation().equals("loc_src"));
            assertTrue(sourceComponentId.getLocationType().equals("HID"));
            assertNull(sourceComponentId.getProcessId());
            assertTrue(sourceComponentId.getSubComponent().equals("sub_src"));
            assertNull(sourceComponentId.getThreadId());

            MsgDataElement msgDataElement = events[0].getMsgDataElement();

            assertNotNull(msgDataElement);

            assertTrue(msgDataElement.getMsgLocale().equals("EN"));
            assertTrue(msgDataElement.getMsgCatalog().equals("msgCatalog"));
            assertTrue(msgDataElement.getMsgCatalogId().equals("msgCatalogId"));
            assertTrue(msgDataElement.getMsgCatalogType().equals("msgCatalogType"));
            assertTrue(msgDataElement.getMsgId().equals("msgId"));
            assertTrue(msgDataElement.getMsgIdType().equals("Name"));

            assertNotNull(msgDataElement.getMsgCatalogTokens());
            assertTrue(msgDataElement.getMsgCatalogTokens().size() == 1);
            assertTrue(((MsgCatalogToken) (msgDataElement.getMsgCatalogTokens().get(0))).getValue().equals(""));

            Situation situation = events[0].getSituation();

            assertNotNull(situation);

            assertTrue(situation.getCategoryName().equals("StartSituation"));

            SituationType situationType = situation.getSituationType();

            assertNotNull(situationType);

            assertTrue(situationType instanceof StartSituation);

            assertTrue(((StartSituation) (situationType)).getReasoningScope().equals("EXTERNAL"));

            assertTrue(((StartSituation) (situationType)).getSituationQualifier().equals("START COMPLETED"));

            assertTrue(((StartSituation) (situationType)).getSuccessDisposition().equals("SUCCESSFUL"));

            String anyElementXMLFragment = "<a:a xmlns:a=\"http://bar\"><expression logicalOperator=\"AND\"><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"equals\"><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName></preCondition><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><testPropertyName>&quot;cbe:CommonBaseEvent:Target:contextDataElements&quot;</testPropertyName></postCondition></expression><answer value=\"0\"></answer></a:a>";

            List anyElements = events[0].getAny();

            assertNotNull(anyElements);

            assertTrue(anyElements.size() == 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testCBEPropertiesAfterXMLFragmentDeserialization_1() {

        String eventXML = "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"i0000000000000000000000000000040\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><reporterComponentId application=\"appl_rep\" componentType=\"compType_rep\" component=\"comp_rep\" componentIdType=\"compIdType_rep\" executionEnvironment=\"exc_rep\" instanceId=\"id_rep\" location=\"loc_rep\" locationType=\"Hostname\" processId=\"pr_rep\" subComponent=\"sub_rep\" threadId=\"th_rep\"/> <sourceComponentId location=\"loc_src\" locationType=\"HID\" component=\"comp_src\" subComponent=\"sub_src\" componentIdType=\"compIdType_src\" componentType=\"compType_src\"/> <msgDataElement msgLocale=\"EN\"> <msgCatalogTokens value=\"value\"/> <msgId>msgId</msgId> <msgIdType>Name</msgIdType> <msgCatalogId>msgCatalogId</msgCatalogId> <msgCatalogType>msgCatalogType</msgCatalogType> <msgCatalog>msgCatalog</msgCatalog></msgDataElement> <contextDataElements name=\"myContext\" type=\"String\"> <contextValue>contextValue</contextValue></contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"> <values>true</values></extendedDataElements><extendedDataElements name=\"ede2\" type=\"stringArray\"> <values>First Value</values><values>Second Value</values></extendedDataElements><associatedEvents resolvedEvents=\"i0000000000000000000000000000040\"> <associationEngine>i0000000000000000000000000000032</associationEngine></associatedEvents> <situation categoryName=\"StartSituation\"> <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\"/> </situation></CommonBaseEvent>";

        try {

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(eventXML);

            assertNotNull(event);

            assertTrue(event.isSetCreationTime());

            assertTrue(event.getCreationTime().equals("2001-12-31T12:00:00"));

            assertTrue(event.getCreationTimeAsLong() == EventHelpers.dateToLong("2001-12-31T12:00:00"));

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
            simpleDateFormat.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "UTC")));

            assertTrue(event.getCreationTimeAsLong() == simpleDateFormat.parse("2001-12-31T12:00:00").getTime());

            assertTrue(event.isSetElapsedTime());

            assertTrue(event.getElapsedTime() == 0);

            assertTrue(event.getExtensionName().equals("Name"));

            assertTrue(event.getGlobalInstanceId().equals("i0000000000000000000000000000040"));

            assertTrue(event.getLocalInstanceId().equals(""));

            assertTrue(event.isSetPriority());

            assertTrue(event.getPriority() == 0);

            assertTrue(event.isSetSeverity());

            assertTrue(event.getSeverity() == 0);

            assertTrue(event.isSetRepeatCount());

            assertTrue(event.getRepeatCount() == 0);

            assertTrue(event.isSetSequenceNumber());

            assertTrue(event.getSequenceNumber() == 0);

            assertTrue(event.getVersion().equals(""));

            List contextDataElements = event.getContextDataElements();

            assertNotNull(contextDataElements);

            assertTrue(contextDataElements.size() == 1);

            assertNotNull(contextDataElements.get(0));

            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getName().equals("myContext"));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getType().equals("String"));
            assertTrue(((ContextDataElement) (contextDataElements.get(0))).getContextValue().equals("contextValue"));
            assertNull(((ContextDataElement) (contextDataElements.get(0))).getContextId());

            List extendedDataElements = event.getExtendedDataElements();

            assertNotNull(extendedDataElements);

            assertTrue(extendedDataElements.size() == 2);

            assertNotNull(extendedDataElements.get(0));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getName().equals("ede1"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getType().equals("boolean"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().get(0).equals("true"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(0))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(1));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getName().equals("ede2"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getType().equals("stringArray"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getValues().size() == 2);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getValues().get(0).equals("First Value"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getValues().get(1).equals("Second Value"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(1))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getChildren().size() == 0);

            List associatedEvents = event.getAssociatedEvents();

            assertNotNull(associatedEvents);

            assertTrue(associatedEvents.size() == 1);

            assertNotNull(associatedEvents.get(0));

            assertTrue(((AssociatedEvent) (associatedEvents.get(0))).getResolvedEvents().equals("i0000000000000000000000000000040"));
            assertTrue(((AssociatedEvent) (associatedEvents.get(0))).getAssociationEngine().equals("i0000000000000000000000000000032"));
            assertNull(((AssociatedEvent) (associatedEvents.get(0))).getAssociationEngineInfo());

            ComponentIdentification reporterComponentId = event.getReporterComponentId();

            assertNotNull(reporterComponentId);

            assertTrue(reporterComponentId.getApplication().equals("appl_rep"));
            assertTrue(reporterComponentId.getComponentType().equals("compType_rep"));
            assertTrue(reporterComponentId.getComponent().equals("comp_rep"));
            assertTrue(reporterComponentId.getComponentIdType().equals("compIdType_rep"));
            assertTrue(reporterComponentId.getExecutionEnvironment().equals("exc_rep"));
            assertTrue(reporterComponentId.getInstanceId().equals("id_rep"));
            assertTrue(reporterComponentId.getLocation().equals("loc_rep"));
            assertTrue(reporterComponentId.getLocationType().equals("Hostname"));
            assertTrue(reporterComponentId.getProcessId().equals("pr_rep"));
            assertTrue(reporterComponentId.getSubComponent().equals("sub_rep"));
            assertTrue(reporterComponentId.getThreadId().equals("th_rep"));

            ComponentIdentification sourceComponentId = event.getSourceComponentId();

            assertNotNull(sourceComponentId);

            assertNull(sourceComponentId.getApplication());
            assertTrue(sourceComponentId.getComponentType().equals("compType_src"));
            assertTrue(sourceComponentId.getComponent().equals("comp_src"));
            assertTrue(sourceComponentId.getComponentIdType().equals("compIdType_src"));
            assertNull(sourceComponentId.getExecutionEnvironment());
            assertNull(sourceComponentId.getInstanceId());
            assertTrue(sourceComponentId.getLocation().equals("loc_src"));
            assertTrue(sourceComponentId.getLocationType().equals("HID"));
            assertNull(sourceComponentId.getProcessId());
            assertTrue(sourceComponentId.getSubComponent().equals("sub_src"));
            assertNull(sourceComponentId.getThreadId());

            MsgDataElement msgDataElement = event.getMsgDataElement();

            assertNotNull(msgDataElement);

            assertTrue(msgDataElement.getMsgLocale().equals("EN"));
            assertTrue(msgDataElement.getMsgCatalog().equals("msgCatalog"));
            assertTrue(msgDataElement.getMsgCatalogId().equals("msgCatalogId"));
            assertTrue(msgDataElement.getMsgCatalogType().equals("msgCatalogType"));
            assertTrue(msgDataElement.getMsgId().equals("msgId"));
            assertTrue(msgDataElement.getMsgIdType().equals("Name"));

            assertNotNull(msgDataElement.getMsgCatalogTokens());
            assertTrue(msgDataElement.getMsgCatalogTokens().size() == 1);
            assertTrue(((MsgCatalogToken) (msgDataElement.getMsgCatalogTokens().get(0))).getValue().equals("value"));

            Situation situation = event.getSituation();

            assertNotNull(situation);

            assertTrue(situation.getCategoryName().equals("StartSituation"));

            SituationType situationType = situation.getSituationType();

            assertNotNull(situationType);

            assertTrue(situationType instanceof StartSituation);

            assertTrue(((StartSituation) (situationType)).getReasoningScope().equals("EXTERNAL"));

            assertTrue(((StartSituation) (situationType)).getSituationQualifier().equals("START COMPLETED"));

            assertTrue(((StartSituation) (situationType)).getSuccessDisposition().equals("SUCCESSFUL"));

            List anyElements = event.getAny();

            assertNotNull(anyElements);

            assertTrue(anyElements.size() == 0);

            assertEquals(event, EventFormatter.eventFromCanonicalXML(EventFormatter.toCanonicalXMLString(event, false)));
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testCBEPropertiesAfterXMLFragmentDeserialization_2() {

        String eventXML = "<CommonBaseEvent creationTime=\"2004-06-06T15:13:21.891Z\" globalInstanceId=\"don6:1086534779703:3\" msg=\"ADMN0015I: AdminService initialized.\" severity=\"10\" version=\"1.0.1\">	<extendedDataElements name=\"sequenceNumber\" type=\"long\">		<values>4</values>	</extendedDataElements>	<extendedDataElements name=\"threadID\" type=\"int\">		<values>10</values>	</extendedDataElements>	<extendedDataElements name=\"level\">		<children name=\"name\" type=\"string\">			<values>AUDIT</values>		</children>		<children name=\"value\" type=\"int\">			<values>850</values>		</children>	</extendedDataElements>	<extendedDataElements name=\"loggerName\" type=\"string\">		<values>com.ibm.ws.management.AdminInitializer</values>	</extendedDataElements>	<extendedDataElements name=\"organization\" type=\"string\">		<values>IBM</values>	</extendedDataElements>	<extendedDataElements name=\"product\" type=\"string\">		<values>WebSphere</values>	</extendedDataElements>	<extendedDataElements name=\"component\" type=\"string\">		<values>Application Server</values>	</extendedDataElements>	<extendedDataElements name=\"version\" type=\"string\">		<values>Platform 6.0 Beta [BASE 6.0.0.0 o0418.30] </values>	</extendedDataElements>	<extendedDataElements name=\"processId\" type=\"string\">		<values>628</values>	</extendedDataElements>	<extendedDataElements name=\"processName\" type=\"string\">		<values>don6\\don6\\server1</values>	</extendedDataElements>	<extendedDataElements name=\"localizable\" type=\"string\">		<values>yes</values>	</extendedDataElements>	<sourceComponentId component=\"com.ibm.ws.management.AdminInitializer\" executionEnvironment=\"Windows 2000[x86]#5.0\" instanceId=\"don6\\don6\\server1\" location=\"don6\" locationType=\"Hostname\" processId=\"1086534779703\" threadId=\"main\" componentType=\"http://www.ibm.com/namespaces/autonomic/WebSphereApplicationServer\"/>	<msgDataElement msgLocale=\"en_US\">		<msgCatalogId>ADMN0015I</msgCatalogId>		<msgCatalogType>Java</msgCatalogType>		<msgCatalog>com.ibm.ws.management.resources.adminservice</msgCatalog>	</msgDataElement></CommonBaseEvent>";

        try {

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(eventXML);

            assertNotNull(event);

            assertTrue(event.isSetCreationTime());

            assertTrue(event.getCreationTime().equals("2004-06-06T15:13:21.891Z"));

            assertTrue(event.getCreationTimeAsLong() == EventHelpers.dateToLong("2004-06-06T15:13:21.891Z"));

            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
            simpleDateFormat.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "UTC")));

            assertTrue(event.getCreationTimeAsLong() == simpleDateFormat.parse("2004-06-06T15:13:21.891Z").getTime());

            assertFalse(event.isSetElapsedTime());

            assertTrue(event.getElapsedTime() == 0);

            assertNull(event.getExtensionName());

            assertTrue(event.getGlobalInstanceId().equals("don6:1086534779703:3"));

            assertNull(event.getLocalInstanceId());

            assertFalse(event.isSetPriority());

            assertTrue(event.getPriority() == 0);

            assertTrue(event.isSetSeverity());

            assertTrue(event.getSeverity() == 10);

            assertFalse(event.isSetRepeatCount());

            assertTrue(event.getRepeatCount() == 0);

            assertFalse(event.isSetSequenceNumber());

            assertTrue(event.getSequenceNumber() == 0);

            assertTrue(event.getVersion().equals("1.0.1"));

            assertTrue(event.getMsg().equals("ADMN0015I: AdminService initialized."));

            List contextDataElements = event.getContextDataElements();

            assertNotNull(contextDataElements);

            assertTrue(contextDataElements.size() == 0);

            List extendedDataElements = event.getExtendedDataElements();

            assertNotNull(extendedDataElements);

            assertTrue(extendedDataElements.size() == 11);

            assertNotNull(extendedDataElements.get(0));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getName().equals("sequenceNumber"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getType().equals("long"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getValues().get(0).equals("4"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(0))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(0))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(1));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getName().equals("threadID"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getType().equals("int"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getValues().get(0).equals("10"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(1))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(1))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(2));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(2))).getName().equals("level"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(2))).getType());
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(2))).getValues().size() == 0);
            assertNull(((ExtendedDataElement) (extendedDataElements.get(2))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().size() == 2);

            assertNotNull(((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(0));

            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(0))).getName().equals("name"));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(0))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(0))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(0))).getValues().get(0).equals("AUDIT"));
            assertNull(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(0))).getHexValue());

            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(0))).getChildren().size() == 0);

            assertNotNull(((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(1));

            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(1))).getName().equals("value"));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(1))).getType().equals("int"));
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(1))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(1))).getValues().get(0).equals("850"));
            assertNull(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(1))).getHexValue());

            assertTrue(((ExtendedDataElement) (((ExtendedDataElement) (extendedDataElements.get(2))).getChildren().get(1))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(3));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(3))).getName().equals("loggerName"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(3))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(3))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(3))).getValues().get(0).equals("com.ibm.ws.management.AdminInitializer"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(3))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(3))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(4));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(4))).getName().equals("organization"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(4))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(4))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(4))).getValues().get(0).equals("IBM"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(4))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(4))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(5));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(5))).getName().equals("product"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(5))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(5))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(5))).getValues().get(0).equals("WebSphere"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(5))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(5))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(6));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(6))).getName().equals("component"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(6))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(6))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(6))).getValues().get(0).equals("Application Server"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(6))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(6))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(7));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(7))).getName().equals("version"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(7))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(7))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(7))).getValues().get(0).equals("Platform 6.0 Beta [BASE 6.0.0.0 o0418.30] "));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(7))).getValuesAsString().equals("Platform 6.0 Beta [BASE 6.0.0.0 o0418.30] "));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(7))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(7))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(8));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(8))).getName().equals("processId"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(8))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(8))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(8))).getValues().get(0).equals("628"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(8))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(8))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(9));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(9))).getName().equals("processName"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(9))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(9))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(9))).getValues().get(0).equals("don6\\don6\\server1"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(9))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(9))).getChildren().size() == 0);

            assertNotNull(extendedDataElements.get(10));

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(10))).getName().equals("localizable"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(10))).getType().equals("string"));
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(10))).getValues().size() == 1);
            assertTrue(((ExtendedDataElement) (extendedDataElements.get(10))).getValues().get(0).equals("yes"));
            assertNull(((ExtendedDataElement) (extendedDataElements.get(10))).getHexValue());

            assertTrue(((ExtendedDataElement) (extendedDataElements.get(10))).getChildren().size() == 0);

            List associatedEvents = event.getAssociatedEvents();

            assertNotNull(associatedEvents);

            assertTrue(associatedEvents.size() == 0);

            ComponentIdentification reporterComponentId = event.getReporterComponentId();

            assertNull(reporterComponentId);

            ComponentIdentification sourceComponentId = event.getSourceComponentId();

            assertNotNull(sourceComponentId);

            assertNull(sourceComponentId.getApplication());
            assertTrue(sourceComponentId.getComponentType().equals("http://www.ibm.com/namespaces/autonomic/WebSphereApplicationServer"));
            assertTrue(sourceComponentId.getComponent().equals("com.ibm.ws.management.AdminInitializer"));
            assertNull(sourceComponentId.getComponentIdType());
            assertTrue(sourceComponentId.getExecutionEnvironment().equals("Windows 2000[x86]#5.0"));
            assertTrue(sourceComponentId.getInstanceId().equals("don6\\don6\\server1"));
            assertTrue(sourceComponentId.getLocation().equals("don6"));
            assertTrue(sourceComponentId.getLocationType().equals("Hostname"));
            assertTrue(sourceComponentId.getProcessId().equals("1086534779703"));
            assertNull(sourceComponentId.getSubComponent());
            assertTrue(sourceComponentId.getThreadId().equals("main"));

            MsgDataElement msgDataElement = event.getMsgDataElement();

            assertNotNull(msgDataElement);

            assertTrue(msgDataElement.getMsgLocale().equals("en_US"));
            assertTrue(msgDataElement.getMsgCatalog().equals("com.ibm.ws.management.resources.adminservice"));
            assertTrue(msgDataElement.getMsgCatalogId().equals("ADMN0015I"));
            assertTrue(msgDataElement.getMsgCatalogType().equals("Java"));
            assertNull(msgDataElement.getMsgId());
            assertNull(msgDataElement.getMsgIdType());

            assertNotNull(msgDataElement.getMsgCatalogTokens());
            assertTrue(msgDataElement.getMsgCatalogTokens().size() == 0);

            Situation situation = event.getSituation();

            assertNull(situation);

            List anyElements = event.getAny();

            assertNotNull(anyElements);

            assertTrue(anyElements.size() == 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"LOCAL INSTANCE ID :-) :-)\" msg=\"msg 2 &lt;&gt;&amp;;&quot;&apos; :-)\" elapsedTime=\"123\" priority=\"3\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"5\" version=\"1.0.1\"><contextDataElements name=\"ettk id\" type=\"anIDTYPE\"><contextId> FIRST</contextId></contextDataElements><contextDataElements name=\"&lt;ettk&amp;value&gt;\" type=\"anVALUETYPE\"><contextValue>SECOND</contextValue></contextDataElements><extendedDataElements name=\"root\" type=\"long\"><children name=\"Child2\" type=\"long\"><children name=\"Child31\" type=\"long\"><children name=\"Child41\" type=\"hexBinary\"><hexValue>818283</hexValue></children></children><children name=\"Child32\" type=\"long\"><children name=\"Child42\" type=\"stringArray\"><values>one</values><values>two</values><values>three</values></children></children><children name=\"Child33\" type=\"long\"><children name=\"Child43\" type=\"longArray\"><values>4</values><values>5</values></children></children></children></extendedDataElements><associatedEvents resolvedEvents=\"cbe1 cbe2\"><associationEngine>AE id</associationEngine></associatedEvents><sourceComponentId application=\"ettkAPP\"/><msgDataElement msgLocale=\"msgLocale\"><msgCatalogTokens value=\"msgCatalogToken1\"/><msgCatalogTokens value=\"msgCatalogToken2\"/><msgId>msgId</msgId><msgIdType>msgIdType</msgIdType><msgCatalogId>msgCatalogId</msgCatalogId><msgCatalogType>msgCatalogType</msgCatalogType><msgCatalog>MsgCatalog</msgCatalog></msgDataElement></CommonBaseEvent>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventFromCanonicalXML(eventXML);
            CommonBaseEvent event2 = EventFormatter.eventFromCanonicalXML(eventXML);

            assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testXMLFragmentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {

            eventUnformatted = EventFormatter.eventFromCanonicalXML(eventXMLUnformatted);
            eventFormatted = EventFormatter.eventFromCanonicalXML(eventXMLFormatted);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testXMLDocumentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {

            eventUnformatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLUnformatted)[0];
            eventFormatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLFormatted)[0];
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }

    public void testEventFactoryXMLTemplateDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><TemplateEvent version=\"1.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"templateEvent.xsd\"><CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent></TemplateEvent>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(event1.getSituation(), event2.getSituation());

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testEventFactoryXMLTemplateFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventFromCanonicalXML(eventXML);
            CommonBaseEvent event2 = EventFormatter.eventFromCanonicalXML(eventXML);

            assertEquals(event1.getSituation(), event2.getSituation());

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testEventFactoryXMLTemplateFragmentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL", "LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {

            eventUnformatted = EventFormatter.eventFromCanonicalXML(eventXMLUnformatted);
            eventFormatted = EventFormatter.eventFromCanonicalXML(eventXMLFormatted);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testEventFactoryXMLTemplateDocumentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL", "LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = null;
        CommonBaseEvent eventFormatted = null;

        try {

            eventUnformatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLUnformatted)[0];
            eventFormatted = EventFormatter.eventsFromCanonicalXMLDoc(eventXMLFormatted)[0];
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }

    public void testExtendedDataElementTypesInXMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];
            CommonBaseEvent event2 = EventFormatter.eventsFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
            assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
            assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

            assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
            assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

            AssociationEngine engine1 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];
            AssociationEngine engine2 = EventFormatter.associationEnginesFromCanonicalXMLDoc(eventXML)[0];

            assertEquals(engine1, engine2);

            assertEquals(engine1.getId(), "i0000000000000000000000000000044");
            assertEquals(engine2.getId(), "i0000000000000000000000000000044");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    public void testExtendedDataElementTypesInXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent>";

        try {

            CommonBaseEvent event1 = EventFormatter.eventFromCanonicalXML(eventXML);
            CommonBaseEvent event2 = EventFormatter.eventFromCanonicalXML(eventXML);

            assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
            assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
            assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

            assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
            assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

            assertEquals(event1, event2);

            assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
            assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }

    }

    public void testDeprecatedXMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements>  <extendedDataElements name=\"ede1\" type=\"noValue\">    <values>values</values>    <children name=\"chname1\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <extendedDataElements name=\"ede2\" type=\"noValue\">    <values>values</values>    <children name=\"chname2\" type=\"noValue\">      <values>values</values>    </children>  </extendedDataElements>  <reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);

        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
    }

    public void testDeprecatedCBE101Sample1XMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\">    <contextDataElements name=\"myContext\" type=\"String\">      <contextValue>contextValue</contextValue>     </contextDataElements>    <extendedDataElements name=\"\" type=\"noValue\">      <values>values</values>        <children name=\"\" type=\"noValue\">          <values>values</values>       </children>       </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngineInfo id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">       <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">     	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />     </situation>	       <a:a xmlns:a=\"http://bar\" />   </CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);

        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
    }

    public void testDeprecatedCBE101Sample2XMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" elapsedTime=\"0\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"\" msg=\"\" priority=\"0\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"0\" version=\"\"><contextDataElements name=\"\" type=\"\"><contextValue>contextValue</contextValue></contextDataElements> <extendedDataElements name=\"\" type=\"noValue\"><values>values</values> <children name=\"\" type=\"noValue\"> <values>values</values>    </children>     </extendedDataElements>    <associatedEvents resolvedEvents=\"i0000000000000000000000000000040\">      <associationEngine>i0000000000000000000000000000044</associationEngine>    </associatedEvents>    <reporterComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <sourceComponentId application=\"\" componentType=\"\" component=\"\" componentIdType=\"\" executionEnvironment=\"\" instanceId=\"\" location=\"\" locationType=\"Name\" processId=\"\" subComponent=\"\" threadId=\"\"/>    <msgDataElement msgLocale=\"EN\">      <msgCatalogTokens value=\"\"/>      <msgId>msgId</msgId>      <msgIdType>Name</msgIdType>      <msgCatalogId>msgCatalogId</msgCatalogId>      <msgCatalogType>msgCatalogType</msgCatalogType>      <msgCatalog>msgCatalog</msgCatalog>    </msgDataElement>    <situation categoryName=\"StartSituation\">    	<situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />	</situation> 	 <a:a xmlns:a=\"http://bar\" > 	     <expression logicalOperator=\"AND\"><preCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\"  comparisonOperator=\"equals\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></preCondition><postCondition xsi:type=\"SimpleNodeType\" propertyName=\"CommonBaseEvent:Source:contextDataElements\" comparisonOperator=\"notNULL\"><testPropertyName>\"cbe:CommonBaseEvent:Target:contextDataElements\"</testPropertyName></postCondition></expression><answer value=\"0\"/></a:a></CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);

        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
    }

    public void testDeprecatedXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\" localInstanceId=\"LOCAL INSTANCE ID :-) :-)\" msg=\"msg 2 &lt;&gt;&amp;;&quot;&apos; :-)\" elapsedTime=\"123\" priority=\"3\" repeatCount=\"0\" sequenceNumber=\"0\" severity=\"5\" version=\"1.0.1\"><contextDataElements name=\"ettk id\" type=\"anIDTYPE\"><contextId> FIRST</contextId></contextDataElements><contextDataElements name=\"&lt;ettk&amp;value&gt;\" type=\"anVALUETYPE\"><contextValue>SECOND</contextValue></contextDataElements><extendedDataElements name=\"root\" type=\"long\"><children name=\"Child2\" type=\"long\"><children name=\"Child31\" type=\"long\"><children name=\"Child41\" type=\"hexBinary\"><hexValue>818283</hexValue></children></children><children name=\"Child32\" type=\"long\"><children name=\"Child42\" type=\"stringArray\"><values>one</values><values>two</values><values>three</values></children></children><children name=\"Child33\" type=\"long\"><children name=\"Child43\" type=\"longArray\"><values>4</values><values>5</values></children></children></children></extendedDataElements><associatedEvents resolvedEvents=\"cbe1 cbe2\"><associationEngine>AE id</associationEngine></associatedEvents><sourceComponentId application=\"ettkAPP\"/><msgDataElement msgLocale=\"msgLocale\"><msgCatalogTokens value=\"msgCatalogToken1\"/><msgCatalogTokens value=\"msgCatalogToken2\"/><msgId>msgId</msgId><msgIdType>msgIdType</msgIdType><msgCatalogId>msgCatalogId</msgCatalogId><msgCatalogType>msgCatalogType</msgCatalogType><msgCatalog>MsgCatalog</msgCatalog></msgDataElement></CommonBaseEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(event1, eventXML);
        EventFormatter.fromCanonicalXMLString(event2, eventXML);

        assertEquals(event1.getExtendedDataElements("root"), event2.getExtendedDataElements("root"));

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }

    public void testDeprecatedXMLFragmentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLString(eventFormatted, eventXMLFormatted);

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testDeprecatedXMLDocumentSerialization() {

        CommonBaseEvent event = Util.generateEvent();

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLDocString(eventFormatted, eventXMLFormatted);

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }

    public void testDeprecatedEventFactoryXMLTemplateDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?><TemplateEvent version=\"1.0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"templateEvent.xsd\"><CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent></TemplateEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(event1.getSituation(), event2.getSituation());

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }

    public void testDeprecatedEventFactoryXMLTemplateFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\"><situation categoryName=\"ReportSituation\"><situationType xsi:type=\"ReportSituation\" reasoningScope=\"INTERNAL\" reportCategory=\"LOG\"/></situation></CommonBaseEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(event1, eventXML);
        EventFormatter.fromCanonicalXMLString(event2, eventXML);

        assertEquals(event1.getSituation(), event2.getSituation());

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }

    public void testDeprecatedEventFactoryXMLTemplateFragmentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL", "LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLString(eventFormatted, eventXMLFormatted);

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLString(eventFormatted, true));
    }

    public void testDeprecatedEventFactoryXMLTemplateDocumentSerialization() {

        CommonBaseEvent event = EventFactory.eINSTANCE.createCommonBaseEvent();
        Situation situation = EventFactory.eINSTANCE.createSituation();
        situation.setCategoryName("ReportSituation");
        situation.setReportSituation("INTERNAL", "LOG");
        event.setSituation(situation);

        String eventXMLUnformatted = EventFormatter.toCanonicalXMLDocString(event, false);
        String eventXMLFormatted = EventFormatter.toCanonicalXMLDocString(event, true);

        CommonBaseEvent eventUnformatted = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent eventFormatted = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(eventUnformatted, eventXMLUnformatted);
        EventFormatter.fromCanonicalXMLDocString(eventFormatted, eventXMLFormatted);

        assertEquals(event, eventUnformatted);
        assertEquals(event, eventFormatted);

        assertEquals(eventXMLUnformatted, EventFormatter.toCanonicalXMLDocString(eventUnformatted, false));
        assertEquals(eventXMLFormatted, EventFormatter.toCanonicalXMLDocString(eventFormatted, true));
    }

    public void testDeprecatedExtendedDataElementTypesInXMLDocumentDeserialization() {

        String eventXML = "<?xml version=\"1.0\" encoding=\"ASCII\"?><CommonBaseEvents xmlns:cbe=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd \"><AssociationEngine id=\"i0000000000000000000000000000044\" name=\"MyCorrelationEngine\" type=\"Correlate\"/><CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent></CommonBaseEvents>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLDocString(event1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(event2, eventXML);

        assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
        assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
        assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

        assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
        assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");

        AssociationEngine engine1 = EventFactory.eINSTANCE.createAssociationEngine();
        AssociationEngine engine2 = EventFactory.eINSTANCE.createAssociationEngine();

        EventFormatter.fromCanonicalXMLDocString(engine1, eventXML);
        EventFormatter.fromCanonicalXMLDocString(engine2, eventXML);

        assertEquals(engine1, engine2);

        assertEquals(engine1.getId(), "i0000000000000000000000000000044");
        assertEquals(engine2.getId(), "i0000000000000000000000000000044");
    }

    public void testDeprecatedExtendedDataElementTypesInXMLFragmentDeserialization() {

        String eventXML = "<CommonBaseEvent creationTime=\"2001-12-31T12:00:00\" extensionName=\"Name\" globalInstanceId=\"CEE736B610030D11D88000A27990EF8D1A\">  <contextDataElements name=\"myContext\" type=\"String\">    <contextValue>contextValue</contextValue>  </contextDataElements><extendedDataElements name=\"ede1\" type=\"boolean\"><values>true</values><children name=\"chname1\" type=\"int\"><values>10</values></children><children name=\"chname2\" type=\"booleanArray\"> <values>[true,false]</values></children></extendedDataElements><extendedDataElements name=\"ede2\" type=\"string\"><values>&quot;stringValue&quot;</values><children name=\"chname3\" type=\"noValue\"></children></extendedDataElements><reporterComponentId application=\"appl1\" componentType=\"compType1\" component=\"comp1\" componentIdType=\"compIdType1\" executionEnvironment=\"exc1\" instanceId=\"id1\" location=\"loc1\" locationType=\"Hostname\" processId=\"pr1\" subComponent=\"sub1\" threadId=\"th1\"/>  <sourceComponentId application=\"app2\" componentType=\"compType2\" component=\"comp2\" componentIdType=\"compIdType2\" executionEnvironment=\"exc2\" instanceId=\"id2\" location=\"loc2\" locationType=\"SNA\" processId=\"pr2\" subComponent=\"sub2\" threadId=\"th2\"/>  <msgDataElement msgLocale=\"EN\">    <msgCatalogTokens value=\"value\"/>    <msgId>msgId</msgId>    <msgIdType>Name</msgIdType>    <msgCatalogId>msgCatalogId</msgCatalogId>    <msgCatalogType>msgCatalogType</msgCatalogType>    <msgCatalog>msgCatalog</msgCatalog>  </msgDataElement>  <situation categoryName=\"StartSituation\">    <situationType reasoningScope=\"EXTERNAL\" successDisposition=\"SUCCESSFUL\" situationQualifier=\"START COMPLETED\" xsi:type=\"StartSituation\" />  </situation></CommonBaseEvent>";

        CommonBaseEvent event1 = EventFactory.eINSTANCE.createCommonBaseEvent();
        CommonBaseEvent event2 = EventFactory.eINSTANCE.createCommonBaseEvent();

        EventFormatter.fromCanonicalXMLString(event1, eventXML);
        EventFormatter.fromCanonicalXMLString(event2, eventXML);

        assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN);
        assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname1").get(0))).getType(), ExtendedDataElement.TYPE_INT);
        assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede1").get(0))).getChildren("chname2").get(0))).getType(), ExtendedDataElement.TYPE_BOOLEAN_ARRAY);

        assertEquals(((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getType(), ExtendedDataElement.TYPE_STRING);
        assertEquals(((ExtendedDataElement) (((ExtendedDataElement) (event1.getExtendedDataElements("ede2").get(0))).getChildren("chname3").get(0))).getType(), ExtendedDataElement.TYPE_NO_VALUE);

        assertEquals(event1, event2);

        assertEquals(event1.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
        assertEquals(event2.getGlobalInstanceId(), "CEE736B610030D11D88000A27990EF8D1A");
    }
}