/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.ISourceReference;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.search.IJavaSearchConstants;
import org.eclipse.jdt.core.search.IJavaSearchResultCollector;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;
import org.eclipse.jface.text.IDocument;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.texteditor.ITextEditor;

import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * A utility class to open a Java file selecting a specific element in the file.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class OpenJavaSource
{
	/**
	 * Eclipse's Java perspective ID.
	 */
	private final static String JAVA_PERSPECTIVE_ID = "org.eclipse.jdt.ui.JavaPerspective";

	/**
	 * Opens the source for a selected object in the text editor of the Java Perspective.
	 * @param String pattern - the object that we wish to display the source for.  The pattern should be the full
	 * package name of the class or method.  For a method, the signature is also part of the pattern.  If the method
	 * is a constructor, the class name is followed by the signature.  
	 * Eg., a typical method would be com.ibm.etools.pd.core.util.OpenSourceUtil.openSource(String, int) void
	 * and a constructor would be com.ibm.ibm.etools.pd.core.util.OpenSourceUtil()
	 * @param javaType The java type identifies whether the pattern represents a Type, a method, or a constructor.
	 * The values are defined in org.eclipse.jdt.core.search.IJavaSearchConstants and can be TYPE, METHOD, or CONSTRUCTOR 
	 * respectively.	
	 * @param searchScope The search scope for the source to be opened.  If null, the WorkspaceScope is be used;
	 * @param switchToJavaPerspective If <code>true</code> the Java perspective is be opened.
	 * @return <code>true</code> if the editor was opened or <code>false</code> otherwise
	 */
	public static boolean openSource(String pattern, int javaType, IJavaSearchScope searchScope, boolean switchToJavaPerspective)
	{
		OpenSourceJavaSearchResultCollector collector = getSearchResults(pattern, javaType, searchScope);
		if (collector != null && collector.getFile() != null) {
			return openWbSource(collector.getFile(), collector.getStart(), collector.getEnd(), switchToJavaPerspective);
		}
		return false;
	}

	public static IFile getSourceFile(String pattern, int javaType, IJavaSearchScope searchScope) {
		OpenSourceJavaSearchResultCollector collector = getSearchResults(pattern, javaType, searchScope);
		if (collector != null) {
			return collector.getFile();
		}
		return null;
	}
	
	private static OpenSourceJavaSearchResultCollector getSearchResults(String pattern, int javaType, IJavaSearchScope searchScope) {
		
		if (searchScope == null)
			searchScope = SearchEngine.createWorkspaceScope(); // Scope the search to the entire workspace

		if (pattern == null)
			pattern = "";

		if (pattern != "")
		{
			// Create a search engine and search the workspace for the selected object.
			SearchEngine searchEngine = new SearchEngine();

			// Handle patterns for inner classes
			int idx = pattern.indexOf("(");
			if (idx != -1)
			{
				idx = pattern.substring(0, idx).lastIndexOf("$");
			}
			else
			{
				idx = pattern.lastIndexOf("$");
			}
			String className = null;
			if (idx != -1)
			{ // We have an inner class
				// Strip out just the class name
				if (javaType == org.eclipse.jdt.core.search.IJavaSearchConstants.METHOD)
				{
					int idx1 = pattern.lastIndexOf(".");
					if (idx1 != -1)
					{
						className = pattern.substring(0, idx1);
					}
				}
				else if (javaType == org.eclipse.jdt.core.search.IJavaSearchConstants.CONSTRUCTOR)
				{
					int idx1 = pattern.indexOf("(");
					if (idx1 != -1)
					{
						className = pattern.substring(0, idx1);
					}
				}
				else
				{
					className = pattern;
				}
				pattern = pattern.substring(idx + 1);
			}
			OpenSourceJavaSearchResultCollector collector = new OpenSourceJavaSearchResultCollector(className);
			// Perform the search
			try
			{
				searchEngine.search(ResourcesPlugin.getWorkspace(), pattern, // The pattern to search for as defined above
				javaType, // CLASS or METHOD
				IJavaSearchConstants.DECLARATIONS, // Only look for definitions of the object
				searchScope, collector // The collector class that catches the search results
				);
			}
			catch (JavaModelException e) {
				// This error will result in falling into the openSourceFailed code below
			}
			return collector;
		}
		return null;
	}
	
	/**
	  * OpenSourceJavaSearchResultCollector is responsible for handling the search results for finding 
	  * the source and source range for the given selected object.  An instance of this class is created and
	  * passed to the SearchEngine.
	  */
	private static class OpenSourceJavaSearchResultCollector implements IJavaSearchResultCollector
	{
		private IResource _resource; // The file that the selected object belongs to
		private int _start; // The starting offset into the source for the selected object
		private int _end; // The ending offset into the source for the selected object
		private String _className;

		public OpenSourceJavaSearchResultCollector(String className)
		{
			_className = className;
		}

		/**
		 * aboutToStart is called when a new search is about to start.  All member variables are initialized
		 * here.
		 */
		public void aboutToStart()
		{
			_resource = null;
			_start = -1;
			_end = -1;
		}

		/**
		 * accept is called for each match of the search criteria.  Since in the context of Open Source
		 * there should be no ambiguity of the searched object, there should only ever be zero or one matches.
		 * @param see IJavaSearchResultCollector for details
		 */
		public void accept(IResource resource, int start, int end, IJavaElement enclosingElement, int accuracy)
		{
			boolean found = false;
			if (_className != null)
			{

				if (enclosingElement instanceof IType && _className.compareTo(((IType) enclosingElement).getFullyQualifiedName()) == 0)
				{
					found = true;
				}
				else if (enclosingElement instanceof IMethod && _className.compareTo(((IMethod) enclosingElement).getDeclaringType().getFullyQualifiedName()) == 0)
				{
					found = true;
				}
			}
			else
			{
				found = true;
			}

			if (found)
			{
				_resource = resource; // Cache the resource
				_start = start; // Cache the returned start offset
				_end = end; // Cache the returned ending offset

				// The returned starting and ending offsets are to the specific name of the selected object that was
				// found.  Here we try to do better than that by finding the source range for the whole enclosingElement
				// for the found object.  If this fails for some reason, we always have the supplied starting and ending
				// offsets.
				try
				{
					if (enclosingElement != null)
					{
						if (enclosingElement instanceof ISourceReference)
						{
							ISourceReference sourceReference = (ISourceReference) enclosingElement;
							ISourceRange sourceRange = sourceReference.getSourceRange();
							if (sourceRange != null)
							{
								_start = sourceRange.getOffset();
								_end = _start + sourceRange.getLength();
							}
						}
					}
				}
				catch (JavaModelException e)
				{
					// We don't care if this didn't work since we have already cached the supplied starting and 
					// ending offsets.
				}
			}
		}

		/**
		 * done is called when the search is finished.  We have nothing special to do here.
		 */
		public void done()
		{
		}

		/**
		 * getProgressMonitor is called to put up a progress monitor while the search is active.  We don't use
		 * this.
		 */
		public IProgressMonitor getProgressMonitor()
		{
			return null;
		}

		/**
		 * getFile is used to return the IFile for the found object.  
		 * @return IFile for the found object.  May return null if the object was not
		 * found or if no IFile could be created from the searched objects resource.
		 */
		public IFile getFile()
		{
			if (_resource != null && _resource.getType() == IResource.FILE)
			{
				return (IFile) _resource;
			}
			else
			{
				return null;
			}
		}

		/**
		 * getStart returns the starting source offset into the IFile of the found object.
		 * @return int - If possible, the starting offset will be the start of the enclosing element, otherwise it will be 
		 * the starting offset of the searched text itself.
		 * If the object was not found, -1 will be returned.
		 */
		public int getStart()
		{
			return _start;
		}

		/**
		 * getEnd returns the ending source offset into the IFile of the found object.
		 * @return int - If possible, the ending offset will be the end of the enclosing element, otherwise it will be 
		 * the ending offset of the searched text itself.
		 * If the object was not found, -1 will be returned.
		 */
		public int getEnd()
		{
			return _end;
		}
	}

	/**
	 * openWbSource is used to open the specified source in the text editor of the Java Perspective.
	 * In addition, it moves the source to the start of the specified source region and highlights
	 * the region.
	 * @param res org.eclipse.core.resources.IResource - the file to open
	 * @param start int - the start of the source region
	 * @param end int - the end of the source region
	 * @return boolean - true if the source was opened successfully, false otherwise.
	 */
	private static boolean openWbSource(IFile res, int start, int end, boolean switchToJavaPerspective)
	{
		IEditorPart editor = null;
		if(switchToJavaPerspective)
		{
			IWorkbenchPage javaPage = getJavaActivePage();
			if(javaPage == null)
				return false;
			
			try
			{
				editor = IDE.openEditor(javaPage, res, true);
			}
			catch (PartInitException e)
			{
				HyadesUIPlugin.logError(e);
			}
		}
		else
		{
			editor = UIUtil.openEditor(res, null, false);
		}
				
		try
		{
			if (editor != null && editor instanceof ITextEditor)
			{
				ITextEditor textEditor = (ITextEditor) editor;
				IDocument document = textEditor.getDocumentProvider().getDocument(editor.getEditorInput());
				int length = document.getLength();

				if (end - 1 < length && start < length)
				{
					textEditor.setHighlightRange(start, end - start, true);
					return true;
				}
			}
		}
		catch (Exception exc)
		{
			HyadesUIPlugin.logError(exc);
		}

		return false;
	}

	/**
	 * getJavaActivePage is used to locate and return the workbench page for the Java Perspective.
	 */
	private static IWorkbenchPage getJavaActivePage()
	{
		IWorkbenchPage page = UIUtil.getActiveWorkbenchPage();
		if (page == null || !page.getPerspective().getId().equals(JAVA_PERSPECTIVE_ID))
		{
			final IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
			page = null;

			try
			{
				IWorkbenchPage[] persps = dwindow.getPages();
				for (int idx = 0; idx < persps.length; idx++)
				{
					if (persps[idx].getPerspective().getId().equals(JAVA_PERSPECTIVE_ID))
					{
						//trace perspective
						page = persps[idx];

						dwindow.setActivePage(page);
						break;
					}
				}

				if (page == null)
				{
					IAdaptable element = ResourcesPlugin.getWorkspace().getRoot();
					IWorkbench workBench = dwindow.getWorkbench();
					if (workBench != null && element != null)
						page = workBench.showPerspective(JAVA_PERSPECTIVE_ID, dwindow, element);
				}
			}
			catch (Exception exc)
			{
				HyadesUIPlugin.logError(exc);
			}
		}

		return page;
	}	
}