/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.common;

import java.util.Date;
import java.util.HashMap;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.fragments.BVRCombinedFragment;
import org.eclipse.hyades.models.common.interactions.BVRExecutionOccurrence;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTypedEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofileFactoryImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.common.util.SaveManager;

/**
 * @author amathur
 *
 * ExecutionContext - This class serves 2 purposes
 * a. As a lookup mechanism for the execution history/result that a 
 *    given event belongs to
 * b. Log and save an execution related event
 */
public class ExecutionContext {
	
	// Static variables
	public static final boolean debug = true;
	public static char   separator = '.';
	public static String root = "ROOT";
	
	
	// Instance variables
	private TPFTestSuite 		  testSuite = null;	  /* Root test suite being executed */
	private ResourceSet			  resourceSet;
	private URI					  testSuiteURI;
	
	protected ExecutionResultData rootResult = null;	  /* Root ExecutionResult */
	protected HashMap 			  resultMap = new HashMap(); /* Map for all execution results */
	private Resource 			  resource = null;	  /* Resource for the execution result */
	private String 				  executionResultLocation;  /* Location for the execution result */
	
	private int agentCount=0;
	
	/**
	 * This class handles the creation of execution results for all tests 
	 * run within this test suite, logging events and also saving the resource.
	 * 
	 * @param suite - TPFTestSuite being run
	 * @param id - ID of the test suite
	 * @param executionResultLocation - folder where the results resource needs to be created
	 * @param executionResultName - name of the execution result specified by the user
	 * 
	 */
	public ExecutionContext(TPFTestSuite suite, String id, String executionResultLocation, String executionResultName) {
		this.testSuite = suite;
		resourceSet = suite.eResource().getResourceSet();
		testSuiteURI = EcoreUtil.getURI(testSuite);
		
		this.executionResultLocation = executionResultLocation;
		
		rootResult = new ExecutionResultData(getTestSuite(), suite.getId());
		long timestamp = new Date().getTime();
		resource = createExecutionResultResource(timestamp);
		resource.getContents().add(rootResult.getResult());
		
		if(executionResultName == null)
			executionResultName = suite.getName() + " " + timestamp;
		rootResult.getResult().setName(executionResultName);
		
		SaveManager.saveResource(resource);
		SaveManager.getInstance().addResource(resource);
		
		resultMap.put(getTestSuite().getId() + "(1)", rootResult);		
	}
	
	/**
	 * This class handles the creation of execution results for all tests 
	 * run within this test suite, logging events and also saving the resource.
	 * 
	 * @param suite - TPFTestSuite being run
	 * @param id - ID of the test suite
	 * @param location - location where the execution will take place.
	 * @param executionResultLocation - folder where the results resource needs to be created
	 * @param executionResultName - name of the execution result specified by the user
	 * 
	 */
	public ExecutionContext(TPFTestSuite suite, String id, CFGLocation location, String executionResultLocation, String executionResultName) {
		this(suite, id, executionResultLocation, executionResultName);	
		rootResult.getResult().setDeployment(location.getDeployment());
	}

	/**
	 * createExecutionResultResource - This creates the resource for the execution result.
	 * @param timestamp
	 * @return
	 */
	protected Resource createExecutionResultResource(long timestamp)
	{
		URI uri = getTestSuite().eResource().getURI().trimFileExtension();
		String filePath = getExecutionResultLocation() + "/" + uri.lastSegment() + "_" + timestamp + "." + ICommonConstants.EXECUTION_FILE_EXTENSION;
		URI executionResourceURI = null;
		
		if ( RegistryReader.isPlatformMode() )
			executionResourceURI = URI.createPlatformResourceURI(filePath);
		else
			executionResourceURI = URI.createFileURI(filePath);
		
		return getTestSuite().eResource().getResourceSet().createResource(executionResourceURI);
	}

	/**
	 * @return
	 */
	public TPFTestSuite getTestSuite() {
		if(testSuite.eResource().getResourceSet() == null)
		{
			EObject eObject = resourceSet.getEObject(testSuiteURI, true);
			if(eObject instanceof TPFTestSuite)
				testSuite = (TPFTestSuite)eObject;
		}
			
		return testSuite;
	}

	/**
	 * @return
	 */
	public Resource getResource()
	{
		return resource;
	}

	/**
	 * @return
	 */
	public String getExecutionResultLocation()
	{
		return executionResultLocation;
	}

	/**
	 * This method logs a given execution event against the execution result it belongs
	 * to and asaves theresource.
	 * 
	 * @param event - TPFExecutionEvent
	 */
	public void logEvent(TPFExecutionEvent event) {
		
		System.out.println("fall back to executionContext logger -- event " + event.toString());
		// Parse the ID
		ExecutionMapData mapData = new ExecutionMapData(event.getOwnerId());
		ExecutionResultData data = mapData.getData();
		
		if (data == null) {
			reportErrorInEvent(event);
			return;
		}
		
		// Get the history to add the event to
		TPFExecutionHistory history = data.getHistory();
		event.setExecutionHistory(history);
		
		// Set the BVRInteractionFragment, if its behavior 
		// Or the test if its that
		EObject eObject = data.getTest().eResource().getEObject(mapData.getOwnerId());

		if (event instanceof TPFInvocationEvent) {
			TPFTest test = getTestSuite();
			
			if (eObject instanceof BVRInteractionFragment) {
				event.setInteractionFragment((BVRInteractionFragment) eObject);
				
				// Also load the suite or test that this is referencing for
				// future lookup! Calling getOtherBehavior() apparently loads it!
				TPFBehavior behavior = ((BVRExecutionOccurrence)eObject).getOtherBehavior();
				if(behavior.getTest() != null)
					test = behavior.getTest();
			}
			
			// Create the ExecutionResultData for this invocation event if needed
			ExecutionResultData child = new ExecutionResultData(test, event.getOwnerId());
			child.setInvocationEvent((TPFInvocationEvent) event);
		
			// Put this new data set in the result map for later lookup
			resultMap.put(child.getId(), child);
			((TPFInvocationEvent)event).setExecutionHistory(data.getResult().getExecutionHistory());
		}
		else if (event instanceof TPFTypedEvent)
		{
			TPFTypedEvent type = (TPFTypedEvent)event;
			// set the test for the execution result that this event belongs to
			if (eObject instanceof TPFTest) {
				data.getResult().setTest((TPFTest)eObject);
			}
			else if (eObject instanceof BVRCombinedFragment) {
				event.setInteractionFragment((BVRInteractionFragment)eObject);
			}
			// JPT: If the eObject is a Loop, address hierarchy here.
		}
		else if (event instanceof TPFVerdictEvent)
		{
			data.getResult().setVerdict(((TPFVerdictEvent) event).getVerdict());
		}
	}
	
	private void reportErrorInEvent(TPFExecutionEvent event) {
		if (debug) {
			System.err.println("Ignoring the following event due to invalid ownerId: ");
			System.err.println("Event    : " + event.getClass().getName());
			System.err.println("ownerId  : " + event.getOwnerId());
			System.err.println("text     : " + event.getText());
			System.err.println("timestamp: " + event.getTimestamp());
		}
	}
		
	public void cleanUp() {
		
		// Firat save the resource
		SaveManager.saveResource(resource);
		SaveManager.getInstance().removeResource(resource);

		// Now cleanup
		//testSuite = null;
		//rootResult = null;
		resultMap.clear();
		//resultMap = null;
	}

	/**
	 * @author amathur
	 *
	 * This class parses an ID into the owning execution result ID and the 
	 * behavior ID of the event.
	 */
	public class ExecutionMapData {
	
		String parentId = null;
		String ownerId = null;		
		ExecutionResultData data = null;	

		/**
		 * 
		 */
		public ExecutionMapData(String id) {
			// If the ID is null, then we cant really locate the execution 
			// result it belongs to, so just quit
			if ((id == null) || (id.equals(""))) {
				return;
			}
			
			// Parse this id into 2 parts
			// 1. ID of the ExecutionOccurence/TPFTest/CombinedFragment(loop) 
			//		- this is the trailing part of the string
			// 2. Result ID - ID of the test result that this event belongs to
			int index = id.lastIndexOf(separator);
			if (index < 0) {
				// This is a root event, which means the map ID needs to
				// be the test suite ID
				// Dont do anything, the mapId & ownerId are already set
				parentId = id;
				ownerId = stripParanthesis(id);
				
				data = (ExecutionResultData) resultMap.get(parentId);
				
				if (data == null) {
					// The parent ID was badly formed. Try to see if the UUID is still
					// correct which means the iteration count was simply missing
					String testSuiteId = getTestSuite().getId();
					String testSuiteBehaviorId = getTestSuite().getBehavior().getId();
					if (parentId.compareTo(testSuiteId) == 0 ||
						parentId.compareTo(testSuiteBehaviorId) == 0 ) {
						data = rootResult;
					}
				}
			}
			else {
				// This is not a root event - does not belong to the root
				// execution result
				
				// Stip trailing ID (which will be the behavior ID)
				ownerId = id.substring(index+1);
				ownerId = stripParanthesis(ownerId);
				
				// Now get the ID of the parent in the hierarchy
				parentId = id.substring(0, index);
				
				// Make sure the execution result for the parent exists
				// If not, then this was an async event and needs to be dealt 
				// accordingly
				data = (ExecutionResultData) resultMap.get(parentId);
				
				// If the immediate parent ID does not have an execution result associated, then 
				// hunt for the execution result up the chain. If one does not exist, then
				// this event came out of order. Create the missing execution results as we 
				// traverse up the chain
				if (data == null) {
					// Try to recover for events that come out of order
					// First parse this parent ID using the data map
					ExecutionMapData map = new ExecutionMapData(parentId);
					ExecutionResultData tempData = map.getData();
					if (tempData != null) {
						// Found a parent result data
						// If the map.getOwnerId() is a loop, then jus tre-use the same execution result data
						// First get the test 
						EObject eObject = tempData.getObjectInTest(map.getOwnerId());
						if (eObject instanceof BVRCombinedFragment) {
							// Its a loop so just use it as is
							data = tempData;
						}
						else if (eObject instanceof TPFTest) {
							data = tempData;
						}
						else if (eObject instanceof BVRExecutionOccurrence) {
							TPFInvocationEvent invocationEvent = Common_TestprofileFactoryImpl.eINSTANCE.createTPFInvocationEvent();
							invocationEvent.setOwnerId(parentId);
							logEvent(invocationEvent);
							
							data = (ExecutionResultData) resultMap.get(parentId);
						}
					}
				}				
			}	
		}
		
		private String stripParanthesis(String id) {
			String str = id;
			int index = str.lastIndexOf('(');
			if (index >= 0) {
				str = str.substring(0, index);
			}
			
			return str;

		}

		/**
		 * @return
		 */
		public String getOwnerId() {
			return ownerId;
		}

		/**
		 * @return
		 */
		public String getParentId() {
			return parentId;
		}

		/**
		 * @return
		 */
		public ExecutionResultData getData() {
			return data;
		}
	}

		
	/**
	 * @return Returns the context's agentCount decremented by 1.
	 */
	public int decrementAgentCount() {
		return --agentCount;
	}
	
	/**
	 * increments the context's agentCount;
	 */
	public void incrementAgentCount() {
		agentCount++;
	}
}
