/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.test.java.internal.junit.wizard;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaConventions;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;

import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.common.TestCommonImages;
import org.eclipse.hyades.test.common.TestCommonPlugin;
import org.eclipse.hyades.test.common.internal.codegen.Generator;
import org.eclipse.hyades.test.java.TestJavaPlugin;
import org.eclipse.hyades.test.java.internal.codegen.JUnitGenerator;
import org.eclipse.hyades.test.java.internal.util.TestJavaUtil;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.wizard.HyadesWizard;

public class GenerateWizard 
extends HyadesWizard implements IRunnableWithProgress
{
	public final static String PLUGIN_ID_JUNIT = "org.junit";
	public final static String PLUGIN_ID_EXECUTION_CORE = "org.eclipse.hyades.execution.core";
	public final static String PLUGIN_ID_EXECUTION_REMOTE = "org.eclipse.hyades.execution.remote";

	private ITestSuite testSuite;
	private AdjustSourceInfoPage adjustSourceInfoPage;
	private ProjectAndSourcePage projectAndSourcePage;

	/**
	 * Constructor for GenerateWizard
	 */
	public GenerateWizard()
	{
		super();

		setWindowTitle(TestJavaPlugin.getString("GEN_WTITLE"));
		setDefaultPageImageDescriptor(TestCommonImages.INSTANCE.getImageDescriptor(TestCommonImages.IMG_WIZBAN_GENERATE));

		IDialogSettings dialogSettings = TestJavaPlugin.getInstance().getDialogSettings();
		IDialogSettings settings = dialogSettings.getSection("junit.GenerateWizard");
		if (settings == null)
			settings = dialogSettings.addNewSection("junit.GenerateWizard");
		setDialogSettings(settings);
	}

	public ITestSuite getTestSuite()
	{
		return testSuite;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
	 */
	protected void initPages()
	{
		IStructuredSelection structuredSelection = getSelection();
		if(structuredSelection.size() == 1)
		{
			Object selection = structuredSelection.getFirstElement();
			if(selection instanceof ITestSuite)
				testSuite = (ITestSuite)selection;
		}
	
		IImplementor implementor = getTestSuite().getImplementor();
		
		IJavaProject[] javaProjects = TestJavaUtil.getJavaProjects();
		projectAndSourcePage = new ProjectAndSourcePage(implementor, javaProjects);
		
		boolean isOK = false;
		String pack = null;
		String cls = null;
		if(javaProjects.length > 0)
		{
			if(implementor.getResource() != null)
			{
				String resource = implementor.getResource();
				int index = resource.lastIndexOf('.');
				if(index > 0)
				{
					pack = resource.substring(0, index);
					isOK = JavaConventions.validatePackageName(pack).isOK();
				}
				
				if(index < (resource.length()-1))
				{
					cls = resource.substring(index+1);
					isOK = isOK && JavaConventions.validateIdentifier(cls).isOK();
				}
				
				isOK = isOK && (pack != null) && (cls != null);
			}
			
			if(!isOK)
				adjustSourceInfoPage = new AdjustSourceInfoPage(implementor);
		}
	}

	/**
	 * @see Wizard#addPages()
	 */
	public void addPages()
	{
		if((projectAndSourcePage.getJavaProjects().length > 0) && (adjustSourceInfoPage != null))
			addPage(adjustSourceInfoPage);
			
		addPage(projectAndSourcePage);
	}

	public void run(IProgressMonitor mon)
	throws InterruptedException, InvocationTargetException
	{
		getDialogSettings().put(ProjectAndSourcePage.SET_PROJECT, getJavaProject().getElementName());
		getDialogSettings().put(ProjectAndSourcePage.SET_SOURCE, getSourceFolder());

		if(getTestSuite() != null)
		{							
			Resource resource = ((EObject)getTestSuite()).eResource();
			boolean isTrackingModification = resource.isTrackingModification();
			resource.setTrackingModification(true);
			
			IContainer location = getJavaProject().getProject();
			if((getSourceFolder() != null) && (getSourceFolder().length() > 0))
				location = location.getFolder(new Path(getSourceFolder()));
			
			String sLocation = location.getFullPath().toString();
			if(!sLocation.equals(getTestSuite().getImplementor().getLocation()))
				getTestSuite().getImplementor().setLocation(sLocation);
			else
				resource.setModified(adjustSourceInfoPage != null);
			
			Generator generator = createGenerator();
			if(generator.getFileHandle(getTestSuite()).exists())
			{
				MessageDialog dialog = new MessageDialog(
				getShell(), TestJavaPlugin.getString("W_QUESTION"), 
				null, TestJavaPlugin.getString("Q_OVR_FILE"), 
				MessageDialog.QUESTION, 
				new String[] {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL}, 1); 

				if(dialog.open() != IDialogConstants.OK_ID)
				{
					mon.setCanceled(true);
					throw new InterruptedException();
				}
			}
				
			addRequiredLibraries(generator);		
			try
			{
				generator.generate(getTestSuite(), mon);
			}
			catch(Exception e)
			{
				TestJavaPlugin.logError(e);
				throw new InvocationTargetException(e);
			}
			
			if(resource.isModified())
			{
				try
				{
					EMFUtil.save(resource);
				}
				catch (Exception e)
				{
					TestJavaPlugin.logError(e);
				}
			}
			resource.setTrackingModification(isTrackingModification);
		}
	}

	public boolean performFinish()
	{
		IRunnableWithProgress op = new WorkspaceModifyDelegatingOperation(this);
		try
		{
			getContainer().run(false, true, op);
		}
		catch(InvocationTargetException e)
		{
			MessageDialog.openError(getShell(), TestJavaPlugin.getString("W_ERROR"), e.getTargetException().getMessage());
			return false;
		}
		catch (InterruptedException e)
		{
			return false;
		}

		return true;
	}

   /**
	* @return Java Project
	*/
	public IJavaProject getJavaProject()
	{
		return projectAndSourcePage.getJavaProject();
	}

   /**
	* @return source folder
	*/
	public String getSourceFolder()
	{
		return projectAndSourcePage.getSourceFolder();
	}
	
	protected Generator createGenerator()
	{
		return new JUnitGenerator();
	}
	
	protected void addRequiredLibraries(Generator generator)
	{
		JUnitGenerator junitGenerator = (JUnitGenerator)generator;
		junitGenerator.addAllLibraries(PLUGIN_ID_JUNIT);
		junitGenerator.addRequiredLibrary(TestCommonPlugin.getID(), "common.runner.jar");
		junitGenerator.addRequiredLibrary(TestJavaPlugin.getID(), "java.runner.jar");
	}
}