/**********************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileFiltersTab;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSet;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetType;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetTypeGroup;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.help.WorkbenchHelp;

/*
 * CONTEXT_ID pfpp0001 for Trace profilings table in Profile profilings ui page
 * CONTEXT_ID pfpp0002 for New button in Profile profilings ui page
 * CONTEXT_ID pfpp0003 for Edit button in Profile profilings ui page
 * CONTEXT_ID pfpp0004 for Delete button in Profile profilings ui page
 * CONTEXT_ID pfpp0005 for Up button in Profile profilings ui page
 * CONTEXT_ID pfpp0006 for Down button in Profile profilings ui page
 * CONTEXT_ID pfpp0007 for Package entry field in Profile profilings add/edit dialog page
 * CONTEXT_ID pfpp0008 for Method Name entry field in Profile profilings for Package entry field in Profile profilings add/edit dialog page
 * CONTEXT_ID pfpp0009 for Rule combobox in Profile profilings for Package entry field in Profile profilings add/edit dialog page
 * CONTEXT_ID pfps0001 for Trace profilings Set table in Profile profilings ui page
 * CONTEXT_ID pfps0002 for New Set button in Profile profilings ui page
 * CONTEXT_ID pfps0003 for Edit Set button in Profile profilings ui page
 * CONTEXT_ID pfps0004 for Delete Set button in Profile profilings ui page
 */
public class TraceProfileOverviewUI extends AbstractChangeable {
	private Composite _result;

	private List _profilingTypeList;
	
	private Table _table;
	private Table _profilingSetTable;
	private TableViewer _tableViewer;
	private CheckboxTableViewer _profilingSetTableViewer;

	private Button _addSetBtn;
	private Button _renameSetBtn;
	private Button _removeSetBtn;
	private Button _editBtn;

	private Label _contentLabel;
	private Label _profilingSetLabel;	
	
	private ProfilingSetsManagerCopy _managerCopy;
	
	private Listener _listener = new Listener() {
		public void handleEvent(Event event) {
			changed();
		}
	};

	private class Sorter implements Comparator {
		public int compare(Object e1, Object e2) {
			if (e1 != null && e2 != null) {
				if (e1 instanceof IProfilingSetTypeGroup && e2 instanceof IProfilingSetTypeGroup) {
					IProfilingSetTypeGroup g1 = (IProfilingSetTypeGroup)e1;
					IProfilingSetTypeGroup g2 = (IProfilingSetTypeGroup)e2;
					return g1.getName().compareTo(g2.getName());
				}
				else if (e1 instanceof IProfilingSetType && e2 instanceof IProfilingSetType) {
					IProfilingSetType t1 = (IProfilingSetType)e1;
					IProfilingSetType t2 = (IProfilingSetType)e2;
					return t1.getName().compareTo(t2.getName());
				}
			}
			return 0;
		}
	}
	
	private static class ProfilingSetLabelProvider extends LabelProvider {
		public String getText(Object element) {
			return ((IProfilingSet)element).getName();
		}

		public Image getImage(Object element) {
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
	}

	private static class ProfilingSetContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object parent) {
			ProfilingSetsManagerCopy copy = (ProfilingSetsManagerCopy)parent;
			return copy.getProfilingSets().values().toArray();
		}

		public void inputChanged(Viewer viewer,	Object oldInput, Object newInput) {
		}

		public void dispose() {
		}
	}

	private class ListContentProvider implements IStructuredContentProvider {
		public Object[] getElements(Object input) {
			if (input != null && input instanceof List) {
				return ((List)input).toArray();
			}
			return new Object[0];
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public void dispose() {
		}
	}
	
	/*
	 * A specialized wizard dialog that updates the manager copy
	 * whenever the user changes page. This is meant as a
	 * temporary solution. (See #60091)
	 */
	private static class EditWizardDialog extends WizardDialog {
		
		private ProfilingSetsManagerCopy _copy;
		
		public EditWizardDialog(Shell parentShell, IWizard newWizard) {
			super(parentShell, newWizard);
			_copy = ((EditWizard)getWizard()).getProfilingSetsManagerCopy();
		}
		
		protected Control createContents(Composite parent) {
			Control control = super.createContents(parent);
			EditWizardPage1 page1 = (EditWizardPage1)getWizard().getPages()[0];
			EditWizardPage2 page2 = (EditWizardPage2)getWizard().getPages()[1];
			page1.initializeFrom(_copy);
			page2.initializeFrom(_copy);
			return control;
		}

		protected void backPressed() {
			EditWizardPage2 page = (EditWizardPage2)getCurrentPage();
			if (page.performApply(_copy)) {
				EditWizardPage1 previous = (EditWizardPage1)page.getPreviousPage();
				previous.initializeFrom(_copy);
				super.backPressed();
			}
		}
		
		protected void nextPressed() {
			EditWizardPage1 page = (EditWizardPage1)getCurrentPage();
			if (page.performApply(_copy)) {
				EditWizardPage2 next = (EditWizardPage2)page.getNextPage();
				next.initializeFrom(_copy);
				super.nextPressed();
			}
		}
		
		protected void finishPressed() {
			boolean ok = false;
			IWizardPage page = getCurrentPage();
			if (page instanceof EditWizardPage1) {
				ok = ((EditWizardPage1)page).performApply(_copy);
			}
			else if (page instanceof EditWizardPage2) {
				ok = ((EditWizardPage2)page).performApply(_copy);
			}

			if (ok) {
				super.finishPressed();
			}
		}
	}
	
    private class EditWizard extends Wizard {
		
		private EditWizardPage1 wizPage1;
		private EditWizardPage2 wizPage2;
		private IStructuredSelection selection;
		private IWorkbench workbench;
		private String parentName;
		private ProfilingSetsManagerCopy original;
		private ProfilingSetsManagerCopy copy;
		
		public EditWizard(String psName, ProfilingSetsManagerCopy options) {
			super();
			parentName = psName;
			original = options;
			copy = original.createCopy();			
		}
		
		public void setParent (String parent) {	
			parentName = parent;
		}

		public void init(IWorkbench workbench, IStructuredSelection selection) {
			this.workbench = workbench;
			this.selection = selection;
		}

		public void addPages() {
			super.addPages();
			
			wizPage1 = new EditWizardPage1("profilingTypes", parentName, copy);			
			wizPage1.setTitle(this.getWindowTitle());
			wizPage1.setDescription(this.getWindowTitle());
			addPage(wizPage1);
			
			wizPage2 = new EditWizardPage2("filters");
			wizPage2.setTitle(this.getWindowTitle());
			wizPage2.setDescription(this.getWindowTitle());
			addPage(wizPage2);
		}

		public ProfilingSetsManagerCopy getProfilingSetsManagerCopy() {
			return copy;
		}
		
		public boolean performFinish() {
			original.resetFrom(copy);
			original.applyChanges();
			changed();
			return true;
		}
	}
	
	private class EditWizardPage2 extends WizardPage {
	
		private ProfileFiltersTab _filterTab;
	
		public EditWizardPage2(String pageId) {
			super(pageId);
			setImageDescriptor(PDPluginImages.getImageDescriptor(PDPluginImages.IMG_UI_WZ_EDITPROFSET));
			_filterTab = new ProfileFiltersTab("name");
		}
		
		public void createControl(Composite parent) {
			Composite composite = _filterTab.createControl(parent);
			_filterTab.addListener(_listener);
			setControl(composite);
		}
		
		public void initializeFrom(ProfilingSetsManagerCopy copy) {
			Collection set = copy.getFilterSets().values();
			FilterSetElement[] array = new FilterSetElement[set.size()];			
			_filterTab.activateFilters((FilterSetElement[])set.toArray(array), copy.getDefaultFilterSet());
		}
		
		public boolean performApply(ProfilingSetsManagerCopy copy) {
			Map profilingFilters = copy.getFilterSets();
			profilingFilters.clear();
			ArrayList filters = _filterTab.getFilterSet();			
			for(int idx=0; idx<filters.size(); idx++)
			{
				FilterSetElement  elem = (FilterSetElement) filters.get(idx);
				profilingFilters.put(elem.getName(), elem);
			}

			String name = _filterTab.getActiveFilterSet();
			copy.setDefaultFilterSet((FilterSetElement)copy.getFilterSets().get(name));
			
			return true;
		}
	}
	
	private class EditWizardPage1 extends WizardPage implements ICheckStateListener, ISelectionChangedListener {
		
		private boolean canFlipToNext;
		private PageContent dc;
		private String profilingSetName;
		private CheckboxTreeViewer treeViewer;
		private TreeContentProvider  provider;
		private ProfilingSetsManagerCopy copy;
		
		public EditWizardPage1(String pageId, String psName, ProfilingSetsManagerCopy managerCopy) {			
			super(pageId);
			profilingSetName = psName;
			copy = managerCopy;
			setImageDescriptor(PDPluginImages.getImageDescriptor(PDPluginImages.IMG_UI_WZ_EDITPROFSET));
		}
				
		public boolean isPageComplete() {
			return true;
		}
		
		public boolean canFlipToNextPage() {
			return canFlipToNext;
		}
		
		public void dispose()
		{
			dc.dispose();
			super.dispose();			
		}
		
		public void createControl(Composite parent) {
			
			Composite result = (Composite) new Composite(parent, SWT.NONE);
			
			GridLayout layout;
			GridData data;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			result.setLayoutData(data);
			
			data = GridUtil.createFill();
			
			GridLayout layout2 = new GridLayout();
			layout2.numColumns = 1;
			
			provider = new TreeContentProvider();
			treeViewer = new CheckboxTreeViewer(result);
			treeViewer.setContentProvider(provider);
			treeViewer.setSorter(new ViewerSorter() {
				private Comparator c = new Sorter(); 
				public int compare(Viewer viewer, Object e1, Object e2) {
					return c.compare(e1, e2);
				}
			});
			
			Object[] profileTypeGroups = getManagerCopy().getProfilingTypesGroups().values().toArray();
			Object[] profilingTypes = getManagerCopy().getProfilingTypes().values().toArray();
			ImageDescriptor[] descriptorList = new ImageDescriptor[profileTypeGroups.length + profilingTypes.length];
			
			//profiling groups icons
			for (int i = 0; i < profileTypeGroups.length; i ++) {
				descriptorList[i] = ((IProfilingSetTypeGroup)profileTypeGroups[i]).getIcon();
			}
			
			//profiling type icons
			for (int j = 0; j < profilingTypes.length; j++)
			{
				IProfilingSetType pType = (IProfilingSetType)profilingTypes[j];
				descriptorList[j + profileTypeGroups.length] = pType.getImage();
			}
			treeViewer.setLabelProvider(new TreeLabelProvider(descriptorList));
			treeViewer.setInput("");
			treeViewer.getTree().setLayout(layout2);
			treeViewer.getTree().setLayoutData(GridUtil.createVerticalFill());
			
			IProfilingSetType[] array = new ProfilingSetType[getManagerCopy().getProfilingTypes().size()];
			dc = new PageContent(result, SWT.NONE);
			dc.createContent((IProfilingSetType[])getManagerCopy().getProfilingTypes().values().toArray(array), copy);
			dc.setVisible(true);
			
			treeViewer.addCheckStateListener(this);
			treeViewer.addSelectionChangedListener(this);
			
			setControl(result);
		
			setPageComplete(isPageComplete());
		}
		
		private void setChecked(IProfilingSetType ptype)
		{
			if(ptype == null)
				return;
			
			treeViewer.expandToLevel(ptype, 0);
			treeViewer.setChecked(ptype, true);
			
			// Get all of this ProfileType's parent's children. If all of them are now checked,
			// the check our parent.
			Object parent = provider.getParent(ptype);
			Object[] parentsChildren = provider.getChildren(parent);
			boolean foundUnchecked = false;
			for(int i = 0; i < parentsChildren.length; i ++) {
				if (treeViewer.getChecked(parentsChildren[i]) == false) foundUnchecked = true;
			}
			
			if (foundUnchecked == false) treeViewer.setChecked(parent, true);
		}
		
		public void checkStateChanged(CheckStateChangedEvent event){
			Object checkedElement = event.getElement();

			if (checkedElement instanceof IProfilingSetType) {

				treeViewer.setSelection(new StructuredSelection(checkedElement));
				Object parent = provider.getParent(checkedElement);
				Object[] parentsChildren = provider.getChildren(parent);
				boolean foundChecked = false;
				for(int i = 0; i < parentsChildren.length; i ++) {
					if (treeViewer.getChecked(parentsChildren[i]) == true) foundChecked = true;
				}
				
				treeViewer.setChecked(parent, foundChecked);

			}
			else if (checkedElement instanceof IProfilingSetTypeGroup) {
				
				treeViewer.setSelection(new StructuredSelection(checkedElement));
				Object[] parentsChildren = provider.getChildren(checkedElement);
				for(int i = 0; i < parentsChildren.length; i ++) {
					treeViewer.setChecked(parentsChildren[i], event.getChecked());
				}
			}
			updateWizardButtons();
		}

		public void selectionChanged(SelectionChangedEvent event) {
			updateWizardButtons();
			IStructuredSelection selection = (IStructuredSelection) event.getSelection();
			if (selection.getFirstElement() instanceof IProfilingSetType) {
				updateGui((IProfilingSetType)selection.getFirstElement());
			} 
			else if (selection.getFirstElement() instanceof IProfilingSetTypeGroup){
				dc.showGroupDetails((IProfilingSetTypeGroup)selection.getFirstElement());
			}else {
				dc.showEmptyDetails();
			}
		}
		

		private void updateGui(IProfilingSetType newProfileType) {
			Control newControl = newProfileType.getProfilingType().getControl(profilingSetName);
			Control currentControl = dc.getCurrentlyDisplayingControl();
			if (newControl.equals(currentControl) == false) {
				dc.showDetailsFor(newControl);
			}
		}
		

		/**
		 * Updates the Next/Back buttons on the wizard.
		 * The buttons will be enabled if one of the following conditions are true:
		 * - if at least one of the checked elements use filters.
		 * - if the first selected valid element uses filters.
		 *
		 * Otherwise, the buttons will be disabled.
		 */
		private void updateWizardButtons() {
			Object[] checkedElementList = treeViewer.getCheckedElements();

			// Get the list of checked elements, then filter out all of the ones that aren't
			// ProfileType objects. For each of the ones that ARE profile type objects, if any
			// of them have filters turned on, then enable the next button and return.
			for(int i = 0; i < checkedElementList.length; i ++) {
				if (checkedElementList[i] instanceof IProfilingSetType) {
					IProfilingSetType nextProfile = (IProfilingSetType) checkedElementList[i];
					if (nextProfile.isUseFilters()) {
						canFlipToNext = nextProfile.isUseFilters();
						getWizard().getContainer().updateButtons();
						return;
					}
				}
			}

			// This condition has been commented out as a result of bugzilla 56785.
			
			// If none of the checked elements have filters turned on, then check that the currently
			// selected element has filters turned on. If it does, then enable the next button. Otherwise,
			// disable the next button.
//			IStructuredSelection selection = (IStructuredSelection) treeViewer.getSelection();
//			Iterator i = selection.iterator();
//			while(i.hasNext() == true) {
//				Object currentIteration = i.next();
//				if (currentIteration instanceof IProfilingSetType) {
//					IProfilingSetType firstSelectedProfile = (IProfilingSetType) currentIteration;
//					canFlipToNext = firstSelectedProfile.isUseFilters();
//					getWizard().getContainer().updateButtons();
//					return;
//				}
//			}
			
			// If none of the conditions hold, disable the buttons.
			canFlipToNext = false;
			getWizard().getContainer().updateButtons();
			
		}		
		
		public void initializeFrom(ProfilingSetsManagerCopy copy) {
			IProfilingSet set = copy.getDefaultSet();
			if(set == null)
			    return;
			
			List types = set.getProfilingTypes();
			for(int idx=0; idx<types.size(); idx++)
			{
				String type = types.get(idx).toString();
				
				Object ptype = copy.getProfilingTypes().get(type);
				if(type != null)
					setChecked((IProfilingSetType)ptype);
			}
			
			/*
			 * #60091: Send the getControl() signal consistently whenever
			 * the profiling type's UI is shown.
			 */
			IStructuredSelection sel = (IStructuredSelection)treeViewer.getSelection();
			if (sel != null && !sel.isEmpty()) {
				Object obj = sel.getFirstElement();
				if (obj != null && obj instanceof IProfilingSetType) {
					IProfilingType type = ((IProfilingSetType)obj).getProfilingType();
					if (type != null) {
						type.getControl(profilingSetName);
					}
				}
			}
		}
		
		public boolean performApply(ProfilingSetsManagerCopy copy)
		{
			//update profiling types
			IProfilingSet set = copy.getDefaultSet();
			
			set.getProfilingTypes().clear();
			Object[] selections = treeViewer.getCheckedElements();
			for(int idx=0; idx<selections.length; idx++)
			{
				Object type = selections[idx];
				if(type != null && type instanceof IProfilingSetType)
				{
					set.getProfilingTypes().add(((IProfilingSetType)type).getId());
				}
			}
			
			return dc.performApply(copy);
		}
		
	}
	
	// This class is responsible for showing the appropriate control when the user clicks on
	// an entry in the TreeViewer.
	private static class PageContent extends Composite {
	
		private StackLayout detailsLayout;
		private Control[] controls;
		private IProfilingSetType[] profilingTypes;
		private Control currentControl;
		private TraceProfileTypeGroupUI _traceProfileTypeGroupUI;
		private Map groupControls;	//a map of the profiling group id, and it's control

		// Constructor.
		public PageContent(Composite composite, int style) {
			super(composite, style);
			currentControl = null;
			groupControls = new HashMap();
		}

		public Composite createContent(IProfilingSetType[] types, ProfilingSetsManagerCopy copy) {

			// Create a new stack layout.
			detailsLayout = new StackLayout();
		    this.setLayout(detailsLayout);
	    	this.setLayoutData(new GridData(GridData.FILL_BOTH));

			controls = new Control[types.length + 1];
			profilingTypes = types;

			for(int i = 0; i < types.length; i ++) {
				controls[i] = types[i].getProfilingType().createControl(this, copy);
				controls[i].setVisible(false);
			}

			controls[controls.length - 1] = new Composite(this, SWT.NONE);
			controls[controls.length - 1].setVisible(false);

			Map groups = copy.getProfilingTypesGroups();
			Object[] keySet = groups.keySet().toArray();
			_traceProfileTypeGroupUI = new TraceProfileTypeGroupUI();
			
			for (int j = 0; j < keySet.length; j++)
			{
				IProfilingSetTypeGroup pTypeGroup = (IProfilingSetTypeGroup)groups.get(keySet[j]);
				groupControls.put(pTypeGroup.getId(), _traceProfileTypeGroupUI.createControl(this, pTypeGroup.getName(), pTypeGroup.getDescription()));
				
			}
			
			return null;
		}

		public boolean performApply(ProfilingSetsManagerCopy copy)
		{
			IProfilingSet set = copy.getDefaultSet();
			if(set == null)
				return true;
			
			StringBuffer errorMsg = new StringBuffer();			
			List types = set.getProfilingTypes();
			for(int i = 0; i < profilingTypes.length; i++) {
				
				if(types.contains(profilingTypes[i].getId()))
				{
					IProfilingType type = profilingTypes[i].getProfilingType();
					String msg = type.validateConfiguration(copy);
					if(msg != null)
					{
						errorMsg.append(msg).append("\n");
					}
					
				}
			}
			
			if(errorMsg.length() > 0)// conflicts for the current configuration
			{
				String msg = UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_MSG");												
				if(!MessageDialog.openQuestion(getShell(), UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_TITLE"), msg+errorMsg.toString()))
				{
					return false;
				}										
			}
			
			List optionsList = ProfilingSetsManager.getDefaultProfilingOptions();
			for(int i = 0; i < profilingTypes.length; i++) {
				
				if(types.contains(profilingTypes[i].getId()))
				{
					IProfilingType type = profilingTypes[i].getProfilingType();
					ProfilingAttribute[] options = type.getAttributes();
					for(int idx=0; idx<options.length; idx++) {
						optionsList.add(options[idx]);
					}
				}
			}
			
			// remove the duplicate entries (with precedence)
			optionsList = ProfilingSetsManager.filterDuplicateOptions(optionsList);
			
			//apply attributes to the manager copy
			set.getAttributes().clear();
			Map attrs = set.getAttributes();
			for(int idx=0; idx<optionsList.size(); idx++)
			{
				ProfilingAttribute option = (ProfilingAttribute) optionsList.get(idx);
				attrs.put(option.getName(), option);
			}
			
			return true;
		}
				
		// Show the control specified on the screen.
		public void showDetailsFor(Control detail) {
			detailsLayout.topControl = detail;
			currentControl = detail;
			detail.setVisible(true);
   			layout();
		}
		
		//show the UI for the profiling group
		public void showGroupDetails(IProfilingSetTypeGroup pTypeGroup) {
			Control newControl = (Control)groupControls.get(pTypeGroup.getId());
			Control currentControl = getCurrentlyDisplayingControl();
			if (newControl.equals(currentControl) == false) {
				showDetailsFor(newControl);
			}
			layout();
		}

		public void showEmptyDetails() {
			detailsLayout.topControl = controls[controls.length - 1];
			currentControl = controls[controls.length - 1];
			controls[controls.length - 1].setVisible(true);
			layout();
		}

		// Dispose all of the controls.
		public void dispose() {
			currentControl = null;
			profilingTypes = null;
			for(int i = 0; i < controls.length; i++) {
				if (controls[i] != null) {
					controls[i].dispose();
				}
			}
			
			Object[] groupControlsArray = groupControls.values().toArray();
			
			for (int j = 0; j < groupControlsArray.length; j++)	{
				if (((Control)groupControlsArray[j]) != null) {
					((Control)groupControlsArray[j]).dispose();
				}
			}
			groupControls.clear();
			groupControls = null;
			controls = null;
			
		}
		
		/**
		 * Returns the control that is currently being displayed in the StackLayout.
		 * Returns null if no control has been displayed.
		 * 
		 * @return the control object currently being displayed.
		 */
		public Control getCurrentlyDisplayingControl() {
			return currentControl;
		}

		public void selectionChanged(SelectionChangedEvent event) {
		}
	}

	private class TreeLabelProvider extends LabelProvider {
		private Hashtable iconList;

		public TreeLabelProvider(ImageDescriptor[] descList) {
			iconList = new Hashtable();
			for(int i = 0; i < descList.length; i ++) {
				if (iconList.containsKey(descList[i]) == false) {
					iconList.put(descList[i], descList[i].createImage());
				}
			}
		}

		public String getText(Object element) {
			if (element instanceof IProfilingSetType) {
				return ((IProfilingSetType)element).getName();
			} else if(element instanceof IProfilingSetTypeGroup){
				return ((IProfilingSetTypeGroup)element).getName();
			}
			return element.toString();
		}

		public Image getImage(Object element) {
			ImageDescriptor imageDesc = null;
			if (element instanceof IProfilingSetType) {
				imageDesc = ((IProfilingSetType)element).getImage();
			}
			else if(element instanceof IProfilingSetTypeGroup){
				imageDesc = ((IProfilingSetTypeGroup)element).getIcon();
			}			
			if(imageDesc == null) {
				return null;
			}
			
			return (Image)iconList.get(imageDesc); 
		}

		public void dispose() {
			Iterator list = iconList.keySet().iterator();
			while(list.hasNext()) {
				((Image)iconList.get(list.next())).dispose();
			}
			iconList.clear();
			iconList = null;
	    }
	}

	private class TreeContentProvider implements ITreeContentProvider {

		public Object[] getChildren(Object element) {
			if(element instanceof IProfilingSetTypeGroup) {
				return getManagerCopy().getProfilingTypesForGroup(((IProfilingSetTypeGroup)element).getId());
			}	
			return new Object[0];
		}

		public Object[] getElements(Object element) {
			return getManagerCopy().getProfilingTypesGroups().values().toArray();
		}

		public boolean hasChildren(Object element) {
			return getChildren(element).length > 0;
		}

		public Object getParent(Object element) {
			if(element instanceof IProfilingSetType) {
				String group = ((IProfilingSetType)element).getGroup();
				if(group != null) {
					return getManagerCopy().getProfilingTypesGroups().get(group);					
				}				
			}
			return "";
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object old_input, Object new_input) {
		}
	}

	private class EditWSetDialog extends Dialog implements ModifyListener {
		private String _name;
		private Text _nameText;
		private String _title;
		private String _description;
		private String _id;
		private Text _descText;

		public EditWSetDialog(Shell shell, String title, String name, String desc, String id) {
			super(shell);

			_name = name;
			_title = title;
			_description = desc;
			_id = id;
		}

		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(_title);
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			Label label1;
			Label label2;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			data.widthHint = 400;
			result.setLayoutData(data);

			label1 = new Label(result, SWT.NONE);
			label1.setText(UIPlugin.getResourceString("STR_PROFILING_SET_NAME"));
			_nameText = new Text(result, SWT.BORDER);
			_nameText.setLayoutData(GridUtil.createHorizontalFill());
			
			label2 = new Label(result, SWT.NONE);
			label2.setText(UIPlugin.getResourceString("STR_PROFILING_SET_DESCRIPTION"));
			label2.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
			data = GridUtil.createHorizontalFill();
			data.heightHint = 80;
			_descText = new Text(result, SWT.BORDER | SWT.WRAP | SWT.V_SCROLL);
			_descText.setLayoutData(data);
			
			_nameText.addModifyListener(this);
			_descText.addModifyListener(this);

			if(_name == null)
				_name = UIPlugin.getResourceString("FILTER_SET_DEFAULT_NAME");
			_nameText.setText(_name);
			
			if (_description == null)
				_description = UIPlugin.getResourceString("FILTER_SET_DEFAULT_DESCRIPTION");
			_descText.setText(_description);
			
			return result;
		}

		public void modifyText(ModifyEvent e) {
			Button ok = getButton(IDialogConstants.OK_ID);
			if (ok != null)
				ok.setEnabled(_nameText.getText().trim().length() != 0 && _descText.getText().trim().length() != 0);
		}

		protected void okPressed() {
			_name = _nameText.getText().trim();
			_description = _descText.getText().trim();
			super.okPressed();
		}
	}
	
	public Composite createControl(Composite parent) {
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 0;

		_result = new Composite(parent, SWT.NONE);
		_result.setLayout(layout);
		_result.setLayoutData(GridUtil.createFill());
			
		createProfilingSetTable(_result);
		createProfilingTypeTable(_result);

		return _result;
	}
	
	private void createProfilingSetTable(Composite parent) {
		GridData data;
		GridLayout layout;
		
		Label label = new Label(parent, SWT.NONE);
		label.setText(UIPlugin.getResourceString("STR_SELECT_PROFILING_SET"));
		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);
		
		Composite tableGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 100;
		tableGroup.setLayoutData(data);

		_profilingSetTable = new Table(tableGroup, SWT.V_SCROLL	| SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION | SWT.CHECK);
		_profilingSetTable.setLayoutData(GridUtil.createFill());
		_profilingSetTable.setLinesVisible(false);

		TableColumn column = new TableColumn(_profilingSetTable, SWT.NONE);
		column.setWidth(340);
		
		_profilingSetTableViewer = new CheckboxTableViewer(_profilingSetTable);
		_profilingSetTableViewer.setContentProvider(new ProfilingSetContentProvider());
		_profilingSetTableViewer.setLabelProvider(new ProfilingSetLabelProvider());
		_profilingSetTableViewer.setSorter(new ViewerSorter() {
			public int compare(Viewer viewer, Object e1, Object e2) {
				String a = ((IProfilingSet)e1).getName();
				String b = ((IProfilingSet)e2).getName();
				return a.compareTo(b);
			}
		});
		
		_profilingSetTableViewer.setInput(getManagerCopy());
		_profilingSetTableViewer.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent event) {
				IProfilingSet set = (IProfilingSet)event.getElement();
				_profilingSetTableViewer.setSelection(new StructuredSelection(set));
			}
		});

		Composite btnGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addSetBtn = new Button(addGroup, SWT.PUSH);
		_addSetBtn.setText(UIPlugin.getResourceString("ADD_PROFILINGSET_TEXT"));
		_addSetBtn.setToolTipText(UIPlugin.getResourceString("ADD_FILTER_TOOLTIP_TEXT"));
		_addSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_renameSetBtn = new Button(addGroup, SWT.PUSH);
		_renameSetBtn.setText(UIPlugin.getResourceString("RENAME_PROFILINGSET_TEXT"));
		_renameSetBtn.setToolTipText(UIPlugin.getResourceString("EDIT_SET_TOOLTIP_TEXT"));
		_renameSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeSetBtn = new Button(addGroup, SWT.PUSH);
		_removeSetBtn.setText(UIPlugin.getResourceString("REMOVE_PROFILINGSET_TEXT"));
		_removeSetBtn.setToolTipText(UIPlugin.getResourceString("REMOVE_TOOLTIP_TEXT"));
		_removeSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_addSetBtn.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				addProfilingSet();
			}
		});

		_removeSetBtn.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				removeProfilingSet();
			}
		});

		_renameSetBtn.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				renameProfilingSet();
			}
		});
		
		_profilingSetTableViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection selection = (IStructuredSelection)event.getSelection();
				if (!selection.isEmpty()) {
					IProfilingSet set = (IProfilingSet)selection.getFirstElement();
					_profilingSetTableViewer.setAllChecked(false);
					_profilingSetTableViewer.setChecked(set, true);
					_profilingSetTableViewer.refresh();
					
					getManagerCopy().setDefaultSet(set);
					showProfilingTypeDetails(set);
					changed();
					
					_profilingSetLabel.setText(set.getDescription());
					_removeSetBtn.setEnabled(getManagerCopy().getProfilingSets().size() > 1);
				}
			}
		});

		data = new GridData();
		data.widthHint = 375;
		_profilingSetLabel = new Label(parent, SWT.WRAP);
		_profilingSetLabel.setLayoutData(data);

		WorkbenchHelp.setHelp(_profilingSetTable, UIPlugin.getPluginId() + ".pfps0001");
	}

	private void createProfilingTypeTable(Composite parent) {
		GridData data;
		GridLayout layout;

		Label label = new Label(parent, SWT.NONE);
		data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);

		_contentLabel = new Label(parent, SWT.NONE);
		_contentLabel.setText(UIPlugin.getResourceString("STR_MODIFY_PROFILINGSET_CONTENTS"));
		_contentLabel.setLayoutData(data);

		Composite tableGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 180;
		tableGroup.setLayoutData(data);

		TableLayout tableLayout = new TableLayout();
		_table = new Table(tableGroup, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.MULTI	| SWT.FULL_SELECTION);
		_table.setLinesVisible(false);
		_table.setLayoutData(GridUtil.createFill());
		_table.setLayout(tableLayout);
		_table.setHeaderVisible(true);

		String headers[] = {
			UIPlugin.getResourceString("HEADER_PROFILING_TYPE")
		};
		
		ColumnLayoutData[] layouts = {
			new ColumnPixelData(300, true),
		};

		TableColumn[] columns = new TableColumn[headers.length];

		for (int i = 0; i < headers.length; i++) {
			tableLayout.addColumnData(layouts[i]);
			TableColumn tc = new TableColumn(_table, SWT.NONE, i);
			tc.setResizable(layouts[i].resizable);
			tc.setText(headers[i]);
			columns[i] = tc;
		}

		_profilingTypeList = new ArrayList();
		_tableViewer = new TableViewer(_table);
		_tableViewer.setContentProvider(new ListContentProvider());
		_tableViewer.setLabelProvider(new LabelProvider());
		_tableViewer.setInput(_profilingTypeList);

		Composite btnGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());
		
		_editBtn = new Button(addGroup, SWT.PUSH);
		_editBtn.setText(UIPlugin.getResourceString("EDIT_PROFILINGSETCONTENTS_TEXT"));
		_editBtn.setToolTipText(UIPlugin.getResourceString("EDIT_FILTER_TOOLTIP_TEXT"));
		_editBtn.setLayoutData(GridUtil.createHorizontalFill());

		Composite moveGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		moveGroup.setLayout(layout);
		moveGroup.setLayoutData(GridUtil.createHorizontalFill());

		label = new Label(parent, SWT.NONE);
		label.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				IStructuredSelection selection = (IStructuredSelection)_profilingSetTableViewer.getSelection();
				if (!selection.isEmpty()) {
					IProfilingSet element = (IProfilingSet)selection.getFirstElement();
					String name = element.getName();
					showEditProfilingSetDialog(UIPlugin.getResourceString("STR_EDIT_PROFILINGSET_DLG"), name, " ");
					showProfilingTypeDetails(element);
				}
			}
		});

		WorkbenchHelp.setHelp(_table, UIPlugin.getPluginId() + ".pfpp0002");
	}

	private void showProfilingTypeDetails(IProfilingSet set) {
		_profilingTypeList.clear();
		
		Collection types = new ArrayList();
		Iterator iter = set.getProfilingTypes().iterator();
		while(iter.hasNext()) {
			String type = (String)iter.next();
			IProfilingSetType pSetType = (IProfilingSetType)getManagerCopy().getProfilingTypes().get(type);
			types.add(pSetType);
		}
		
		/*
		 * The types must be sorted at this stage, not in the viewer.
		 * Viewers sort individual items, but in this case, we expand
		 * each item's description into potentially several new items,
		 * but the description must remain next to its type.
		 */
		IProfilingSetType[] sortedTypes = new IProfilingSetType[types.size()];
		types.toArray(sortedTypes);
		Arrays.sort(sortedTypes, new Comparator() {
			public int compare(Object o1, Object o2) {
				IProfilingSetType t1 = (IProfilingSetType)o1;
				IProfilingSetType t2 = (IProfilingSetType)o2;
				return t1.getName().compareTo(t2.getName());
			}
		});
		
		for(int i=0;i<sortedTypes.length;i++) {
			IProfilingSetType pSetType = sortedTypes[i];
			
			if(pSetType != null) {
				pSetType.setDescription(pSetType.getProfilingType().getDescription(getManagerCopy()));
				boolean descriptionExists = (pSetType.getDescription() != null) &&  (pSetType.getDescription().length() > 0);

				if (descriptionExists) {
					if (i != 0) {
						_profilingTypeList.add("");
					}
					_profilingTypeList.add(pSetType.getName());
					addDescription(_profilingTypeList, pSetType.getDescription());
					_profilingTypeList.add("");
				}
				else {
					_profilingTypeList.add(pSetType.getName());
				}
			}
		}

		_table.setRedraw(false);
		_tableViewer.refresh();
		_table.setRedraw(true);
	}
	
	/**
	 * adds the description of the profiling type to the List
	 * handles multi line descriptions
	 * the delimiter for the new line is \n
	 */
	private void addDescription(List profilingList, String description)
	{
		String delim = "\n";
		
		StringTokenizer tokens = new StringTokenizer(description, delim, false);
		while (tokens.hasMoreElements())
		{
			String token = tokens.nextToken();
			profilingList.add(token);
		}
	}

	private void showEditProfilingSetDialog(String title, String parentName, String profilingType) {
		EditWizard wizard = new EditWizard(parentName, getManagerCopy());
		EditWizardDialog dialog = new EditWizardDialog(new Shell(Display.getDefault()), wizard);

		wizard.init(org.eclipse.ui.PlatformUI.getWorkbench(), null);
		wizard.setParent(parentName);
		wizard.setWindowTitle(title);
		dialog.create();
		dialog.open();
	}

	private void addProfilingSet() {
		String name = UIPlugin.getResourceString("FILTER_SET_DEFAULT_NAME");
		String description = UIPlugin.getResourceString("FILTER_SET_DEFAULT_DESCRIPTION");
		EditWSetDialog dialog =	new EditWSetDialog(_result.getShell(), UIPlugin.getResourceString("STR_ADD_PROFILINGSET_DLG"), name, description, name);
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {
			IProfilingSet set = new ProfilingSet(dialog._name, dialog._name, dialog._description);
			
			// set the default options on the set
			Map map = set.getAttributes();
			Iterator iter = ProfilingSetsManager.getDefaultProfilingOptions().iterator();
			while (iter.hasNext()) {
				ProfilingAttribute attr = (ProfilingAttribute)iter.next();
				map.put(attr.getName(), attr);
			}
			
			getManagerCopy().getProfilingSets().put(dialog._name, set);
			getManagerCopy().setDefaultSet(set);
			ProfilingSetsManager manager = ProfilingSetsManager.instance();
			manager.getProfilingSets().put(dialog._name, set);
			manager.setDefaultSet(set);
			manager.writeSetsToPreferences();
			_profilingSetTableViewer.refresh(true);
			_profilingSetTableViewer.setSelection(new StructuredSelection(set));
		}
	}
	
    private void renameProfilingSet() {
		IStructuredSelection selection = (IStructuredSelection)_profilingSetTableViewer.getSelection();
		if (!selection.isEmpty()) {
			IProfilingSet set = (IProfilingSet)selection.getFirstElement();		
			EditWSetDialog dialog =	new EditWSetDialog(_result.getShell(), UIPlugin.getResourceString("STR_EDIT_PROFILINGSET_DLG"), set.getName(), set.getDescription(), set.getId());
			dialog.open();

			if (dialog.getReturnCode() == Window.OK) {
				ProfilingSetsManager manager = ProfilingSetsManager.instance();
				set.setName(dialog._name);
				set.setDescription(dialog._description);
				manager.writeSetsToPreferences();
				_profilingSetTableViewer.refresh(true);
				_profilingSetTableViewer.setSelection(new StructuredSelection(set));
			}
		}
    }
	
	private void removeProfilingSet() {
		IStructuredSelection selection = (IStructuredSelection)_profilingSetTableViewer.getSelection();
		if (!selection.isEmpty()) {
			IProfilingSet set = (IProfilingSet)selection.getFirstElement();		
			getManagerCopy().getProfilingSets().remove(set.getId());
			ProfilingSetsManager manager = ProfilingSetsManager.instance();
			manager.getProfilingSets().remove(set.getId());
			manager.writeSetsToPreferences();
			_profilingSetTableViewer.refresh();
			
			IProfilingSet newSet = (IProfilingSet)_profilingSetTableViewer.getElementAt(0);
			_profilingSetTableViewer.setSelection(new StructuredSelection(newSet));
		}
	}
	
	/**
	 * Initialize ui content using selected configuration attributes
	 * @param conf
	 */
	public void initializeFrom(ILaunchConfiguration conf) {
		getManagerCopy().initializeFrom(conf);
		
		IProfilingSet set = getManagerCopy().getDefaultSet();
		if(set != null)	{
			_profilingSetTableViewer.setSelection(new StructuredSelection(set));
		}
	}
	
	public void performApply(ILaunchConfigurationWorkingCopy wc) {
		_managerCopy.performApply(wc);
	}
	
	public boolean isValid(ILaunchConfiguration conf) {
		return true;
	}
	
	private ProfilingSetsManagerCopy getManagerCopy() {
		if(_managerCopy == null)
			_managerCopy = new ProfilingSetsManagerCopy();
		return _managerCopy;
	}
}
