/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.edit.datapool;

/**
 * A datapool factory manages the construction and sharing of 
 * {@link IDatapool} objects.  This interface supports sharing
 * of editable datapools.
 */
public interface IDatapoolFactory
{
	
	/**
	 * Load the datapool from the specified file.  If a
	 * <code>sharedInstance</code> is requested then the cache of
	 * other datapools that have been opened is checked for an 
	 * existing copy of the datapool.  Note that modifying a shared
	 * instance of the datapool may effect other uses of the datapool
	 * and should only be done in a confined execution environment.
	 * 
	 * @param	dpFile			The datapool file name.
	 * @param	sharedInstance	Available for sharing instances of
	 *						the datapool, which can save significantly 
	 *						on datapool loading overhead.
	 * 
	 * @see	#unload(IDatapool)
	 */
	public IDatapool loadForEdit(java.io.File dpFile, boolean sharedInstance);
	
	/**
	 * Removes a reference to an instance of a datapool.  If the
	 * datapool is shared the reference count is reduced.  Once 
	 * all references to a datapool have been unloaded the datapool
	 * instance will be made available to garbage collection.
	 * 
	 * @param	datapool	The datapool instance being unloaded.
	 * 
	 * @see	#load(java.io.File,boolean)
	 */
	public void unload(IDatapool datapool);
	
	/**
	 * Persists the specified instance of a datapool.  The file 
	 * used is the same as the file from which the datapool was 
	 * loaded.  If the datapool was constructed from scratch the
	 * {@link #saveAs(IDatapool,java.io.File) saveAs} method
	 * should be used to save the file.
	 * 
	 * @param	datapool	The datapool instance being saved.
	 * 
	 * @see	#load(java.io.File,boolean)
	 * @see	#unload(IDatapool)
	 * @see	#saveAs(IDatapool,java.io.File)
	 */
	public void save(IDatapool datapool);
	
	/**
	 * Persists the specified instance of a datapool to the 
	 * specified file.
	 * 
	 * @param	datapool	The datapool instance being saved.
	 * @param	dpFile		The file to which the datapool is to be saved.
	 * 
	 * @see	#load(java.io.File,boolean)
	 * @see	#unload(IDatapool)
	 * @see	#save(IDatapool)
	 */
	public void saveAs(IDatapool datapool, java.io.File dpFile);

	/**
	 * Constructs and returns an empty IDataPool
	 */
	public IDatapool constructDatapool(); 	
	
	
}
