/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.impl;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.AdapterPlugin;
import org.eclipse.hyades.logging.adapter.IComponent;
import org.eclipse.hyades.logging.adapter.IContext;
import org.eclipse.hyades.logging.adapter.IContextListener;
import org.eclipse.hyades.logging.adapter.IProcessUnit;
import org.eclipse.hyades.logging.adapter.parsers.PreparationException;
import org.eclipse.hyades.logging.adapter.util.AdapterUtilities;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
/**
 * Controller manages a set of Context by setting the configurations and then starting each context.
 * @version 0.10
 */
public class Controller implements Runnable {
   
    private boolean singleFileInputMode = false;
    
    /* Keep our contexts and thier associated threads within a list. It is imperative that the index
     *  of each list be kept consistant.  That is, the thread running context at index n must be in the
     *  thread list at index n
     */
    private IContext[] contexts;
    private Thread[]   contextThreads;
    
    private String contextConfigurationFile = Messages.getString("HyadesGADefaultContextConfigurationFile");
    private String componentConfigurationFile = Messages.getString("HyadesGADefaultComponentConfigurationsFile");
    private DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
    private InputStream inContextParams = null;
    private InputStream inAppParams = null;
    private boolean running = false;
    private long sleepTime = 500;
    
    /**
     * Constructor for Controller.
     */
    public Controller() {
        super();
    }
    
    public boolean isRunning() {
        return running;
    }
    
    public void start() {
    	/* RKD:  We will run all of the contexts accept the first one (our internal logging context). This was already started
    	 * when we went through the prepare stage.
         */
        for (int i = 1; i < contextThreads.length; ++i) {
            if (contexts[i] != null) {
            	contextThreads[i]=new Thread(contexts[i]);
            	contextThreads[i].setName(contexts[i].getName() + Messages.getString("HyadesGA") + contexts[i].getUniqueID());
            	contextThreads[i].setDaemon(true);
            	contextThreads[i].start();
            }
        }
        running = true;
    }
    
    
    public void stop() {
        if (contexts != null)
        	/* stop them in reverse order so that the logging context gets stopped last */
            for (int i=contexts.length-1; i<=0; i--) {
                if (contexts[i] != null)
                    contexts[i].stop();
            }
        running = false;
    }
	/**
	 * This stop method will still cause the contexts to be stopped but will
	 * not call the context listener methods when flushing the components.
	 */    
	public void hardStop() {
		if (contexts != null)
			/* stop them in reverse order so that the logging context gets stopped last */
			for (int i=contexts.length-1; i<=0; i--) {
				if (contexts[i] != null) {
					// Set hardStop flag before stopping context
					((Context)contexts[i]).setHardStop();
					contexts[i].stop();
				}
			}
		running = false;
	}
     
    public IContext[] getContexts() {
        return contexts;
    }
    
    public void setContextConfigPath(String config) {
        contextConfigurationFile = config;
    }
    
    public void setComponentConfigPath(String config) {
        componentConfigurationFile = config;
    }
    
    /**
     * creates in Parameters Streams and initiates contexts
     */
    public void prepareAdapter() throws AdapterException {
        try {
            // built the contexts based on configurations
            inContextParams = new FileInputStream(contextConfigurationFile);
            contexts = createContextsAndComponents(inContextParams);
           
            // build the components based on configurations 
            if (singleFileInputMode) {
                inAppParams = new FileInputStream(contextConfigurationFile);
            }
            else {
                inAppParams = new FileInputStream(componentConfigurationFile);
            }
            
            setComponentConfigurations(inAppParams);

            /* Before we start the contexts locate any IContextListeners that
             * are registered with the plugin.  This only works when we are running
             * as a plugin inside of Eclipse.  Outside of Eclipse there is no notion
             * of IContextListeners.
             */
            try {
                List contextListeners = AdapterPlugin.getContextListeners();

                /* If this context has a listener then set it */
                if (contextListeners != null) {
                    Iterator listenerIterator = contextListeners.iterator();
                    while (listenerIterator.hasNext()) {
                        IContextListener listener = (IContextListener) listenerIterator.next();
                        String[] targetContexts = listener.getTargetContextUniqueIds();
                        for (int j = 0; j < targetContexts.length; j++) {
                            for (int k = 0; k < contexts.length; k++) {
                                if (contexts[k].getUniqueID().equals(targetContexts[j])) {
                                    contexts[k].setContextListener(listener);
                                }
                            }
                        }
                    }
                }
            }
            catch (Throwable e) {
                /* We are not running inside of Eclipse so ignore this */
            }
            
            /* Start the logging context in its entirty before we continue with anything else */
            IProcessUnit logger=startInternalLoggingContext();
            

            for (int i = 1; i < contexts.length; ++i) {
                if (contexts[i] != null) {
                	
                	/* Set the logger for this context to be the internal logging sensor */
                	contexts[i].setLogger(logger);
                	
                    // set all the context config info
                    contexts[i].init();
                    // set all the component config info
                    
                    /* Setup the configuration information for this context.  At this point in time the context
                     * may fail to initialize itself properly.  If that is the case then this context should be 
                     * disabled.
                     */
                    try {
                    	contexts[i].update();
                    }
                    catch(AdapterException e) {
                    	contexts[i]=null;
                    	/* Make sure we log that this context will not be started */
                    	log(Messages.getString("HyadesGAContextFatal_ERROR_"));
                    	throw e;
                    }
                }
            }
        }
        catch (Exception e) {
        	/* RKD:  Our caller will need to use our exception information for now to handle the error.*/
            log(Messages.getString("HyadesGAAdapterFatal_ERROR_"));
        	log(e.getMessage());
            throw new AdapterException(e.getMessage());
        }
        finally {
        	/* Close our configuration file */
            try {
                if (inAppParams != null) {
                	inAppParams.close();
                }
                if (inContextParams != null) {
                	inContextParams.close();
                }
            }
            catch (IOException e) {
            	/* We couldn't close the configuration files.  This should only occur if we failed to 
            	 * open the file in the first place.
            	 */
                log(e.toString());
            }
        }
    }
    
    /**
     * updates configuration of the application
     * TODO: HS implement update method
     */
    public void update(Element temp) {
    }
    
    /**
     * parses Context configurations and build the Context objects
     * as well as set teh context config for each component
     */
    private Context[] createContextsAndComponents(InputStream inContextStream) throws PreparationException {

        Document doc = null;
        Context[] contexts = null;
        try {
            doc = docFactory.newDocumentBuilder().parse(inContextStream);

            //Add the adapter logging context as default:
            Element loggingContext = doc.createElement(Messages.getString("HyadesGAContextElementTagName"));
            loggingContext.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "Context instance for the current component");
            loggingContext.setAttribute(Messages.getString("HyadesGAExecutableClassAttributeName"), "org.eclipse.hyades.logging.adapter.impl.AdapterContext");
            loggingContext.setAttribute(Messages.getString("HyadesGAImplemenationCreationDateAttributeName"), "Fri Jan 09 15:27:17 EST 2004");
            loggingContext.setAttribute(Messages.getString("HyadesGALoggingLevelAttributeName"), "60");
            loggingContext.setAttribute(Messages.getString("HyadesGANameAttributeName"), "AdaptorContext");
            loggingContext.setAttribute(Messages.getString("HyadesGARoleAttributeName"), "context");
            loggingContext.setAttribute(Messages.getString("HyadesGARoleCreationDateAttributeName"), "Fri Jan 09 15:27:17 EST 2004");
            loggingContext.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "NB1F4ED002DA11D8A519FBE7C98C2F53");

            Element loggingSensor = doc.createElement(Messages.getString("HyadesGAComponentElementTagName"));
            loggingSensor.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "Adapter logging sensor");
            loggingSensor.setAttribute(Messages.getString("HyadesGAExecutableClassAttributeName"), "org.eclipse.hyades.logging.adapter.internal.util.AdapterSensor");
            loggingSensor.setAttribute(Messages.getString("HyadesGAImplemenationCreationDateAttributeName"), "Fri Jan 09 15:27:17 EST 2004");
            loggingSensor.setAttribute(Messages.getString("HyadesGALoggingLevelAttributeName"), "0");
            loggingSensor.setAttribute(Messages.getString("HyadesGANameAttributeName"), "AdapterSensor");
            loggingSensor.setAttribute(Messages.getString("HyadesGARoleAttributeName"), "sensor");
            loggingSensor.setAttribute(Messages.getString("HyadesGARoleCreationDateAttributeName"), "Fri Jan 09 15:27:17 EST 2004");
            loggingSensor.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "NF991E0004FF11D8930381B6A308BEB5");

            loggingContext.appendChild(loggingSensor);

            Element loggingOutputter = doc.createElement(Messages.getString("HyadesGAComponentElementTagName"));
            loggingOutputter.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "Adapter logging outputter");
            loggingOutputter.setAttribute(Messages.getString("HyadesGAImplemenationCreationDateAttributeName"), "Fri Jan 09 15:27:17 EST 2004");
            loggingOutputter.setAttribute(Messages.getString("HyadesGAImplementationVersionAttributeName"), "");
            loggingOutputter.setAttribute(Messages.getString("HyadesGAImplemenationVersionDescriptionAttributeName"), "");
            loggingOutputter.setAttribute(Messages.getString("HyadesGARoleAttributeName"), "outputter");
            loggingOutputter.setAttribute(Messages.getString("HyadesGARoleCreationDateAttributeName"), "Fri Jan 09 15:27:17 EST 2004");
            loggingOutputter.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "N5F286A002DA11D8BC799C6AF4352915");

            //Add the Eclipse error dialog outputter if in Eclipse mode:
            if (AdapterUtilities.isWorkbench()) {
            	//Set the logging level so we get any messages that are at least informational
                loggingOutputter.setAttribute(Messages.getString("HyadesGALoggingLevelAttributeName"), "10");
                loggingOutputter.setAttribute(Messages.getString("HyadesGAExecutableClassAttributeName"), "org.eclipse.hyades.logging.adapter.internal.util.GlaTaskOutputter");
                loggingOutputter.setAttribute(Messages.getString("HyadesGANameAttributeName"), "CBEEclipseErrorDialogOutputter");
            }
            //Add the log file outputter if in standalone mode:
            else {
            	//Set the logging level so we get Warning, Critical and Fatal error messages
                loggingOutputter.setAttribute(Messages.getString("HyadesGALoggingLevelAttributeName"), "30");
                loggingOutputter.setAttribute(Messages.getString("HyadesGAExecutableClassAttributeName"), "org.eclipse.hyades.logging.adapter.outputters.CBEFileOutputter");
                loggingOutputter.setAttribute(Messages.getString("HyadesGANameAttributeName"), "CBEFileOutputter");
            }

            loggingContext.appendChild(loggingOutputter);

            NodeList contextsNodeList = doc.getElementsByTagName(Messages.getString("HyadesGAContextsElementTagName"));

            if (contextsNodeList.getLength() == 0) {

                //Create a contexts element:	
                Element contextsElement = doc.createElement(Messages.getString("HyadesGAContextsElementTagName"));
                
                /* Insert the logging context at the first of the child list of elements */
                NodeList children=contextsElement.getChildNodes();
                for(int i=0; i<children.getLength(); i++) {
                	if(children.item(i).getNodeType()==Node.ELEMENT_NODE) {
                		contextsElement.insertBefore(loggingContext, children.item(i));
                		break;
                	}	
                }

                doc.getDocumentElement().appendChild(contextsElement);
            }
            else {

                //ASSUMPTION:  Only one contexts element permitted.	

                //Retrieve the contexts element:	
                Node contextsElement = contextsNodeList.item(0);
                /* Insert the logging context at the first of the child list of elements */
                NodeList children=contextsElement.getChildNodes();
                for(int i=0; i<children.getLength(); i++) {
                	if(children.item(i).getNodeType()==Node.ELEMENT_NODE) {
                		contextsElement.insertBefore(loggingContext, children.item(i));
                		break;
                	}	
                }

                doc.getDocumentElement().appendChild(contextsElement);
            }

            try {
                NodeList contextList = doc.getElementsByTagName(Messages.getString("HyadesGAContextElementTagName"));
                // check to see if this file contains everything or just the context config
                NodeList tempInstanceList = doc.getElementsByTagName(Messages.getString("HyadesGAContextInstanceElementTagName"));
                if (tempInstanceList.getLength() > 0)
                    singleFileInputMode = true;
                else
                    singleFileInputMode = false;
                int count = contextList.getLength();
                contexts = new Context[count];
                int j = 0;
                Element element = null;
                Context context = null;
                Component component = null;
                
                for (int i = 0; i < count; ++i) {
                    //TODO: HS need to support imbedded/nested contexts/components
                    // Extract each root context and copy the expected attributes
                    element = (Element) contextList.item(i);
                	/* try and build the context.  A number of things can go wrong here as we use relection to build the instance. */
                	context = ContextFactory.buildContext(element.getAttribute(Messages.getString("HyadesGAExecutableClassAttributeName")), element.getAttribute(Messages.getString("HyadesGAUniqueIDAttributeName")), element.getAttribute(Messages.getString("HyadesGANameAttributeName")));	
                	context.setContextConfiguration(element);

                    //
                    // get a list of all sub elements and assume they are components
                    NodeList componentList = element.getElementsByTagName(Messages.getString("HyadesGAComponentElementTagName"));
                    IComponent[] compArray = null;
                    int componentCount = componentList.getLength();
                    compArray = new IComponent[componentCount];
                    for (int k = 0, l = 0; k < componentCount; k++) {
                        // Extract each component config
                        element = (Element) componentList.item(k);
                        String executableClass1 = element.getAttribute(Messages.getString("HyadesGAExecutableClassAttributeName"));
                        String componentName1 = element.getAttribute(Messages.getString("HyadesGANameAttributeName"));
                        component = ComponentFactory.buildComponent(element.getAttribute(Messages.getString("HyadesGAExecutableClassAttributeName")), element.getAttribute(Messages.getString("HyadesGAUniqueIDAttributeName")), element.getAttribute(Messages.getString("HyadesGANameAttributeName")));
                        component.setContextConfiguration(element);
                        compArray[l++] = component;
                    }
                    // put the components in the context
                    context.setComponents(compArray);
                    // put this context in the controller's array
                    contexts[j++] = context;
                }
            }
            catch (Exception e) {
                log(e.toString());
                throw new PreparationException(e.getMessage());
            }
        }
        catch (SAXException e) {
            log(e.toString());
        }
        catch (ParserConfigurationException e) {
            log(e.toString());
        }
        catch (IOException e) {
            log(e.toString());
        }
        return contexts;
    }

    /**
     * CreateContextsConfig takes in an entire component configurations file and
     * separates out the context instances and gives each context the sub tree for itself and
     * it's components.
     */
    private void setComponentConfigurations(InputStream inAppStream) throws AdapterInvalidConfig {

        Document doc = null;

        try {

            doc = docFactory.newDocumentBuilder().parse(inAppStream);

            //Add the adapter logging instance as default:
            Element loggingSensorInstance = doc.createElement(Messages.getString("HyadesGASensorTagName"));
            loggingSensorInstance.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "An adapter CBE sensor");
            loggingSensorInstance.setAttribute(Messages.getString("HyadesGAmaximumBlockingAttributeName"), "5");
            loggingSensorInstance.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "NF991E0004FF11D8930381B6A308BEB5");

            Element loggerContextInstance = doc.createElement(Messages.getString("HyadesGAContextInstanceElementTagName"));
            loggerContextInstance.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "Context instance for the current component");
            loggerContextInstance.setAttribute(Messages.getString("HyadesGAcontinuousOperationAttributeName"), "true");
            loggerContextInstance.setAttribute(Messages.getString("HyadesGAmaximumIdleTimeAttributeName"), "600000");
            loggerContextInstance.setAttribute(Messages.getString("HyadesGApauseIntervalAttributeName"), "10");
            loggerContextInstance.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "NB1F4ED002DA11D8A519FBE7C98C2F53");

            loggerContextInstance.appendChild(loggingSensorInstance);

            if (AdapterUtilities.isWorkbench()) {
            	
				Element loggingOutputterInstance = doc.createElement(Messages.getString("HyadesGAOutputterTagName"));
				loggingOutputterInstance.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "Eclipse error dialog outputter");
				loggingOutputterInstance.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "N5F286A002DA11D8BC799C6AF4352915");
				
				/* RKD:  We need to inform the outputter of the configuration path so it can create markers for the resource */
                Element resourcePathProperty=doc.createElement(Messages.getString("HyadesGAPropertyElementTagName"));
                resourcePathProperty.setAttribute(Messages.getString("HyadesGAPropertyNameAttributeName"), "resourceName");
                resourcePathProperty.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), contextConfigurationFile);
                
                loggingOutputterInstance.appendChild(resourcePathProperty);
                loggerContextInstance.appendChild(loggingOutputterInstance);
				
				
            }
            else{

                Element loggingOutputterInstance = doc.createElement(Messages.getString("HyadesGAOutputterTagName"));
                loggingOutputterInstance.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "Single file outputter");
                loggingOutputterInstance.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "N5F286A002DA11D8BC799C6AF4352915");

				Element outputterProperty = doc.createElement(Messages.getString("HyadesGAPropertyElementTagName"));
				outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), ".");
				outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyNameAttributeName"),Messages.getString("HyadesGAdirectoryAttributeName"));
                loggingOutputterInstance.appendChild(outputterProperty);
                
				outputterProperty = doc.createElement(Messages.getString("HyadesGAPropertyElementTagName"));
				outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), "hgla.log");
				outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyNameAttributeName"),Messages.getString("HyadesGAfileNameAttributeName"));
				loggingOutputterInstance.appendChild(outputterProperty);                
				
                loggerContextInstance.appendChild(loggingOutputterInstance);
            }

            NodeList configurationNodeList = doc.getElementsByTagName(Messages.getString("HyadesGAConfigurationTagName"));

            if (configurationNodeList.getLength() == 0) {

                //Create a configuration element:	
                Element loggerConfigurationInstance = doc.createElement(Messages.getString("HyadesGAConfigurationTagName"));
                loggerConfigurationInstance.setAttribute(Messages.getString("HyadesGADescriptionAttributeName"), "The component level configurations for this adapter");
                loggerConfigurationInstance.setAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"), "N06FBD3004FF11D8BCF4CFA9EA8F31E7");

                loggerConfigurationInstance.appendChild(loggerContextInstance);

                doc.getDocumentElement().appendChild(loggerConfigurationInstance);
            }
            else {

                //ASSUMPTION:  Only one configuration element permitted.	

                //Retrieve the configuration element:	
                Node loggerConfigurationInstance = configurationNodeList.item(0);
                loggerConfigurationInstance.appendChild(loggerContextInstance);

                doc.getDocumentElement().appendChild(loggerConfigurationInstance);
            }

            try {
                // get a list of all the context instances and associates each with a context
                NodeList contextInstanceList = doc.getElementsByTagName(Messages.getString("HyadesGAContextInstanceElementTagName"));

                int contextInstanceCount = contextInstanceList.getLength();
                for (int i = 0; i < contextInstanceCount; ++i) {
                    try {
                        //find the matching Context and Context Instance
                        Element contextElement = (Element) contextInstanceList.item(i);
                        String contextInstanceID = contextElement.getAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"));

                        int contextCount = contexts.length;
						/* Throw an exception if we have a mismatch in the number of Context's and
						 * the number of ContextInstance's.
						 */
						if (contextInstanceCount != contextCount) {
							throw new AdapterInvalidConfig(Messages.getString("HyadesGAContextConfigurationErrorContextCountMismatchFatal_ERROR_"));
						}                       
                        
                        boolean contextFound = false;
                        for (int j = 0; j < contextCount; j++) {
                            if (contexts[j].getUniqueID().equals(contextInstanceID)) {
                                contexts[j].setConfiguration(contextElement);
                                contextFound = true;
                                // so now we have a good context which may or may not have components
                                //assume each child of the contextInstance is a component
                                NodeList componentInstanceList = contextElement.getChildNodes();
                                int numberOfComponentInstances = componentInstanceList.getLength();
                                // get the components from the context
								IComponent components[] = contexts[j].getComponents();
								int componentCount = components.length;

								int componentInstanceCount = 0;
                                for (int k = 0; k < numberOfComponentInstances; k++) {
                                    if (componentInstanceList.item(k).getNodeType() == Node.ELEMENT_NODE) {
                                        Element componentElement = (Element) componentInstanceList.item(k);
                                        String componentInstanceID = componentElement.getAttribute(Messages.getString("HyadesGAUniqueIDAttributeName"));
                                        boolean componentFound = false;
                                        componentInstanceCount++;
                                        // walk through the components of the context and set the config
                                        for (int l = 0; l < componentCount; l++) {
                                            if (components[l].getUniqueID().equals(componentInstanceID)) {
                                                components[l].setConfiguration(componentElement);
                                                componentFound = true;
                                                break;
                                            }
                                            //TODO: HS if there was not a match throw a config exception
                                        }
                                        if (!componentFound) {
											throw new AdapterInvalidConfig(Messages.getString("HyadesGAContextInstanceConfigurationErrorComponentIdNotFoundFatal_ERROR_",componentInstanceID , contexts[j].getUniqueID()));
                                        }
                                    }
                                }
								/* Throw an exception if we have a mismatch in number of components between
								 * the context and the contextInstance.
								 */
								if (componentInstanceCount != componentCount) {
									throw new AdapterInvalidConfig(Messages.getString("HyadesGAContextInstanceConfigurationErrorComponentMismatchFatal_ERROR_",contexts[j].getUniqueID()));
								}
                                break;
                            }
                        }
						if (!contextFound) {
							throw new AdapterInvalidConfig(Messages.getString("HyadesGAContextInstanceConfigurationErrorContextIdNotFoundFatal_ERROR_", contextInstanceID));
						}
                        //TODO: HS if there was not a match throw a config exception
                    }
                    catch (AdapterInvalidConfig e) {
                    	throw e;
                    }
                    catch (Exception e) {
                        log(e.toString());
                    }
                }
            }
            catch (AdapterInvalidConfig e) {
            	throw e;
            }
            catch (Exception e) {
                log(e.toString());
                return;
            }
        }
        catch (SAXException e) {
            log(e.toString());
        }
        catch (ParserConfigurationException e) {
            log(e.toString());
        }
        catch (IOException e) {
            log(e.toString());
        }
    }
    /**
     * run provides the basic loop for a Controller thread. 
     */
    public void run() {
        start();
        while (isRunning()) {
            synchronized (this) {
                try {
                    wait(sleepTime);
                }
                catch (Exception e) {
                    log(e.toString());
                }
                if (areContextsDone()) {
                    stop();
                }
            }
        }
    }
    /**
     * return false if any context is still alive, otherwise return true 
     */
    private boolean areContextsDone() {
        if (contextThreads != null) {
        	/* Ignore the internal logging context.  hense we start at index 1 */
            for (int i = 1; i < contextThreads.length; i++) {
                if (contextThreads[i] != null) {
                    if (contextThreads[i].isAlive() && !(contexts[i] instanceof AdapterContext))
                        return false;
                }
            }
        }
        return true;
    }
    /**
     * clean a controller by nulling out everything
     */
    private void clean() {
        if (contexts != null) {
            for (int i = 0; i < contexts.length; i++) {
                contexts[i] = null;
            }
        }
        contexts = null;
        //		hashContextConfig.clear();
    }
    
    /**
     * Starts the internal logging context.
     *
     */
    private IProcessUnit startInternalLoggingContext() {
    	contexts[0].init();
    	contextThreads=new Thread[contexts.length];
        try {
        	contexts[0].update();
        	contextThreads[0]=new Thread(contexts[0]);
        	contextThreads[0].setName(contexts[0].getName() + Messages.getString("HyadesGA") + contexts[0].getUniqueID());
        	contextThreads[0].setDaemon(true);
        	contextThreads[0].start();
        	return (IProcessUnit)contexts[0].getComponents()[0];
        	
        }
        catch(AdapterException e) {
        	/*  Since we are genning the context content we should not have configuration problems */
        }
        return null;
    }
    
    /**
     * Log the string message to standard error since the logging outputter(s)
     * associated with the <code>AdapterSensor</code> logger will never be
     * started.
     * 
     * @param logRecord The string message to be logged to standard error.
     */
    public void log(String logRecord) {
    	if(!AdapterUtilities.isWorkbench()) {
    		System.err.println(logRecord);
    	}
       
    }
}
