package org.eclipse.hyades.logging.adapter.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.StringTokenizer;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.IDirectedGraph;
import org.eclipse.hyades.logging.adapter.IParser;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author rduggan
 *
 * This class represents a particular attribute to be
 * parsed from the record.  It can have multiple substitution
 * rules that are used to determine the value for the
 * attribute based on the data in the record.
 *  
 */
public class AttributeStatement implements IStatement {

	/* The parser that this AttributeStatement is a child of */
	private IParser parser;
	
	List substitutions;

	/* The name of this attribute */
	private String name;
	
	/* The default value for this attribute */
	private String defaultValue;
	
	/* Whether ths Attribute keeps state between rule runs */
	private boolean usePreviousMatchSubstitutionAsDefault=false;
	
	/* Our previously successful SubstuteAtttrbite value */
	private IDirectedGraph previousMatch=null;
	 
	/**
	 * 
	 */
	public AttributeStatement() {
	}
	
	/**
	 * 
	 */
	public AttributeStatement(IParser parser) {
		this.parser=parser;
	}
	


	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#prepare(org.w3c.dom.Element)
	 */
	public void prepare(Element node, String path) throws PreparationException {
	
		if(path==null) {
			this.name=node.getAttribute(Messages.getString("HyadesGANameAttributeName"));
		}
		else {
			this.name=path+"."+node.getAttribute(Messages.getString("HyadesGANameAttributeName"));
		}
		
		/* Retrieve the index if there is one specified and embed this into the name */
		String index=node.getAttribute(Messages.getString("HyadesGAIndexAttributeName"));
		if(index!=null && !index.equals("")) {
			this.name=this.name+"["+index+"]";
		}
		
		/* Retrieve the default value for this attribute */
		defaultValue = node.getAttribute(Messages.getString("HyadesGAdefaultValueAttributeName"));
		if(defaultValue == null || defaultValue.equals("")) {
			defaultValue = null;
		}
		
		/* Retrieve whether we are keeping previous matches around */
		String previous=node.getAttribute(Messages.getString("HyadesGAUsePreviousMatchSubstitutionAsDefault"));
		if(previous==null || previous.equals("") || previous.equals("false")) {
			usePreviousMatchSubstitutionAsDefault=false;
		}
		else {
			usePreviousMatchSubstitutionAsDefault=true;
		}
		
		/* Create a SubstituteStatement per Substitution rule */
		NodeList children=node.getChildNodes();
		int count=children.getLength();
		if(count>0) {
			substitutions=new ArrayList();
			for(int i=0; i<count; i++) {
				Element child=null;
				try {
					child=(Element)children.item(i);
				}
				catch(ClassCastException e) {
					/* We can ignore this child as it is not a Element */	
					continue; 	
				}
				if(child.getNodeName().endsWith(Messages.getString("HyadesGASubstitutionRuleTagName"))) {

					IStatement statement=new SubstituteStatement();
					
					try {
						statement.prepare(child, name);
						substitutions.add(statement);
					}
					catch(PreparationException e) {
						/* SubstitutionStatement rule has a problem.  We will log the fact this is a problem and ignore this rule during the run */
						if(parser!=null && parser instanceof Parser) {
							CommonBaseEvent event = ((Parser)parser).getEventFactory().createCommonBaseEvent();
							if(e.getDetails()!=null) {
								event.setMsg(e.getMessage()+" "+e.getDetails());
							}
							else {
								event.setMsg(e.getMessage());
							}
						
						    event.setSeverity(CommonBaseEvent.SEVERITY_CRITICAL);
							
							ExtendedDataElement ede=((Parser)parser).getEventFactory().createExtendedDataElement();
							ede.setName("EXCEPTION DETAILS");
							ede.setType(ExtendedDataElement.TYPE_STRING);
							ede.setValues(new String[] {e.getLocalizedMessage()});
							event.addExtendedDataElement(ede);
							
							ede=((Parser)parser).getEventFactory().createExtendedDataElement();
							ede.setName("PATH");
							ede.setType(ExtendedDataElement.TYPE_STRING);
							ede.setValues(new String[] {e.getDigraph()});
							event.addExtendedDataElement(ede);
							
						    parser.log(event);
							
						}
						else {
							throw e;
						}
					}
				}
			}
			if (substitutions.isEmpty())
				substitutions = null;	
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#run(java.lang.String)
	 */
	public boolean run(String line, HashMap inputs, List outputs) throws AdapterException {
		/*
		 * Iterate through each of the child rules looking for a match.  If there
		 * is a match then stop the iteration and return.
		 */
		if (substitutions!=null) {
			ListIterator iter=substitutions.listIterator();
			SubstituteStatement statement = null;
			while(iter.hasNext()) {
				statement=(SubstituteStatement)iter.next();
				if(statement.run(line, inputs, outputs)) {
					/* lookup our match result and store it */
					if(usePreviousMatchSubstitutionAsDefault) {
						previousMatch=statement.getResultGraph();
					}
					return true;
				}
			}
			/* If there is no match and previous is enabled then place the previous in the
			 * outputs
			 */
			if(usePreviousMatchSubstitutionAsDefault && previousMatch!=null) {
				outputs.add(previousMatch);
				return true;
			}
			/* Else if there is a default value then add this value to the outputs */
			else if (defaultValue != null) {
				outputs.add(statement.createResultGraph(defaultValue));
				return true;
			}
			
		}
		/* If there are no substitution rules, check whether there is a default value */
		else if (defaultValue != null) {
			/* Create a result graph containing the default value and add it to the outputs */
			IDirectedGraph result = new DirectedGraphImpl();
			List path = new ArrayList();			
			StringTokenizer s = new StringTokenizer(name, ".");
			
			while (s.hasMoreTokens()) {
				String subPath=s.nextToken();
				int indexStart=subPath.indexOf('[');
				if(indexStart>0) {
					String index=subPath.substring(indexStart+1, subPath.length()-1);
					subPath=subPath.substring(0, indexStart);
					path.add(subPath);
					path.add(index);
				}
				else {
					path.add(subPath);
				}
			}			
			result.setPath(path);
			result.setValue(defaultValue);
			outputs.add(result);
			return true; 
		}
		
		return false;
	}
}
