/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.security.util;

import java.io.*;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.*;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPluginRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;

import org.eclipse.hyades.execution.security.*;
import org.eclipse.hyades.internal.execution.local.control.*;
import org.eclipse.hyades.internal.execution.security.*;
import org.eclipse.hyades.security.SecurityPlugin;


public class ConnectUtil
{
  public static final int CONNECTION_SUCCESS = 0;
  public static final int CONNECTION_CONTROLLER_ERROR = 1;
  public static final int CONNECTION_HOST_ERROR = 2;
  public static final int CONNECTION_PORT_ERROR = 3;
  public static final int CONNECTION_SECURITY_NOT_SUPPORTED = 4;
  
  private Application _app;
  private String _port;
  private String _hostName;
  private InetAddress _hostAddr;  
  private String _userId;
  private Node   _node;
  private ISecureClientParameters _parameters;
    
  
    public ConnectUtil(String hostName, String port, Application app)
    {
    	this(hostName, port, null, app);
    }

    public ConnectUtil(String hostName, String port
    							, String userId
    							, Application app)
    {
    	_hostName = hostName;
    	_port     = port;
    	_userId   = userId;
    	_app      = app;
    }

    public ConnectUtil(InetAddress hostAddr, String port, Application app)
    {
    	this(hostAddr, port, null, app);
    }

    public ConnectUtil(InetAddress hostAddr, String port
    							, String userId
    							, Application app)
    {
    	_hostAddr = hostAddr;
    	_hostName = _hostAddr.getHostName();
    	_port     = port;
    	_userId   = userId;
    	_app      = app;
    }
    
    private ISecureClientParameters getClientParms() {
		IPluginRegistry reg = Platform.getPluginRegistry();

		IConfigurationElement[] config =
		Platform.getPluginRegistry().getConfigurationElementsFor(SecurityPlugin.getPluginId(), "JSSESecurityProviders");

		for (int idx = 0; idx < config.length; idx++) {
			IConfigurationElement elem = config[idx];
	
			String typeAttr = elem.getAttribute("name");
			String classAttr = elem.getAttribute("class");
			
			/* For each of our security providers, instantiate the instances */	
			if(classAttr!=null) {
				try {
					Object realization=elem.createExecutableExtension("class");
					if(realization instanceof ISecureClientParameters) {
						ISecureClientParameters parms=(ISecureClientParameters)realization;
						/* Add the provider from the realization */
						Security.addProvider(parms.getSecurityProvider());
						parms.getKeystoreManager().setProvider(parms.getSecurityProvider());
						
						
						return (ISecureClientParameters)realization;	
					}
				}
				catch(CoreException e) {
				}
			}
		}
		return null;
    }

	public final int connect()
	{
		try {
			if(_hostAddr == null)
				_hostAddr = InetAddress.getByName(_hostName);
		}
		catch(UnknownHostException exc)
		{
			resetConnection();
						
			String text = SecurityPlugin.getResourceString("TIMEOUT_NODE_ERROR_");
			text = TString.change(text, "%1", _hostName);
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);		
					       
			return CONNECTION_HOST_ERROR;			
		}
		
		try {
			
			/* Determine if we have aconnection with this node/user/app existing */
			if(checkConnectionExists()) {
				return CONNECTION_SUCCESS;
			}
			
			
			if(_node == null)
			{
				_node = NodeFactory.createNode(_hostAddr, _app);
			}
			
			_node.connect(Integer.parseInt(_port));
		}
		catch(UnknownHostException exc)
		{
			resetConnection();
						
			String text = SecurityPlugin.getResourceString("TIMEOUT_NODE_ERROR_");
			text = TString.change(text, "%1", _hostName);
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);		
					       
			return CONNECTION_HOST_ERROR;
		}
		catch(SecureConnectionRequiredException exc) {  
			
			return connectUnderException(exc);
		}
		catch (LoginFailedException exc) {
			return connectUnderException(exc);	
		}
		
		catch(UntrustedAgentControllerException exc) {
			
			/* RKD:  First we open a dialog stating that the server is not trusted and ask the user to continue */
			String text = SecurityPlugin.getResourceString("STR_CERTIF_UNAV_ERROR_");    
			text = TString.change(text, "%1", _hostName);
			
			
			String txt = TString.change(SecurityPlugin.getResourceString("STR_UNTRUSTED_AGENT_CONTROLLER"), "%1", _hostName);
			boolean ok = openQuestionDialog(SecurityPlugin.getResourceString("SEC_MSG"), txt);
			
			if(!ok) {
				
				resetConnection();
				
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
				openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
		
				return CONNECTION_CONTROLLER_ERROR;
			}
			else {
				ISecureClientParameters params=_node.getSecurityParameters();
				if(params!=null) {
					params.disableServerAuthentication();
				}
				return connect();
			
			}
			
		}
		catch(AgentControllerUnavailableException exc)
		{
			String text = SecurityPlugin.getResourceString("STR_AGENT_CONTROLLER_UNAV_ERROR_");
			    
			text = TString.change(text, "%1", _hostName);
						
			resetConnection();
			
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
		
			return CONNECTION_CONTROLLER_ERROR;
		}		
		catch(NumberFormatException exc)
		{
			resetConnection();
			
			String text = SecurityPlugin.getResourceString("INVALID_PORT_ERROR_");
			text = TString.change(text, "%1", String.valueOf(_port));
			String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
			msg = TString.change(msg, "%1", _hostName);
			
			Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,exc);
			openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
		
			return CONNECTION_PORT_ERROR;
		}
				
		return CONNECTION_SUCCESS;
	}
	
	public String getUserId()
	{
		return _userId;
	}
	
	public Node getNode()
	{
		return _node;
	}
	
	private void resetConnection()
	{
	    if(_userId != null) {
	       UserFactory.removeUser(_app, _userId);
	    }
	    if(_node!=null) {
	    	_node.setSecurityParameters(null);
	    }
		
	}
	
	private ConnectivityDialog dlg; 
	private int connectUnderException(AgentControllerUnavailableException exc) {
		try
		 {			 	
			 	User user = null;
			 	if(_userId != null && !_userId.equals("")) {
			 	 	user = UserFactory.getUser(_app, _userId);
			 	}
			 	 	
			 	if(user == null)
			 	{
			 		/* RKD: This dialog should be only put up if the server is using password authentication
			 		 */
			 		 if(exc instanceof SecureConnectionRequiredException && ((SecureConnectionRequiredException)exc).isPasswordProtected()) {
				 		 	
				 		/* RKD: This dialog needs to state the security requirements of the server */
			 		 	Display.getDefault().syncExec(new Runnable() {
							public void run() {
					 		 	dlg = new ConnectivityDialog(getValidShell(), _hostName, _userId, (_userId == null || _userId.equals("")));
						 		dlg.setDescription(SecurityPlugin.getResourceString("STR_PWD_REQ_INFO_"));
						 		dlg.create();
						 		openDialog(dlg);
							}
			 		 	});
				 		
				 		if(dlg.getReturnCode() != Window.OK) {
				 			
							resetConnection();
										
							String text = SecurityPlugin.getResourceString("STR_PWD_REQ_INFO_");
							String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
							msg = TString.change(msg, "%1", _hostName);
							
							Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,null);
							openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);		
				 			
				 		   return CONNECTION_HOST_ERROR;
				 		}
				 		   		
				 		_userId = dlg.getUserId();
				 		 
				 		if(UserFactory.getUser(_app, _userId) != null)
				 		{//remove the user first
				 			UserFactory.removeUser(_app, _userId);
				 		}
			 		
				 		user=UserFactory.createUser( _app, _userId, dlg.getPassword());	
			 		 }
			 		 else {
			 		 	/* RKD:  I don't think this is required anymore */
			 		 	_userId="ignoredUserId";
			 		 	user=UserFactory.createUser( _app, _userId, "dummyPass");
			 		 }	 		
			 	}  
			 	else if(exc instanceof LoginFailedException) {
			 		
			 		/* This dialog should state that the previous login failed */
			 		Display.getDefault().syncExec(new Runnable() {
						public void run() {
					 		dlg = new ConnectivityDialog(getValidShell(), _hostName, _userId);
					 		dlg.setDescription(SecurityPlugin.getResourceString("STR_LOGIN_FAILED_INFO_"));
					 		dlg.create();
						 	openDialog(dlg);
						}
			 		});
				 		
			 		if(dlg.getReturnCode() != Window.OK) {
			 			resetConnection();
									
						String text = SecurityPlugin.getResourceString("STR_PWD_REQ_INFO_");
						String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
						msg = TString.change(msg, "%1", _hostName);
						
						Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,null);
						openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"),msg,err);		
			 			
			 		   return CONNECTION_HOST_ERROR;
			 		}
			 		   	
			 		   		
			 		_userId = dlg.getUserId();
			 		
					user=UserFactory.getUser(_app, _userId);
			 		if(user != null)
			 		{//remove the user first
			 			user.setPassword(dlg.getPassword());
			 		}
			 		else {
		 		
			 			try {
			 				user=UserFactory.createUser( _app, _userId, dlg.getPassword());	
			 			}
			 			catch(DuplicateUserException e) {
						 }
			 		}			 		
		 		} 	 	
		 		
		 		if(exc instanceof SecureConnectionRequiredException) {
			 	 	_port=new Long(((SecureConnectionRequiredException)exc).getSecurePort()).toString();
			 	 
			 	}
			 	else if(exc instanceof LoginFailedException) {
			 		_port=new Long(((LoginFailedException)exc).getSecurePort()).toString();
			 	}
			 	
			 	if(_node.getSecurityParameters()==null) {
					ISecureClientParameters parms=getClientParms();
			 		if(parms==null) {
						resetConnection();
			 		
						String text = SecurityPlugin.getResourceString("STR_CLIENT_CANNOT_USE_SECURITY_ERROR_");
						text = TString.change(text, "%1", _hostName);
						String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
						msg = TString.change(msg, "%1", _hostName);
	
						Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text, null);
						openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg, err);
						return CONNECTION_SECURITY_NOT_SUPPORTED;
			 			
			 		}
			 		else {
						/* Load the keystore */ 
					 	_node.setSecurityParameters(getClientParms());
					 	_node.getSecurityParameters().getKeystoreManager().loadKeystore(_node.getSecurityParameters().getKeystoreFileName(), _node.getSecurityParameters().getKeystoreFilepassword());			 			
			 		}
			 		
			 	}

		 		
			 	/* Update the user info on this node */
			 	_node.setUser(user);
			 	
			 	return connect();
			 	 
			 	
			 }
			 catch(DuplicateUserException e)
			 {
						 				 	
			 	resetConnection();
			 }		 				 	 
			 catch(IOException e)
			 {
				
				try {
					 				
				 	if(_node.getSecurityParameters()==null) {
				 		resetConnection();
			 		
			 			String text = SecurityPlugin.getResourceString("STR_CLIENT_CANNOT_USE_SECURITY_ERROR_");
						text = TString.change(text, "%1", _hostName);
						String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
						msg = TString.change(msg, "%1", _hostName);
						
						Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text, e);
						openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg, err);
				 		return CONNECTION_SECURITY_NOT_SUPPORTED;
				 	}
					else {
						/* Create an empty keystore  keystore */
						KeystoreHelper.createKeyStore(SecurityPlugin.getKeyStoreLocation(), SecurityPlugin.getWorkspaceName());
						/* Load the keystore */ 
						 _node.setSecurityParameters(getClientParms());
						 _node.getSecurityParameters().getKeystoreManager().loadKeystore(_node.getSecurityParameters().getKeystoreFileName(), _node.getSecurityParameters().getKeystoreFilepassword());		
	
	
					}
				 	
				 	
				 	// Update the user info on this node 
				 	_node.setUser(UserFactory.getUser(_app, _userId));
					
					return connect();
					
				}
				catch(Exception e1) {
					
			 		resetConnection();
			 		
					String text = SecurityPlugin.getResourceString("STR_KEY_NOT_FOUND_ERROR_");
					text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
					String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
					msg = TString.change(msg, "%1", _hostName);
					
					Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
					openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				
					return CONNECTION_HOST_ERROR;
				}
			 }
			 
			 catch(CertificateException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_KEY_LOAD_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			
				return CONNECTION_HOST_ERROR;
			 }	
			 catch(KeyManagementException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_KEY_MANAG_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			
				return CONNECTION_HOST_ERROR;
			 }			
			 catch(KeyStoreException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_KEY_STORE_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			
				return CONNECTION_HOST_ERROR;
			 }
			 catch(NoSuchAlgorithmException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_ALGORITHM_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
			
				return CONNECTION_HOST_ERROR;							
			 }			
			 catch(UnrecoverableKeyException e)
			 {
				resetConnection();
			 	
				String text = SecurityPlugin.getResourceString("STR_UNREC_KEY_ERROR_");
				text = TString.change(text, "%1", SecurityPlugin.getKeyStoreLocation());
				String msg = SecurityPlugin.getResourceString("CONNECTION_FAIL_TEXT");
				msg = TString.change(msg, "%1", _hostName);
				
				Status err = new Status(Status.WARNING,ResourcesPlugin.PI_RESOURCES,IResourceStatus.WARNING,text,e);
				openErrorDialog(SecurityPlugin.getResourceString("SEC_MSG"), msg,err);
				
				return CONNECTION_HOST_ERROR;				
			 }	
			 
			 
			 return CONNECTION_SUCCESS;
			 					 		
	}
	
	private boolean checkConnectionExists() {
	
		Node node=NodeFactory.getNode(_hostAddr, _app);
		if(node!=null) {
			if(node.isConnected()) {
				_node=node;
				return true;
			}	
		}
		
		return false;
	}

	private void openErrorDialog(final String title, final String msg, final Status err)
	{
		Display.getDefault().syncExec(new Runnable()
		{
			public void run()
			{
				ErrorDialog.openError(getValidShell(),title, msg,err);
			}
		});
	}
	
	private void openDialog(final Dialog dialog)
	{
		dialog.getShell().getDisplay().syncExec(new Runnable()
		{
			public void run()
			{
				dialog.open();
			}
		});
	}

	private static Shell getValidShell()
	{
		Shell shell = null;
		IWorkbench workbench = PlatformUI.getWorkbench();
		if(workbench != null)
		{
			if(workbench.getActiveWorkbenchWindow() != null)
			{
				shell = workbench.getActiveWorkbenchWindow().getShell();
				if((shell != null) && (!shell.isDisposed()))
					return shell;
			}
			
			if(workbench.getWorkbenchWindows().length > 0)
			{
				shell = workbench.getWorkbenchWindows()[0].getShell();
				if((shell != null) && (!shell.isDisposed()))
					return shell;
			}
		}
		
		return null;
	}		
		
	private boolean okPressed = false;
	private boolean openQuestionDialog(final String title, final String msg)
	{		
		Display.getDefault().syncExec(new Runnable()
		{
			public void run()
			{
				okPressed = MessageDialog.openQuestion(getValidShell(),title, msg);
			}
		});
		
		return okPressed;
	}
}
