/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.config.sensors;
import java.util.Hashtable;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.impl.Sensor;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.impl.SimpleEventFactoryHomeImpl;
import org.eclipse.hyades.logging.parsers.IParser;
import org.eclipse.hyades.logging.parsers.Parser;
import org.eclipse.hyades.logging.parsers.ParserConstants;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
/**
 * A sensor that calls a static parser that generates CBEs
 * 
 * @version 0.1
 */
public class StaticParserSensor extends Sensor
{
	public final static String PATH_SEPARATOR = System.getProperty("file.separator");
	
	/** 
	 * class name for the static parser
	 */
	protected String parserClassName = null;


	/** 
	 * instance of static parser class
	 */
	protected Parser parser = null;
		
	/**
	 * directory address to start with, maybe relative or absolute 
	 */
	private String directory = null;
	
		
	/**
	 * relative file name (just name), maybe just regular expression 
	 */
	private String fileName = null;
	protected String rawLogFileName = null;
    
	/**
	 * Constructor for SingleOSFileSensor.
	 */
	public StaticParserSensor()
	{
		super();
	}
	/**
	 * @see org.eclipse.hyades.logging.adapter.IComponent#stop()
	 */
	public void stop()
	{
		super.stop();
		
		//NOTE:  The clean() API MUST be called here:
		clean();
	}

	/**
	 * update the configuration based on the configuration Element
	 */
	public void update() throws AdapterInvalidConfig
	{
		// first get the basic configuration set
		super.update();
		// maximumBlocking is set by the sensor config
		String directory = null;
		String fileName = null;
		String className = null;

		String propertyName = null;
		String propertyValue = null;
		
		Element sensorProperty = null;
		Element sensorTypeInstance = null;
		Element sensorNode;
		Element element = getConfiguration();
		
		// Use the sensor properties as the parserParameters
		Hashtable parserParameters = getProperties();
		
		// This sensor may be configured with a sub element of the config
		// or the sensor may be configured with sensor properties only.

		// Get the sensor parameters from the sensor properties
		if (parserParameters != null && !parserParameters.isEmpty()) {
			directory = (String)parserParameters.get(Messages.getString("HyadesGAdirectoryAttributeName"));

			fileName = (String)parserParameters.get(Messages.getString("HyadesGAfileNameAttributeName"));
			
			className = (String)parserParameters.get(Messages.getString("HyadesGAstaticParserClassAttributeName"));									
		}	
		else {
			parserParameters = new Hashtable();		
	
			/**
			 * If we have an old style sensor configuration (eg. StaticParserSensor sensor type instance) 
			 * then get the sensor parameters from it.
			 */
			NodeList sensorNodes = element.getChildNodes();
			for (int i = 0; i < sensorNodes.getLength(); i++) {
				if (sensorNodes.item(i).getNodeType() == Node.ELEMENT_NODE) {
					sensorNode = (Element) sensorNodes.item(i);
	
					if (sensorNode.getTagName().equals(Messages.getString("HyadesGAStaticParserSensorTagName"))) {
						// Get the parameters from the sensor type instance attributes
						sensorTypeInstance = sensorNode;
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAdirectoryAttributeName"))) {
							String dir = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAdirectoryAttributeName"));
							if (dir != null && dir.length() > 0 && directory == null)
								directory = dir;
						}
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAfileNameAttributeName"))) {
							String file = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAfileNameAttributeName"));
							if (file != null && file.length() > 0 && fileName == null)
								fileName = file;
						}	
						if (sensorTypeInstance.hasAttribute(Messages.getString("HyadesGAstaticParserClassAttributeName")))
							className = sensorTypeInstance.getAttribute(Messages.getString("HyadesGAstaticParserClassAttributeName"));
					
					    // Get the parameters from the sensor type instance properties
						sensorNodes = sensorNode.getChildNodes();
						for (int j = 0; j < sensorNodes.getLength(); j++) {
							if (sensorNodes.item(j).getNodeType() == Node.ELEMENT_NODE) {
								sensorProperty = (Element) sensorNodes.item(j);
								// get name and value from property and add the property to the hash table.
								if (sensorProperty.hasAttribute(Messages.getString("HyadesGASensorPropertyNameAttributeName")))
									propertyName = sensorProperty.getAttribute(Messages.getString("HyadesGASensorPropertyNameAttributeName"));
		
								if (sensorProperty.hasAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName")))
									propertyValue = sensorProperty.getAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName"));
							
								if (propertyName != null && 
									propertyName.length() > 0 &&
									propertyValue != null &&
									propertyValue.length() > 0) {
									parserParameters.put(propertyName, propertyValue);
								}
							}
						}
					}
				}
			}
		}
		
		// We must have a directory and fileName
		if (directory == null || directory.length() == 0 || fileName == null || fileName.length() == 0 || className == null || className.length() == 0) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGA_CBE_StaticParserSensor_Invalid_Config_File_ERROR_"));
		}
		
		setDirectory(directory);
		setFileName(fileName);
		setParserClassName(className);
		
		// Construct the full input log file name
		if (getDirectory().endsWith(PATH_SEPARATOR)) {
			rawLogFileName = getDirectory() + getFileName();
		}
		else {
			rawLogFileName = getDirectory() + PATH_SEPARATOR + getFileName();
		}

		// Create the parameter Hashtable to be passed to the parser
		
		parserParameters.put(ParserConstants.FILE_PATH_KEY, rawLogFileName);
		parserParameters.put(ParserConstants.MESSAGE_ARRAY_SIZE_KEY, new Integer(maximumBlocking));
			
		parser=(Parser)getParser(getParserClassName());
		if(parser!=null) {
			
			try {
				parser.setConfiguration(parserParameters);
				parser.preParse();
			}
			catch(Throwable e) {
				throw new AdapterInvalidConfig(e);
			}
		}
		
	}
	
	/**
	 * 	simulates a getNext
	 * @see com.ibm.acad.general.sensor.ISensor#testGetNext()
	 */
	public Object[] testGetNext()
	{
		CommonBaseEvent[] eventArray = new CommonBaseEvent[1];
		eventArray[0] = new SimpleEventFactoryHomeImpl().getEventFactory("org.eclipse.hyades.logging.adapter.config.sensors.StaticParserSensor").createCommonBaseEvent();
		return eventArray;
	}

	/**
	 * 	returns last read line
	* @see com.ibm.acad.general.sensor.ISensor#getNext()
	 */
	public Object[] getNext()
	{

		if (parser == null || flushingMode) {
			return null;	
		}
		
		Object[] messages = null;
		
		try {
			
			messages=parser.parseNext();

		}
		catch (Throwable t) {
			// TODO Handle this exception
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    		    
		    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_File_Parsing_ERROR_");
		    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{rawLogFileName ,t.toString()});							    

		    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);

		    log(event);
		}
		

		if (messages != null && messages[0] != null) {
			return messages;
		}
		else {
			return null;
		}
	}
	/**
	 * do all cleaning here
	 */
	public void clean()
	{
		try
		{
			if (parser != null) {
				parser.postParse();
			}
		}
		catch (Exception e)
		{
		    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
		    
		    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_File_Parsing_ERROR_");
		    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{rawLogFileName ,e.toString()});							    

		    event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);

		    log(event);
		}
	}

	/**
	 * Returns the directory.
	 * @return String
	 */
	final public String getDirectory()
	{
		return directory;
	}
	/**
	 * Returns the fileName.
	 * @return String
	 */
	final public String getFileName()
	{
		return fileName;
	}
	/**
	 * Sets the directory.
	 * @param directory The directory to set
	 */
	final public void setDirectory(String directory)
	{
		this.directory = directory;
	}
	/**
	 * Sets the fileName.
	 * @param fileName The fileName to set
	 */
	final public void setFileName(String fileName)
	{
		this.fileName = fileName;
	}
	/**
	 * @return
	 */
	public String getParserClassName() {
		return parserClassName;
	}


	/**
	 * @param string
	 */
	public void setParserClassName(String string) {
		parserClassName = string;
	}
	
	public IParser getParser(String classname) {
		IParser parser=null;
		
		try {
			// Create a parser instance
			parser = (IParser)Class.forName(getParserClassName()).newInstance();
		}
		catch (Throwable t) {
			// Get parser instance using plugin registry
			IConfigurationElement[] elements = Platform.getPluginRegistry().getConfigurationElementsFor("org.eclipse.hyades.logging.adapter.config", "staticParser");
			for(int i=0; i<elements.length; i++) {
				IConfigurationElement elem = elements[i];
				if(elem.getAttribute("name").equals(classname)) {
					try {
						parser=(IParser)elem.createExecutableExtension("name");
						break;
					}
					catch(Throwable e) {	

						/* If we can't instantiate it then ignore it */
					    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
					    event.setMsg(e.toString());
					    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

					    log(event);
					}	
				}
			}
		}
		return parser;
	}

}
