/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.extensions;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.BatchUpdateException;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.resources.database.internal.DBCollectedExceptions;
import org.eclipse.hyades.resources.database.internal.impl.DBHyadesResourceExtension;
import org.eclipse.hyades.resources.database.internal.impl.SQLStatement;
/**
 * This class provides basic JDBC functionality.
 */
public abstract class JDBCHelper {

	public class DriverClassLoader extends URLClassLoader {

		public DriverClassLoader(URL urls[], ClassLoader parent) {
			super(urls, parent);
		}

		//		public Class getDriverClass(String name) throws
		// ClassNotFoundException {
		//			try {
		//				return loadClass(name, true);
		//			} catch (ClassNotFoundException ex) {
		//				ex.printStackTrace();
		//			}
		//			return Class.forName(name);
		//		}
		protected Class findClass(final String name) throws ClassNotFoundException {
			return super.findClass(name);
		}

	}

	protected boolean debug = false;
	protected boolean toFileOnly = false;
	protected String protocol;
	protected String driver;
	protected Connection connection;
	protected DatabaseType type;
	protected String urlString;
	protected Driver driverInstance;
	protected Writer file;
//	protected Map preparedStatements = new HashMap();

	public JDBCHelper(DatabaseType type) {
		this.type = type;
		try {
			String s = System.getProperties().getProperty("JDBCHelper.debug"); // use
			debug = Boolean.valueOf(s).booleanValue();
		} catch (Exception e) {
			//ignore and leave the default cacheSize
		}
		try {
			String s = System.getProperties().getProperty("JDBCHelper.toFileOnly"); // use
			toFileOnly = Boolean.valueOf(s).booleanValue();
		} catch (Exception e) {
			//ignore and leave the default cacheSize
		}
	}

	public Driver loadJDBCDriver() throws Exception {
		//		Class.forName(driver).newInstance();
		if (driverInstance != null)
			return driverInstance;
		return loadJDBCDriverSpecial();
	}

	protected Driver loadJDBCDriverSpecial() throws Exception {
		//		ClassLoader cl = new URLClassLoader(new URL[]{new
		// URL("file://E:/test/workspaces/defects/hyades-3.0-rdb/com.ibm.etools.resources.database.extension/lib/db2j.jar")},
		// getClass().getClassLoader());
		if(driverInstance!=null)
			return driverInstance;
		DriverClassLoader cl = new DriverClassLoader(new URL[]{new URL(urlString)}, getClass().getClassLoader());
		//		Class clazz = cl.getDriverClass(driver);
		//		System.out.println("JDBCHelper.loadJDBCDriverSpecial() - before");
		//		driverInstance = (Driver) clazz.newInstance();
		//		DriverManager.registerDriver(driverInstance);
		//		System.out.println("JDBCHelper.loadJDBCDriverSpecial() - after");
		try {
			// If this is the first time the class is loaded, it registers
			// itself
			// with DriverManager
			Class driverClass = Class.forName(driver, true, cl);
			// In case we unloaded our own driver:

			Enumeration enum = DriverManager.getDrivers();

			while (enum.hasMoreElements()) {
				Driver element = (Driver) enum.nextElement();
				if (element.getClass() == driverClass) {
					driverInstance = element;
				}
			}
			
			if (driverInstance == null) {
				driverInstance = (Driver) driverClass.newInstance();
				DriverManager.registerDriver(driverInstance);
			}
		} catch (Exception exc) {
			throw exc;
		}

		return driverInstance;
	}

	/**
	 * Load the JDBC driver and open the connection to the named database.
	 */
	public void openForExistingDB(String name, Properties properties) throws Exception {
		loadJDBCDriver();
		connection = driverInstance.connect(protocol + getLocation(properties) + name, properties);
	}
	/**
	 * @param properties
	 * @return
	 */
	protected String getLocation(Properties properties) {
		String location = properties.getProperty("location");
		if (location == null || location.length() == 0)
			location = "";
		else
			location = "//" + location + "/";
		return location;
	}

	/**
	 * Load the JDBC driver and open the connection to the named database,
	 * creating it if it is not already created. NOTE: Currently, this only
	 * works for Cloudscape.
	 */
	public void openForNewDB(String name, Properties properties) throws Exception {
		loadJDBCDriver();
		connection = driverInstance.connect(protocol + getLocation(properties) + name + ";create=true", properties);
	}
	/**
	 * Close the database connection and shutdown the database. Returns true if
	 * the database was shut down successfully and false otherwise.
	 */
	public void close() throws SQLException {
		if (!getConnection().getAutoCommit())
			getConnection().commit();
		getConnection().close();
	}
	/**
	 * Shuts down the database. This is used by Cloudscape in embedded mode.
	 * Returns true if the shutdown was successful; false otherwise.
	 */
	public boolean shutdown() {
		boolean exceptionThrown = false;
		try {
			DriverManager.getConnection(protocol + ";shutdown=true");
		} catch (SQLException exception) {
			exceptionThrown = true;
		}
		return exceptionThrown;
	}
	public synchronized void executeStatement(String statement) throws Exception {
		Statement s = getConnection().createStatement();
		execute(s, statement);
		s.close();
	}
	public synchronized void executeUpdateStatement(String statement) throws Exception {
		Statement s = getConnection().createStatement();
		executeUpdate(s, statement);
		s.close();
	}
	/**
	 * @param s
	 * @param statement
	 */
	public synchronized void executeUpdate(Statement statement, String sqlString) throws Exception {
		if (debug == true)
			System.out.println(sqlString);
		if (toFileOnly == true)
			writeToFile(sqlString);
		else
			statement.executeUpdate(sqlString);
	}

	/**
	 * @param sqlString
	 */
	protected void writeToFile(String sqlString) {
		try {
			if (file == null) {
				file = new FileWriter("D:/temp/log_" + (new java.util.Date()).getTime() + ".sql");
			}

			file.write(sqlString+"\n");
			file.flush();
		} catch (IOException e) {
			e.printStackTrace();
			throw new RuntimeException(e);
		}
	}

	public void executeUpdateStatements(String[] statements) throws Exception {
		for (int i = 0; i < statements.length; i++)
			executeUpdateStatement(statements[i]);
	}
	public void executeUpdateStatements(List statements) throws Exception {
		for (int i = 0, l = statements.size(); i < l; i++) {
			SQLStatement statement = (SQLStatement) statements.get(i);
			executeUpdateStatement(statement.getStatement());
		}
	}

	public PreparedStatement createPreparedStatement(String statement) throws Exception {
//		PreparedStatement preparedStatement = (PreparedStatement)preparedStatements.get(statement);
//		if(preparedStatement==null)
//			preparedStatement = getConnection().prepareStatement(statement);
//		return preparedStatement;
		return getConnection().prepareStatement(statement);
	}
	public Statement createStatement() throws Exception {
		return getConnection().createStatement();
	}
	public Statement createStatement(int resultSetType, int resultSetConcurrency) throws Exception {
		return getConnection().createStatement(resultSetType, resultSetConcurrency);
	}
	public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws Exception {
		return getConnection().createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
	}
	public void setAutoCommit(boolean value) throws Exception {
		getConnection().setAutoCommit(value);
	}

	/**
	 * @param statement
	 * @param sqlString
	 */
	public synchronized void execute(Statement statement, String sqlString) throws Exception {
		if (debug == true)
			System.out.println(sqlString);
		if (toFileOnly == true)
			writeToFile(sqlString);
		else
			statement.execute(sqlString);
	}

	/**
	 * @param statement
	 * @param statement2
	 * @return
	 */
	public synchronized ResultSet executeQuery(Statement statement, String sqlString) throws Exception {
		if (debug == true)
			System.out.println(sqlString);
		ResultSet res=null;
		try {
			res = statement.executeQuery(sqlString);
		} catch (Exception e) {
			System.err.println("------------ JDBCHelper.executeQuery sqlString="+sqlString);
			e.printStackTrace();
			throw e;
		}
		return res;
	}
	/**
	 * @return Returns the database type.
	 */
	public DatabaseType getDatabaseType() {
		return type;
	}
	public Connection getConnection()
	{
		try {
			if(connection==null || connection.isClosed())
			{
				Properties properties = HyadesResourceExtensions.getInstance().getProperties();
				openForExistingDB(DBHyadesResourceExtension.DB_NAME, properties);
			}
		} catch (Exception e) {
			throw new DBCollectedExceptions(e);
		}
		return connection;
	}

	/**
	 * @param ps
	 */
	public synchronized boolean executePreparedStatemet(PreparedStatement ps) throws Exception {
		if (debug == true)
			System.out.println(ps);
		if (toFileOnly == true)
		{
			writeToFile(printPreparedStatementParameters(ps));
			return true;
		}
		else
			return ps.execute();
	}
	public synchronized int[] executeBatchedPreparedStatement(PreparedStatement ps) throws Exception {
		if (debug == true)
			System.out.println(ps);
		if (toFileOnly == true)
		{
			writeToFile(printPreparedStatementParameters(ps));
			return new int[0];
		}
		else
		{
			int[] res =null;
			try {
				res = ps.executeBatch();
			} catch (BatchUpdateException e) {
				System.err.println("------------ JDBCHelper.executeBatchedPreparedStatement");
				e.printStackTrace();
				SQLException e1=null;
				int i=1;
				while((e1=e.getNextException())!=null){
					System.err.println("------------ "+(i++)+"-------------");
					e1.printStackTrace();
				}
				throw e;
			}
			return res ;
		}
	}

	/**
	 * @param ps
	 * @return
	 */
	private String printPreparedStatementParameters(PreparedStatement ps) throws Exception {
		String res=ps.toString();
		boolean first=true;
		try {
			res+=": PreparedStatement.resultSetMetaData=";
			ResultSetMetaData metaData = ps.getMetaData();
			if(metaData==null)
			{
				res+=metaData;
			}
			else
				for (int i = 0; i < metaData.getColumnCount(); i++) {
					if(first)
					{
						first=false;
					}
					else
					{
						res+=", ";
					}
					res+="col"+i+"="+metaData.getTableName(i)+"."+metaData.getColumnLabel(i);
				}
//			ParameterMetaData parameterMetaData = ps.getParameterMetaData();
//			res+="\n     PreparedStatement.parameterMetaData=";
//			first=true;
//			if(metaData==null)
//			{
//				res+=parameterMetaData;
//			}
//			else
//				for (int i = 0; i < parameterMetaData.getParameterCount(); i++) {
//					if(first)
//					{
//						first=false;
//					}
//					else
//					{
//						res+=", ";
//					}
//					res+="param"+i+"="+parameterMetaData.getParameterTypeName(i);
//				}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return res;
	}

	/**
	 * 
	 */
	public void commitTransaction() throws Exception {
		if (!getConnection().getAutoCommit())
			getConnection().commit();
	}

	/**
	 * 
	 */
	public void rollbackTransaction() throws Exception {
		if (!getConnection().getAutoCommit())
			getConnection().rollback();
	}
	
	
	
} // JDBCHelper
