/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.provider;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.model.IWorkbenchAdapter;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.adapter.IHyadesWorkbenchAdapter;
import org.eclipse.hyades.ui.util.ILabelAndDescriptionProvider;

/**
 * Label provider that allows the definition of which subclass of 
 * {@link org.eclipse.ui.model.IWorkbenchAdapter} adapter class
 * should be used with the adaptable instances.
 * 
  * <p>This implementation is based on the 
 * {@link org.eclipse.ui.model.WorkbenchLabelProvider} class.
 * 
 * @author marcelop
 * @since 0.0.1 
 */
public class WorkbenchAdapterLabelProvider 
extends LabelProvider implements ILabelAndDescriptionProvider
{
	private Class adapterClass;

	/**
	 * The cache of images that have been dispensed by this provider.
	 * Maps ImageDescriptor->Image.
	 */
	private Map imageTable;
	
	/**
	 * Constructor for WorkbenchAdapterLabelProvider
	 * @param adapterClass The adapter class to be used.
	 * @throws IllegalArgumentException if the <code>adapterClass</code> is
	 * <code>null</code> or if it is not an instance of IWorkbenchAdapter. 
	 */
	public WorkbenchAdapterLabelProvider(Class adapterClass)
	throws IllegalArgumentException
	{
		super();
		
		if((adapterClass == null) || !IWorkbenchAdapter.class.isAssignableFrom(adapterClass))
			throw new IllegalArgumentException(HyadesUIPlugin.getString("_ERROR_ARG_INV"));
			
		this.adapterClass = adapterClass;
	}

	/**
	 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
	 */
	public void dispose()
	{
		if (imageTable != null)
		{
			for(Iterator i = imageTable.values().iterator(); i.hasNext();)
				((Image) i.next()).dispose();
			imageTable = null;
		}

		adapterClass = null;

		super.dispose();
	}
	
	protected IWorkbenchAdapter getAdapter(Object object)
	{
		if(object == null)
			return null;
			
		if(adapterClass.isInstance(object))
			return (IWorkbenchAdapter)object;
			
		if(object instanceof IAdaptable)
			return (IWorkbenchAdapter)((IAdaptable)object).getAdapter(adapterClass);
			
		return (IWorkbenchAdapter)Platform.getAdapterManager().getAdapter(object, adapterClass);
	}

	/**
	 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
	 */
	public String getText(Object element)
	{
		IWorkbenchAdapter adapter = getAdapter(element);
		if(adapter == null)
			return "";
			
		return adapter.getLabel(element);
	}

	/**
	 * @see org.eclipse.hyades.ui.util.ILabelAndDescriptionProvider#getDescription(java.lang.Object)
	 */
	public String getDescription(Object element)
	{
		IWorkbenchAdapter adapter = getAdapter(element);
		if((adapter == null) && (!(adapter instanceof IHyadesWorkbenchAdapter)))
			return "";

		return ((IHyadesWorkbenchAdapter)adapter).getDescription(element);
	}

	/**
	 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
	 */
	public Image getImage(Object element)
	{
		IWorkbenchAdapter adapter = getAdapter(element);
		if(adapter == null)
			return null;

		ImageDescriptor descriptor = adapter.getImageDescriptor(element);
		if (descriptor == null)
			return null;

		//obtain the cached image corresponding to the descriptor
		if (imageTable == null)
			imageTable = new Hashtable(40);
		
		Image image = (Image) imageTable.get(descriptor);
		if(image == null)
		{
			image = descriptor.createImage();
			imageTable.put(descriptor, image);
		}
		return image;
	}
}
