/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.common;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.loaders.hierarchy.IgnoredXMLFragmentLoader;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.common.common.CMNExtendedProperty;
import org.eclipse.hyades.models.common.common.CommonFactory;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofileFactoryImpl;

/**
 * @author amathur
 *
 * This is the base class for the execution event loaders and handles 
 * attributes of the base event.
 */
public abstract class XMLexecutionEventLoader extends IgnoredXMLFragmentLoader implements IXMLEventConstants {

	// Final static debug variables
	public static final boolean DEBUG = false;		// set 'true' to turn on debugging in fragment loaders
	public static final String ROOT_PARENT = "ROOT";
	
	// Static variables -----------------------------
	protected final static String EVENT_ID = "id";
	protected final static String EVENT_OWNERID = "ownerId";
	protected final static String EVENT_TIMESTAMP = "timestamp";
	protected final static String EVENT_TEXT = "text";
	protected final static String EVENT_PARENTID = "parentId";
	protected final static String EVENT_NAME = "name";
	protected final static String EVENT_EVENT_TYPE = "eventType";
	protected final static String EVENT_SORT_BY = "sortBy";
	protected final static String EVENT_CONFLICT = "conflict";
	protected final static String EVENT_PROPERTY = "property";
	protected final static String EVENT_PROPERTY_NAME = "pname";
	protected final static String EVENT_PROPERTY_TYPE = "ptype";
	protected final static String EVENT_PROPERTY_VALUE = "pvalue";
	
	// Instance variables
	protected String 			id = null;
	protected String 			ownerId = null;
	protected long 				timestamp = -1;
	protected String 			text;
	protected TPFExecutionEvent event = null;
	protected String			parentId = null;
	protected String 			name = null;
	protected String			eventType = null;
	protected String 			sortBy = null;
	protected int				conflict = CONFLICT_NONE;
	protected CMNExtendedProperty		property = null;
	protected ArrayList			properties = new ArrayList();
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#initialize(org.eclipse.hyades.loaders.util.HierarchyContext, java.lang.String)
	 */
	public void initialize(HierarchyContext context, String name) {
		super.initialize(context, name);
		event = null;
		id = null;
		ownerId = null;
		timestamp = 0;
		text = null;
		parentId = null;
		this.name = null;
		eventType = null;
		sortBy = null;
		conflict = CONFLICT_NONE;
		
		property = null;
		properties.clear();
		
		printTag(name);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#addAttribute(java.lang.String, java.lang.String)
	 */
	public void addAttribute(String name, String value) {
		/* The attributes belonging to the base execution event are:
		 * id
		 * ownerId
		 * timestamp
		 * text
		 */
		switch (LoadersUtils.getHashCode(name)) {
			case ExecutionConstants.EVENT_ID_int :
				id = value;
				break;

			case ExecutionConstants.EVENT_OWNERID_int :
				ownerId = value;
				break;

			case ExecutionConstants.EVENT_PARENTID_int :
				parentId = value;
				break;

			case ExecutionConstants.EVENT_TIMESTAMP_int :
				timestamp = Long.parseLong(value);
				break;

			case ExecutionConstants.EVENT_TEXT_int :
				text = value;
				break;
				
			case ExecutionConstants.EVENT_NAME_int :
				this.name = value;
				break;
				
			case ExecutionConstants.EVENT_EVENT_TYPE_int :
				eventType = value;
				break;

			case ExecutionConstants.EVENT_SORT_BY_int :
				sortBy = value;
				break;
				
			case ExecutionConstants.EVENT_CONFLICT_int :
				conflict = Integer.parseInt(value);
				break;
				
			case ExecutionConstants.EVENT_PROPERTY_NAME_int :
				if (property != null) {
					property.setName(value);
				}
				break;
			
			case ExecutionConstants.EVENT_PROPERTY_TYPE_int :
				if (property != null) {
					property.setType(value);
				}
				break;
			
			case ExecutionConstants.EVENT_PROPERTY_VALUE_int :
				if (property != null) {
					property.setValue(value);
				}
				break;

			default :
				break;
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#startChild(java.lang.String)
	 */
	public void startChild(String name) {
		switch (LoadersUtils.getHashCode(name)) {
			case ExecutionConstants.EVENT_PROPERTY_int:
				property = CommonFactory.eINSTANCE.createCMNExtendedProperty();
				break;
		}
		super.startChild(name);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#endChild(java.lang.String)
	 */
	public void endChild(String name) {
		switch (LoadersUtils.getHashCode(name)) {
			case ExecutionConstants.EVENT_PROPERTY_int:
				properties.add(property);
				property = null;
				break;
		}
		super.endChild(name);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#addYourselfInContext()
	 */
	public void addYourselfInContext() {
		if (event == null) {
			// Cast the specialized event to the base event before calling this method
			return;
		}
		
		printEndTag();
		
		// Set the base event attributes
		event.setId(id);
		event.setOwnerId(ownerId);
		event.setTimestamp(timestamp);
		event.setText(text);
		event.setName(name);
		event.setEventType(eventType);
		
		// Now add all the properties of that event
		if (properties != null) {
			Iterator iter = properties.iterator();
			while (iter.hasNext()) {
				CMNExtendedProperty p = (CMNExtendedProperty) iter.next();
				if (p != null) {
					// @akmathur: Inserting a null is BAD, should never happen 
					// but would like to avoind it here
					event.getProperties().add(p);
				}
			}
		}
		
		if ( parentId == null )
		{
			// This condition is here to accommodate the old XML fragment grammar for 
			// execution history in which there is no parent ID, and in which the
			// limited expressible hierarchy is encoded in the ownerId string (and
			// must be parsed out to determine the containing event element and
			// the related interaction fragment (if any)).
			ExecutionContext eContext = (ExecutionContext) context.getCustomData().get(ExecutionContext.root);
			// @amathur: 
			// - sortBy is not supported in the old style runner. So no need 
			//   to change code here for this feature.
			// - conflict is not supported since its a 3.0 feature
			eContext.logEvent(event);
		}
		else
		{
		}
		
	}
	
	
	/**
	 * This method adds the event to the correct container within the execution result.
	 * The container may be specified by a parentId, or may be the root container
	 * (which is the outermost execution result, and is signified by an empty parentId.)
	 * 
	 * If the parentId specifies an Invocation Event, then the invocation event is
	 * not actually the container.  The real container is the execution history that is
	 * contained by the execution result contained by the invocation event.  (Got that?)
	 * To make more sense of this, look at the data model, and see that an execution event
	 * can be contained by either another execution event, or by an execution history.  
	 * If the parentId specifies an invocation event, then the container is really the
	 * execution history.
	 * 
	 * If the sortBy field in the XML is not set, then the even is merely added to the 
	 * execution history. If the sortBy field is set, then the event is inserted into the 
	 * execution history in the right place based on the field its sorted on.
	 */
	public void addYourselfToContainer()
	{
		ExecutionContext eContext = (ExecutionContext) context.getCustomData().get(ExecutionContext.root);
		// Put this object into our cache of EObjects by ID
		if ( id != null )
			eContext.putEObjectByID(id, event);

		
		if ( parentId != null && parentId.equals(ROOT_PARENT))
		{
			// This event has no specified parent, so it must be a child of
			// the root execution result.
			TPFExecutionResult exResult = eContext.rootResult.getResult();
			if ( exResult != null )
			{
				TPFExecutionHistory history = exResult.getExecutionHistory();
				if ( history == null )
				{
					// TODO: Determine if this condition can ever be met and
					// remove the code if it can not.
					history = Common_TestprofileFactoryImpl.eINSTANCE
						.createTPFExecutionHistory();
					exResult.setExecutionHistory(history);
				}
				// @amathur: Changed in 3.0. Try to insert and if not, just add it
				if (!insertEvent(history.getExecutionEvents(), event)) {
					event.setExecutionHistory(history);
				}
			}
		}
		else if ( parentId != null )
		{
			// Get this EObjec by its ID (using the Context's cached lookup mechanism)
			EObject obj = eContext.getEObjectByID(parentId);
			if ( obj != null && obj instanceof TPFInvocationEvent )
			{
				// If the event identified as the parent of this event is 
				// an invocation event, then add this event as a child of the
				// invoked execution result's execution history object.

				TPFExecutionHistory history = ((TPFInvocationEvent) obj).getInvokedExecutionResult().getExecutionHistory();
				if ( history != null )
				{
					// @amathur: Changed in 3.0. Try to insert and if not, just add it
					if (!insertEvent(history.getExecutionEvents(), event)) {
						event.setExecutionHistory(history);						
					}
				}
			}
			else if ( obj != null )
			{
				// Otherwise simply add this event to its parent.
				TPFExecutionEvent parentEvent = (TPFExecutionEvent) obj; 
				// @amathur: Changed in 3.0. Try to insert and if not, just add it
				if (!insertEvent(parentEvent.getChildren(), event)) {
					event.setParent(parentEvent);					
				}
				
			}
		}
		// ownerId (if populated) specifies the GUID of the test model element whose
		// execution gave rise to this event.  Find that element in the test model
		// and setthe reference from this execution event to that model element.
		if ( parentId != null && ownerId != null && ownerId.length() != 0 )
		{
			TPFTest theTest = getReferencedTest();
			if ( theTest != null )
			{
				EObject testElement = ExecutionResultData.getObjectFromTest(
					theTest, ownerId);
				
				if ( testElement != null && testElement instanceof BVRInteractionFragment )
				{
					event.setInteractionFragment((BVRInteractionFragment)testElement);
				}
			}
		}
	}
	
	/**
	 * @return
	 */
	protected TPFTest getReferencedTest() {
		TPFTest referencedTest = null;
		
		if ( event != null )
		{
			TPFExecutionHistory history = getContainingHistory(event);
			if ( history != null )
			{
				TPFExecutionResult result = history.getExecutionResult();
				if ( result != null )
				{
					referencedTest = result.getTest();
				}
			}
		}
		
		return referencedTest;
	}

	/**
	 * @param executionEvent
	 * @return
	 */
	private TPFExecutionHistory getContainingHistory(TPFExecutionEvent executionEvent) {
		TPFExecutionHistory history = executionEvent.getExecutionHistory();
		for ( history = executionEvent.getExecutionHistory(); history == null; 
			  history = executionEvent.getExecutionHistory())
		{
			executionEvent = executionEvent.getParent();
			if ( executionEvent == null )
				break;
		}
		return history;
	}

	/**
	 * This method retrieves the EAttribute for the sortBy field. 
	 * @return EAttribute if valid, null otherwise
	 */
	private EAttribute getSortEAttribute() {
		if (sortBy == null) {
			return null;
		}
		
		EStructuralFeature esf = (EStructuralFeature) event.eClass().getEStructuralFeature(sortBy);
		
		// If it is an invalid sort field, then just add the event
		if ((esf == null) || !(esf instanceof EAttribute)) {
			return null;
		}
		else {
			return (EAttribute) esf;
		}
	}
	
	/**
	 * This method determines if a sort is needed. It utilizes multiple citeria to figure this out
	 * A sorting is not needed if
	 * 	a. the sortBy attributed in the stream is null
	 * 	b. the sortBy attribute is an empty string
	 * 	c. the value of the sortBy attribute is not a valid attribute of TPFExecutionEvent
	 * 	d. if the value of the attribute specified by sortBy is null
	 * @return
	 */
	private boolean isSortNeeded() {
		
		if ((sortBy == null) || (sortBy.length() == 0)) {
			// check for basic invalidity
			return false;
		}
		else {
			// Check to see if this attribute exists in the model
			EAttribute ea = getSortEAttribute();
			if (ea == null) {
				return false;
			}
			
			Object obj = event.eGet(ea);
			if (obj == null) {
				return false;			
			}
		}
		
		return true;

	}

	/**
	 * This method attempts to insert the execution event into an execution history. If it is
	 * successful in inserting the event, it returns a true. If not it returns a false and then 
	 * the event will need to be added manually to the end of the list of events for the parent.
	 * Since the parent could be either TPFExecutionHistory or TPFExecutionEvent, this is 
	 * adding is done outside this method call.
	 * @param history
	 * @param event
	 */
	private boolean insertEvent(EList events, TPFExecutionEvent event) {
		
		// Do we need to resolve conflicts?
		if (resolveConflict(events))
			return true;

		// Simply add the event to the execution history if one of these holds true:
		//   a. the sortBy field is not set
		//   b. the sortBy field is empty OR
		//   c. the sortBy field is an invalid attribute of the event

		if (!isSortNeeded()) {
			return false;
		}
		
		// If its come here, then the attribute is valid and its value is non null
		// Find the index to insert at
		int index = getIndex(events, event);
		
		if (index >= events.size()) {
			return false;
		}
		
		events.add(index, event);
		return true;
	}
	
	/**
	 * This method figures out if there is a conflict to resolve. 
	 * This makes sense only on typed, timed & wait events for now. 
	 * However, this may be made useful for other types of events in the future. 
	 * This method returns false by default, which 
	 * indicates that there is no conflict resolution needed.
	 * @param events
	 * @return false (default) if no conflict resolution required, true if its 
	 * been taken care of
	 */
	protected boolean resolveConflict(EList events) {
		// If there is no conflict resolution needed, then operate normally
		if (conflict == CONFLICT_NONE) {
			return false;
		}
		
		// If the ID of the event is null, I will have nothing to compare against
		// So abort conflict resolution and plain insert
		if (event.getId() == null) {
			return false;
		}
		
		// So conflict resolution is indeed needed
		
		// 1. Find an event that matches this description
		//   a. Find the parent event
		TPFExecutionEvent match = null;
		Iterator iter = events.iterator();
		while (iter.hasNext()) {
			TPFExecutionEvent ev = (TPFExecutionEvent) iter.next();
			if (isIdentical(ev)) {
				// Found a match!
				match = ev;
				break;
			}
		}
		
		// If we did not find a match, let the caller know
		// so it can be inserted
		if (match == null) {
			return false;
		}
		
		// We found a match, figure out if we need to overwrite
		// and perform the action. If it does not need to be overwritten,
		// then it is just dropped.
		if (needOverwrite(match)) {
			overwrite(match);
		}
		
		return true;
	}
		
	/**
	 * Check to see if there exists an event in the excution history that's 
	 * identical to the current event being logged
	 * @param match
	 * @return
	 */
	protected boolean isIdentical(TPFExecutionEvent match) {
		// Make sure the ID and the type are identical
		if ((event.getId().equals(match.getId())) &&
				(event.getClass().getName().equals(match.getClass().getName()))) {
			return true;
		}
		return false;
	}
	
	/**
	 * Based on the value of the 'conflict' attribute, figures out if an
	 * overwrite is required. Specialized events could override this method
	 * if there are more specific ways to determine the need to overwrite.
	 * if this method determines that the event need not be overwritten,
	 * then it is dropped (ignored).
	 * @param match
	 * @return
	 */
	protected boolean needOverwrite(TPFExecutionEvent match) {
		boolean overwrite = false;
		switch (conflict) {
			case CONFLICT_IGNORE :
				// Do not do anything with the event
				break;
				
			case CONFLICT_OVERWRITE :
				// Definetly overwrite, irrespective of any comparison
				overwrite = true;
				break;
				
			case CONFLICT_OVERWRITE_IF_EARLIEST :
				// Overwrite only if the current event occured before 
				// "in time" to the match we found
				if (event.getTimestamp() < match.getTimestamp()) {
					overwrite = true;
				}
				break;
				
			case CONFLICT_OVERWRITE_IF_LATEST :
				// Overwrite only if the current event occured later 
				// "in time" to the match we found
				if (event.getTimestamp() > match.getTimestamp()) {
					overwrite = true;
				}
				break;
			default:
				// The 'conflict' value was some fictitious number
				// so ignore it all-together. The event will be 
				// discarded, so make sure the conflict value is correct
				return false;
		}
		
		return overwrite;

	}
	
	/**
	 * Overwrite the properties of the basic TPFExecutionEvent
	 * @param match - an existing identical event
	 */
	protected void overwrite(TPFExecutionEvent match) {
		// @akmathur: Do not know the side-effects of removing an event and adding 
		// and anticipate a ConcurrentModificationException, not knowing the viewer code.
		// Hence overwriting attributes.
		match.setTimestamp(event.getTimestamp());
		match.setText(event.getText());
		match.setEventType(event.getEventType());
		match.setName(event.getName());
		match.setInteractionFragment(event.getInteractionFragment());
		match.setOwnerId(event.getOwnerId());
	}
	
	/**
	 * This method locates the index where this event needs to be inserted into a list of events
	 * based on the sortBy field. This method utilizes linear search starting at the beginning
	 * of the list.
	 * @param events
	 * @param event
	 * @return
	 */
	private int getIndex(EList events, TPFExecutionEvent event) {
		// Get the structural feature for the sortBy string
		// Make sure this method is called for a valid feature.
		// This si made sure in the code above in its maiden usage.
		EAttribute ea = (EAttribute) event.eClass().getEStructuralFeature(sortBy);
		String newValue = event.eGet(ea).toString();

		// Find the index where this event needs to be inserted
		int index = 0;
		Object obj = null;
		for (index = 0; index < events.size(); index++) {
			TPFExecutionEvent ev = (TPFExecutionEvent) events.get(index);
			obj = ev.eGet(ea);
			if (obj == null) continue;
			String val = obj.toString();
			if (val.compareTo(newValue) > 0)
				break;
		}
		
		return index;
	}
		
	protected void printTag(String str) {
		if (DEBUG) {
			System.out.println("<" + str);
		}
	}
	
	protected void printAttribute(String name, String value) {
		if (DEBUG) {
			System.out.println(name + "=\"" + value + "\"");
		}
	}
	
	protected void printEndTag() {
		if (DEBUG) {
			System.out.println("/>");
		}
	}
	
}
