/**********************************************************************
 * Copyright (c) 2003 Hyades project. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Common
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.log.ui.internal.views;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.analysis.engine.IAnalysisMonitor;
import org.eclipse.hyades.analysis.engine.ILogAnalyzer;
import org.eclipse.hyades.log.ui.internal.LogUIPlugin;
import org.eclipse.hyades.log.ui.internal.util.ContextIds;
import org.eclipse.hyades.log.ui.internal.util.ProgressMonitorAdapter;
import org.eclipse.hyades.log.ui.internal.util.SymptomDBDialog;
import org.eclipse.hyades.log.ui.internal.views.ExtensionPointHandler.ConfigurationElement;
import org.eclipse.hyades.models.cbe.CBECommonBaseEvent;
import org.eclipse.jface.action.*;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.help.WorkbenchHelp;

public class LogPaneTreeViewer extends TreeViewer {

	private LogViewerUI _logViewUI = null;
	private ExtensionPointHandler eph = ExtensionPointHandler
			.getExtensionPointHandler();
	private Collection _analyzedObjects;
	private ISelection sel;
	protected Action _updateAction;
	protected Action _analyzeAllAction;
	protected Action _analyzeAction;
	public static final int NO_ANALYZE = -1;
	public static final int ANALYZE_SEL = 0;
	public static final int ANALYZE_ALL = 1;

	//-------------------------------
	// UpdateAction popup action
	//-------------------------------
	class UpdateAction extends Action {
		public UpdateAction(String name) {
			super(name);

			WorkbenchHelp.setHelp(this, ContextIds.ACTLOG_VIEW_POPUP_REFRESH);
		}

		/**
		 * Invoked when an action occurs. Argument context is the Window which
		 * contains the UI from which this action was fired. This default
		 * implementation prints the name of this class and its label.
		 */
		public void run() {
			_logViewUI.update();
		}
	}
	public LogPaneTreeViewer(LogViewerUI logViewUI, Composite parent,
			String role) {
		super(parent);

		_logViewUI = logViewUI;

		_updateAction = new UpdateAction(LogUIPlugin
				.getResourceString("REFRESH_VIEWS"));
	}

	// Attaches a context menu listener to the table.
	public void setMenuListener(IMenuListener menuListener) {
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(menuListener);
		Menu menu = menuMgr.createContextMenu(getControl());
		getControl().setMenu(menu);
	}

	public void addSelectionListener(SelectionListener listener) {
		((Tree) getControl()).addSelectionListener(listener);
	}

	public void fillContextMenu(IMenuManager menu) {

		ISelection sel = getSelection();

		if (sel instanceof IStructuredSelection) {

			if (!sel.isEmpty())
				updateMenu(menu, ((IStructuredSelection) sel).getFirstElement());
		}

		menu.add(new Separator());
		menu.add(_updateAction);
		menu.add(new Separator());

		//popup menu contribution
		menu.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	public void updateMenu(IMenuManager menu, Object logObject) {

		final ISelection sel = getSelection();

		if (sel instanceof IStructuredSelection) {

			//Action analyzeAction = new
			// AnalyzeAction(LogUIPlugin.getResourceString("STR_ANALYZE_ACT"),
			// logType, false);

			//menu.add(analyzeAction);
			MenuManager manager = new MenuManager(LogUIPlugin
					.getResourceString("STR_ANALYZE_ACT"));
			menu.add(manager);
			fillMenu(manager, logObject);
			//WorkbenchHelp.setHelp(analyzeAction,
			// ContextIds.ACTLOG_VIEW_POPUP_ANALYZE);
			menu.add(new Separator());

			//Action analyzeAllAction = new
			// AnalyzeAction(LogUIPlugin.getResourceString("STR_ANALYZE_ALL_ACT"),
			// logType, true);
			// menu.add(analyzeAllAction);
			manager = new MenuManager(LogUIPlugin
					.getResourceString("STR_ANALYZE_ALL_ACT"));
			menu.add(manager);
			fillMenu(manager);
		}
	}

	public void dispose() {

		List logAnalyzers = eph.getLogAnalyzers();
		for (Iterator iter = logAnalyzers.iterator(); iter.hasNext();) {
			ILogAnalyzer element = ((ConfigurationElement) iter.next())
					.getAnalyzer();
			element.unloadDatabase();
		}
	}

	public Collection getAnalyzedObjects(){
		if(_analyzedObjects==null){
			_analyzedObjects = new HashSet();
		}
		return _analyzedObjects;
	}
	
	private List getObjectsToAnalyze(int option) {

		if (option == ANALYZE_SEL) {
			
			Display.getDefault().syncExec(new Runnable(){
				public void run(){
				sel = getSelection();
				}
			});
			List objects = new ArrayList();
			if (sel instanceof IStructuredSelection) {
				Iterator iterator = ((IStructuredSelection) sel).iterator();
				Object object = null;

				while (iterator.hasNext()) {
					object = iterator.next();
					if (object != null && object instanceof CBECommonBaseEvent) {
						getAnalyzedObjects().add(((CBECommonBaseEvent) object)
								.getGlobalInstanceId());
					}
				}
			}
			List input = ((LogContentProvider) getContentProvider()).getPageList(getInput());
			boolean inputXMI = ((LogContentProvider) getContentProvider()).isInputXMI(getInput());
			for (int i = 0; i < input.size(); i++) {
				Object object = input.get(i);
				if(!(object instanceof CBECommonBaseEvent)){
					continue;
				}
				CBECommonBaseEvent element = (CBECommonBaseEvent) object;
				
				if (getAnalyzedObjects().contains(element.getGlobalInstanceId())) {
					objects.add(element);
				}else if(inputXMI && element.isAnalyzed()){
					objects.add(element);
				}
			}
			return objects;
		} else if (option == ANALYZE_ALL) {
			_analyzedObjects.clear();
			return ((LogContentProvider) getContentProvider())
					.getPageList(getInput());
		}
		return Collections.EMPTY_LIST;
	}

	private boolean isObjectInList(String globalInstanceID, List input) {
		int s = input.size();
		for (int i = 0; i < s; i++) {
			if (((CBECommonBaseEvent) input.get(i)).getGlobalInstanceId()
					.equals(globalInstanceID))
				return true;
		}
		return false;
	}

	private void fillMenu(IContributionManager innerMgr) {

		innerMgr.removeAll();
		List logAnalyzers = eph.getLogAnalyzers();
		if (logAnalyzers != null) {
			for (Iterator iter = logAnalyzers.iterator(); iter.hasNext();) {
				ConfigurationElement config = (ConfigurationElement) iter
						.next();
				_analyzeAllAction = new AnalyzeAction(config.getName(), config
						.getAnalyzer(), true);
				innerMgr.add(_analyzeAllAction);
			}
		}

	}

	private void fillMenu(IContributionManager innerMgr, Object logObject) {

		innerMgr.removeAll();
		List logAnalyzers = eph.getLogAnalyzer(logObject);
		if (logAnalyzers != null) {
			for (Iterator iter = logAnalyzers.iterator(); iter.hasNext();) {
				ConfigurationElement config = (ConfigurationElement) iter
						.next();
				_analyzeAction = new AnalyzeAction(config.getName(), config
						.getAnalyzer(), false);
				innerMgr.add(_analyzeAction);
			}
		}
	}

	class BasicLoadAction extends Action {

		private ILogAnalyzer logAnalyzer = null;
		private boolean reload = false;

		public BasicLoadAction(String label, ILogAnalyzer logAnalyzer) {
			super(label);
			this.logAnalyzer = logAnalyzer;
		}

		public ILogAnalyzer getLogAnalyzer() {
			return logAnalyzer;
		}

		public boolean getReload() {
			return reload;
		}

		public void setReload(boolean newVal) {
			reload = newVal;
		}

		public void run() {

			BusyIndicator.showWhile(null, new Runnable() {

				public void run() {

					logAnalyzer.loadDatabase();

					if (logAnalyzer.errorMsg() != null) {
						MessageDialog.openError(getControl().getShell(),
								LogUIPlugin.getResourceString("STR_LOG_MSG"),
								logAnalyzer.errorMsg());
						return;
					}
				}

			});
		}
		public boolean isEnabled() {
			return true;
		}
	}

	class LoadAction extends BasicLoadAction {

		public LoadAction(String label, ILogAnalyzer logAnalyzer) {
			super(label, logAnalyzer);

		}

		public void run() {

			SymptomDBDialog dialog = new SymptomDBDialog(getControl()
					.getShell(), LogUIPlugin
					.getResourceString("STR_SYMPTOM_DB_DLG_TITLE"), null);
			setReload(false);
			dialog.open();

			if (dialog.getReturnCode() == Window.OK) {
				setReload(true);
			}
		}

	}

	class AnalyzeAction extends LoadAction {

		boolean analyzeAll = false;
		//	protected List sel;
		protected ILogAnalyzer aLogAnalyzer;

		public AnalyzeAction(String label, ILogAnalyzer logAnalyzer) {
			super(label, logAnalyzer);
		}

		public AnalyzeAction(String label, ILogAnalyzer logAnalyzer,
				boolean analyzeAll) {
			super(label, logAnalyzer);
			this.analyzeAll = analyzeAll;

		}

		public void run() {

			IRunnableWithProgress analyze = new IRunnableWithProgress() {
				public void run(IProgressMonitor m) {
					IAnalysisMonitor monitor = new ProgressMonitorAdapter(m);
					((LogContentProvider) getContentProvider()).setAnalyzeRunnable(this);
					List listToAnalyze = getObjectsToAnalyze(analyzeAll ? 1 : 0);
					monitor.beginTask(LogUIPlugin.getResourceString("STR_ANALYZE_PROGRESS"),listToAnalyze.size());
					monitor.subTask(LogUIPlugin
							.getResourceString("STR_ANALYZE_LOAD_DB"));
					getLogAnalyzer().loadDatabase();

					if (getLogAnalyzer().errorMsg() == null) {
						getLogAnalyzer().analyze(listToAnalyze, monitor);
					}
					monitor.done();
				}
			};

			try {
				LogUIPlugin.getActiveWorkbenchWindow().run(true, true, analyze);
			} catch (Exception e) {
				LogUIPlugin.log(e);
			}

			if (getLogAnalyzer().errorMsg() != null
					&& getLogAnalyzer()
							.errorMsg()
							.equals(
									LogUIPlugin
											.getResourceString("STR_NO_SYMPTOM_DB_ERROR"))) {
				if (MessageDialog.openQuestion(getControl().getShell(),
						LogUIPlugin.getResourceString("STR_LOG_MSG"),
						getLogAnalyzer().errorMsg())) {
					super.run();
					if (getReload()) {
						try {
							LogUIPlugin.getActiveWorkbenchWindow().run(true,
									true, analyze);
						} catch (Exception e) {
							LogUIPlugin.log(e);
						}
					}

				} else {
					return;
				}
			}

			//Refresh after analyzing all selected records.
			((LogContentProvider) getContentProvider()).setAnalyzed(true);
			refresh();
			((LogContentProvider) getContentProvider()).setAnalyzed(false);

			//Need to send an event so that the Analysis Result area will be
			// updated.
			Event event = new Event();
			event.item = ((Tree) getControl()).getSelection()[0];
			getControl().notifyListeners(SWT.Selection, event);
		}

	}

	/**
	 * @return Returns the _analyzeAction.
	 */
	public Action get_analyzeAction() {
		return _analyzeAction;
	}
	/**
	 * @return Returns the _analyzeAllAction.
	 */
	public Action get_analyzeAllAction() {
		return _analyzeAllAction;
	}
}