/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.search.IJavaSearchScope;
import org.eclipse.jdt.core.search.SearchEngine;

import org.eclipse.hyades.ui.util.IDisposable;

/**
 * Implementation of the {@link org.eclipse.jdt.core.search.IJavaSearchScope}.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class JavaSearchScope
implements IJavaSearchScope, IDisposable
{
	private Collection elements;
	private Collection resourcePaths;
	private Collection projectAndJarPaths;
	
	private boolean includesBinaries;
	private boolean includesClasspaths;
	
	private IJavaSearchScope workspaceScope;
	
	/**
	 * Constructor for ResourceSearchJavaScope.
	 */
	public JavaSearchScope()
	{
		elements = new HashSet();
		resourcePaths = new HashSet();
		projectAndJarPaths = new HashSet();
		
		workspaceScope = SearchEngine.createWorkspaceScope();
		includesBinaries = true;
		includesClasspaths = true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		elements.clear();
		resourcePaths.clear();
		projectAndJarPaths.clear();
		workspaceScope = null;
	}

	/**
	 * Adds a Java element to this scope.
	 * @param element
	 * @return <code>true</code> if the element was added or 
	 * <code>false</code> otherwise.
	 */
	public boolean addJavaElement(IJavaElement element)
	{
		if(element == null)
			return false;
		
		return elements.add(element);
	}

	/**
	 * Adds a resource to this scope.
	 * @param IResource resource
	 * @return <code>true</code> if the resource was added or 
	 * <code>false</code> otherwise.
	 */
	public boolean addResource(IResource resource)
	{
		return addResource(resource.getFullPath().toString());
	}

	/**
	 * Adds a resource to this scope.
	 * @param String resource
	 * @return <code>true</code> if the resource was added or 
	 * <code>false</code> otherwise.
	 */
	public boolean addResource(String resource)
	{
		if(resource == null)
			return false;
		
		return resourcePaths.add(resource);
	}

	/**
	 * Adds the project and the jar paths.
	 * @param path
	 * @return <code>true</code> if the resource was added or 
	 * <code>false</code> otherwise.
	 */
	public boolean addProjectAndJarPath(IPath path)
	{
		if(path == null)
			return false;
		
		return projectAndJarPaths.add(path);
	}

	/**
	 * @see org.eclipse.jdt.core.search.IJavaSearchScope#encloses(IJavaElement)
	 */
	public boolean encloses(IJavaElement element)
	{
		if(elements.isEmpty())
			return workspaceScope.encloses(element);
			
		return elements.contains(element);
	}

	/**
	 * @see org.eclipse.jdt.core.search.IJavaSearchScope#encloses(String)
	 */
	public boolean encloses(String resourcePath)
	{
		if(resourcePaths.isEmpty())
			return workspaceScope.encloses(resourcePath);
			
		return resourcePaths.contains(resourcePath);
	}

	/**
	 * @see org.eclipse.jdt.core.search.IJavaSearchScope#enclosingProjectsAndJars()
	 */
	public IPath[] enclosingProjectsAndJars()
	{
		if(projectAndJarPaths.isEmpty())
			return workspaceScope.enclosingProjectsAndJars();
			
		return (IPath[])projectAndJarPaths.toArray(new IPath[projectAndJarPaths.size()]);
	}

	/**
	 * @see org.eclipse.jdt.core.search.IJavaSearchScope#includesBinaries()
	 * @deprecated
	 */
	public boolean includesBinaries()
	{
		return includesBinaries;
	}

	/**
	 * @see org.eclipse.jdt.core.search.IJavaSearchScope#includesClasspaths()
	 * @deprecated
	 */
	public boolean includesClasspaths()
	{
		return includesClasspaths;
	}

	/**
	 * @see org.eclipse.jdt.core.search.IJavaSearchScope#setIncludesBinaries(boolean)
	 * @deprecated
	 */
	public void setIncludesBinaries(boolean includesBinaries)
	{
		this.includesBinaries = includesBinaries;
	}

	/**
	 * @see org.eclipse.jdt.core.search.IJavaSearchScope#setIncludesClasspaths(boolean)
	 * @deprecated
	 */
	public void setIncludesClasspaths(boolean includesClasspaths)
	{
		this.includesClasspaths = includesClasspaths;
	}
}
