/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal;

import java.util.Collection;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
/**
 * This interface can be used to set criteria for getting objects from the
 * database. Objects that are instances of the given EClass are retrieved from
 * the database. If you do not specify attribute values using the values map,
 * all instances of the class are returned. If you specify attribute values,
 * objects that have attributes with those values are returned. By default, all
 * contained objects for each matching object are retrieved; however, you can
 * specify that only the matching objects are retrieved.
 * <p>
 * If you know the database id for an object, you can use it to get that object.
 * <p>
 * For each matching object, all of its attribute values are set, not just the
 * ones specified in the query.
 */
public interface ObjectQuery {
	/**
	 * Objects that are instances of the EClass are returned.
	 * 
	 * @return Objects that are instances of this EClass are retrieved from the
	 *         database.
	 */
	EClass getEClass();

	/**
	 * You must set the EClass using this method; otherwise, no objects are
	 * retrieved from the database.
	 * 
	 * @param eClass
	 *            Objects that are instances of this EClass are retrieved.
	 */
	void setEClass(EClass eClass);

	/**
	 * You can specify attribute values by adding entries to the map. The key is
	 * an EAttribute, and the map value is the value of the attribute. Only
	 * objects that have attribute values that match all values in the map are
	 * returned.
	 */
	Map getAttributeValues();

	/**
	 * This method returns whether references are set for the objects loaded
	 * from the database. This includes loading contained objects. By default,
	 * isSetReferences returns true.
	 */
	boolean isSetReferences();

	/**
	 * Objects loaded from the database will not have their references set (and
	 * contained objects will not be loaded) if setReferences is false.
	 * 
	 * @param setReferences
	 *            Whether to set references for the objects loaded from the
	 *            database.
	 */
	void setReferences(boolean setReferences);

	/**
	 * You can choose to restrict the objects that are loaded from the database.
	 * You might want to do this, for example, if you are using paging lists to
	 * load certain objects on demand. The given collection should be a list of
	 * classes whose instances will not be created to provide values for
	 * multi-valued references.
	 */
	void setNotLoadedClasses(Collection classes);

	/**
	 * Get the classes whose instances are not to be created to provide values
	 * for multi-valued references, because paging lists are used to get those
	 * values.
	 */
	Collection getNotLoadedClasses();

	/**
	 * Specify the URI of the resource that has the object.
	 */
	void setURI(URI uri);

	/**
	 * Get the URI of the resource that has the object.
	 */
	URI getURI();
} // ObjectQuery
