/**********************************************************************
 * Copyright (c) 2003,2004 Scapa Technologies Limited and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import java.io.IOException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.URI;
import org.eclipse.hyades.model.statistical.SDDescriptor;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.swt.graphics.Image;

import org.eclipse.hyades.statistical.ui.widgets.zoomslider.internal.*;

/**
 * This is the interface passed to all modules to represent an instance of StatCon.
 * This interface represents what a StatCon module can do with StatCon.
 */
public interface StatConInterface {

	public static final int TAB_GRAPHS = GraphWindow.TAB_GRAPHS;
	public static final int TAB_CONTROLS = GraphWindow.TAB_CONTROLS;
	public static final int TAB_LOG = GraphWindow.TAB_LOG;

	/**
	 * Bring a specific tab to the front.  The tabs are currently the graph information tab (TAB_GRAPH),
	 * the control information tab (TAB_CONTROLS), and the CBE log tab (TAB_LOG)
	 * @param tab which tab to bring to the front
	 */
	public void showTab(int tab);

	/**
	 * Completely reloads all CBE log events, sorting them in timestamped order and showing them in the CBE log
	 * tab.
	 */
	public void reloadCBELog();
	
	/**
	 * Updates the CBE log tab.  Causes StatCon to check all registered TRCAgents for extra messages and append 
	 * all newly added messages.
	 */
	public void updateCBELog();

	/**
	 * Get all modules loaded by this StatCon instance
	 * @return an array of String references which represent all the modules loaded by this StatCon instance
	 */
	public String[] getModuleRefs();
	
	/**
	 * Get a StatConModule by reference
	 * @param ref the String reference for the required module
	 * @return the StatConModule with the specified reference or null if no module could be found with the reference
	 */
	public StatConModule getModuleByRef(String ref);

	/**
	 * Re-reads the agents statistical models and adds any missing descriptors, representations and observations
	 * @param agent the agent to update
	 */
	public SDDescriptor[] updateAgent(TRCAgent agent);

	/**
	 * Re-reads the statistical model underneath this node and adds any missing descriptors, representations and observations
	 * @param model_node
	 */
	public void updateModel(SDDescriptor model_node);
	
	/**
	 * Causes StatCon to generate a configuration file for a specific TRCAgent.
	 * The configuration file will be only for the specific agent and will not contain the root level <smgconfig>
	 * tags.  Adding these tags around the string will make it a valid StatCon file and will allow the file to be used
	 * in applyConfig().
	 * @param agent the agent to generate the configuration file for
	 * @return the XML configuration file for this TRCAgent.
	 */	
	public String generateConfig(TRCAgent agent);
	/**
	 * Causes StatCon to generate a configuration file for a specific statistical model (SDDescriptor).
	 * The configuration file will be only for the specific model and will not contain the root level <smgconfig>
	 * tags.  Adding these tags around the string will make it a valid StatCon file and will allow the file to be used
	 * in applyConfig().
	 * @param model_node the statistical model root node to generate this configuration data for
	 * @return the XML configuration file for this statistical model.
	 */
	public String generateConfig(SDDescriptor model_node);
	/**
	 * Apply a configuration file to StatCon.
	 * @param xml the XML configuration file to apply
	 * @throws Exception
	 */
	public void applyConfig(String xml) throws Exception;

	/**
	 * Set whether to constantly update the Graph area.  Each module may make a request to update the graph area
	 * and should pass in a specific request ID.  StatCon then keeps track of these IDs and their corresponding request.
	 * If any ID has requested constant updating then constant updating is set to ON.  If all IDs have not requested 
	 * constant updating, then constant updating is set to OFF.
	 * @param request_id the ID of this request
	 * @param update whether this request ID needs constant updating or not
	 * @param ms how frequently the updating is to be (in milliseconds)
	 */
	public void setConstantUpdate(String request_id, boolean update, long ms);

	/**
	 * Get the GraphWindow widget associated with this StatCon instance 
	 * @return the GraphWindow widget associated with this StatCon instance
	 */
	public GraphWindow getGraphWindow();

	/**
	 * Add a global action (global menuitem) to this StatCon instance.  This adds a menuitem to the global StatCon
	 * menu.  This menu is shown when the tree area is right clicked anywhere
	 * @param name the name of the action (the name of the menuitem)
	 * @param runnable the action to run
	 */
	public void addGlobalAction(String id, String name, Image image, Runnable runnable);
	/**
	 * Remove the specified global action (global menuitem) from this StatCon instance.
	 * @param name the name of the global action to remove
	 */
	public void removeGlobalAction(String id);

	/**
	 * Add a file drop listener to this StatCon instance.  This allows a module to listen for drops of module-related
	 * files.  A module may add itself as a drop listener more than once (for more than one file extension).
	 * @param extension the file extension to listen out for
	 * @param listener the StatConDropListener implementation to notify upon drops
	 */
	public void addDropTargetListener(String extension, StatConDropListener listener);
	/**
	 * Removes a file drop listener from this StatCon interface.
	 * @param extension the file extension to remove the listener for
	 */
	public void removeDropTargetListener(String extension);

	/**
	 * Add a module-specific top level node to this StatCon instance. The node has a name and an image.  The
	 * node can be used to represent static things added to StatCon such as testsuites etc.
	 * @param name the name of the module-specific node to add
	 * @param image the image to show alongside the node
	 * @param id the id of this top level node
	 * @throws EntityExistsException
	 */	
	public void addModuleNode(String name, String id, Image image) throws EntityExistsException;
	/**
	 * Add a module-specific top level node to this StatCon instance. The node has a name and an image.  The
	 * node can be used to represent static things added to StatCon such as testsuites etc.
	 * @param name the name of the module-specific node to add
	 * @param image the image to show alongside the node
	 * @param id the id of this top level node
	 * @param parent_id the id of the top level node to put this node underneath
	 * @throws EntityExistsException
	 */	
	public void addModuleNode(String name, String id, String parent_id, Image image) throws EntityExistsException;
	/**
	 * Add an action (menuitem) to be associated with a top level node.  This works in the same way as global actions
	 * but the action is only shown when the node is right-clicked.
	 * @param node the node to associate the action with.
	 * @param action_name the name of the action (menuitem)
	 * @param runnable the Runnable implementation to run when the action is selected
	 */
	public void addModuleNodeAction(String node, String action_id, String action_name, Image image, Runnable runnable);
	/**
	 * Remove an action (menuitem) associated with a top level node.
	 * @param node the node to associate the action with.
	 * @param action_name the name of the action (menuitem)
	 */
	public void removeModuleNodeAction(String node, String action_id);
	/**
	 * Removes a top level node from StatCon
	 * @param name the name of the top level node to remove
	 */
	public void removeModuleNode(String name);

	/**
	 * Add an action (menuitem) to be associated with a slider.
	 * @param action_id the id of the action
	 * @param action_name the name of the action (menuitem)
	 * @param image the image of the action
	 * @param runnable the Runnable implementation to run when the action is selected 
	 * @param vertical whether this action should appear on vertical sliders (true) or horizontal (false)
	 */
	public void addSliderAction(String action_id, String action_name, Image image, Runnable runnable, boolean vertical);
	
	/**
	 * Remove an action from a slider
	 * @param action_id the ID of the action
	 */
	public void removeSliderAction(String action_id);
	
	/**
	 * Adds an node change listener.  Allows a module to listen for changes to module nodes. 
	 * @param listener the ModuleNodeChangeListener to add.
	 */
	public void addModuleNodeChangeListener(ModuleNodeChangeListener listener);	
	/**
	 * Removes a node change listener.
	 * @param listener the ModuleNodeChangeListener to remove.
	 */
	public void removeModuleNodeChangeListener(ModuleNodeChangeListener listener);	

	/**
	 * Adds a trace agent to StatCon.  StatCon expects to find a valid .trcaxmi or .trcmxmi resource at the specified
	 * IFile.
	 * @param file the .trcaxmi or .trcmxmi TRCAgent resource to load
	 * @return an array of statistical models loaded from this file
	 * @throws IOException
	 */
	public SDDescriptor[] addAgent(IFile file) throws IOException;	
	/**
	 * Adds a trace agent to StatCon.  StatCon expects to find a valid .trcaxmi or .trcmxmi resource at the specified
	 * IPath.
	 * @param path the .trcaxmi or .trcmxmi TRCAgent resource to load
	 * @return an array of statistical models loaded from this path
	 * @throws IOException
	 */
	public SDDescriptor[] addAgent(IPath path) throws IOException;	
	/**
	 * Adds a trace agent to StatCon.  
	 * @param agent the TRCAgent to load
	 * @return an array of statistical models loaded from this agent
	 * @throws IOException
	 */
	public SDDescriptor[] addAgent(TRCAgent agent);
	/**
	 * Adds a trace agent to StatCon.  StatCon will save the URI information to file and try to load
	 * the agent from the same place when the editor is closed and reopened.
	 * @param agent the TRCAgent to load
	 * @param agent_uri the URI associated with this TRCAgent
	 * @return an array of statistical models loaded from this agent
	 * @throws IOException
	 */
	public SDDescriptor[] addAgent(TRCAgent agent, URI agent_uri);

	/**
	 * removes a TRCAgent from StatCon
	 * @param agent
	 */
	public void removeAgent(TRCAgent agent);
	/**
	 * Sets an agents image in StatCon
	 * @param agent the agent to set the image of
	 * @param image the image to set to the trace agent
	 */
	public void setAgentImage(TRCAgent agent, Image image);
	/**
	 * Sets the associated file of an agent in StatCon
	 * @param agent the agent to set the file of
	 * @param agent_uri the URI where this agent can be loaded from 
	 */
	public void setAgentFile(TRCAgent agent, URI agent_uri);
	/**
	 * Adds an action (menuitem) to be shown when an agent is right-clicked.  Works in the same way as
	 * global actions but the menuitem is only shown when the agent is right-clicked. 
	 * @param agent the agent to add the menuitem to
	 * @param action_name the name of the action (menuitem)
	 * @param runnable the Runnable to run when the menuitem is pressed
	 */
	public void addAgentAction(TRCAgent agent, String action_id, String action_name, Image image, Runnable runnable);
	/**
	 * Removes an action from an agent.
	 * @param agent the agent to remove the action from.
	 * @param action_name the name of the action to remove
	 */
	public void removeAgentAction(TRCAgent agent, String action_id);

	/**
	 * Add a model action (menuitem that appears on any model tree item) to this StatCon instance.
	 * @param name the name of the action (the name of the menuitem)
	 * @param runnable the action to run
	 */
	public void addModelAction(TRCAgent agent, String id, String name, Image image, Runnable runnable);
	/**
	 * Remove the specified model action (model menuitem) from this StatCon instance.
	 * @param name the name of the global action to remove
	 */
	public void removeModelAction(TRCAgent agent, String id);

	/**
	 * Get the most recently selected slider
	 * @return
	 */
	public ZoomSlider getSelectedSlider();
	
	/**
	 * Get the most recently selected slider value 
	 * @return
	 */
	public double getSelectedSliderValue();
	
	/**
	 * Can be used in conjunction with model actions to get the exact selected descriptor
	 * @return
	 */
	public SDDescriptor getSelectedDescriptor();

	public void setDescriptorImage(SDDescriptor descriptor, Image image);

	/**
	 * Adds a model change listener.  Allows a module to listen for changes to statistical models. 
	 * @param listener the ModelChangeListener to add.
	 */
	public void addModelChangeListener(ModelChangeListener listener);	
	/**
	 * Removes a model change listener.
	 * @param listener the ModelChangeListener to remove.
	 */
	public void removeModelChangeListener(ModelChangeListener listener);	

	/**
	 * Adds an agent change listener.  Allows a module to listen for changes to agents. 
	 * @param listener the AgentChangeListener to add.
	 */
	public void addAgentChangeListener(AgentChangeListener listener);	
	/**
	 * Removes a agent change listener.
	 * @param listener the AgentChangeListener to remove.
	 */
	public void removeAgentChangeListener(AgentChangeListener listener);	

	/**
	 * Set whether this StatCon editor is dirty.
	 * @param b whether to make this StatCon editor dirty.
	 */	
	public void setDirty(boolean b);
	
	/**
	 * Returns the URI of the project containing this StatCon file
	 * @return the URI of the project containing this StatCon file
	 */
	public URI getCurrentProjectURI();
}