/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.adapter;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionType;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFLoopEvent;
import org.eclipse.hyades.models.common.testprofile.TPFMessageEvent;
import org.eclipse.hyades.models.common.testprofile.TPFSUT;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTypedEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.adapter.HyadesWorkbenchAdapter;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.ResourceUtil;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * Extension for the <code>HyadesWorkbenchAdapter</code> to handle the test 
 * specific objects.
 * 
 * @author marcelop
 * @since 0.0.1 
 */
public class TestWorkbenchAdapter
extends HyadesWorkbenchAdapter
{
	public final static String TEST_SUITE_LOGICAL_FOLDER_NAME = TestUIPlugin.getString("LOG_FLD_TST_SUITES_NME"); //$NON-NLS-1$
	public final static String DEPLOYMENT_LOGICAL_FOLDER_NAME = TestUIPlugin.getString("LOG_FLD_DEPLOYS_NME"); //$NON-NLS-1$

	/**
	 * Creates the logical folder to group all the test suites. This method
	 * doesn't add any child to the folder.
	 * @return LogicalFolder
	 */
	private static LogicalFolder createTestSuitesFolder()
	{
		LogicalFolder logicalFolder = new LogicalFolder(TEST_SUITE_LOGICAL_FOLDER_NAME)
		{
			protected List createChildren()
			{
				return new UniqueEList();
			}
		};
		
		logicalFolder.setDescription(TestUIPlugin.getString("LOG_FLD_TST_SUITES_DSC")); //$NON-NLS-1$
		return logicalFolder;
	}
	
	/**
	 * Creates the logical folder to group all the deployments. This method
	 * doesn't add any child to the folder.
	 * @return LogicalFolder
	 */
	private static LogicalFolder createDeploymentsFolder()
	{
		LogicalFolder logicalFolder = new LogicalFolder(DEPLOYMENT_LOGICAL_FOLDER_NAME)
		{
			protected List createChildren()
			{
				return new UniqueEList();
			}
		};
		logicalFolder.setDescription(TestUIPlugin.getString("LOG_FLD_DEPLOYS_DSC")); //$NON-NLS-1$
		return logicalFolder;
	}

	/**
	 * Finds the test navigator view in the current perspective, returning 
	 * <code>null</code> if it is not available.
	 * @return TestNavigator
	 */
	protected TestNavigator findTestNavigator()
	{
		if(PlatformUI.getWorkbench() != null)
			return TestNavigator.getTestNavigator(PlatformUI.getWorkbench().getActiveWorkbenchWindow());
		return null;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.adapter.IHyadesWorkbenchAdapter#getDescription(java.lang.Object)
	 */
	public String getDescription(Object object)
	{
		if(object instanceof Resource)
			return ""; //$NON-NLS-1$
		
		return super.getDescription(object);
	}

	/**
	 * @see org.eclipse.ui.model.IWorkbenchAdapter#getLabel(java.lang.Object)
	 */
	public String getLabel(Object object)
	{
		if(object instanceof Resource)
		{
			IFile file = EMFUtil.getWorkspaceFile((Resource)object);
			if(file != null)
				return ((IWorkbenchAdapter)file.getAdapter(IWorkbenchAdapter.class)).getLabel(file);
			return ""; //$NON-NLS-1$
		}
		
		if(object instanceof TPFExecutionEvent)
		{
			if(object instanceof TPFTypedEvent)
			{
				TPFTypedEvent typedEvent = (TPFTypedEvent)object;
				if(typedEvent.getType() == null)
					return TestUIPlugin.getString("W_EVT_TYPED"); //$NON-NLS-1$
				return typedEvent.getType().getLabel();
			}

			if(object instanceof TPFMessageEvent)
				return TestUIPlugin.getString("W_EVT_MSG"); //$NON-NLS-1$

			if(object instanceof TPFVerdictEvent)
			{
				TPFVerdictEvent verdictEvent = (TPFVerdictEvent)object;
				if(verdictEvent.getVerdict() == null)
					return TestUIPlugin.getString("W_EVT_VER"); //$NON-NLS-1$
				return verdictEvent.getVerdict().getLabel();
			}

			if(object instanceof TPFInvocationEvent)
				return TestUIPlugin.getString("W_EVT_INV"); //$NON-NLS-1$
				
			//added new label for loop event :BGB(12/1/03)
			if(object instanceof TPFLoopEvent)
			{
				//now getting loop name first for label, if no name, label is generic object name :BGB(12/4/03)
				TPFLoopEvent			loopEvent			= null;
				BVRInteractionFragment	interactionFragment	= null;
				loopEvent	= ((TPFLoopEvent)object);
				interactionFragment	= loopEvent.getInteractionFragment();
				if ( interactionFragment != null )
				{
					if ( ( interactionFragment.getName() != null) &&
					     ( interactionFragment.getName().length() > 0 ) )
						return( interactionFragment.getName() );
				}
				return TestUIPlugin.getString("W_EVT_LOOP"); //$NON-NLS-1$
			}
				
			return TestUIPlugin.getString("W_EVT_EXEC"); //$NON-NLS-1$
		}
		
		if((object instanceof TPFExecutionResult) && (((TPFExecutionResult)object).getName() == null))
			return TestUIPlugin.getString("W_TST_EXE"); //$NON-NLS-1$

		return super.getLabel(object);
	}

	/**
	 * @see org.eclipse.ui.model.IWorkbenchAdapter#getImageDescriptor(java.lang.Object)
	 */
	public ImageDescriptor getImageDescriptor(Object object)
	{
		if(object instanceof Resource)
			return PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(ISharedImages.IMG_OBJ_FILE);
		
		if(object instanceof EObject)
		{
			if(object instanceof TPFTestSuite)
			{
				ImageDescriptor imageDescriptor = null;
				TPFTestSuite testSuite = (TPFTestSuite)object;
				String type = testSuite.getType();
				if(type != null)
				{
					IAssociationMapping mapping = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS);
					IAssociationDescriptor descriptor = mapping.getDefaultAssociationDescriptor(type);
					if(descriptor != null)
						imageDescriptor  = descriptor.getImageDescriptor();
				}
				if(imageDescriptor == null)
					imageDescriptor = TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_DEFAULT);
				return imageDescriptor;
			}

			if(object instanceof TPFTestCase)
			{
				ImageDescriptor imageDescriptor = null;
				TPFTestCase testCase = (TPFTestCase)object;
				String type = testCase.getType();
				if(type != null)
				{
					IAssociationMapping mapping = TestUIExtension.getTestCaseMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS);
					IAssociationDescriptor descriptor = mapping.getDefaultAssociationDescriptor(type);
					if(descriptor != null)
						imageDescriptor = descriptor.getImageDescriptor();
						
					if(imageDescriptor == null)
					{
						mapping = TestUIExtension.getTestCaseMappingRegistry().getAssociationMapping(TestUIPlugin.EP_TEST_CASE_WIZARD);
						descriptor = mapping.getDefaultAssociationDescriptor(type);
						if(descriptor != null)
							imageDescriptor = descriptor.getImageDescriptor();						
					}
				}
				if(imageDescriptor == null)
					imageDescriptor = TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_DEFAULT);
				return imageDescriptor;
			}
			
			if(object instanceof TPFExecutionResult)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EXECUTION);

			if(object instanceof TPFExecutionEvent)
			{
				if(object instanceof TPFTypedEvent)
				{
					//modified the icons to have separate images for different typed events :BGB(12/1/03)
					TPFExecutionType	exeType	= null;
					exeType = ((TPFTypedEvent)object).getType();
					if ( exeType != null )
					{
						if ( exeType.getValue() == TPFExecutionType.START )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_TYPED_START);
						else if ( exeType.getValue() == TPFExecutionType.STOP )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_TYPED_STOP);
					}
					return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_TYPED);
				}
				
				//added new loop event :BGB(12/1/03)
				if(object instanceof TPFLoopEvent)
					return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_LOOP);

				if(object instanceof TPFMessageEvent)
					return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_MESSAGE);

				if(object instanceof TPFVerdictEvent)
				{
					TPFVerdict	verdict	= null;
					verdict	= ((TPFVerdictEvent)object).getVerdict();
					if ( verdict != null )
					{
						if ( verdict.getValue() == TPFVerdict.PASS )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_PASS);
						if ( verdict.getValue() == TPFVerdict.FAIL )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_FAIL);
						if ( verdict.getValue() == TPFVerdict.INCONCLUSIVE )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_INCONCLUSIVE);
						if ( verdict.getValue() == TPFVerdict.ERROR )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_ERROR);
					}
					return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_VERDICT_PASS);
				}

				if(object instanceof TPFInvocationEvent)
				{
					//added support for rollup icons in invocation events :BGB(12/1/03)
					TPFExecutionResult	exeResult	= null;
					TPFExecutionHistory	exeHistory	= null;
					EList				eventList	= null;
					exeResult	= ((TPFInvocationEvent)object).getInvokedExecutionResult();
					if ( exeResult != null )
						exeHistory	= exeResult.getExecutionHistory();
					if ( exeHistory != null )
						eventList	= exeHistory.getExecutionEvents();
					if ( eventList != null & eventList.isEmpty() == false )
					{
						TPFVerdict	curVerdict		= null;
						int			rollupVerdict	= TPFVerdict.PASS,
									i				= 0;
						boolean		bRollupVerFound	= false;
						Object		curObject		= null;
						
						//traverse through the list of events to get the final verdict; this
						//final verdict represents the rollup verdict
						i = eventList.size() - 1;
						while( ( i >= 0 ) && ( bRollupVerFound == false ) )
						{
							curObject	= eventList.get(i);
							if ( curObject instanceof EObject )
							{
								if ( curObject instanceof TPFVerdictEvent )
								{
									curVerdict = ((TPFVerdictEvent)curObject).getVerdict();
									if ( curVerdict.getValue() != TPFVerdict.PASS )
										rollupVerdict	= curVerdict.getValue();
									bRollupVerFound = true;
								}
							}
							i--;
						}
						if ( rollupVerdict == TPFVerdict.PASS )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_INVOCATION_PASS);
						else if ( rollupVerdict == TPFVerdict.FAIL )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_INVOCATION_FAIL);
						if ( rollupVerdict == TPFVerdict.ERROR )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_INVOCATION_ERROR);
						if ( rollupVerdict == TPFVerdict.INCONCLUSIVE )
							return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_INVOCATION_INCONCLUSIVE);
					}
					return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_INVOCATION);
				}
				
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_EVENT_EXECUTION);
			}

			if(object instanceof TPFTestComponent)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_TEST_COMPONENT);

			if(object instanceof TPFBehavior)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_BEHAVIOR);

			if(object instanceof TPFSUT)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_SUT);
				
			if(object instanceof TPFDeployment)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_DEPLOYMENT);

			if(object instanceof DPLDatapool)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_DATAPOOL);

			if(object instanceof CFGLocation)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_LOCATION);
			
			if(object instanceof CFGArtifact)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_ARTIFACT);
			
			if(object instanceof CFGPropertyGroup)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_PROPERTY_GROUP);
			
			if(object instanceof CFGComparableProperty)
				return TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_PROPERTY);
		}
		
		return super.getImageDescriptor(object);
	}

	/**
	 * @see org.eclipse.ui.model.IWorkbenchAdapter#getParent(java.lang.Object)
	 */
	public Object getParent(Object object)
	{
		if(object instanceof EObject)
			return getEObjectParent((EObject)object);

		if(object instanceof IWorkspaceRoot)
			return null;

		return super.getParent(object);
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IHyadesWorkbenchAdapter#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element)
	{
		if(element instanceof EObject)
		{
			TestNavigator testNavigator = findTestNavigator();
			if((testNavigator == null) || (!testNavigator.isShowingEObjectChildren()))
				return false;
				
			if(element instanceof TPFTestSuite)
				return true;
	
			if(element instanceof TPFDeployment)
				return true;
				
			return false;
		}
			
		if(element instanceof IWorkspaceRoot)
			return true;
			
		return super.hasChildren(element);
	}

	/**
	 * @see org.eclipse.ui.model.IWorkbenchAdapter#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object object)
	{
		if(object instanceof EObject)
			return getEObjectChildren((EObject)object);
		
		if(object instanceof IWorkspaceRoot)
			return getWorkspaceRootChildren((IWorkspaceRoot)object);
				
		return adjustChildren(super.getChildren(object));
	}
	
	/**
	 * Adjusts the elements of the children array replacing them by meaninful
	 * objects.
	 * @param children
	 * @return Object[]
	 */
	protected Object[] adjustChildren(Object[] children)
	{
		TestNavigator testNavigator = findTestNavigator();
		if(testNavigator == null)
			return children;

		List validChildren = new ArrayList(children.length);
		for(int i = 0, maxi = children.length; i < maxi; i++)
		{
			if(children[i] instanceof IFile)
			{
				IFile file = (IFile)children[i];
				if(TestUI.isSupportedFileExtension(file.getFileExtension()))
				{
					EObject[] eObjects = EMFUtil.load(null, file);
					for (int j = 0, maxj = eObjects.length; j < maxj; j++)
					{
						if((eObjects[j] instanceof TPFTestSuite) || (eObjects[j] instanceof TPFDeployment))
							validChildren.add(eObjects[j]);
					}
				}
			}
			else
				validChildren.add(children[i]);
		}
		
		return validChildren.toArray();
	}
	
	/**
	 * Returns the workspace root children.
	 * @param root
	 * @return Objetc[]
	 */
	protected Object[] getWorkspaceRootChildren(IWorkspaceRoot root)
	{
		TestNavigator testNavigator = findTestNavigator();
		if(testNavigator == null)
			return new Object[0];
			
		if(testNavigator.isShowingFolders())
			return getProjects(root);
		
		return getLogicalFolders(root);
	}
	
	/**
	 * Returns all the projects in the workspace.
	 * @param root
	 * @return IProject[]
	 */
	protected IProject[] getProjects(IWorkspaceRoot root)
	{
		return root.getProjects();
	}

	/**
	 * Returns the top level logical folders.
	 * @param root
	 * @return LogicalFolder[]
	 */
	protected LogicalFolder[] getLogicalFolders(IWorkspaceRoot root)
	{
		TestNavigator testNavigator = findTestNavigator();
		if(testNavigator == null)
			return new LogicalFolder[0];
		
		if(!EMFUtil.getResourceSet().getResources().isEmpty())
		{
			LogicalFolder[] logicalFolders = getLogicalFolders(testNavigator);
			if(logicalFolders.length != 0)
				return logicalFolders;
		}
		
//		final TestNavigator finalTestNavigator = testNavigator;
		ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
		{
			public Object visit(IFile file)
			{
				return new ResourceSetFileAdaptable(EMFUtil.getResourceSet(), file);
			}
		};
			
		Map filesByExtensions = ResourceUtil.getFiles(root, new String[]{ICommonConstants.TEST_SUITE_FILE_EXTENSION, ICommonConstants.DEPLOYMENT_FILE_EXTENSION}, visitor);
		List testSuiteResources = (List)filesByExtensions.get(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
		List deploymentResources = (List)filesByExtensions.get(ICommonConstants.DEPLOYMENT_FILE_EXTENSION);
		
		LogicalFolder[] logicalFolders = new LogicalFolder[2];

		logicalFolders[0] = testNavigator.getRootLogicalFolder(TEST_SUITE_LOGICAL_FOLDER_NAME);
		if(logicalFolders[0] == null)
		{
			logicalFolders[0] = createTestSuitesFolder();
			testNavigator.registerRootLogcalFolder(logicalFolders[0]);
		}
		else
		{
			if(!logicalFolders[0].isLinked())
				logicalFolders[0].getChildren().clear();
		}
		logicalFolders[0].getChildren().addAll(ResourceUtil.getValidAdapters(testSuiteResources, CMNNamedElement.class));
		//
		logicalFolders[1] = testNavigator.getRootLogicalFolder(DEPLOYMENT_LOGICAL_FOLDER_NAME);
		if(logicalFolders[1] == null)
		{
			logicalFolders[1] = createDeploymentsFolder();
			testNavigator.registerRootLogcalFolder(logicalFolders[1]);
		}
		else
		{
			logicalFolders[1].getChildren().clear();
		}
		logicalFolders[1].getChildren().addAll(ResourceUtil.getValidAdapters(deploymentResources, CMNNamedElement.class));
				
		
		testSuiteResources.clear();
		deploymentResources.clear();
		filesByExtensions.clear();
		return logicalFolders;
	}
	
	/**
	 * Returns the top level logical folders based on the testNavigator information.
	 * @param testNavigator
	 * @return LogicalFolder[]
	 */
	protected LogicalFolder[] getLogicalFolders(TestNavigator testNavigator)
	{
		LogicalFolder[] logicalFolders = new LogicalFolder[2];
		logicalFolders[0] = testNavigator.getRootLogicalFolder(TEST_SUITE_LOGICAL_FOLDER_NAME);
		logicalFolders[1] = testNavigator.getRootLogicalFolder(DEPLOYMENT_LOGICAL_FOLDER_NAME);
		if((logicalFolders[0] == null) || (logicalFolders[1] == null))
			return new LogicalFolder[0];

		ResourceSet resourceSet = EMFUtil.getResourceSet();
		for(Iterator i = resourceSet.getResources().iterator(); i.hasNext(); )
		{
			Resource resource = (Resource)i.next();
			logicalFolders[0].getChildren().addAll(EcoreUtil.getObjectsByType(resource.getContents(), Common_TestprofilePackage.eINSTANCE.getTPFTestSuite()));
			logicalFolders[1].getChildren().addAll(EcoreUtil.getObjectsByType(resource.getContents(), Common_TestprofilePackage.eINSTANCE.getTPFDeployment()));
		}
		
		return logicalFolders;
	}

	/**
	 * Returns the parent of the specified <code>eObject</code>
	 * @param eObject
	 * @return Object
	 */
	protected Object getEObjectParent(EObject eObject)
	{
		if(eObject instanceof TPFTestCase)
		{
			TPFTestCase testCase = (TPFTestCase)eObject;
			TPFTestSuite testSuite = testCase.getTestSuite();
			if(testSuite == null)
				return null;
			return getEObjectChildren(testSuite)[0];
		}
		
		TestNavigator testNavigator = findTestNavigator();
		if(((testNavigator != null)) && (!testNavigator.isShowingFolders()))
		{
			if(eObject instanceof TPFTestSuite)
				return testNavigator.getRootLogicalFolder(TEST_SUITE_LOGICAL_FOLDER_NAME);

			if(eObject instanceof TPFDeployment)
				return testNavigator.getRootLogicalFolder(DEPLOYMENT_LOGICAL_FOLDER_NAME);
				
			return null;
		}
		
		IFile file = EMFUtil.getWorkspaceFile(eObject);
		if(file != null)
			return file.getParent();
			
		return null;
	}

	/**
	 * Returns the eObject children.
	 * @param eObject
	 * @return LogicalFolder[]
	 */
	protected LogicalFolder[] getEObjectChildren(EObject eObject)
	{
		TestNavigator testNavigator = findTestNavigator();
		if((testNavigator == null) || (!testNavigator.isShowingEObjectChildren()))
			return new LogicalFolder[0];
			
		/*
		 * It is necessary to adjust getEObjectParent(EObject) if the logical folder 
		 * order changes. 
		 */
		
		if(eObject instanceof TPFTestSuite)
		{
			TPFTestSuite testSuite = (TPFTestSuite)eObject;
			LogicalFolder[] logicalFolders = new LogicalFolder[3];
			
			logicalFolders[0] = new LogicalFolder(eObject, TestUIPlugin.getString("LOG_FLD_TST_CASES_NME")); //$NON-NLS-1$
			logicalFolders[0].setDescription(TestUIPlugin.getString("LOG_FLD_TST_CASES_DSC")); //$NON-NLS-1$
			logicalFolders[0].linkChildren(testSuite.getTestCases());
			
			logicalFolders[1] = new LogicalFolder(eObject, TestUIPlugin.getString("LOG_FLD_TST_COMP_NME")); //$NON-NLS-1$
			logicalFolders[1].setDescription(TestUIPlugin.getString("LOG_FLD_TST_COMP_DSC")); //$NON-NLS-1$
			logicalFolders[1].linkChildren(testSuite.getTestComponents());

			logicalFolders[2] = new LogicalFolder(eObject, TestUIPlugin.getString("LOG_FLD_SUT_NME")); //$NON-NLS-1$
			logicalFolders[2].setDescription(TestUIPlugin.getString("LOG_FLD_SUT_DSC")); //$NON-NLS-1$
			logicalFolders[2].linkChildren(testSuite.getSUTs());

			return logicalFolders;
		}

		if(eObject instanceof TPFDeployment)
		{
			TPFDeployment deployment = (TPFDeployment)eObject;
			LogicalFolder[] logicalFolders = new LogicalFolder[2];
			
			logicalFolders[0] = new LogicalFolder(eObject, TestUIPlugin.getString("LOG_FLD_LOCS_NME")); //$NON-NLS-1$
			logicalFolders[0].setDescription(TestUIPlugin.getString("LOG_FLD_LOCS_DSC")); //$NON-NLS-1$
			logicalFolders[0].linkChildren(deployment.getLocations());

			logicalFolders[1] = new LogicalFolder(eObject, TestUIPlugin.getString("LOG_FLD_DEPLY_SPEC_NME")); //$NON-NLS-1$
			logicalFolders[1].setDescription(TestUIPlugin.getString("LOG_FLD_DEPLY_SPEC_DSC")); //$NON-NLS-1$
			logicalFolders[1].linkChildren(deployment.getPropertyGroups());

			return logicalFolders;
		}
		
		return new LogicalFolder[0];
	}
}
