/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace;

import javax.rmi.CORBA.Tie;

import org.eclipse.hyades.collection.correlation.ITransport;
import org.eclipse.hyades.collection.correlation.ServiceRegistry;
import org.eclipse.hyades.collection.correlation.TransportMediator;
import org.omg.CORBA.LocalObject;
import org.omg.CORBA.Object;
import org.omg.IOP.ServiceContext;
import org.omg.PortableInterceptor.ClientRequestInfo;
import org.omg.PortableInterceptor.ClientRequestInterceptor;
import org.omg.PortableInterceptor.ForwardRequest;
import org.omg.PortableInterceptor.ORBInitInfo;
import org.omg.PortableInterceptor.ORBInitializer;
import org.omg.PortableInterceptor.ServerRequestInfo;
import org.omg.PortableInterceptor.ServerRequestInterceptor;


/**
 * This is the transport layer for the profiler.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public class IiopTransport extends LocalObject
        implements ORBInitializer, ClientRequestInterceptor, ServerRequestInterceptor, ITransport {

    /**
     * the service ID for transport
     */
    private static int DISTRIBUTED_TRACE_SERVICE_ID = 0xEEF8AC;

    /**
     * the buffer for context data being transported
     */
    private byte[] _buffer;

    /**
     * the first empty byte in the buffer
     */
    private int _offset;

    /**
     * the number of remaining empty bytes in the buffer
     */
    private int _length;

    /**
     * the profile which uses this transport
     */
    private IiopProfiler _profiler;


    /**
     * Creates a transport for the profiler.
     */
    public IiopTransport() {

        _profiler = IiopProfiler.getInstance();
        ServiceRegistry.getRegistry().registerApplication(_profiler);
        _buffer = new byte[0];
    }


    /**
     * @see org.omg.PortableInterceptor.ORBInitializerOperations#pre_init(ORBInitInfo)
     */
    public void pre_init(ORBInitInfo initInfo) {

        try {
            initInfo.add_server_request_interceptor(this);
            initInfo.add_client_request_interceptor(this);
        } catch (Throwable e) {
        }
    }


    /**
     * @see org.omg.PortableInterceptor.ORBInitializerOperations#post_init(ORBInitInfo)
     */
    public void post_init(ORBInitInfo info) {
    }


    /**
     * @see org.omg.PortableInterceptor.ClientRequestInterceptorOperations#send_request(ClientRequestInfo)
     */
    public void send_request(ClientRequestInfo ri) throws ForwardRequest {

        /* Create the correlator for this call event. */
        TraceCorrelator local = (TraceCorrelator) _profiler.createCorrelatorData();
        local.setInvocation(ri.target().hashCode(), ri.target().getClass(),
            ri.operation(), new Class[0], new Object[0], null);

        /* Package the correlator with the request itself. */
        ServiceContext serviceContext = null;
        ThreadComparator key = new ThreadComparator();
        synchronized (_buffer) {
            TransportMediator.getDataStream(_profiler.getId(), key, this, local);
            serviceContext = new ServiceContext(DISTRIBUTED_TRACE_SERVICE_ID, _buffer);
            ri.add_request_service_context(serviceContext, false);
        }
    }


    /**
     * @see org.omg.PortableInterceptor.ClientRequestInterceptorOperations#send_poll(ClientRequestInfo)
     */
    public void send_poll(ClientRequestInfo ri) {
    }


    /**
     * @see org.omg.PortableInterceptor.ClientRequestInterceptorOperations#receive_reply(ClientRequestInfo)
     */
    public void receive_reply(ClientRequestInfo ri) {
        
        try {

            /* Create the correlator for the local return event. */
            TraceCorrelator local = (TraceCorrelator) _profiler.createCorrelatorData();
            local.setInvocation(ri.target().hashCode(), ri.target().getClass(), ri.operation(),
                new Class[0], new Object[0], null);

            /* Unload the correlator from the peer. */
            ThreadComparator key = new ThreadComparator();
            synchronized (_buffer) {

                /* This is a temporary workaround.  Should fix this properly. */ 
                try {
                    _buffer = ri.get_reply_service_context(DISTRIBUTED_TRACE_SERVICE_ID).context_data;
                } catch (Exception e) {
                    _buffer = ri.get_request_service_context(DISTRIBUTED_TRACE_SERVICE_ID).context_data;
                }
                _offset = 0;
                _length = _buffer.length;
                TransportMediator.setDataStream(key, _buffer, _offset, _length, local);
            }

        } catch (Exception e) {
            System.err.println("exception in receive reply");
            e.printStackTrace();
        }
    }


    /**
     * @see org.omg.PortableInterceptor.ClientRequestInterceptorOperations#receive_exception(ClientRequestInfo)
     */
    public void receive_exception(ClientRequestInfo ri) throws ForwardRequest {
    }


    /**
     * @see org.omg.PortableInterceptor.ClientRequestInterceptorOperations#receive_other(ClientRequestInfo)
     */
    public void receive_other(ClientRequestInfo ri) throws ForwardRequest {
    }


    /**
     * @see org.omg.PortableInterceptor.InterceptorOperations#name()
     */
    public String name() {
        return IiopTransport.class.getName();
    }


    /**
     * @see org.omg.PortableInterceptor.InterceptorOperations#destroy()
     */
    public void destroy() {
    }


    /**
     * @see org.omg.PortableInterceptor.ServerRequestInterceptorOperations#receive_request_service_contexts(ServerRequestInfo)
     */
    public void receive_request_service_contexts(ServerRequestInfo ri) throws ForwardRequest {
    }


    /**
     * @see org.omg.PortableInterceptor.ServerRequestInterceptorOperations#receive_request(ServerRequestInfo)
     */
    public void receive_request(ServerRequestInfo ri) throws ForwardRequest {
        
        try {

            /* Due to the limitation of the ServerRequestInfo interface, the target of an RMI call on the server side
               cannot be determined properly.  As a workaround, the class will alwyas be assumed to be the
               javax.rmi.CORBA.Tie interface.  As a result, the relevant stattistics will obviously be incorrect. */
            TraceCorrelator local = (TraceCorrelator) _profiler.createCorrelatorData();
            local.setInvocation(new String(ri.object_id()).hashCode(), Tie.class, ri.operation(),
                new Class[0], new Object[0], null);

            /* Get the context information from the peer. */
            ThreadComparator key = new ThreadComparator();
            synchronized (_buffer) {
                _buffer = ri.get_request_service_context(DISTRIBUTED_TRACE_SERVICE_ID).context_data;
                _offset = 0;
                _length = _buffer.length;
                TransportMediator.setRemoteDataStream(key, _buffer, _offset, _length, local);
            }

        } catch (Exception e) {
            System.err.println("exception in receive request");
            e.printStackTrace();
        }
    }


    /**
     * @see org.omg.PortableInterceptor.ServerRequestInterceptorOperations#send_reply(ServerRequestInfo)
     */
    public void send_reply(ServerRequestInfo ri) {

        /* Create the correlator for this reply event. */
        TraceCorrelator local = (TraceCorrelator) _profiler.createCorrelatorData();
        local.setInvocation(new String(ri.object_id()).hashCode(), Tie.class, ri.operation(),
            new Class[0], new Object[0], null);

        /* Send the context from the peer. */
        ThreadComparator key = new ThreadComparator();
        synchronized (_buffer) {
            TransportMediator.getRemoteDataStream(_profiler.getId(), key, this, local);
            ServiceContext serviceContext = new ServiceContext(DISTRIBUTED_TRACE_SERVICE_ID, _buffer);
            ri.add_reply_service_context(serviceContext, false);
        }
    }


    /**
     * @see org.omg.PortableInterceptor.ServerRequestInterceptorOperations#send_exception(ServerRequestInfo)
     */
    public void send_exception(ServerRequestInfo ri) throws ForwardRequest {
    }


    /**
     * @see org.omg.PortableInterceptor.ServerRequestInterceptorOperations#send_other(ServerRequestInfo)
     */
    public void send_other(ServerRequestInfo ri) throws ForwardRequest {
    }


    /**
     * @see org.eclipse.hyades.collection.correlation.ITransport#sendRequest(byte[], int, int)
     */
    public void sendRequest(byte[] buffer, int offset, int length) {
        this._buffer = buffer;
        this._offset = offset;
        this._length = length;
    }


    /**
     * @see org.eclipse.hyades.collection.correlation.ITransport#sendReply(byte[], int, int)
     */
    public void sendReply(byte[] buffer, int offset, int length) {
        this._buffer = buffer;
        this._offset = offset;
        this._length = length;
    }
    

    /**
     * @see org.eclipse.hyades.collection.correlation.ITransport#receiveReply()
     */
    public byte[] receiveReply() {
        return null;
    }


    /**
     * @see org.eclipse.hyades.collection.correlation.ITransport#receiveRequest()
     */
    public byte[] receiveRequest() {
        return null;
    }
}
