/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import org.eclipse.emf.ecore.EObject;
/**
 * The purpose of this class is to associate EObjects in the database with their
 * database ids, without having strong references to the objects. If a client
 * does not hold a strong reference to an EObject, when it is garbage collected,
 * the EObject can be removed from this cache. This implementation reuses the
 * entry objects for both directions to save memory.
 */
public class WeakObjectCache {
	protected static final int INITIAL_CAPACITY = 11;
	protected static final float LOAD_FACTOR = 0.75f;
	protected static class Entry {
		WeakEObject key;
		Integer value;
		Entry nextIdToObject, nextObjectToId;
		protected Entry(WeakEObject key, Integer value, Entry nextIdToObject, Entry nextObjectToId) {
			this.key = key;
			this.value = value;
			this.nextIdToObject = nextIdToObject;
			this.nextObjectToId = nextObjectToId;
		}
		public String toString() {
			return "<" + key.toString() + ", " + value + ">";
		}
	}
	/**
	 * A weak object is a weak reference that refers to an EObject. The hash
	 * code is stored with the weak object, since it is needed to cleanup the
	 * tables after the EObject has been garbage collected.
	 */
	protected class WeakEObject extends WeakReference {
		int hash;
		protected WeakEObject(EObject object, ReferenceQueue queue) {
			super(object, queue);
			hash = object.hashCode();
		}
		public boolean equals(Object weakObject) {
			if (this == weakObject)
				return true;
			if (!(weakObject instanceof WeakEObject))
				return false;
			Object thisEObject = this.get();
			Object otherEObject = ((WeakEObject) weakObject).get();
			if ((thisEObject == null) || (otherEObject == null))
				return false;
			if (thisEObject == otherEObject)
				return true;
			else
				return false;
		}
		public int hashCode() {
			return hash;
		}
		public String toString() {
			if (get() == null)
				return "null";
			else
				return get().toString();
		}
	}
	protected transient Entry objectToIdTable[];
	protected transient Entry idToObjectTable[];
	protected transient int threshold = (int) (INITIAL_CAPACITY * LOAD_FACTOR);
	protected transient int count;
	protected ReferenceQueue queue = new ReferenceQueue();
	public WeakObjectCache() {
	}
	public Integer getId(EObject object) {
		if (idToObjectTable == null)
			return null;
		Entry entry = find(null, object, null);
		return entry == null ? null : entry.value;
	}
	public EObject getObject(Integer id) {
		if (objectToIdTable == null)
			return null;
		Entry entry = find(id, null, null);
		return entry == null ? null : (EObject) entry.key.get();
	}
	public boolean add(EObject object, Integer id) {
		if (idToObjectTable == null) {
			idToObjectTable = new Entry[INITIAL_CAPACITY];
			objectToIdTable = new Entry[INITIAL_CAPACITY];
		}
		if (object == null || id == null)
			return false;
		return put(object, id);
	}
	protected Entry find(Integer id, EObject object, WeakEObject weakObject) {
		int hash;
		try {
			if (id != null)
				hash = id.hashCode();
			else if (object != null)
				hash = object.hashCode();
			else
				hash = weakObject.hash;
			int index = (hash & 0x7FFFFFFF) % idToObjectTable.length;
			Entry entry;
			if (id != null)
				entry = idToObjectTable[index];
			else
				entry = objectToIdTable[index];
			while (entry != null) {
				if (equals(entry, id, object, weakObject))
					return entry;
				if (id != null)
					entry = entry.nextIdToObject;
				else
					entry = entry.nextObjectToId;
			}
		} catch (NullPointerException e) {
			// ignore this exception
		}
		return null;
	}
	protected boolean equals(Entry entry, Integer id, EObject object, WeakEObject weakObject) {
		if (id != null && entry.value.equals(id))
			return true;
		if (object != null && entry.key.get() == object)
			return true;
		if (weakObject != null && entry.key == weakObject)
			return true;
		return false;
	}
	protected boolean put(EObject object, Integer id) {
		if (find(null, object, null) != null)
			return false;
		int hash1 = object.hashCode();
		int hash2 = id.hashCode();
		int index1 = (hash1 & 0x7FFFFFFF) % objectToIdTable.length;
		int index2 = (hash2 & 0x7FFFFFFF) % idToObjectTable.length;
		if (count >= threshold) {
			rehash();
			index1 = (hash1 & 0x7FFFFFFF) % objectToIdTable.length;
			index2 = (hash2 & 0x7FFFFFFF) % idToObjectTable.length;
		}
		WeakEObject weakObject = new WeakEObject(object, queue);
		Entry entry = new Entry(weakObject, id, idToObjectTable[index2], objectToIdTable[index1]);
		idToObjectTable[index2] = entry;
		objectToIdTable[index1] = entry;
		count++;
		return true;
	}
	public boolean remove(EObject object) {
		if (idToObjectTable == null)
			return false;
		Entry entry = find(null, object, null);
		if (entry == null)
			return false;
		remove(entry, idToObjectTable);
		remove(entry, objectToIdTable);
		--count;
		return true;
	}
	protected void remove(Entry entry, Entry[] table) {
		int hash;
		if (table == idToObjectTable)
			hash = entry.value.hashCode();
		else
			hash = entry.key.hashCode();
		int index = (hash & 0x7FFFFFFF) % table.length;
		Entry current = table[index];
		Entry previous = null;
		while (current != null) {
			if (current == entry) {
				if (previous != null) {
					if (table == idToObjectTable)
						previous.nextIdToObject = current.nextIdToObject;
					else
						previous.nextObjectToId = current.nextObjectToId;
				} else {
					if (table == idToObjectTable)
						table[index] = current.nextIdToObject;
					else
						table[index] = current.nextObjectToId;
				}
				return;
			}
			previous = current;
			if (table == idToObjectTable)
				current = current.nextIdToObject;
			else
				current = current.nextObjectToId;
		}
	}
	protected void rehash() {
		rehash(idToObjectTable);
		rehash(objectToIdTable);
	}
	protected void rehash(Entry[] table) {
		int newCapacity = table.length * 2 + 1;
		Entry[] newTable = new Entry[newCapacity];
		threshold = (int) (newCapacity * LOAD_FACTOR);
		for (int i = 0; i < table.length; i++) {
			Entry entry = table[i];
			while (entry != null) {
				Entry next;
				int hash;
				if (table == idToObjectTable) {
					next = entry.nextIdToObject;
					hash = entry.value.hashCode();
				} else {
					next = entry.nextObjectToId;
					hash = entry.key.hashCode();
				}
				int index = (hash & 0x7FFFFFFF) % newCapacity;
				if (table == idToObjectTable)
					entry.nextIdToObject = newTable[index];
				else
					entry.nextObjectToId = newTable[index];
				newTable[index] = entry;
				entry = next;
			}
		}
		if (table == idToObjectTable)
			idToObjectTable = newTable;
		else
			objectToIdTable = newTable;
	}
	/**
	 * Removes objects from the cache that have been garbage collected.
	 */
	public void refresh() {
		WeakEObject weakObject;
		weakObject = (WeakEObject) queue.poll();
		while (weakObject != null) {
			Entry entry = find(null, null, weakObject);
			if (entry != null) {
				remove(entry, idToObjectTable);
				remove(entry, objectToIdTable);
				--count;
			}
			weakObject = (WeakEObject) queue.poll();
		}
	}
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("objectToIdTable:\n");
		printTable(buffer, objectToIdTable);
		buffer.append("idToObjectTable:\n");
		printTable(buffer, idToObjectTable);
		return buffer.toString();
	}
	public int getObjectCount() {
		return count;
	}
	protected void printTable(StringBuffer buffer, Entry[] table) {
		if (table == null)
			return;
		for (int i = 0; i < table.length; i++) {
			Entry entry = table[i];
			if (entry != null) {
				buffer.append("  " + i + ": [");
				buffer.append(entry.toString());
				if (table == objectToIdTable)
					entry = entry.nextObjectToId;
				else
					entry = entry.nextIdToObject;
				while (entry != null) {
					buffer.append(", ");
					buffer.append(entry.toString());
					if (table == objectToIdTable)
						entry = entry.nextObjectToId;
					else
						entry = entry.nextIdToObject;
				}
				buffer.append("]\n");
			}
		}
	}
} // WeakObjectCache
