/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.facades.behavioral;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * 
 * The verdict is a predefined enumeration datatype which contains at least the 
 * values fail, inconclusive, pass, error indicating how this test case execution 
 * has performed. 
 * 
 * - A pass verdict indicates that the test case is successful and that the SUT 
 *   has behaved as expected. 
 * - A fail verdict on the other hand shows that the SUT is not behaving 
 *   according to the specification. 
 * - An inconclusive verdict means that the test execution cannot determine 
 *   whether the SUT performs correctly or not. 
 * - An error verdict tells that the test system itself and not the SUT fails.
 * 
 * @author marcelop
 * @author psun
 * @since 0.0.1
 */
public class Verdict extends AbstractEnumerator
{
	public static final int PASS = 0;
	public static final int FAIL = 1;
	public static final int INCONCLUSIVE = 2;
	public static final int ERROR = 3;
	
	public static final Verdict PASS_LITERAL = new Verdict(PASS, "pass");
	public static final Verdict FAIL_LITERAL = new Verdict(FAIL, "fail");
	public static final Verdict INCONCLUSIVE_LITERAL = new Verdict(INCONCLUSIVE, "inconclusive");
	public static final Verdict ERROR_LITERAL = new Verdict(ERROR, "error");
	 
	private static final Verdict[] VALUES_ARRAY = 
	new Verdict[]{PASS_LITERAL, FAIL_LITERAL, INCONCLUSIVE_LITERAL, ERROR_LITERAL};
	
	public static final List VALUES = Collections.unmodifiableList(Arrays.asList(VALUES_ARRAY));

	/**
	 * Gets the verdict by specifying the name.
	 * 
	 * @param name
	 * @return Verdict
	 */
	public static Verdict getVerdict(String name)
	{
		for (int i = 0, maxi = VALUES_ARRAY.length; i < maxi; i++)
		{
			if(VALUES_ARRAY[i].toString().equals(name))
				return VALUES_ARRAY[i];
		}
		return null;
	}

	/**
	 * Gets the verdict by specifying the value value.
	 * 
	 * @param value
	 * @return Verdict
	 */
	public static Verdict getVerdict(int value)
	{
		switch(value)
		{
			case PASS: return PASS_LITERAL;
			case FAIL: return FAIL_LITERAL;
			case INCONCLUSIVE: return INCONCLUSIVE_LITERAL;
			case ERROR:	return ERROR_LITERAL;
		}
		return null;
	}
	
	/**
	 * Constructor for Verdict
	 * 
	 * @param value
	 * @param name
	 */
	private Verdict(int value, String name)
	{
		super(value, name);
	}	
}
