/**********************************************************************
Copyright (c) 2003 Hyades project.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html

Contributors:
 IBM Corporation - initial implementation
 IBM Rational - initial implementation
**********************************************************************/
package org.eclipse.hyades.execution.local;

import java.io.IOException;

import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.invocation.CallData;
import org.eclipse.hyades.execution.invocation.IRemoteObject;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;

/**
 * This is the parent of the hierarchy of stubs -- objects that handle remote
 * invocation semantics for other objects.
 */
public abstract class RemoteObjectStub implements IRemoteObject {
	
	/**
	 * a unique identifier used to reference this stub (and hence the
	 * target object) 
	 */
	protected Integer uniqueId = new Integer(hashCode());
	
	/**
	 * Remote objects are created in the context of the same session as the
	 * factory that creates them.
	 */
	protected ISession sessionContext;
	
	/**
	 * the object for which this stub is providing remote execution semantics
	 */
	protected Object delegate;
	
	public RemoteObjectStub(){}
	
	/**
	 * get the id unique to this instance
	 * @return
	 */
	public Integer getUniqueId() {
		return uniqueId;
	}

	/**
	 * Delegate a remote method call to the target instance. This method
	 * orchestrates the marshalling of the method call and the remote invocation
	 * of the method against the target.
	 * 
	 * @param callArgs
	 * @param call
	 * @return
	 * @throws RemoteInvocationException
	 */
	public ReturnData delegateRemoteCall(Class[] argTypes, Object[] callArgs, String call)
		throws RemoteInvocationException {

		// Gather the necessary data, marshal it and submit it for remote
		// invocation.
		CallData callData = new CallData(getUniqueId(), argTypes, callArgs, call);
	
		/* Create our custom command */
		BinaryCustomCommand command = new BinaryCustomCommand();
		try {
			command.setData(Marshaller.marshalMethodCall(callData));
		}
		catch ( IOException e ) {
			throw new RemoteInvocationException(e);
		}
		
		/* Find our session to send the invocation over */		
		try {
			SessionStub sessStub = ((SessionStub)getSessionContext());
			((SessionImpl)sessStub.getDelegate()).invokeRemote(command);
			
		}
		catch(ClassCastException e) {
			throw new RemoteInvocationException("Class is not a child of a ISession instance");
		}
	
		// Wait for notification that the remote call has returned, unmarshal
		// the return data, return the result of the remote call.
		ReturnData returnData = null;
		while (true) {
			Marshaller.waitForReturnData();
			ReturnData tmp = Marshaller.peekReturnValue();
			if ( callData.isForSameCallAs(tmp) ) {
				returnData = Marshaller.unqueueReturnValue();
				break;
			}
		}

		return returnData;
	}
	
	/**
	 * Get the instance for which this stub is providing remote invocation
	 * semantics.
	 *  
	 * @return
	 */
	public Object getDelegate() {
		return delegate;
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#setDelegate(java.lang.Object)
	 */
	public void setDelegate(Object delegate) {
		this.delegate = delegate;
	}

	/**
	 * @see org.eclipse.hyades.execution.invocation.IRemoteObject#init()
	 */
	public void init() {}
	
	/**
	 * Get the local session context for this remote object.
	 * Remote objects are created in the context of a session. The session
	 * context of a remote object is the same as that of the factory that
	 * created the remote object.
	 * 
	 * @return a <code>SessionImpl</code> instance
	 * 
	 * @see org.eclipse.hyades.execution.local.SessionImpl
	 */
	public ISession getSessionContext() {
		return sessionContext;
	}
	
	/**
	 * Set the session context for this remote object. The session context
	 * should be set to the same as that of the factory that created the
	 * remote object.
	 */
	public void setSessionContext(ISession session) {
		sessionContext = session;
	}

}
