package org.eclipse.hyades.logging.events.cbe.util.tests;

import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import junit.framework.TestCase;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

import org.eclipse.hyades.logging.events.cbe.impl.tests.Util;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * 
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 */
public class EventHelpersTest extends TestCase {

    /**
     * Constructor for EventHelpersTest.
     * @param arg0
     */
    public EventHelpersTest(String testcase) {
        super(testcase);
    }

    /*
     * @see TestCase#setUp()
     */
    protected void setUp() throws Exception {
        super.setUp();
    }

    /*
     * @see TestCase#tearDown()
     */
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    /**
     * Tests whether the comparison between two EMF objects.
     */
    public void testCompareEvent() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        assertTrue(cbe1.equals(cbe1));
        CommonBaseEvent cbe2 = (CommonBaseEvent) (Util.clone(cbe1));

        assertTrue(cbe1.equals(cbe2));
    }

    public void testConvertingLocaleSensitiveDates(){
                
        final long CURRENT_TIME_MILLIS = System.currentTimeMillis();

        Locale[] locales = Locale.getAvailableLocales();
        
        Locale defaultLocale = Locale.getDefault();
        
        ExtendedDataElement extendedDataElement = null;
        
        for (int counter = 0; counter < locales.length; counter++) {

            //Change the locale:
            Locale.setDefault(locales[counter]);

            extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();

            extendedDataElement.setValuesAsDateAsLong(CURRENT_TIME_MILLIS);            
            
            assertTrue(CURRENT_TIME_MILLIS == extendedDataElement.getValuesAsDateAsLong());
        }        
        
        //Reset the default locale:
        Locale.setDefault(defaultLocale);
    }
    
    /**
     * Tests whether order of elements is considered 
     * important for ExtendedDataElement.values
     */
    public void testCompareExtendedDataElementValues() {
        EventFactory eFx = EventFactory.eINSTANCE;

        ExtendedDataElement ede1 = eFx.createExtendedDataElement();
        ede1.getValues().add("A");
        ede1.getValues().add("B");

        ExtendedDataElement ede2 = eFx.createExtendedDataElement();
        ede2.getValues().add("A");
        ede2.getValues().add("B");

        assertEquals(ede1, ede2);

        ede1.getValues().clear();
        ede1.getValues().add("A");
        ede1.getValues().add("B");
        ede2.getValues().clear();
        ede2.getValues().add("B");
        ede2.getValues().add("A");

        assertTrue(ede1.equals(ede2) == false);
    }

    /**
     * Tests whether order of elements is considered 
     * important for MsgDataElement.msgCatalogTokens
     */
    public void testCompareMsgDataElementMsgCatalogTokens() {
        EventFactory eFx = EventFactory.eINSTANCE;

        MsgDataElement mde1 = eFx.createMsgDataElement();
        mde1.addMsgCatalogTokenAsString("A");
        mde1.addMsgCatalogTokenAsString("B");

        MsgDataElement mde2 = eFx.createMsgDataElement();
        mde2.addMsgCatalogTokenAsString("A");
        mde2.addMsgCatalogTokenAsString("B");

        assertEquals(mde1, mde2);

        mde1.getMsgCatalogTokens().clear();
        mde1.addMsgCatalogTokenAsString("A");
        mde1.addMsgCatalogTokenAsString("B");
        mde2.getMsgCatalogTokens().clear();
        mde2.addMsgCatalogTokenAsString("B");
        mde2.addMsgCatalogTokenAsString("A");

        assertTrue(mde1.equals(mde2) == false);
    }

    /**
     * Tests whether the comparison between two EMF objects fails when
     * the elements of internal lists have different numbers of copies
     * of the same element (AABA is not the same as AABB).
     */
    public void testCompareEObjectDuplicateListElements() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        // The idea is to get two lists (AABBB) and (AAABB)
        final int listDivide = 2;
        final int listElements = listDivide + 2;
        for (int i = 0; i < listElements; i++) {
            ContextDataElement cde1 = EventFactory.eINSTANCE.createContextDataElement();
            String contextId = "A";
            if (i >= listDivide) {
                contextId = "B";
            }
            cde1.setContextId(contextId);
            cbe1.getContextDataElements().add(cde1);

            if (i == listDivide) {
                contextId = "A";
            }
            ContextDataElement cde2 = EventFactory.eINSTANCE.createContextDataElement();
            cde2.setContextId(contextId);
            cbe2.getContextDataElements().add(cde2);
        }

        assertFalse(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison between two EMF objects succeeds when
     * the elements of internal lists don't follow the same order.
     */
    public void testCompareEObjectDifferentListOrder() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        final int listElements = 5;
        for (int i = 0; i < listElements; i++) {
            ContextDataElement cde1 = EventFactory.eINSTANCE.createContextDataElement();
            cde1.setContextId(Integer.toString(i));
            cbe1.getContextDataElements().add(cde1);

            ContextDataElement cde2 = EventFactory.eINSTANCE.createContextDataElement();
            cde2.setContextId(Integer.toString(listElements - i - 1));
            cbe2.getContextDataElements().add(cde2);
        }

        assertTrue(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison between two EMF objects fails when 
     * there are differences in their basic attributes.
     */
    public void testCompareEObjectDifferentAttribute() {
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        ComponentIdentification cit = EventFactory.eINSTANCE.createComponentIdentification();
        cbe2.setSourceComponentId(null);
        assertFalse(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison between two EMF objects fails 
     * when the contents of the contained objects differ.
     */
    public void testCompareEObjectDifferentContainment() {
        EventPackage ePck = EventPackage.eINSTANCE;
        CommonBaseEvent cbe1 = createReferenceEvent();
        CommonBaseEvent cbe2 = createReferenceEvent();

        // Testing whether different values will be caught.
        cbe1.getSourceComponentId().setApplication("something");
        cbe2.getSourceComponentId().setApplication("something different");
        assertFalse(cbe1.equals(cbe2));

        // Testing whether different pointer values will be caught.
        cbe1.getSourceComponentId().setApplication(null);
        cbe2.getSourceComponentId().setApplication("something different");
        assertFalse(cbe1.equals(cbe2));

        // Thread id is the last attribute inside the component id, may be
        // a boundary condition.
        cbe1.getSourceComponentId().setApplication("same");
        cbe2.getSourceComponentId().setApplication("same");
        cbe1.getSourceComponentId().setThreadId("id");
        cbe2.getSourceComponentId().setThreadId("id2");
        assertFalse(cbe1.equals(cbe2));
    }

    /**
     * Tests whether the comparison returns false when the object types
     * are not EMF types.
     */
    public void testCompareNotEObjects() {
        assertFalse(EventHelpers.compareEObject(new String(), new String()));
        CommonBaseEvent cbe = createReferenceEvent();
        assertFalse(cbe.equals(new String()));
        assertFalse(EventHelpers.compareEObject(new String(), cbe));
        String base = "AAA";
        assertFalse(EventHelpers.compareEObject(base, base));
    }

    /**
     * Tests whether the comparison returns false when either or both objects
     * is a null pointer.
     */
    public void testCompareEObjectsNull() {
        CommonBaseEvent cbe = createReferenceEvent();
        assertFalse(EventHelpers.compareEObject(cbe, null));
        assertFalse(EventHelpers.compareEObject(null, cbe));
    }

    /**
     * Tests whether the comparison returns false when either or both objects
     * is a null pointer.
     */
    public void testCompareEObjectsNotSet() {
        CommonBaseEvent cbe = createReferenceEvent();
        CommonBaseEvent cbe1 = createReferenceEvent();
        cbe.unsetRepeatCount();
        cbe1.setRepeatCount((short) 0);
        assertFalse(EventHelpers.compareEObject(cbe, cbe1));
        assertFalse(EventHelpers.compareEObject(cbe1, cbe));
    }

    /**
     * Tests whether the comparison returns false when either or both objects
     * is a null pointer.
     */
    public void testCompareSituationTypes() {
        EventPackage ePck = EventPackage.eINSTANCE;
        EClass situationClass = ePck.getSituationType();
        EFactory eFx = ePck.getEFactoryInstance();

        List l = ePck.eContents();
        Iterator i = l.iterator();
        while (i.hasNext()) {
            EClass c = (EClass) i.next();
            if (c.isInstance(situationClass)) {
                EObject o1 = eFx.create(c);
                EObject o2 = eFx.create(c);
                assertTrue(o1.eClass().getName() + ".equals failed", o1.equals(o2));
            }
        }
    }

    /**
     * 
     */
    public void testDifferentClasses() {
        EventFactory efx = EventFactory.eINSTANCE;

        CommonBaseEvent cbe = efx.createCommonBaseEvent();
        ExtendedDataElement ebe = efx.createExtendedDataElement();
        EventHelpers.compareEObject(cbe, ebe);
    }

    /*
     * Private methods.
     */

    /**
     * Creates a reference event for all comparison.
     * <P>
     * The individual test fixture modify the attributes in order to match
     * the test scenario.
     */
    private CommonBaseEvent createReferenceEvent() {
        CommonBaseEvent result = EventFactory.eINSTANCE.createCommonBaseEvent();
        result.setCreationTime("1999-05-31T13:20:00-05:00");
        result.setGlobalInstanceId("AAA");
        result.setElapsedTime(65);
        result.setExtensionName("Event_class");
        result.setLocalInstanceId("id");
        result.setMsg("msg");
        result.setPriority((short) 32);
        result.setRepeatCount((short) 12);
        result.setSequenceNumber(88);
        result.setSeverity((short) 67);
        result.setVersion("1.0.1");

        MsgDataElement mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog("cat");
        mde.setMsgCatalogId("msgCatId");
        mde.setMsgCatalogType("type");
        mde.setMsgId("abc");
        mde.setMsgIdType("idType");
        mde.setMsgLocale("locale");
        mde.addMsgCatalogTokenAsString("token1");
        result.setMsgDataElement(mde);

        ComponentIdentification cit1 = EventFactory.eINSTANCE.createComponentIdentification();
        cit1.setApplication("App");
        cit1.setComponent("comp");
        cit1.setComponentIdType("idType");
        cit1.setComponentType("comp");
        cit1.setExecutionEnvironment("env");
        cit1.setInstanceId("instId");
        cit1.setLocation("location");
        cit1.setLocationType("type");
        cit1.setProcessId("process");
        cit1.setSubComponent("sub");
        cit1.setThreadId("threadid");
        result.setReporterComponentId(cit1);

        ComponentIdentification cit2 = EventFactory.eINSTANCE.createComponentIdentification();
        cit2.setApplication("App2");
        cit2.setComponent("comp2");
        cit2.setComponentIdType("idType2");
        cit2.setComponentType("comp2");
        cit2.setExecutionEnvironment("env2");
        cit2.setInstanceId("instId2");
        cit2.setLocation("location2");
        cit2.setLocationType("type2");
        cit2.setProcessId("process2");
        cit2.setSubComponent("sub2");
        cit2.setThreadId("threadid2");
        result.setSourceComponentId(cit2);

        result.getAny().add("BSD1");
        result.getAny().add("BSD2");

        for (int i = 0; i < 5; i++) {
            AssociatedEvent ae1 = EventFactory.eINSTANCE.createAssociatedEvent();
            ae1.setAssociationEngine("ABC" + i);
            AssociationEngine aei = EventFactory.eINSTANCE.createAssociationEngine();
            aei.setId("A" + i);
            aei.setName("B" + i);
            aei.setType("CFG" + i);
            ae1.setAssociationEngineInfo(aei);
            ae1.setResolvedEvents("tr YU IO" + i);
            result.getAssociatedEvents().add(ae1);
        }

        for (int i = 0; i < 5; i++) {
            ContextDataElement cde1 = EventFactory.eINSTANCE.createContextDataElement();
            cde1.setContextId("id" + i);
            cde1.setContextValue("bfg" + i);
            cde1.setName("rew" + i);
            cde1.setType("tr4" + i);
            result.getContextDataElements().add(cde1);
        }

        for (int i = 0; i < 5; i++) {
            ExtendedDataElement ede1 = EventFactory.eINSTANCE.createExtendedDataElement();
            ede1.setName("tre" + i);
            ede1.setType("string" + i);
            ede1.getValues().add("text" + i);
            result.getExtendedDataElements().add(ede1);
        }

        Situation sit = EventFactory.eINSTANCE.createSituation();
        AvailableSituation sitType = EventFactory.eINSTANCE.createAvailableSituation();
        sitType.setAvailabilityDisposition("av");
        sitType.setOperationDisposition("op");
        sitType.setProcessingDisposition("pro");
        sitType.setReasoningScope("reason");
        sit.setCategoryName("cat");
        sit.setSituationType(sitType);
        result.setSituation(sit);

        return result;
    }

    /**
     * Tests whether the comparison for ExtendedDataElement.hexValue
     * is case insensitive.
     */
    public void testCompareExtendedDataElementHexValue() {
        ExtendedDataElement ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setHexValue("ABc123");
        ExtendedDataElement ede1 = EventFactory.eINSTANCE.createExtendedDataElement();
        ede1.setHexValue("abC123");

        assertEquals(ede, ede1);

    }
}
