/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.java.internal.junit.wizard;

import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.help.WorkbenchHelp;

import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.test.java.TestJavaPlugin;
import org.eclipse.hyades.test.java.internal.util.ContextIds;
import org.eclipse.hyades.test.java.internal.util.TestJavaUtil;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;

/**
 * @author marcelop
 * @since 0.3.0
 */
public class ProjectAndSourcePage 
extends WizardPage implements ModifyListener, SelectionListener
{
	protected static final String SET_PROJECT = "PROJECT_NAME";
	protected static final String SET_SOURCE = "SOURCE_FOLDER_NAME";

	private IImplementor implementor;
	private IJavaProject[] javaProjects;
	
	private Text javaProjectText;
	private Button selectJavaProjectButton;
	
	private Text sourceFolderText;
	private Button selectSourceFolderButton;
	
	/**
	 * Constructor for ProjectAndSourcePage
	 */
	public ProjectAndSourcePage(IImplementor implementor, IJavaProject[] javaProjects)
	{
		super("ProjectAndSourcePage");
		setTitle(TestJavaPlugin.getString("GEN_WTITLE"));
		setDescription(TestJavaPlugin.getString("GEN_LDESC"));
		
		this.implementor = implementor;
		this.javaProjects = javaProjects;
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
	 */
	public void dispose()
	{
		implementor = null;
		super.dispose();
	}
	
	public IJavaProject[] getJavaProjects()
	{
		return javaProjects;
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent)
	{
		WorkbenchHelp.setHelp(parent, TestJavaPlugin.getID() + ContextIds.INV_TEST_DLG);
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(GridDataUtil.createFill());
		
		Label label = new Label(composite, SWT.NONE);
		label.setText(TestJavaPlugin.getString("STR_WB_PROJECT"));
		
		javaProjectText = new Text(composite, SWT.BORDER | SWT.SINGLE);
		javaProjectText.setLayoutData(GridDataUtil.createHorizontalFill());
		
		selectJavaProjectButton = new Button(composite, SWT.PUSH);
		GridData gridData = new GridData(GridData.HORIZONTAL_ALIGN_END);
		gridData.horizontalSpan = 2;
		selectJavaProjectButton.setLayoutData(gridData);
		selectJavaProjectButton.setText(TestJavaPlugin.getString("STR_WB_BROWSE_BTN"));
		selectJavaProjectButton.addSelectionListener(this);
		selectJavaProjectButton.setEnabled(javaProjects.length > 0);
		
		label = new Label(composite, SWT.NONE);
		label.setText(TestJavaPlugin.getString("STR_WB_SRC_FOLDER"));

		sourceFolderText = new Text(composite, SWT.BORDER | SWT.SINGLE);
		sourceFolderText.setLayoutData(GridDataUtil.createHorizontalFill());
		sourceFolderText.setEditable(false);

		selectSourceFolderButton = new Button(composite, SWT.PUSH);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_END);
		gridData.horizontalSpan = 2;
		selectSourceFolderButton.setLayoutData(gridData);
		selectSourceFolderButton.setText(TestJavaPlugin.getString("STR_WB_SOURCE_BTN"));
		selectSourceFolderButton.addSelectionListener(this);
		
		
		if(selectJavaProjectButton.isEnabled())
		{
			if(implementor.getLocation() != null)
			{
				IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(implementor.getLocation());
				if((resource != null) && resource.exists())
				{
					if(resource.getType() == IResource.PROJECT)
					{
						javaProjectText.setText(resource.getName());
					}
					else if(resource.getType() == IResource.FOLDER)
					{
						javaProjectText.setText(resource.getProject().getName());
						sourceFolderText.setText(resource.getFullPath().removeFirstSegments(1).toString());
					}
				}
			}
			
			if(javaProjectText.getText().length() == 0)
			{
				String value = getDialogSettings().get(SET_PROJECT);
				if ((value != null) && (getJavaProject(value) != null))
					javaProjectText.setText(value);
					
				value = getDialogSettings().get(SET_SOURCE);
				if (value != null)
					sourceFolderText.setText(value);
			}

			setPageComplete(validatePage());
			setErrorMessage(null);

			javaProjectText.addModifyListener(this);			
			sourceFolderText.addModifyListener(this);
			javaProjectText.setFocus();
		}
		else
		{
			setPageComplete(false);
			selectSourceFolderButton.setEnabled(false); 
			javaProjectText.setEditable(false);
			setErrorMessage(TestJavaPlugin.getString("NO_JAVA_PRJ"));
			composite.setFocus();
		}
			
		setControl(composite);
	}
	
	/**
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e)
	{
		setPageComplete(validatePage());
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		if(e.widget == selectJavaProjectButton)
		{
			IJavaProject javaProject = chooseJavaProject();
			if(javaProject != null)
				javaProjectText.setText(javaProject.getElementName());
		}
		else if(e.widget == selectSourceFolderButton)
		{
			Object object = chooseSourceFolder();
			if(object instanceof IJavaProject)
			{
				javaProjectText.setText(((IJavaProject)object).getElementName());
			}	
			else if(object instanceof IPackageFragmentRoot)
			{
				IPackageFragmentRoot packageFragmentRoot = (IPackageFragmentRoot)object;
				javaProjectText.removeModifyListener(this);
				javaProjectText.setText(packageFragmentRoot.getJavaProject().getElementName());
				javaProjectText.addModifyListener(this);
				sourceFolderText.setText(packageFragmentRoot.getPath().removeFirstSegments(1).toString());
			}
		}
	}

	protected boolean validatePage()
	{
		String value = javaProjectText.getText();
		if(value.trim().length() == 0)
		{
			setErrorMessage(TestJavaPlugin.getString("_ERROR_WIZ_EMPTY_PROJECT"));
			return false;
		}
		
		IJavaProject javaProject = getJavaProject(value);
		if(javaProject == null)
		{
			setErrorMessage(TestJavaPlugin.getString("_ERROR_WIZ_NOT_EXIST_PROJECT"));
			return false;
		}
		
		
		List sourceFolders = TestJavaUtil.getSourceFolders(javaProject);
		if(sourceFolders.isEmpty())
		{
			sourceFolderText.removeModifyListener(this);
			sourceFolderText.setEditable(false);
			sourceFolderText.addModifyListener(this);
		}
		else
		{
			sourceFolderText.setEditable(true);
			value = sourceFolderText.getText();
			if(value.trim().length() == 0)
			{
				setErrorMessage(TestJavaPlugin.getString("_ERROR_WIZ_EMPTY_FOLDER"));
				return false;
			}
			
			if(!isValidFolderName())
			{
				setErrorMessage(TestJavaPlugin.getString("_ERROR_INV_FLD_NME"));
				return false;
			}
		}
		
		setErrorMessage(null);
		return true;	
	}
	
	protected boolean isValidFolderName()
	{
		String folder = getSourceFolder();
		if(folder.startsWith("/"))
			return false;

		if(folder.endsWith("/"))
			return false;
			
		return true;
	}
	
	public String getSourceFolder()
	{
		if(sourceFolderText.getEditable())
			return sourceFolderText.getText().trim().replace('\\', '/');
			
		return "";
	}

	protected IJavaProject getJavaProject()
	{
		return getJavaProject(javaProjectText.getText());
	}

	/**
	 * @return Java Project
	 */
	protected IJavaProject getJavaProject(String projectName)
	{
		if((projectName != null) && (projectName.trim().length() > 0))
		{
			IJavaProject javaProject = JavaCore.create(ResourcesPlugin.getWorkspace().getRoot()).getJavaProject(projectName);
			if(javaProject.exists())
				return javaProject;
		}
		
		return null;
	}

	protected IJavaProject chooseJavaProject()
	{
		ILabelProvider labelProvider = new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_DEFAULT);
		ElementListSelectionDialog dialog = new ElementListSelectionDialog(getShell(), labelProvider);
		dialog.setTitle(TestJavaPlugin.getString("TITLE_PROJECT_SELECT"));
		dialog.setMessage(TestJavaPlugin.getString("STR_PROJECT_SELECT"));
		dialog.setElements(javaProjects);

		IJavaProject javaProject = getJavaProject(javaProjectText.getText());
		if (javaProject != null)
			dialog.setInitialSelections(new Object[] {javaProject});

		if (dialog.open() == Window.OK)
			return (IJavaProject)dialog.getFirstResult();

		return null;
	}
	
	protected Object chooseSourceFolder()
	{
		ElementTreeSelectionDialog dialog = TestJavaUtil.createSourceFolderDialog(getShell(), null);
		dialog.setTitle(TestJavaPlugin.getString("TITLE_SRC_SELECT"));
		dialog.setMessage(TestJavaPlugin.getString("STR_SRC_SELECT"));

		IJavaProject javaProject = getJavaProject(javaProjectText.getText());
		if (javaProject != null)
			dialog.setInitialSelections(new Object[]{javaProject});

		if(dialog.open() == Window.OK)
			return dialog.getFirstResult();

		return null;
	}	
}
