package org.eclipse.hyades.logging.core;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import org.eclipse.hyades.internal.logging.core.XmlGenerator;

/** 
 * Implementation of the <code>org.eclipse.hyades.logging.core.ILogger</code>
 * interface which represents an XML logger for logging messages to a Logging Agent
 * with the same name as the logger.
 * <p>
 * A logger instance is created via the constructor.
 * For example,
 * <p>
 * <code>
 * XmlLogger xmlLogger = new XmlLogger("My XML Logger");
 * 
 * xmlLogger.write("My XML log message.");
 * </code>
 *  
 * 
 * @author      Paul Slauenwhite
 * @author      Richard Duggan
 * @version	    March 7, 2003
 * @see		    org.eclipse.hyades.logging.core.ILogger
 * @deprecated  Replaced by org.eclipse.hyades.logging.commons.Logger (Hyades implementation of the Apache Commons' org.apache.commons.logging.Log interface). 
 */
public class XmlLogger implements ILogger, IPreemptedLogger {

    /**
     * The Logging Agent with the same name as the XML logger to which logged messages are sent.
     */
    private LoggingAgent loggingAgent;

    /**
     * Constructor to create an XML logger instance with a Logging Agent using 
     * the parameter name.
     * 
     * @param name The name of the newly created XML logger.
     */
    public XmlLogger(String name) {
        loggingAgent = new LoggingAgent(name);
    }

    /**
     * Deregisters this Logging Agent instance from the Agent Controller.
     */
    public final void finalize() {
        loggingAgent.deregister();
    }

    /**
     * Suspends the current thread until the Logging Agent is monitored or the 
     * the parameter maximum time (milliseconds) has expired.
     *
     * @param maxTime the maximum amount of time (milliseconds) to suspend the current thread
     * @return true if the logger is being monitored
     */
    public boolean waitUntilLogging(long maxTime) {
        return loggingAgent.waitUntilLogging(maxTime);
    }

    /**
     * Checks if the Logging Agent is logging (e.g. Agent Controller is available and the Logging Agent is being monitored).
     *
     * @return True if the Logging Agent is logging, otherwise false.
     */
    public boolean isLogging() {
        return loggingAgent.isLogging();
    }

    /**
     * Logs the parameter boolean primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The boolean primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The boolean primitive to be logged to the Logging Agent.
     */
    public void write(boolean msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter char primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The char primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The char primitive to be logged to the Logging Agent.
     */
    public void write(char msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter byte primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The byte primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The byte primitive to be logged to the Logging Agent.
     */
    public void write(byte msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter short primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The short primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The short primitive to be logged to the Logging Agent.
     */
    public void write(short msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter int primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The int primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The int primitive to be logged to the Logging Agent.
     */
    public void write(int msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter long primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The long primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The long primitive to be logged to the Logging Agent.
     */
    public void write(long msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter float primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The float primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The float primitive to be logged to the Logging Agent.
     */
    public void write(float msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter double primitive to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The double primitive is first converted to XML and then sent to the Logging Agent.
     * 
     * @param msg The double primitive to be logged to the Logging Agent.
     */
    public void write(double msg) {
        loggingAgent.write(XmlGenerator.primitiveToXML(msg));
    }

    /**
     * Logs the parameter <code>java.lang.Object</code> log record to the Logging Agent 
     * with the same name as the XML logger.
     * 
     * The log record is first converted to XML and then sent to the Logging Agent.
     * 
     * @param record The log record to be logged to the Logging Agent.
     */
    public void write(Object record) {

        if (record instanceof IExternalizableToXml)
            loggingAgent.write(((IExternalizableToXml) record).externalizeCanonicalXmlString());

        else {
            XmlGenerator xmlGenerator = new XmlGenerator(loggingAgent.getName());
            xmlGenerator.reset(null, false, 4);
            loggingAgent.write(xmlGenerator.objectToXML(record));
        }
    }

    /**
     * The following method is used to only log error messages to an active
     * logging agent.
     *
     * @param message	-	The message to be logged
     *		  messageId -	The message ID corresponding to the error message
     *		  severity	-	The level of severity of the message (e.g. default = DEBUG, 1 = INFORMATION, 2 = WARNING, 3 = SEVERE, 4 = MAX ERROR LEVEL)
     */
    public void logErrorMessage(String message, String messageId, int severity) {
        loggingAgent.logErrorMessage(message, messageId, severity);
    }
}
