package org.eclipse.hyades.logging.core;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.hyades.internal.logging.core.Constants;
import org.w3c.dom.Document;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Utility class for working with XML data.
 * <p>
 * NOTE: The serialization APIs invoked in JRE 1.3.x and below run-time
 * environments use the <code>org.apache.xml.serialize.*</code> classes which
 * <b>MUST </b> be supplied by the user on the classpath (e.g. Xerces). The
 * <code>org.apache.xml.serialize.*</code> classes required to serialize a
 * Document Object Model (DOM) to an XML document are loaded reflectively at
 * run-time so as not eliminate the Xerces dependency at compilation time.
 * Conversely, the serialization APIs invoked in JRE 1.4.x and above run-time
 * environments use the <code>javax.xml.transform.*</code> classes which are
 * available on the classpath (e.g. JRE 1.4.x and above).
 * <p>
 * 
 * 
 * @author Paul E. Slauenwhite
 * @version June 2, 2004
 * @since April 15, 2004
 * @see org.eclipse.hyades.logging.core.SerializationExceptionn
 */
public class XmlUtility implements Constants {

    /**
     * Static flag for quickly determining if the
     * <code>javax.xml.transform.*</code> classes are available on the
     * classpath (e.g. JRE 1.4.x and above).
     * <p>
     * By default, a JRE 1.4.x and above run-time environment is assumed.
     */
    private static boolean isJavaXMLTransformerAvailable = true;

    /**
     * Serializes a Document Object Model (DOM) to an XML document and writes
     * the XML document to an output file on the local file system.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to an XML
     * document, formatted (e.g. line breaks and indentation) and written to an
     * output file on the local file system.
     * <p>
     * The encoding for the serialized XML document is explicitly set to "UTF-8"
     * for all platforms excluding z/OS and OS/390 platforms. The encoding for
     * the serialized XML document is explicitly set to "IBM-1047" for z/OS and
     * OS/390 platforms only.
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            formatted XML document and written to the output file.
     * @param outputFile
     *            The file on the local file system where the formatted XML
     *            document is written.
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static void serialize(Document document, File outputFile) throws SerializationException {
        serialize(document, outputFile, true);
    }

    /**
     * Serializes a Document Object Model (DOM) to an XML document and writes
     * the XML document to an output file on the local file system.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to an XML
     * document, which may be potentially formatted and written to an output
     * file on the local file system.
     * <p>
     * The serialized XML document is formatted (e.g. line breaks and
     * indentation) if the parameter <code>format</code> flag is true.
     * <p>
     * The encoding for the serialized XML document is explicitly set to "UTF-8"
     * for all platforms excluding z/OS and OS/390 platforms. The encoding for
     * the serialized XML document is explicitly set to "IBM-1047" for z/OS and
     * OS/390 platforms only.
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            potentially formatted XML document and written to the output
     *            file.
     * @param outputFile
     *            The file on the local file system where the potentially
     *            formatted XML document is written.
     * @param format
     *            If the serialized XML document is formatted (e.g. line breaks
     *            and indentation).
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static void serialize(Document document, File outputFile, boolean format) throws SerializationException {

        FileWriter writer = null;

        try {

            writer = new FileWriter(outputFile);

            serialize(document, writer, format);
        } catch (Exception e) {
            throw (new SerializationException(e.toString()));
        } finally {

            if (writer != null) {

                try {
                    writer.close();
                } catch (IOException i) {
                    //Ignore since only attempting to close the FileWriter.
                }
            }
        }
    }

    /**
     * Serializes a Document Object Model (DOM) to an XML document string.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to the returned
     * XML document string, which is formatted (e.g. line breaks and
     * indentation).
     * <p>
     * The encoding for the serialized XML document string is explicitly set to
     * "UTF-8" for all platforms excluding z/OS and OS/390 platforms. The
     * encoding for the serialized XML document string is explicitly set to
     * "IBM-1047" for z/OS and OS/390 platforms only.
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            formatted XML document string.
     * @return The Document Object Model (DOM) serialized as a formatted XML
     *         document string, otherwise null.
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static String serialize(Document document) throws SerializationException {
        return (serialize(document, true));
    }

    /**
     * Serializes a Document Object Model (DOM) to an XML document string.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to the returned
     * XML document string, which may be potentially formatted.
     * <p>
     * The returned XML document string is formatted (e.g. line breaks and
     * indentation) if the parameter <code>format</code> flag is true.
     * <p>
     * The encoding for the serialized XML document string is explicitly set to
     * "UTF-8" for all platforms excluding z/OS and OS/390 platforms. The
     * encoding for the serialized XML document string is explicitly set to
     * "IBM-1047" for z/OS and OS/390 platforms only.
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            potentially formatted XML document string.
     * @param format
     *            If the serialized XML document string is formatted (e.g. line
     *            breaks and indentation).
     * @return The Document Object Model (DOM) serialized as a potentially
     *         formatted XML document string, otherwise null.
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static String serialize(Document document, boolean format) throws SerializationException {

        StringWriter writer = new StringWriter();

        try {
            serialize(document, writer, format);
        } catch (Exception e) {
            throw (new SerializationException(e.toString()));
        } finally {

            try {
                writer.close();
            } catch (IOException i) {
                //Ignore since only attempting to close the StringWriter.
            }
        }

        return (writer.toString());
    }

    /**
     * Serializes a Document Object Model (DOM) to an XML document array of
     * bytes.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to the returned
     * XML document array of bytes, which is formatted (e.g. line breaks and
     * indentation).
     * <p>
     * The encoding for the serialized XML document array of bytes is explicitly
     * set to "UTF-8" for all platforms excluding z/OS and OS/390 platforms. The
     * encoding for the serialized XML document array of bytes is explicitly set
     * to "IBM-1047" for z/OS and OS/390 platforms only.
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            formatted XML document array of bytes.
     * @return The Document Object Model (DOM) serialized as a formatted XML
     *         document array of bytes, otherwise null.
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static byte[] serializeAsByteArray(Document document) throws SerializationException {
        return (serializeAsByteArray(document, true));
    }

    /**
     * Serializes a Document Object Model (DOM) to an XML document array of
     * bytes.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to the returned
     * XML document array of bytes, which may be potentially formatted.
     * <p>
     * The returned XML document array of bytes is formatted (e.g. line breaks
     * and indentation) if the parameter <code>format</code> flag is true.
     * <p>
     * The encoding for the serialized XML document array of bytes is explicitly
     * set to "UTF-8" for all platforms excluding z/OS and OS/390 platforms. The
     * encoding for the serialized XML document array of bytes is explicitly set
     * to "IBM-1047" for z/OS and OS/390 platforms only.
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            potentially formatted XML document array of bytes.
     * @param format
     *            If the serialized XML document array of bytes is formatted
     *            (e.g. line breaks and indentation).
     * @return The Document Object Model (DOM) serialized as a potentially
     *         formatted XML document array of bytes, otherwise null.
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static byte[] serializeAsByteArray(Document document, boolean format) throws SerializationException {
        return (serialize(document, format).getBytes());
    }

    /**
     * Serializes a Document Object Model (DOM) to an XML document and writes
     * the XML document to a writer.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to an XML
     * document, formatted (e.g. line breaks and indentation) and written to a
     * writer.
     * <p>
     * The encoding for the serialized XML document is explicitly set to "UTF-8"
     * for all platforms excluding z/OS and OS/390 platforms. The encoding for
     * the serialized XML document is explicitly set to "IBM-1047" for z/OS and
     * OS/390 platforms only.
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            potentially formatted XML document and written to the writer.
     * @param writer
     *            The writer where the potentially formatted XML document is
     *            written.
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static void serialize(Document document, Writer writer) throws SerializationException {
        serialize(document, writer, true);
    }

    /**
     * Serializes a Document Object Model (DOM) to an XML document and writes
     * the XML document to a writer.
     * <p>
     * The parameter Document Object Model (DOM) is serialized to an XML
     * document, which may be potentially formatted and written to a writer.
     * <p>
     * The serialized XML document is formatted (e.g. line breaks and
     * indentation) if the parameter <code>format</code> flag is true.
     * <p>
     * The encoding for the serialized XML document is explicitly set to "UTF-8"
     * for all platforms excluding z/OS and OS/390 platforms. The encoding for
     * the serialized XML document is explicitly set to "IBM-1047" for z/OS and
     * OS/390 platforms only.
     * <p>
     * NOTE: This API is modeled as a facade design pattern. The facade provide
     * a unified API which is delegated to one of several specialized APIs
     * relative to the run-time environment.
     * <p>
     * This facade uses the following specialized APIs relative to the run-time
     * environment:
     * <p>
     * <ol>
     * <li>JRE 1.4.x and above run-time environments use the
     * <code>JavaXMLTransformer</code> which uses the
     * <code>javax.xml.transform.*</code> classes which are available on the
     * classpath (e.g. JRE 1.4.x and above).</li>
     * <li>JRE 1.3.x and below run-time environments use the
     * <code>XercesXMLSerializer</code> which uses the
     * <code>org.apache.xml.serialize.*</code> classes which <b>MUST </b> be
     * supplied by the user on the classpath (e.g. Xerces).</li>
     * </ol>
     * <p>
     * 
     * @param document
     *            The Document Object Model (DOM) to be serialized to a
     *            potentially formatted XML document and written to the writer.
     * @param writer
     *            The writer where the potentially formatted XML document is
     *            written.
     * @param format
     *            If the serialized XML document is formatted (e.g. line breaks,
     *            indentation, etc.).
     * @throws SerializationException
     *             If an error occurs during serialization.
     */
    public static void serialize(Document document, Writer writer, boolean format) throws SerializationException {

        if (isJavaXMLTransformerAvailable) {

            try {

                JavaXMLTransformer.serialize(document, writer, format);

                return;
            } catch (NoClassDefFoundError n) {
                isJavaXMLTransformerAvailable = false;
            } catch (Exception e) {
                throw (new SerializationException(e.toString()));
            }
        }

        try {
            XercesXMLSerializer.serialize(document, writer, format);
        } catch (Exception e) {
            throw (new SerializationException(e.toString()));
        }
    }

    /**
     * Normalizes the parameter string according to the XML specification for
     * attribute-value normalization ( <a
     * href="http://www.w3.org/TR/REC-xml">http://www.w3.org/TR/REC-xml </a>)
     * and valid characters ( <a
     * href="http://www.w3.org/TR/REC-xml#charsets">http://www.w3.org/TR/REC-xml#charsets
     * </a>).
     * <p>
     * Valid characters, according to the XML specification, in all Unicode
     * characters, excluding the surrogate blocks, 0xFFFE, and 0xFFFF.
     * <p>
     * 
     * @param string
     *            The string to be normalized.
     * @return The normalized string.
     */
    public static String normalize(String string) {

        //Return 'null' if the string is null:
        if (string == null) return "null";

        //Return an empty string if the string is empty:
        if (string.length() == 0) return "";

        StringBuffer normalizedString = new StringBuffer();
        char character;

        //Check if any characters require normalization or replacement of
        // non-valid characters:
        for (int counter = 0; counter < string.length(); counter++) {

            character = string.charAt(counter);

            //0x003C:
            if (character == '<')
                normalizedString.append("&lt;");

            //0x003E:
            else if (character == '>')
                normalizedString.append("&gt;");

            //0x0026:
            else if (character == '&')
                normalizedString.append("&amp;");

            //0x0022:
            else if (character == '"')
                normalizedString.append("&quot;");

            //0x0027:
            else if (character == '\'')
                normalizedString.append("&apos;");

            //0x0009:
            else if (character == '\t')
                normalizedString.append("&#x9;");

            //0x000A:
            else if (character == '\n')
                normalizedString.append("&#xA;");

            //0x000D:
            else if (character == '\r')
                normalizedString.append("&#xD;");

            /*
             * //0x0020: else if (character == ' ')
             * normalizedString.append("&#x20;");
             */

            //Valid character range:
            else if (((((int) (character)) >= 0x0020) && (((int) (character)) <= 0xD7FF)) || ((((int) (character)) >= 0xE000) && (((int) (character)) <= 0xFFFD)) || ((((int) (character)) >= 0x10000) && (((int) (character)) <= 0x10FFFF)))
                normalizedString.append(character);

            else
                normalizedString.append('?');
        }

        return (normalizedString.toString());
    }

    /**
     * De-normalizes the parameter string.
     * <p>
     * 
     * @param string
     *            The String to be de-normalized.
     * @return The de-normalized String.
     */
    public static String denormalize(String string) {

        if (string == null) return "null";

        StringBuffer denormalizedString = new StringBuffer();
        char character = 0;
        int semiColonIndex = -1;
        String name = null;

        //Locate and de-normalize all entity references:
        for (int counter = 0; counter < string.length(); counter++) {

            character = string.charAt(counter);

            //Check if this character is the start of a possible entity
            // reference (e.g. ampersand in &<name>;) and find a possible end to
            // the possible entity reference (e.g. semi-solon in &<name>;):
            if ((character == '&') && ((semiColonIndex = string.indexOf(';', (counter + 1))) != -1)) {

                name = string.substring((counter + 1), semiColonIndex).trim();

                if (name.equals("lt"))
                    denormalizedString.append('<');

                else if (name.equals("gt"))
                    denormalizedString.append('>');

                else if (name.equals("amp"))
                    denormalizedString.append('&');

                else if (name.equals("quot"))
                    denormalizedString.append('"');

                else if (name.equals("apos"))
                    denormalizedString.append('\'');

                else if (name.equals("#x9"))
                    denormalizedString.append('\t');

                else if (name.equals("#xA"))
                    denormalizedString.append('\n');

                else if (name.equals("#xD"))
                    denormalizedString.append('\r');

                /*
                 * else if (name.equals("#x20")) denormalizedString.append(' ');
                 */

                //Not a supported entity reference:
                else {
                    denormalizedString.append('&');
                    denormalizedString.append(name);
                    denormalizedString.append(';');
                }

                counter = semiColonIndex;
            } else
                denormalizedString.append(character);
        }

        return (denormalizedString.toString());
    }

    /**
     * A specialized implementation that uses a
     * <code>javax.xml.transform.Transformer</code> to serialize a Document
     * Object Model (DOM) to an XML document.
     * <p>
     * This specialized implementation uses the Java
     * <code>javax.xml.transform.*</code> classes which are available on the
     * classpath (e.g. JRE 1.4.x and above).
     * <p>
     * 
     * 
     * @author Paul E. Slauenwhite
     * @version June 2, 2004
     * @since June 2, 2004
     */
    private static class JavaXMLTransformer implements Constants {

        /**
         * Static instance of the <code>javax.xml.transform.Transformer</code>
         * to serialize a Document Object Model (DOM) to an XML document.
         */
        private static Transformer transformer = null;

        /**
         * Serializes a Document Object Model (DOM) to an XML document using a
         * <code>javax.xml.transform.Transformer</code> and writes the XML
         * document to a writer.
         * <p>
         * The parameter Document Object Model (DOM) is serialized to an XML
         * document using a <code>javax.xml.transform.Transformer</code>,
         * which may be potentially formatted and written to a writer.
         * <p>
         * The serialized XML document is formatted (e.g. line breaks and
         * indentation) if the parameter <code>format</code> flag is true.
         * <p>
         * The encoding for the serialized XML document is explicitly set to
         * "UTF-8" for all platforms excluding z/OS and OS/390 platforms. The
         * encoding for the serialized XML document is explicitly set to
         * "IBM-1047" for z/OS and OS/390 platforms only.
         * <p>
         * 
         * @param document
         *            The Document Object Model (DOM) to be serialized to a
         *            potentially formatted XML document and written to the
         *            writer.
         * @param writer
         *            The writer where the potentially formatted XML document is
         *            written.
         * @param format
         *            If the serialized XML document is formatted (e.g. line
         *            breaks and indentation).
         * @throws Exception
         *             If an error occurs during serialization.
         */
        private static void serialize(Document document, Writer writer, boolean format) throws Exception {

            if (transformer == null) {

                transformer = TransformerFactory.newInstance().newTransformer();

                transformer.setOutputProperty(OutputKeys.METHOD, "xml");

                if ((OS_NAME.equals("z/OS")) || (OS_NAME.equals("OS/390"))) {
                    transformer.setOutputProperty(OutputKeys.ENCODING, "IBM-1047");
                } else {
                    transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
                }
            }

            if (format) {

                transformer.setOutputProperty(OutputKeys.INDENT, "yes");

                //Unless a width is set, there will be only line breaks but no
                //indentation.
                //NOTE: The IBM and Sun JDK do not agree on the property name
                //so both are set.
                transformer.setOutputProperty("{http://xml.apache.org/xalan}indent-amount", "2");
                transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
            } else {

                transformer.setOutputProperty(OutputKeys.INDENT, "no");

                //Unless a width is set, there will be only line breaks but no
                //indentation.
                //NOTE: The IBM and Sun JDK do not agree on the property name
                //so both are set.
                transformer.setOutputProperty("{http://xml.apache.org/xalan}indent-amount", "0");
                transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "0");
            }

            transformer.transform(new DOMSource(document), new StreamResult(writer));
        }
    }

    /**
     * A specialized implementation that uses a
     * <code>org.apache.xml.serialize.XMLSerializer</code> to serialize a
     * Document Object Model (DOM) to an XML document.
     * <p>
     * This specialized implementation uses the Xerces
     * <code>org.apache.xml.serialize.*</code> classes which <b>MUST </b> be
     * supplied by the user on the classpath.
     * <p>
     * The <code>org.apache.xml.serialize.*</code> classes required to
     * serialize a Document Object Model (DOM) to an XML document are loaded
     * reflectively at run-time so as not eliminate the Xerces dependency at
     * compilation time.
     * <p>
     * 
     * @author Paul E. Slauenwhite
     * @version June 2, 2004
     * @since June 2, 2004
     */
    private static class XercesXMLSerializer implements Constants {

        /**
         * Static instance of the
         * <code>org.apache.xml.serialize.OutputFormat</code> formatter used
         * to serialize a Document Object Model (DOM) to an XML document.
         */
        private static Object outputFormat = null;

        /**
         * Serializes a Document Object Model (DOM) to an XML document using a
         * <code>org.apache.xml.serialize.XMLSerializer</code> and writes the
         * XML document to a writer.
         * <p>
         * The parameter Document Object Model (DOM) is serialized to an XML
         * document using a <code>org.apache.xml.serialize.XMLSerializer</code>,
         * which may be potentially formatted and written to a writer.
         * <p>
         * The serialized XML document is formatted (e.g. line breaks and
         * indentation) if the parameter <code>format</code> flag is true.
         * <p>
         * The encoding for the serialized XML document is explicitly set to
         * "UTF-8" for all platforms excluding z/OS and OS/390 platforms. The
         * encoding for the serialized XML document is explicitly set to
         * "IBM-1047" for z/OS and OS/390 platforms only.
         * <p>
         * The <code>org.apache.xml.serialize.*</code> classes required to
         * serialize a Document Object Model (DOM) to an XML document are loaded
         * reflectively at run-time so as not eliminate the Xerces dependency at
         * compilation time.
         * <p>
         * 
         * @param document
         *            The Document Object Model (DOM) to be serialized to a
         *            potentially formatted XML document and written to the
         *            writer.
         * @param writer
         *            The writer where the potentially formatted XML document is
         *            written.
         * @param format
         *            If the serialized XML document is formatted (e.g. line
         *            breaks and indentation).
         * @throws Exception
         *             If an error occurs during serialization.
         */
        private static void serialize(Document document, Writer writer, boolean format) throws Exception {

            if (outputFormat == null) {

                Class outputFormatClass = Class.forName("org.apache.xml.serialize.OutputFormat");

                outputFormat = outputFormatClass.newInstance();

                outputFormatClass.getMethod("setMethod", new Class[] { String.class}).invoke(outputFormat, new Object[] { "xml"});

                if ((OS_NAME.equals("z/OS")) || (OS_NAME.equals("OS/390"))) {
                    outputFormatClass.getMethod("setEncoding", new Class[] { String.class}).invoke(outputFormat, new Object[] { "IBM-1047"});
                } else {
                    outputFormatClass.getMethod("setEncoding", new Class[] { String.class}).invoke(outputFormat, new Object[] { "UTF-8"});
                }

                //Required to align attributes from the same element on the
                // same line:
                outputFormatClass.getMethod("setLineWidth", new Class[] { int.class}).invoke(outputFormat, new Object[] { new Integer(Integer.MAX_VALUE)});
            }

            if (format) {
                outputFormat.getClass().getMethod("setIndent", new Class[] { int.class}).invoke(outputFormat, new Object[] { new Integer(2)});
            } else {
                outputFormat.getClass().getMethod("setIndent", new Class[] { int.class}).invoke(outputFormat, new Object[] { new Integer(0)});
            }

            Class xmlSerializerClass = Class.forName("org.apache.xml.serialize.XMLSerializer");

            Object xmlSerializer = xmlSerializerClass.getConstructor(new Class[] { Writer.class, outputFormat.getClass()}).newInstance(new Object[] { writer, outputFormat});

            xmlSerializerClass.getMethod("serialize", new Class[] { Document.class}).invoke(xmlSerializer, new Object[] { document});
        }
    }
}