package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import org.w3c.dom.CDATASection;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * <code>SVGXYChart</code> is an abstract base class providing implementation of methods
 * common to classes generating SVG graphics which have x,y axes.
 * 
 *  @version 1.51.1.22
 */
public abstract class SVGXYChart extends SVGDocumentGenerator implements IDataInputConstants, Serializable {
	/**
	 * Provides methods for retrieving the data from
	 * the configuration and data documents.
	 */
	protected final XYChartDataRetriever dataRetriever;
	
	// default axis data label font size
	protected final int axisLabelFontSize = DEFAULT_FONTSIZE;
	protected String [] legendFlyOverLabels = null;
	protected DataRange xAxis = null;
	protected DataRange yAxis = null;	
	protected SegmentMarker[] xMarkers = null;
	protected SegmentMarker[] yMarkers = null;
	protected DataSet[] dataSets = null;
		
				
	/**
	 * Sole constructor.	  
	 */ 
	SVGXYChart() {
		super();
		dataRetriever = new XYChartDataRetriever();
	}	
		
	/** 
	 * Adds the JavaScript functions required by this object's graphic. 
	 * 
	 * @param parent element to which the script-related elements are appended
	 * @param numberOfDataSets the number of data sets in the input
	 */
	protected void addJavaScriptFunctions(Document generatedDocument, GraphicAttributes attrs, int numberOfSets, int gridXOffset, int gridYOffset, int xAxisLength, int yAxisLength) {		
		Element parent = generatedDocument.getDocumentElement();	
		super.addJavaScriptFunctions(generatedDocument, attrs, numberOfSets, gridXOffset, gridYOffset, xAxisLength, yAxisLength);
		
		Element newElement = generatedDocument.createElement("script");
		newElement.setAttribute("type", JAVASCRIPT_VERSION);
		parent.appendChild(newElement);
	
		ECMAScriptGenerator jscript = new ECMAScriptGenerator();
		StringBuffer functions = new StringBuffer(jscript.generateXYChartHelperFunctions());				
			
		CDATASection newCDATASection = generatedDocument.createCDATASection(functions.toString());
		newElement.appendChild(newCDATASection);
	}	
	
	/**
	 * Add one text box (with a value inside it)
	 * 
	 * @param parent element to which the textbox-related elements are appended
	 * @param x x-coordinate of the text box
	 * @param y y-coordinate of the text box
	 * @param width width of the text box
	 * @param height height of the text box
	 */
	protected void addEachTextBox(Document generatedDocument, GraphicAttributes attrs, Element parent, double x, double y, double width, double height, String value, int dataSet, int next) {
		String stylesheetClass = "dataValues anchorAtMiddle";
		Element rectElement = generatedDocument.createElement("rect");
		rectElement.setAttribute("x", Double.toString(x));
		rectElement.setAttribute("y", Double.toString(y));
		rectElement.setAttribute("width", Double.toString(width));
		rectElement.setAttribute("height", Double.toString(height));
		rectElement.setAttribute("fill", "#FFFFCC");
		rectElement.setAttribute("stroke", "gray");
		rectElement.setAttribute("stroke-width", "1");
		rectElement.setAttribute("id", "textboxRect" + Integer.toString(dataSet) + Integer.toString(next));
		//rectElement.setAttribute("crossXAxisFlag", crossXAxisFlag);
		parent.appendChild(rectElement);

		// Set the text position inside the box		
		/*
		double textX = x + width / 2.0;
		double textY = y + 13;

		Element textElement = addLabel(generatedDocument, attrs, parent, value, stylesheetClass, textX, textY, 0);
		textElement.setAttribute("id", "textboxText" + dataSet + next);
		*/
		Element textElement = generatedDocument.createElement("text");
		textElement.setAttribute("id", "textboxText" + dataSet + next);
		textElement.setAttribute("style", "text-anchor: middle");
		textElement.setAttribute("class", stylesheetClass);
		//textElement.setAttribute("textLength", Integer.toString(value.length()));
		textElement.appendChild(generatedDocument.createTextNode(value));
		parent.appendChild(textElement);
	}	
	
//========================================================================================================	


	private Element genNamedPolyLine(Document d, String className, String points)
	{
		Element e = d.createElement("polyline");
		e.setAttribute("class", className);
		e.setAttribute("fill", "none");
		e.setAttribute("points", points);
		return e;
	}
	
	private Element genNamedPolyLineGroup(Document d, String id, String className, String points)
	{
		Element e = d.createElement("g");
		e.setAttribute("id", id);
		e.appendChild(genNamedPolyLine(d, className, points));
		return e;
	}

	private Element gen2NamedPolyLineGroup(Document d, String id, String className1, String points1, String className2, String points2)
	{
		Element e = d.createElement("g");
		e.setAttribute("id", id);
		e.appendChild(genNamedPolyLine(d, className1, points1));
		e.appendChild(genNamedPolyLine(d, className2, points2));
		return e;
	}
	
	private Element genUseWithOffsets(Document d, String id, double xOffset, double yOffset)
	{
		Element e = d.createElement("use");
		e.setAttribute("xlink:href", "#"+id);
		e.setAttribute("x", Double.toString(xOffset));
		e.setAttribute("y", Double.toString(yOffset));
		return e;
	}

	/**
	 * Adds the grid for the x,y axes and draws tick lines on them,
	 * positioning them based on the y-axis marker values.
	 * 
	 * @param parent element to which the grid-related elements are appended
	 * @param xAxisLength the length of the xAxis
	 * @param yAxisLength the length of the yAxis
	 * @param gridXOffset the actual x position of the coordinate (0,0)
	 * @param gridYOffset the actual y position of the coordinate (0,0)
	 * @param xAxis the x axis
	 * @param yAxis the y axis
	 */
	protected void addGrid(Document generatedDocument, Element parent, 
	                         int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset,
	                         DataRange xAxis, DataRange yAxis) {
	                         	
		Element defs = generatedDocument.createElement("defs");
		parent.appendChild(defs);
		
		// a few definitions
		// #solidgridline: solid line across chart
		defs.appendChild(genNamedPolyLineGroup(generatedDocument, 
		    "solidgridline", "gridline", "0 0 " + xAxisLength + " 0"));

		// #dashgridline: dash line across chart
		defs.appendChild(genNamedPolyLineGroup(generatedDocument, 
		    "dashgridline", "griddashline", "0 0 " + xAxisLength + " 0"));

		// #tickline: tick line on the y axis
		defs.appendChild(gen2NamedPolyLineGroup(generatedDocument, 
		    "tickline", "grid", "0 0 5 0", "grid", xAxisLength - 5 + " 0 " + xAxisLength + " 0"));

		// #xtickline: tick line on the x axis
		defs.appendChild(genNamedPolyLineGroup(generatedDocument, 
		    "xtickline", "grid", "0 0 0 -5"));


		// Enclose the whole chart with a group to specify offsets
		Element top = generatedDocument.createElement("g");
		parent.appendChild(top);
		top.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		
		// draw the border lines, left, bottom and right
		top.appendChild(genNamedPolyLine(generatedDocument, "grid",
								    	"0 0 " 
									    + "0 " + yAxisLength + " " 
									    + xAxisLength + " " + yAxisLength + " "
									    + xAxisLength + " 0"));

		boolean xIsCategory = (xAxis.getType() == DataRange.CATEGORIZATION);




		// draw the x ticks
		double xMin = xAxis.getMinValue();
		double xMax = xAxis.getMaxValue();
        SegmentMarker[] xsms = xAxis.getSegmentMarkers();
        for(int i=0; i < xsms.length; i++) {
        	SegmentMarker sm = xsms[i];
        	double x = 0.0;
        	if (xIsCategory) {
        		double position = xsms[i].getPosition();
        		if (position < 0) continue;
        		if (position > 1) continue;
        		x = xAxisLength * position;
        	}
        	else {
        	    double value = xsms[i].getValue();
        	    if (value < xMin) continue;
        	    if (value > xMax) continue;
        	    x = xAxisLength * (value - xMin) / (xMax - xMin);
        	}
			top.appendChild(genUseWithOffsets(generatedDocument, "xtickline", x, yAxisLength));
        }

		// draw the y ticks and y solid lines
		double yMin = yAxis.getMinValue();
		double yMax = yAxis.getMaxValue();
        SegmentMarker[] ysms = yAxis.getSegmentMarkers();
        for(int i=0; i < ysms.length; i++) {
        	SegmentMarker sm = ysms[i];
        	double value = ysms[i].getValue();
        	if (value < yMin) continue;
        	if (value > yMax) continue;
        	double y = (int)(yAxisLength * (yMax - value) / (yMax - yMin));
			top.appendChild(genUseWithOffsets(generatedDocument, "tickline", 0, y));
        	//if (sms[i].getType() == SegmentMarker.LINE) {
				top.appendChild(genUseWithOffsets(generatedDocument, "solidgridline", 0, y));
        	//}
        }
		
		// draw the y dash lines
		double[] dashGridYPos = calculateDashGridPositions(ysms);
        for(int i=0; i < dashGridYPos.length; i++) {
        	double value = dashGridYPos[i];
        	if (value < yMin) continue;
        	if (value > yMax) continue;
        	int y = (int)(yAxisLength * (yMax - value) / (yMax - yMin));
			top.appendChild(genUseWithOffsets(generatedDocument, "dashgridline", 0, y));
        }
	}


	protected double[] calculateDashGridPositions(SegmentMarker[] sms) {
		double prevY = 0;
		double yLabelValue = 0;		
		double nextDashY = 0;
		
		if (sms.length <= 1) {
			return new double[0];
		}
		int n = sms.length - 1;
		double[] dashpos = new double[n]; 
		
		double prev = sms[0].getValue();
		for(int i=0; i < n; i++) {
			double value = sms[i+1].getValue();
			dashpos[i] = (prev + value) / 2.0;
			prev = value;
		}
		
		return dashpos;
	}
	                         
	                         
	/**
	 * Adds the titles to the SVG graphic. This includes the main title,
	 * legend title (if appropriate), x-axis label and y-axis label.
	 * 
	 * @param parent element to which the title-related elements are appended
	 * @param xAxisLength the length of the xAxis
	 * @param yAxisLength the length of the yAxis
	 * @param gridXOffset the actual x position of the coordinate (0,0)
	 * @param gridYOffset the actual y position of the coordinate (0,0)
	 * @param xAxis the x axis
	 * @param yAxis the y axis
	 * @param numberOfXLabelRows the maximum number of rows that x labels fit into
	 * @param rangeMaxValue the maximum value of range
	 */	

	protected void addTitles(Document generatedDocument, GraphicAttributes attrs, Element parent, 
	                         int xAxisLength, int yAxisLength, int gridXOffset, int gridYOffset, 
	                         DataRange xAxis, DataRange yAxis,
	                         int numberOfXLabelRows, String rangeMaxValue) {		

		String title, stylesheetClass = null;		
		Element gElement = null;		
		int xAxisLabelYValue = yAxisLength + axisLabelFontSize + 5;	

		// add a font group for the labels and then add the labels
		// x-axis title
		title = xAxis.getLabel();
		if (title != null) {
			stylesheetClass = "xyTitleLabels anchorAtMiddle";
			gElement = generatedDocument.createElement("g");
			gElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
			gElement.setAttribute("id", "labelsValues0");
			gElement.setAttribute("class", stylesheetClass);
			//gElement.setAttribute("font-family", attrs.getGraphicFont());
			parent.appendChild(gElement);				
			// default axis label font size
			//fontSize = DEFAULT_FONTSIZE + LABEL_FONTSIZE_DELTA;
			// double xTitleYPosition = xAxisLabelYValue + fontSize + numberOfXLabelRows * 12;
			double xTitleYPosition = xAxisLabelYValue + (numberOfXLabelRows + 1) * 12;
			
			double graphicHeight = (double)attrs.getGraphicHeight();
			
			if (xTitleYPosition + 14 > graphicHeight) {
				addLabel(generatedDocument, attrs, gElement, title, null, xAxisLength/2, graphicHeight - 8.0, 0);
			} else {
				addLabel(generatedDocument, attrs, gElement, title, null, xAxisLength/2, xTitleYPosition, 0);
			}
		}
		// y-axis title
		title = yAxis.getLabel();
		if (title != null) {
			stylesheetClass = "xyTitleLabels anchorAtMiddle";
			gElement = generatedDocument.createElement("g");
			gElement.setAttribute("transform", "translate(" + 30 + "," + gridYOffset + ")");
			gElement.setAttribute("id", "labelsValues1");
			gElement.setAttribute("class", stylesheetClass);
			//gElement.setAttribute("font-family", attrs.getGraphicFont());
			parent.appendChild(gElement);				
			addLabel(generatedDocument, attrs, gElement, title, null, 0 , yAxisLength / 2.0, 270);
		}
	}	

//========================================================================================================	

    /**
     * Evenly distribute the segment markers of a data range by updating the position attribute of each segment markers
     * 
     * @param dataRange the data range whose segment markers are to be evenly distributed
     */
    protected void evenlyDistributeSegmentMarkers(DataRange dataRange) {
    	if (dataRange == null) return;
    	SegmentMarker[] sms = dataRange.getSegmentMarkers();
    	if (sms == null) return;
    	double w = 1.0 / sms.length;
    	double o = w / 2;
    	for(int i = 0; i < sms.length; i++) {
            sms[i].setPosition(o + w * i);
    	}
    }
    

	/**
	 * Adds the axis data labels to the x- and y-axes.
	 * 
	 * @param generatedDocument the document
	 * @param parent element to which the label-related elements are appended
	 * @param xAxisLength the length of the xAxis
	 * @param yAxisLength the length of the yAxis
	 * @param gridXOffset the actual x position of the coordinate (0,0)
	 * @param gridYOffset the actual y position of the coordinate (0,0)
	 * @param xAxis the x axis
	 * @param yAxis the y axis
	 */
	protected void addAxisLabels(Document generatedDocument, GraphicAttributes attrs, Element parent, 
									int xAxisLength, int yAxisLength, 
									int gridXOffset, int gridYOffset, 
									DataRange xAxis, DataRange yAxis)
	{
		String XstylesheetClass = "dataValues anchorAtMiddle";
		String YstylesheetClass = "dataValues anchorAtEnd";		

		// x axis labels
		Element gElement = generatedDocument.createElement("g");
		parent.appendChild(gElement);
		gElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		gElement.setAttribute("id", "dataValues0");
		gElement.setAttribute("class", XstylesheetClass);
		//gElement.setAttribute("font-family", attrs.getGraphicFont());
		
		int xAxisLabelYValue = yAxisLength + axisLabelFontSize + 5;

		double xMin = xAxis.getMinValue();
		double xMax = xAxis.getMaxValue();
        SegmentMarker[] xsms = xAxis.getSegmentMarkers();

		boolean xIsCategory = (xAxis.getType() == DataRange.CATEGORIZATION);
		
        for(int i=0; i < xsms.length; i++) {
        	SegmentMarker sm = xsms[i];
        	double x = 0.0;
        	if (xIsCategory) {
        		double position = xsms[i].getPosition();
        		if (position < 0) continue;
        		if (position > 1) continue;
        		x = xAxisLength * position;
        	}
        	else {
        	    double value = xsms[i].getValue();
        	    if (value < xMin) continue;
        	    if (value > xMax) continue;
        	    x = xAxisLength * (value - xMin) / (xMax - xMin);
        	}
        	String label = xsms[i].getLabel();
        	if (label == null) {
        		label = attrs.formatNumber(xsms[i].getValue());
        	}
			String[] newXLabels = wrapXLabel(label);

			double y = xAxisLabelYValue;
			for (int j = 0; j < newXLabels.length; j++) {
				addLabel(generatedDocument, attrs, gElement, newXLabels[j], XstylesheetClass, x, y, 0);
				y += 12;            // increment the y position by 12 (move to the next row)
			}


        }
        
        // y axis labels
		gElement = generatedDocument.createElement("g");
		parent.appendChild(gElement);	
		gElement.setAttribute("transform", "translate(" + gridXOffset + "," + gridYOffset + ")");
		gElement.setAttribute("id", "dataValues1");
		gElement.setAttribute("class", YstylesheetClass);
		//gElement.setAttribute("font-family", attrs.getGraphicFont());

		double x = -5;
		
		double yMin = yAxis.getMinValue();
		double yMax = yAxis.getMaxValue();
        SegmentMarker[] ysms = yAxis.getSegmentMarkers();
        double maxY = yAxisLength + (DEFAULT_FONTSIZE/2.0);
        for(int i=0; i < ysms.length; i++) {
        	SegmentMarker sm = ysms[i];
        	double value = ysms[i].getValue();
        	if (value < yMin) continue;
        	if (value > yMax) continue;
        	double y = yAxisLength * (yMax - value) / (yMax - yMin) + (DEFAULT_FONTSIZE/2.0);
        	String label = ysms[i].getLabel();
        	if (label == null) {
        		label = attrs.formatNumber(ysms[i].getValue());
        	}
			addLabel(generatedDocument, attrs, gElement, label, YstylesheetClass, x, y, 0);
		}


	}


	/**
	 * get the number of x axis data labels to the x- and y-axes.
	 * @param generatedDocument the document
	 * @param xAxisLength the length of the xAxis
	 * @param xAxis the x axis
	 * @return the maximum number of rows that x labels fit into
	 */
	protected int getNumberOfAxisLabels(GraphicAttributes attrs, int xAxisLength, DataRange xAxis)
	{
		
		double xMin = xAxis.getMinValue();
		double xMax = xAxis.getMaxValue();
        SegmentMarker[] xsms = xAxis.getSegmentMarkers();

		int maxXLabelRows = 0;
		
		boolean xIsCategory = (xAxis.getType() == DataRange.CATEGORIZATION);
		
        for(int i=0; i < xsms.length; i++) {
        	SegmentMarker sm = xsms[i];
        	double x = 0.0;
        	if (xIsCategory) {
        		double position = xsms[i].getPosition();
        		if (position < 0) continue;
        		if (position > 1) continue;
        		x = xAxisLength * position;
        	}
        	else {
        	    double value = xsms[i].getValue();
        	    if (value < xMin) continue;
        	    if (value > xMax) continue;
        	    x = xAxisLength * (value - xMin) / (xMax - xMin);
        	}
        	String label = xsms[i].getLabel();
        	if (label == null) {
        		label = attrs.formatNumber(xsms[i].getValue());
        	}
			String[] newXLabels = wrapXLabel(label);

			if (newXLabels.length > maxXLabelRows) {
				maxXLabelRows = newXLabels.length;	
			}
        }
 	    return maxXLabelRows;
	}

	
	/**
	 * Text wrapping: wrap the x label into rows if the width of the label is too long
	 * @param label x Label
	 */
	private String[] wrapXLabel(String label) {
		ArrayList wrapStrings = new ArrayList();

		StringTokenizer st = new StringTokenizer(label, " ");
		while (st.hasMoreElements()) {
			wrapStrings.add(st.nextElement().toString());
		}
		
		return (String[])wrapStrings.toArray(new String[wrapStrings.size()]);
	}
	
	/**
	 * Calculate the y position of all dash grid lines
	 * 
	 * @param yAxisLabels the set of y-axis range marker values
	 */
	protected Vector calculateDashGridYPositions(String [] yAxisLabels) {
		double prevY = 0;
		double yLabelValue = 0;		
		double nextDashY = 0;
		Vector dashY = new Vector(); 
		
		// Calculate the y position for each dashed grid line
		for (int i = 0; i < yAxisLabels.length; i++) {	
			// NumberFormatException not possible here
			yLabelValue = Double.parseDouble(yAxisLabels[i]);
			nextDashY = (prevY + yLabelValue) / 2.0;
			dashY.add(Double.toString(nextDashY));
			prevY = yLabelValue;
		}
		return dashY;
	}
	
	/**
	 * Retrieves all configuration attributes from the data retriever
	 */
	protected void getConfigAttributes(GraphicAttributes attrs) throws DataInputProcessingException {
		Hashtable configProperties = dataRetriever.getConfigurationAttributes(attrs.getConfigDocument(), attrs.getDataDocument());	
		if (configProperties == null) {
			return;
		}	
		String width = (String)configProperties.get(WIDTH);
		String height = (String)configProperties.get(HEIGHT);
		//String fontFamily = (String)configProperties.get(FONT);
		String title = (String)configProperties.get(TITLE);
		String timestampPrefix = (String)configProperties.get(TIMESTAMP_PREFIX);
		//String suppressLegend = (String)configProperties.get(SUPPRESS_LEGEND);
		String legendTitle = (String)configProperties.get(LEGEND_TITLE);
					
		if (width != null) {
			attrs.setGraphicWidth(width);
		}	
		
		if (height != null) {
			attrs.setGraphicHeight(height);
		}	
		
		/*
		if (fontFamily != null) {
			attrs.setGraphicFont(fontFamily);			
		}			
		*/
		
		if (title != null) {
			attrs.setGraphicTitle(title);			
		}
			
		if (timestampPrefix != null) {
			attrs.setGraphicTimestampPrefix(timestampPrefix);			
		}	
		
		//if (suppressLegend != null) {
		//	attrs.setLegendSuppressed(suppressLegend);			
		//}	
		
		if (legendTitle != null) {
			attrs.setGraphicLegendTitle(legendTitle);			
		}	
		/* Test 

		System.out.println("From the input documents Graphic Color Palette: " + attrs.getGraphicColorPalette());
		System.out.println("From the input documents Graphic Color Palette Name: " + attrs.getGraphicColorPaletteName());
		System.out.println("From the input documents Graphic Font: " + attrs.getGraphicFont());
		System.out.println("From the input documents Graphic Height: " + attrs.getGraphicHeight());
		System.out.println("From the input documents Graphic Legend Title: " + attrs.getGraphicLegendTitle());
		System.out.println("From the input documents Graphic Timestamp Prefix: " + attrs.getGraphicTimestampPrefix());
		System.out.println("From the input documents Graphic Title: " + attrs.getGraphicTitle());
		System.out.println("From the input documents Graphic Type: " + attrs.getGraphicType());
		System.out.println("From the input documents Graphic Width: " + attrs.getGraphicWidth());
		System.out.println("From the input documents Preferences Icon Text: " + attrs.getPreferencesIconText());
		System.out.println("From the input documents Preferences Page: " + attrs.getPreferencesPage());
		System.out.println("From the input documents Preferences Page Height: " + attrs.getPreferencesPageHeight());
		System.out.println("From the input documents Preferences Page Width: " + attrs.getPreferencesPageWidth());
		/* Test */	
	}
	
	/**
	 * Resets the horizontal or vertical offset of the default graphics such that
	 * the graph is centered in the chart.	 
	 * 
	 * @param dimensionLength length of the graphic dimension 
	 * @param axisLength length of the axis
	 * 
	 * @return int new offset value
	 */
	protected int resetGridOffsetForCentering(short dimensionLength, int axisLength) {
		return (int)((dimensionLength - axisLength) / 2.0);
	}
	
		/**
	 * Resets the horizontal or vertical offset of the default graphics such that
	 * the graph is centered in the chart.	 
	 * 
	 * @param dimensionLength length of the graphic dimension 
	 * @param axisLength length of the axis
	 * @param offset for the graph center
	 * 
	 * @return int new offset value
	 */
	protected int resetGridOffsetForCenter(short dimensionLength, int axisLength, double offset) {
		return (int)((dimensionLength - axisLength) / offset);
	}

	/**
	 * Calculate the area the Y markers needs
	 * @return double the maximum width of the Y markers
	 */
	protected double getMaxLengthOfYmarker(SegmentMarker[] yMarkers, GraphicAttributes graphicAttrs) {
		int maxLengthOfYmarker = 0;
		for(int i = 0; i < yMarkers.length; i++)
		{
			String label = yMarkers[i].getDisplayLabel(graphicAttrs);
			if(label != null) {
				int length = label.length();
				if (maxLengthOfYmarker < length) {
					maxLengthOfYmarker = length;
				}
			}
		}
		
		//Calculate the max size of the Y marker text width in pixel. 16 gap between Header and text and 6.0 is a fond size.
		return (double)(16 + (maxLengthOfYmarker)  * 6.0);
	}	

}
