/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.launcher;

import java.util.*;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.*;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.ui.internal.launcher.*;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.util.*;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.ui.HyadesUIPlugin;

/**
 * 
 * @author znensi
 *
 * This class is used to store changes to the LCD, before the user hits apply.
 * Necessary for the case when user reverts, or cancel.
 * During a save action, this manager copy instance gets sent to the manager.
 * During a cancel action, the manager copy info is not persisted.
 */
public class ProfilingSetsManagerCopy extends ProfilingSetsManager {
	
	/**
	 * This class will be used to keep temporary information ragarding profiling sets
	 */
	public ProfilingSetsManagerCopy() {
		super();
		initialize();
	}
	
	public void initializeFrom(ILaunchConfiguration conf)
	{
		String defaultProfilingSet = getDefaultSet().getId();
		String defaultFilterSet = getDefaultFilterSet().getName();
		
		try {
			String profilingSetId = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, defaultProfilingSet);
			String filterSetName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET, defaultFilterSet);
			
			IProfilingSet profilingSet = (IProfilingSet)getProfilingSets().get(profilingSetId);
			FilterSetElement filterSet = (FilterSetElement)getFilterSets().get(filterSetName);
			
			if (profilingSet == null) {
				Map map = getProfilingSets();
				if (map.size() > 0) {
					profilingSet = (IProfilingSet)map.values().iterator().next();
				}
			}
			
			if (filterSet == null) {
				Map map = getFilterSets();
				if (map.size() > 0) {
					filterSet = (FilterSetElement)map.values().iterator().next();
				}
			}
			
			setDefaultSet(profilingSet);
			setDefaultFilterSet(filterSet);
		}
		catch(CoreException exc) {
			HyadesUIPlugin.logError(exc);
		}
	}
	
	public static ProfilingSetsManager instance()
	{
		//this method is not supposed to be called on a copy instance
		return null;
	}
	
	/**
	 * Apply any changes to the main profiling set manager
	 */
	public void applyChanges()
	{
		//update profiling set manager
		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		manager.getProfilingSets().clear();
		manager.getProfilingSets().putAll(getProfilingSets());
		
		manager.getProfilingTypes().clear();
		manager.getProfilingTypes().putAll(getProfilingTypes());
		
		manager.getProfilingTypesGroups().clear();
		manager.getProfilingTypesGroups().putAll(getProfilingTypesGroups());
		
		manager.getFilterSets().clear();
		manager.getFilterSets().putAll(getFilterSets());
		
		manager.setDefaultSet(getDefaultSet());
		manager.setDefaultFilterSet(getDefaultFilterSet());
		
		writeSetsToPreferences();				
	}
		
	protected void initializeProfilingTypesGroups()
	{
		profilingTypesGroups = new HashMap();
		
		Map groups = ProfilingSetsManager.instance().getProfilingTypesGroups();
		Iterator keys = groups.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSetTypeGroup group = (IProfilingSetTypeGroup) groups.get(key);
			profilingTypesGroups.put(key, new ProfilingSetTypeGroup(group.getId(), group.getName(), group.getDescription(), group.getIcon()));
		}		
	}
	
	protected void initializeProfilingTypes()
	{
		profilingTypes = new HashMap();
		
		Map types = ProfilingSetsManager.instance().getProfilingTypes();
		Iterator keys = types.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSetType type = (IProfilingSetType) types.get(key);
			profilingTypes.put(key, 
					    new ProfilingSetType(type.getId(), type.getName(), type.getDescription()
					    		, type.getImage(), type.isUseFilters(), type.getGroup()
								, type.getProfilingType()));
		}		
	}

	protected void initializeProfilingSets()
	{
		profilingSets = new HashMap();
		
		IProfilingSet defaultSet = ProfilingSetsManager.instance().getDefaultSet();
		Map sets = ProfilingSetsManager.instance().getProfilingSets();
		Iterator keys = sets.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSet set = (IProfilingSet) sets.get(key);
			
			IProfilingSet copy = new ProfilingSet(set.getId(), set.getName(), set.getDescription());
            copy.getProfilingTypes().addAll(set.getProfilingTypes());
            copy.getAttributes().putAll(set.getAttributes());
			
			profilingSets.put(key, copy);
			
			if (set == defaultSet) {
				setDefaultSet(copy);
			}
		}				
	}
	
	protected void initializeFilterSets()
	{
		filterSets = new HashMap();
		
		Map sets = ProfilingSetsManager.instance().getFilterSets();
		Iterator keys = sets.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			FilterSetElement set = (FilterSetElement) sets.get(key);
			
			FilterSetElement copy = new FilterSetElement(set.getName());
			ArrayList children = set.getChildren();
			ArrayList ccopy = new ArrayList();
			for(int idx=0; idx<children.size(); idx++)
			{
				FilterTableElement child = (FilterTableElement)children.get(idx);
				ccopy.add(new FilterTableElement(child.getText(), child.getMethod(), child.getVisibility()));
			}
			copy.setChildren(ccopy);
			
			filterSets.put(key, copy);
		}		
	}
	
	public void performApply(ILaunchConfigurationWorkingCopy wc)
	{
		IProfilingSet set = getDefaultSet();

		if(set != null) {
			FilterSetElement filterSet = getDefaultFilterSet();
			String filterSetName = null;
			if (filterSet != null) {
				filterSetName = filterSet.getName();
			}
			
			/*
			 * #59620: All we should be saving in the launch configuration
			 * is the reference to the profiling and filter set, not their
			 * actual contents.
			 */
			wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, set.getId());
			wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET, filterSetName);
		}
	}
	
	public ProfilingSetsManagerCopy createCopy() {
		ProfilingSetsManagerCopy copy = new ProfilingSetsManagerCopy();
		copy.resetFrom(this);
		return copy;
	}
	
	public void resetFrom(ProfilingSetsManagerCopy other) {
		getProfilingSets().clear();
		getProfilingSets().putAll(other.getProfilingSets());
		
		getProfilingTypes().clear();
		getProfilingTypes().putAll(other.getProfilingTypes());
		
		getProfilingTypesGroups().clear();
		getProfilingTypesGroups().putAll(other.getProfilingTypesGroups());
		
		getFilterSets().clear();
		getFilterSets().putAll(other.getFilterSets());
		
		setDefaultSet(other.getDefaultSet());
		setDefaultFilterSet(other.getDefaultFilterSet());
	}
}
