/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.control;

import java.lang.reflect.Constructor;
import java.text.MessageFormat;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolCell;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.edit.datapool.IDatapoolFactory;
import org.eclipse.hyades.edit.datapool.IDatapoolListener;
import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.hyades.edit.datapool.IDatapoolSuggestedType;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.test.ui.datapool.DatapoolPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.action.CopyAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.CutAction;
import org.eclipse.hyades.test.ui.datapool.internal.action.PasteAction;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolAddDialogSimple;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolColumnDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolDeleteColumnDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolRowDialog;
import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolRowGroupDialog;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPart;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IDatapoolPartExtended;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IValidateValueClass;
import org.eclipse.hyades.test.ui.datapool.internal.interfaces.IValueClassFactory;
import org.eclipse.hyades.test.ui.datapool.internal.util.TypeChecker;
import org.eclipse.hyades.test.ui.datapool.internal.util.ValueClassMap;
import org.eclipse.hyades.test.ui.datapool.internal.util.ValueObject;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellEditorListener;
import org.eclipse.jface.viewers.ICellEditorValidator;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.ACC;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.ControlEditor;
import org.eclipse.swt.custom.TableCursor;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.IKeyBindingService;
import org.eclipse.ui.IWorkbenchPartSite;

/**
 * A Datapool Table is a Excel like table editor that wraps the JFace 
 * TableViewer and SWT Table widget and edits the DPLDatapool object.  It 
 * provides the ability to insert, delete, and edit variables, equivalence 
 * classes, and records.  A varible is represented as a column in the 
 * table.  An equivalence class is a grouping of rows with the same name,
 * and a record is a single row in the table.  The table enforces that an 
 * equivalence class always has one or more records. 
 * 
 * @author psun
 */
public class DatapoolTable implements IMenuListener, SelectionListener, IDatapoolListener {

	private static final String START_LITERAL = "<Literal>"; //$NON-NLS-1$
	private static final String END_LITERAL = "</Literal>"; //$NON-NLS-1$
	private static final String TAG_DATA = "data"; //$NON-NLS-1$
	private static final String TAG_RECORD = "record"; //$NON-NLS-1$
	private static final String TAG_VARIABLE = "variable"; //$NON-NLS-1$
	private static final String TAG_VARIABLE_INDEX = "variableIndex"; //$NON-NLS-1$
	private static final String TAG_EQUIVALENCE_CLASS = "equivalenceClass"; //$NON-NLS-1$
	
	private static final int MAX_TABLE_WIDTH = 60;
	private static final int MAX_TABLE_HEIGHT = 60;
		 
	private IDatapool datapool = null;
	private Table table = null;
	private DatapoolTableUtil tableUtil = null;
	private TableViewer viewer = null;
	private Composite parent = null;
	private Menu contextMenu = null;
	private TableCursor cursor = null;
	private ControlEditor controlEditor = null;
	private IDatapoolPart datapoolPart = null;
	private IDatapoolFactory datapoolFactory = null;
	private CellEditor cellEditor = null;
	private ValueObject valueObject = null;
	private DatapoolMenuManager datapoolMenuManager = null;
	private boolean ignoreECNotification = false;
	private boolean controlKeyDown = false;
	private boolean altKeyDown = false;
	private boolean alreadySaved = false;
	private boolean isInputValid = true;
	private boolean showEquivlenceClasses = true;
	private boolean showVariables = true;
	private boolean showRecords = true;
	private boolean isF2Mode = false;
	
	private FocusListener focusListener;
	private ICellEditorListener	cellEditorListener;

	private String vendorID = null;
	
	private CutAction cutAction;
	private CopyAction copyAction;
	private PasteAction pasteAction;
	
	private void clearCellEditor()
	{
		//clearCellEditor(this.cellEditor); // this doesn't assign this.cellEdior to null.
		if(this.cellEditor != null)
		{
			this.cellEditor.dispose();
			this.cellEditor = null;
		}
		if(this.valueObject != null)
		{
			this.valueObject = null;
		}
		
		if(cursor != null && !cursor.isDisposed())
			cursor.setFocus();
	}

	private void clearCellEditor(CellEditor cellEditor)
	{
		if(cellEditor != null)
		{
			cellEditor.dispose();
			cellEditor = null;
		}
		if(valueObject != null)
		{
			valueObject = null;
		}
		if(cursor != null && !cursor.isDisposed())
			cursor.setFocus();
	}

	/*
	 * Handles the selection of the header area.
	 * <p>Clicking on the header area bring up the dialog to change and/or 
	 * move the selected variable/column.
	 */
	protected SelectionListener headerListener = new SelectionAdapter()
	{
		public void widgetSelected(SelectionEvent event)
		{
			datapoolPart.notifyEdit();
			if(datapoolPart.isReadOnly())
				return;
			clearCellEditor();
			TableColumn tableColumn = (TableColumn) event.widget;
			editColumnAux(tableColumn);
		}		
	};

	/*
	 * Handles the resizing of scroll bars when the ScrollableComposite 
	 * containing the table is resized.  The datapool table uses the 
	 * composite's scroll bars. 
	 */
	private ControlListener resizeListener = new ControlAdapter()
	{
		public void controlMoved(ControlEvent e) 
		{
		}

		public void controlResized(ControlEvent e) 
		{
			//resetScrollBarSize(true);
		}				
	};

	/*
	 * Handles the resizing of scroll bars when columns in the table are 
	 * resized.  
	 */
	protected ControlListener resizeColumnListener = new ControlAdapter()
	{
		public void controlMoved(ControlEvent e) 
		{
		}

		public void controlResized(ControlEvent e) 
		{
			if(cursor != null && !cursor.isDisposed())
			{
				if(table.getSelectionIndex() >= 0 && cursor.getColumn() >= 0)
					cursor.setSelection(table.getSelectionIndex(), cursor.getColumn());
			}
			//resetScrollBarSize(false);
		}				
	};

	class TableCellEditorAdapter implements ICellEditorListener
	{
		private CellEditor editor = null;
		private ValueObject valueObject = null;
		private TableItem row = null;
		private int rowIndex = -1;
		private int column = -1;
		
		public TableCellEditorAdapter(CellEditor cellEditor, ValueObject valueObject, TableItem row, int column)
		{
			this.editor = cellEditor;
			this.valueObject = valueObject;
			this.row = row;
			this.column = column;
			if(row != null)
				this.rowIndex = table.indexOf(row);
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ICellEditorListener#applyEditorValue()
		 */
		public void applyEditorValue() 
		{
			isF2Mode = false;
			if(!isInputValid)
			{
				return;
			}
			applyEditingValue(true);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ICellEditorListener#cancelEditor()
		 */
		public void cancelEditor() 
		{
			isF2Mode = false;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ICellEditorListener#editorValueChanged(boolean, boolean)
		 */
		public void editorValueChanged(boolean oldValidState, boolean newValidState) 
		{
			if(newValidState != true)
				isInputValid = false;
			else 
				isInputValid = true;
			editor.getControl().redraw();
		}
	}
	
	class TableCellPaintAdapter implements PaintListener
	{
		/**
		 * @see org.eclipse.swt.events.PaintListener#paintControl(PaintEvent)
		 */
		public void paintControl(PaintEvent event)
		{
			if( isInputValid == false )
				drawPolyline( event.gc, (Widget)event.getSource() );
			else
				event.gc.drawPolyline( new int[0] );
		}
		
		private void drawPolyline(GC gc, Widget widget ) 
		{
			if (gc == null) 
				return;
					
			Rectangle rect = gc.getClipping();
			String str = "";
			if( widget instanceof Text )
			{
				str = ((Text)widget).getText();
			}
			else if( widget instanceof CCombo )
			{
				str = ((CCombo)widget).getText();
			}
			else if( widget instanceof Combo )
			{
				str = ((Combo)widget).getText();
			}
			else
				return;

			int offset = 2;
			char c;
			for( int i = 0; i < str.length(); i++ )
			{
				c = str.charAt( i );
				if( Character.isWhitespace( c ) )
					offset += gc.getAdvanceWidth( c );
				else
					break;
			}

			int length = gc.textExtent( str.trim() ).x;
			
			Point left = new Point( offset, 0 );				
			Point right = new Point( (length + offset + 2), 0 );
			
			gc.setForeground(Display.getCurrent().getSystemColor( SWT.COLOR_RED ));
			int[] polyline= computePolyline(left, right, rect.height );//gc.getFontMetrics().getHeight());
			gc.drawPolyline(polyline);
		}
		
		private int[] computePolyline(Point left, Point right, int height) 
		{		
			final int WIDTH= 4; // must be even
			final int HEIGHT= 2; // can be any number
			
			int leftX= left.x;
			int peeks= (right.x - left.x) / WIDTH;
					
			// compute (number of point) * 2
			int length= ((2 * peeks) + 1) * 2;
			if (length < 0)
				return new int[0];
				
			int[] coordinates= new int[length];
			
			// cache peeks' y-coordinates
			int bottom= left.y + height - 1;
			int top= bottom - HEIGHT;
			
			// populate array with peek coordinates
			for (int i= 0; i < peeks; i++) {
				int index= 4 * i;
				coordinates[index]= leftX + (WIDTH * i);
				coordinates[index+1]= bottom;
				coordinates[index+2]= coordinates[index] + WIDTH/2;
				coordinates[index+3]= top;
			}
			
			// the last down flank is missing
			coordinates[length-2]= left.x + (WIDTH * peeks);
			coordinates[length-1]= bottom;
			
			return coordinates;
		}

	}
	
	/*
	 * Handles the keystrokes of the cell editor when it is in edit mode.
	 */
	class TableCellKeyAdapter implements KeyListener
	{
		private CellEditor editor = null;
		private ValueObject valueObject = null;
		private TableItem row = null;
		private int rowIndex = -1;
		private int column = -1;
		
		public TableCellKeyAdapter(CellEditor cellEditor, ValueObject valueObject, TableItem row, int column)
		{
			this.editor = cellEditor;
			this.valueObject = valueObject;
			this.row = row;
			this.column = column;
			if(row != null)
				this.rowIndex = table.indexOf(row);
		}
		
		/* 
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */		
		public void keyPressed(KeyEvent e) 
		{
			if(datapool == null)
				return;
			// Close the text editor and copy the data over when the user hits
			// "ENTER" or any of the arrow keys.
			if(e.character == SWT.CR)
			{
				if(!isInputValid)
				{
					e.doit = false;
					return;
				}
				return;
			}

			// Bring up the context menu
			if(e.keyCode == SWT.F10 && (e.stateMask & SWT.SHIFT) != 0)
			{
				if(contextMenu == null || contextMenu.isDisposed())
				{
					createMenu();
				}
				contextMenu.setEnabled(true);
				contextMenu.setVisible(true);
				return;
			}

			// Handle behavior when the arrow keys are hit.
			if (!isF2Mode &&
				(e.keyCode == SWT.ARROW_DOWN ||
				 e.keyCode == SWT.ARROW_UP ||
				 e.keyCode == SWT.ARROW_LEFT ||
				 e.keyCode == SWT.ARROW_RIGHT)) 
			{
				if(!isInputValid)
				{
					e.doit = false;
					return;
				}				
				applyEditingValue(true);
				// Move the cursor if any of the arrow keys are hit.  This is to
				// simulate excel behavior.
				if(cursor != null && !cursor.isDisposed())
				{
					if(e.keyCode == SWT.ARROW_DOWN)
					{
						if(rowIndex + 1 < table.getItemCount())
						{
							table.setSelection(rowIndex + 1);
							cursor.setSelection(rowIndex + 1, column);	
						}
					}
					if(e.keyCode == SWT.ARROW_UP)
					{
						if(rowIndex > 0)
						{
							table.setSelection(rowIndex - 1);
							cursor.setSelection(rowIndex - 1, column);	
						}
					}					
					if(e.keyCode == SWT.ARROW_RIGHT)
					{
						if(column + 1 < tableUtil.getColumnCount())
						{
							cursor.setSelection(rowIndex, column + 1);
						}
					}
					if(e.keyCode == SWT.ARROW_LEFT)
					{
						if(column > 0)
						{
							cursor.setSelection(rowIndex, column - 1);
						}
					}
					cursor.setFocus();
				}
				return;				
			}
		}	

		/* 
		 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
		 */
		public void keyReleased(KeyEvent e) 
		{			
			if (e.character == SWT.ESC)
			{
				clearCellEditor(editor);
				return;
			}	
			if (e.keyCode == SWT.ARROW_DOWN ||
				e.keyCode == SWT.ARROW_UP ||
				e.keyCode == SWT.ARROW_LEFT ||
				e.keyCode == SWT.ARROW_RIGHT) 
			{
				if(!isInputValid)
				{
					e.doit = false;
					return;
				}
			}
		}			
	}

	class TableCellValidator implements ICellEditorValidator
	{

		IValidateValueClass validator = null;
		
		public TableCellValidator(IValidateValueClass validator)
		{
			this.validator = validator;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ICellEditorValidator#isValid(java.lang.Object)
		 */
		public String isValid(Object value) 
		{
			if(this.validator == null)
				return null;
			if(this.validator.checkValue(value))
				return null;
			else
			{
				Error error = this.validator.getError();
				if(error != null)
					return error.toString();
				else 
					return "Error"; //$NON-NLS-1$
			}
		}		
	}
	
	class CursorKeyAdapter implements KeyListener
	{
		private CellEditor editor = null;
		private TableCursor cursor = null;
		private IDatapoolPart datapoolPart = null;
		
		public CursorKeyAdapter(TableCursor cursor, CellEditor cellEditor, IDatapoolPart datapoolPart)
		{
			this.editor = cellEditor;
			this.cursor = cursor;
			this.datapoolPart = datapoolPart;
		}
		
		public void keyPressed(KeyEvent e)
		{
			if(datapool == null)
				return;
			// Bring up the context menu
			if(e.keyCode == SWT.F10 && (e.stateMask & SWT.SHIFT) != 0)
			{
				if(contextMenu == null || contextMenu.isDisposed())
				{
					createMenu();
				}
				contextMenu.setEnabled(true);
				contextMenu.setVisible(true);
				return;
			}
			
			if(e.keyCode == SWT.ALT)
			{
				altKeyDown = true;
			}
			if(e.keyCode == SWT.CTRL)
			{
				controlKeyDown = true;
				DatapoolTable.this.connectCutCopyPasteActions(DatapoolTable.this.datapoolPart);
			}
			int selectedColumnIndex = this.cursor.getColumn();
			TableItem row = this.cursor.getRow();

			if(e.keyCode == SWT.ESC || 
				e.keyCode == SWT.CR ||
				e.keyCode == SWT.ALT ||
				e.keyCode == SWT.PAGE_DOWN ||
				e.keyCode == SWT.PAGE_UP ||
				e.keyCode == SWT.HOME ||
				e.keyCode == SWT.END ||
				e.keyCode == SWT.CTRL ||
				e.keyCode == SWT.SHIFT ||
				e.keyCode == SWT.F1 ||
				(e.keyCode >= SWT.F3 && e.keyCode <= SWT.F12))
			{
				return;
			}
			if(!isF2Mode &&
			    e.keyCode == SWT.ARROW_DOWN ||
			    e.keyCode == SWT.ARROW_UP ||
				e.keyCode == SWT.ARROW_LEFT ||
				e.keyCode == SWT.ARROW_RIGHT)
			{
				if(!isInputValid)
					return;
				clearCellEditor();
				int selectedRowIndex = table.indexOf(row);
				// Code to prevent the cursor from moving too far to the
				// right.  This is to accomidate the fact that when a 
				// table column deleted, it is really hidden as a zero sized
				// column to the right
				if(e.keyCode == SWT.ARROW_DOWN)
				{
					int maxRowIndex = table.getItemCount();
					if(selectedRowIndex < maxRowIndex)
						table.setSelection(selectedRowIndex);
				}
				if(e.keyCode == SWT.ARROW_UP)
				{
					if(selectedRowIndex >= 0)
						table.setSelection(selectedRowIndex);
				}
				if(e.keyCode == SWT.ARROW_RIGHT)
				{
					int visibleColumnCount = tableUtil.getColumnCount();
					if(selectedColumnIndex > visibleColumnCount - 1)
						cursor.setSelection(selectedRowIndex, selectedColumnIndex - 1);
				}
				return;	
			}
			
			datapoolPart.notifyEdit();
			if(this.datapoolPart.isReadOnly())
				return;
			if(e.keyCode == SWT.INSERT && (e.stateMask & SWT.CTRL) != 0)
			{
				if(selectedColumnIndex == 0)
					insertRowGroup();
				if(selectedColumnIndex > 0)
					insertColumn();
				return;
			}
			if(e.keyCode == SWT.DEL && (e.stateMask & SWT.CTRL) != 0)
			{
				if(selectedColumnIndex == 0)
					deleteRowGroup();
				if(selectedColumnIndex > 0)
					deleteColumn();
				return;
			}
			if(((e.stateMask & SWT.CTRL) != 0) ||
				((e.stateMask & SWT.ALT) != 0))
			{
				return;
			}
			if(e.keyCode == SWT.INSERT)
			{
				if(selectedColumnIndex == 0)
					insertRow();
				if(selectedColumnIndex <= 0)
					return;
			}
			if(e.keyCode == SWT.DEL)
			{
				if(selectedColumnIndex == 0)
					deleteRow();
				if(selectedColumnIndex > 0)
					clearCell();
				return;
			}
			else 
			{
				char inputCharacter = e.character;

				// No editing of equivalence class/record name;
				if(selectedColumnIndex == 0)
				{
					editRowAux(row);
					return;					
				}
				
				startCellEditing(row, selectedColumnIndex);
				if(cellEditor instanceof TextCellEditor)
				{
					if(e.keyCode == SWT.BS)
						cellEditor.setValue(new String());
					else if(e.keyCode == SWT.F2)
					{
						String currentText = row.getText(selectedColumnIndex);
						cellEditor.setValue(currentText);
						((Text)(cellEditor.getControl())).setSelection(currentText.length());
						isF2Mode = true;
					}
					else 
					{
						String characterValue = String.valueOf(inputCharacter);
						if(cellEditor.getValidator() != null && cellEditor.getValidator().isValid(characterValue) != null)
						{
							isInputValid = false;
						}
						cellEditor.setValue(characterValue);
						((Text)(cellEditor.getControl())).setSelection(2);
					}
				}

			}
			return;
		}
		
		public void keyReleased(KeyEvent e) 
		{
			if(e.keyCode == SWT.CTRL)
			{
				controlKeyDown = false;
				return;
			}
			if(altKeyDown)
			{
				altKeyDown = false;
				return;				
			}
//			special handling on linux where keyPressed() doesn't get this event.
			if(e.keyCode == SWT.INSERT && (e.stateMask & SWT.CTRL) != 0)
			{
				int selectedColumnIndex = this.cursor.getColumn();
				if(selectedColumnIndex == 0)
					insertRowGroup();
				if(selectedColumnIndex > 0)
					insertColumn();
				return;
			}
		}		
	}
		
	class CursorMouseAdapter implements MouseListener
	{
		private CellEditor editor = null;
		private TableCursor cursor = null;
		private IDatapoolPart datapoolPart = null;
		
		public CursorMouseAdapter(TableCursor cursor, CellEditor cellEditor, IDatapoolPart datapoolPart)
		{
			this.editor = cellEditor;
			this.cursor = cursor;
			this.datapoolPart = datapoolPart;
		}
		
		// Double clicking in a cell creates an editor and begins editing 
		// for that cell
		public void mouseDoubleClick(MouseEvent event) 
		{
			if(datapool == null)
				return;
			datapoolPart.notifyEdit();
			if(datapoolPart.isReadOnly())
				return;				
			TableItem row = this.cursor.getRow();
			int column = this.cursor.getColumn();

			// No editing of equivalence class/record name;
			if(column == 0)
			{
				editRowAux(row);
				return;					
			}

			startCellEditing(row, column);
			if(DatapoolTable.this.cellEditor != null)
				DatapoolTable.this.isF2Mode = true;
		}
	
		public void mouseDown(MouseEvent event) 
		{	
			if(event.button != 1)				
				return;
			clearCellEditor();
		}
		
		public void mouseUp(MouseEvent event) 
		{
			// Bring up the context menu if the user hits the right mouse
			// button on a cursor.
			if(event.button == 3)
			{
				contextMenu.setEnabled(true);
				contextMenu.setVisible(true);
			}
		}	
	}
	
	class CursorSelectionAdapter implements SelectionListener
	{
		private CellEditor editor = null;
		private TableCursor cursor = null;
		private IDatapoolPart datapoolPart = null;
		
		public CursorSelectionAdapter(TableCursor cursor, CellEditor cellEditor, IDatapoolPart datapoolPart)
		{
			this.editor = cellEditor;
			this.cursor = cursor;
			this.datapoolPart = datapoolPart;
		}
		
		// When the TableEditor is over a cell, select the corresponding row in 
		// the table.
		public void widgetSelected(SelectionEvent event) 
		{
			if(datapoolPart.isReadOnly())
				return;
			clearCellEditor();
			if(!isInputValid)
				return;
			table.setSelection(new TableItem[] {cursor.getRow()});
			//setScrollBars();
			if(datapoolMenuManager != null)
				setMenuMode(datapoolMenuManager);
			
		}
		
		// Create an editor when the user hits "ENTER".
		public void widgetDefaultSelected(SelectionEvent event)
		{
			clearCellEditor();
			TableItem row = this.cursor.getRow();
			int columnIndex = this.cursor.getColumn();
			int rowIndex = table.indexOf(row);
			int maxRowIndex = table.getItemCount();
			if(rowIndex + 1 < maxRowIndex)
			{
				cursor.setSelection(rowIndex + 1, columnIndex);		
				table.setSelection(rowIndex + 1);
			}
			else
				return;
			//setScrollBars();			
		}
	}

	class TableMouseAdapter implements MouseListener
	{

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseListener#mouseDoubleClick(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseDoubleClick(MouseEvent event) 
		{
			
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseDown(MouseEvent event) 
		{		
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseListener#mouseUp(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseUp(MouseEvent event) 
		{
			if(event.button == 3)
			{
				if(contextMenu.isDisposed())
				{
					createMenu();
					contextMenu.setEnabled(true);
					contextMenu.setVisible(true);
				}
			}				
		}
		
	}
	
	/**
	 * Default constructor for the Datapool Table.  Registers various key, 
	 * menu, and selection listeners.  Creates context menu and populates 
	 * the table.
	 * 
	 * @param parent
	 * @param datapoolPart
	 * @param datapool
	 * @param datapoolFactory
	 * @param vendorID
	 */
	public DatapoolTable(Composite parent, 
			 IDatapoolPart datapoolPart, 
			 IDatapool datapool, 
			 IDatapoolFactory datapoolFactory, 
			 String vendorID)
	{
		this.vendorID = vendorID;
		ValueClassMap.setVendorID(vendorID);
		DatapoolTableConstructor(parent, datapoolPart, datapool, datapoolFactory);
	}
	
	
	
	/**
	 * Default constructor for the Datapool Table.  Registers various key, 
	 * menu, and selection listeners.  Creates context menu and populates 
	 * the table.
	 * 
	 * @param parent
	 * @param datapoolPart
	 * @param datapool
	 * @param datapoolFactory
	 */
	public DatapoolTable(Composite parent, 
						 IDatapoolPart datapoolPart, 
						 IDatapool datapool, 
						 IDatapoolFactory datapoolFactory)
	{
		DatapoolTableConstructor(parent, datapoolPart, datapool, datapoolFactory);
	}

	/**
	 * @param parent
	 * @param datapoolPart
	 * @param datapool
	 * @param datapoolFactory
	 */
	private void DatapoolTableConstructor(Composite parent, IDatapoolPart datapoolPart, IDatapool datapool, IDatapoolFactory datapoolFactory) {
		if(parent == null || datapoolPart == null || datapoolFactory == null)
			return;
		this.datapool = datapool;
		this.datapoolFactory = datapoolFactory;
		
		this.parent = new Composite(parent, SWT.NONE);
		this.parent.setLayoutData(new GridData(GridData.FILL_BOTH));
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = layout.marginWidth = 0;
		this.parent.setLayout(layout);
		
		setWaitCursor();
		setVendorConfiguration();
		
		this.datapoolPart = datapoolPart;
		DatapoolClipboard.getInstance().setClipboard(datapoolPart.getClipboard());
		createTable(this.parent);
		tableUtil = new DatapoolTableUtil(this);

		createMenu();
		table.addSelectionListener(this);
		if(datapool != null)
			datapool.addDatapoolListener(this);
		this.datapoolMenuManager = new DatapoolMenuManager(this, vendorID, showEquivlenceClasses, showVariables, showRecords);
		setMenuMode(datapoolMenuManager);
		
		/*try
		{
			//parent.getParent().getParent().addControlListener(resizeListener);
		}
		catch(Exception e)
		{
			// do nothing
		}*/
		
		unsetWaitCursor();
	}

	/**
	 * 
	 */
	private void createMenu() {
		MenuManager menuMgr= new MenuManager("#DatapoolEditor"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(this);
		contextMenu = menuMgr.createContextMenu(table);
		table.setMenu(contextMenu);
	}

	/* 
	 * Creates the table widget and viewer.  Populates the table from the datapool.
	 */
	private void createTable(Composite parent)
	{					
		table = new Table(parent, SWT.BORDER | SWT.FULL_SELECTION);
		viewer = new TableViewer(table);

		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.FILL;
		gd.grabExcessHorizontalSpace = true;
		gd.verticalAlignment = GridData.FILL;
		gd.grabExcessVerticalSpace = true;
		gd.widthHint = MAX_TABLE_WIDTH;
		gd.heightHint = MAX_TABLE_HEIGHT;
		table.setLayoutData(gd);
		
		TableLayout layout = new TableLayout();
		layout.addColumnData(new ColumnPixelData(100, false));
		table.setLayout(layout);
				
		// Populate table		
		if(isWithoutColumns())
			makeHeaderColumn();	
		else
			makeColumns();
		
		if(!isWithoutRows())
			makeRows();

		createCursor();
			
		// Show the TableCursor when the user releases the "SHIFT" or "CTRL" key.
		// This signals the end of the multiple selection task.
		table.addKeyListener(new KeyAdapter() {
			
			public void keyReleased(KeyEvent e) {
				if (e.keyCode == SWT.CONTROL && (e.stateMask & SWT.SHIFT) != 0) return;
				if (e.keyCode == SWT.SHIFT && (e.stateMask & SWT.CONTROL) != 0) return;
				if (e.keyCode != SWT.CONTROL && (e.stateMask & SWT.CONTROL) != 0) return;
				if (e.keyCode != SWT.SHIFT && (e.stateMask & SWT.SHIFT) != 0) return;
			
				if(table.getItemCount() != 0)
				{
					TableItem[] selection = table.getSelection();
					TableItem row = (selection.length == 0) ? table.getItem(table.getTopIndex()) : selection[0];
					table.showItem(row);
					if(cursor != null && !cursor.isDisposed())
					{
						cursor.setSelection(row, 0);
						cursor.setVisible(true);
						cursor.setFocus();
					}
				}
			}
		});
		table.addMouseListener(new TableMouseAdapter());
		
		table.getAccessible().addAccessibleListener(new AccessibleAdapter() {
			public void getName(AccessibleEvent e) {
				if(e.childID == ACC.CHILDID_MULTIPLE)
				{
					TableItem[] selection = table.getSelection();
					if(selection != null && selection.length > 0 && 
					   cursor != null && !cursor.isDisposed())
					{
						try
						{
							e.result = selection[0].getText(cursor.getColumn());
						}
						catch (Exception ex) {}
					}
				}
				else
					e.result = DatapoolPlugin.getResourceString("W_DATATABLE");
			}
		});		
		
		if(datapool == null)
			hideTable();
	}

	/*
	 * Creates the table cursor and its assicated event listeners.
	 */
	private void createCursor() 
	{
		if(datapool == null || datapool.getEquivalenceClassCount() == 0 || 
		   (datapool.getEquivalenceClassCount() == 1 && datapool.getEquivalenceClass(0).getRecordCount() == 0))
			return;
		cursor = new TableCursor(table, SWT.NONE);
		// Create an editor to edit the cell when the user hits "ENTER" 
		// while over a cell in the table
		controlEditor = new ControlEditor(cursor);
		controlEditor.grabHorizontal = true;
		controlEditor.grabVertical = true;
		
		cursor.setMenu(contextMenu);
		cursor.addSelectionListener(new CursorSelectionAdapter(cursor, cellEditor, datapoolPart));
		cursor.addKeyListener(new CursorKeyAdapter(cursor, cellEditor, datapoolPart));
		cursor.addMouseListener(new CursorMouseAdapter(cursor, cellEditor, datapoolPart));
		cursor.getAccessible().addAccessibleListener(new AccessibleAdapter() {
			public void getName(AccessibleEvent e) {
				TableItem[] selection = table.getSelection();
				if(selection != null && selection.length > 0 && 
				   cursor != null && !cursor.isDisposed())
				{
					try
					{
						e.result = selection[0].getText(cursor.getColumn());
					}
					catch (Exception ex) {}
				}
			}
		});
		
		if(table.getItemCount() > 0)
		{
			table.select(0);
			cursor.setSelection(0,0);
		}
	}
	
	private void resetCursor()
	{
		if(cursor == null || cursor.isDisposed())
			createCursor();
		else
		{
			if(table.getItemCount() > 0)
			{
				table.select(0);
				cursor.setSelection(0,0);
				cursor.redraw();
			}
		}
	}
	
	/*
	 * Creates and displays a wait cursor.
	 */
	private void setWaitCursor()
	{
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Cursor waitCursor = new Cursor(display, SWT.CURSOR_WAIT);
		setDisplayCursor(waitCursor);		
	}

	/*
	 * Cancels the display of the wait cursor.
	 */
	private void unsetWaitCursor()
	{
		setDisplayCursor(null);
	}

	/*
	 * Creates and sets the cursor to c.
	 */
	private void setDisplayCursor(Cursor c) {
		if(parent == null)
			return;
		Display display = parent.getShell().getDisplay();
		Shell[] shells = display.getShells();
		for (int i = 0; i < shells.length; i++)
			shells[i].setCursor(c);
	}
	
	/**
	 * @param row
	 * @param column
	 */
	private void startCellEditing(TableItem row, int column) {

		IDatapoolCell[] cells = (IDatapoolCell[])row.getData(TAG_DATA);
		IDatapoolCell cell = cells[column - 1];

		IDatapoolVariable variable = (IDatapoolVariable)cell.getCellVariable();
		IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
		Object rawValue = cell.getCellValue();
	
		// TODO to be added after 3.0.1.1
		// bugzilla70477/RATLC458847 didn't get CCB approval.
		// provide combo editor for IDatapoolSuggestedType.TYPE_ENUMERATION
/*		if(suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_ENUMERATION)
		{			
			String[] enum = suggestedType.getEnumerationLiterals();
			if(enum != null && enum.length > 0)
			{
				cellEditor = new ComboBoxCellEditor(cursor, enum);
					
				if(rawValue instanceof String)
				{
					((CCombo)cellEditor.getControl()).setText((String)rawValue);
				}
			}
		}
		else if(suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_BOOLEAN)
		{
			String[] boolList = new String[]{Boolean.toString(true), Boolean.toString(false)};
			cellEditor = new ComboBoxCellEditor(cursor, boolList, SWT.READ_ONLY);
			((CCombo)cellEditor.getControl()).setText((String)rawValue);
		}
*/		
		if(cellEditor == null)
		{
			if(rawValue == null)
			{	
				String typeName = suggestedType.getSuggestedClassName();					
				if(typeName == null || typeName.length() == 0)
				{
					rawValue = new String();
				}
				else
				{
					rawValue = createEmptyCellObject(typeName);
				}
			}
			
			valueObject = new ValueObject(rawValue);
			cellEditor = (CellEditor)valueObject.getPropertyDisplay(cursor);
		}		
		
		if(cellEditor != null)
		{
			if(cellEditor.getControl() != null)
			{
				cellEditor.getControl().addKeyListener(new TableCellKeyAdapter(cellEditor, valueObject, row, column));
				cellEditor.getControl().addPaintListener(new TableCellPaintAdapter());
			}
			cellEditor.addListener(new TableCellEditorAdapter(cellEditor, valueObject, row, column));				
			controlEditor.setEditor(cellEditor.getControl());
			cellEditor.setValidator(new TableCellValidator(ValueClassMap.getValueValidatorClass(rawValue)));
			cellEditor.setFocus();
			
			if(cellEditor instanceof TextCellEditor)
			{
				String currentText = row.getText(column);
				((Text)(cellEditor.getControl())).setSelection(currentText.length());												
			}
			
			this.datapoolPart.markDirty();
		}
	}

	/**
	 * @param typeName
	 * @return
	 */
	private Object createEmptyCellObject(String typeName) 
	{
		Object value;
		IValueClassFactory valueFactory = ValueClassMap.getValueClassFactory(typeName);
		if(valueFactory != null)
			value = valueFactory.createEmptyObject();
		else
		{						
			try
			{
				Class objectClass  = Class.forName(typeName);
				Constructor objectConstructor = objectClass.getConstructor(new Class[0]);
		        value = objectConstructor.newInstance(new Object[0]);		  		    					
			}
			catch(Exception exception)
			{
				value = new String();
			}
		}
		return value;
	}

	/*
	 * Takes the value text from the editor and sets it to the table item as
	 * well as the datapool cell.
	 */ 
	private void applyEditingValue(boolean dispose)	
	{
		if(cellEditor != null)
		{
			if (cursor != null && cursor.getColumn() > 0)
			{					
				Object updatedValue = null;
				String newDescription = null;
				boolean update = true;
				
				TableItem selectedTableItem = cursor.getRow();
				int selectedColumnIndex = cursor.getColumn();
				if(selectedTableItem == null || selectedTableItem.isDisposed())
					return;
				IDatapoolCell[] rowData = (IDatapoolCell[])selectedTableItem.getData(TAG_DATA); 
				IDatapoolCell cell = rowData[cursor.getColumn() - 1];
				
				// TODO to be added after 3.0.1.1
				// bugzilla70477/RATLC458847 didn't get CCB approval.
				// check suggested type first
/*				IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)cell.getCellVariable().getSuggestedType();
				if(suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_ENUMERATION ||
					suggestedType.getSuggestedType() == IDatapoolSuggestedType.TYPE_BOOLEAN)
				{
					updatedValue = ((CCombo)cellEditor.getControl()).getText();
					newDescription = updatedValue.toString();
				}
				else
				{
*/				
					if(valueObject == null)
						update = false;
					else
					{
						String oldDescription = valueObject.getDescription();
						updatedValue = valueObject.updateObject();
						newDescription = valueObject.getDescription();
						
						// only make changes when the value has changed.		
						if((newDescription == null && oldDescription == null) ||
						   (newDescription != null && newDescription.equals(oldDescription)))
						{
							update = false;
						}
					}
				//}
				
				if(update)
				{
					cell.setCellValue(updatedValue);
					selectedTableItem.setText(selectedColumnIndex, newDescription);
					cursor.setSelection(table.getSelectionIndex(), selectedColumnIndex);
					datapoolPart.markDirty();
				}
			}
			
			if(dispose)
				clearCellEditor();
		}						
	}

	/*
	 * Creates TableColumn objects from the DPLVariable objects in the 
	 * datapool.  The indexes are shifted by one because there is a 
	 * column for the row headers.
	 */
	private void makeColumns()
	{
		makeHeaderColumn();
		for(int i = 0; i < datapool.getVariableCount(); i++)
		{
			TableColumn tableColumn = new TableColumn(table, SWT.NONE, i + 1);
			IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(i);
			tableColumn.setResizable(true);
			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
			String type = new String();
			if(suggestedType != null);
				type = suggestedType.getSuggestedClassName();			
			tableColumn.setText(variable.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + type); //$NON-NLS-1$
			tableColumn.setData(TAG_VARIABLE, variable); 
			tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(i)); 
			tableColumn.setWidth(100);
			tableColumn.addSelectionListener(headerListener);	
			tableColumn.addControlListener(resizeColumnListener);
		}		
    }

	/*
	 * Creates the column header.
	 */
	private void makeHeaderColumn() {
		TableColumn recordTableColumn = new TableColumn(table, SWT.NONE, 0);
		recordTableColumn.setResizable(true);
		if(showEquivlenceClasses == false)
			recordTableColumn.setWidth(20);
		else
			recordTableColumn.setWidth(100);
		recordTableColumn.setData(TAG_VARIABLE, null);
		recordTableColumn.addControlListener(resizeColumnListener); 
	}

	/*
	 * Creates TableItem objects from the DPLEquivalenceClass and the IDatapoolRecord
	 * objects in the datapool.  Records do not need to have cells for every
	 * variable.
	 */
	private void makeRows()
	{
		for(int i = 0; i < datapool.getEquivalenceClassCount(); i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i);
			for(int j = 0; j < equivalenceClass.getRecordCount(); j++)
			{
				IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(j);
				int cellCount = record.getCellCount();
								
				String rowContents[] = new String[datapool.getVariableCount() + 1];
				IDatapoolCell[] rowData = new IDatapoolCell[datapool.getVariableCount()];
				if(showEquivlenceClasses)
					rowContents[0] = equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(j);  //$NON-NLS-1$
				else
					rowContents[0] = String.valueOf(j);

				for(int k = 0; k < cellCount; k++)
				{
					IDatapoolCell cell = (IDatapoolCell)record.getCell(k);
					IDatapoolVariable cellVariable = (IDatapoolVariable)cell.getCellVariable();
					int index = findColumnIndex(cellVariable.getName());
					String cellValue = cell.getStringValue();
					ValueObject valueObject = new ValueObject(cell.getCellValue());
					if(valueObject != null)
						cellValue = valueObject.getDescription();
					rowContents[index] = cellValue;
					rowData[index - 1] = cell;
				}				
							
				TableItem item = new TableItem(table, SWT.NULL);
				item.setText(rowContents);
				item.setData(TAG_DATA, rowData);
				item.setData(TAG_RECORD, record); 
				item.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass); 
			}	
		}
	}

	/*
	 * Checks to see if the datapool is empty of equivalence classes causing
	 * the table to be without rows.
	 */
	private boolean isWithoutRows()
	{
		if(datapool == null)
			return true;
		if(datapool.getEquivalenceClassCount() > 0)
		{
			if(datapool.getEquivalenceClassCount() == 1)
			{
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(0);
				if(equivalenceClass.getRecordCount() == 0)
					return true;
				else
					return false;
			}
			else
				return false;
			
		}
		else
		{
			return true;
		}
	}

	/*
	 * Checks to see if the datapool is empty of variables causing the table 
	 * to be without columns.
	 */
	private boolean isWithoutColumns()
	{
		if(datapool == null)
			return true;
		if(datapool.getVariableCount() == 0)
			return true;
		else
			return false;
	}
	
	/*
	 * Finds the column index from the varible's ID.
	 */
	private int findColumnIndex(String name)
	{
		for(int i = 0; i < table.getColumnCount(); i++)
		{
			TableColumn tableColumn = table.getColumn(i);
			IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(TAG_VARIABLE); 
			if(variable != null && name.equals(variable.getName()))
				return i;	
		}	
		return -1;
	}
	/*
	 * Sets the scroll bars location when the cursor or selection moves.
	 *
	private void setScrollBars() {
		Point cursorPoint = cursor.getLocation();		
		Composite greatGrandParent = parent.getParent().getParent();
		if(greatGrandParent != null && greatGrandParent instanceof ScrolledComposite) 
		{
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
			
			int xAdjustment = cursorPoint.x - scrollableComponentWidth + cursor.getBounds().width + 30;
			int yAdjustment = cursorPoint.y - scrollableComponentHeight + cursor.getBounds().height + 53;
			((ScrolledComposite)greatGrandParent).setOrigin(xAdjustment, yAdjustment);
		}
	}*/

	/*
	 * Resize the scrollbar by xChange and yChange.
	 *
	private void resizeScrollBars()
	{
		try
		{
			Composite greatGrandParent = parent.getParent().getParent();
						
			int currentHorizontalMaximum = greatGrandParent.getHorizontalBar().getMaximum();
			int currentVerticalMaximum = greatGrandParent.getVerticalBar().getMaximum();
	
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
	
			int tableWidth = table.getBounds().width;
			int tableHeight = table.getBounds().height;
			
			int horizontalAdjust = 0;
			int verticalAdjust = 68;
			for(int i = 0; i < table.getColumnCount(); i++)
				horizontalAdjust += table.getColumn(i).getWidth();
			for(int i = 0; i < table.getItemCount(); i++)
			{
				TableItem item = table.getItem(i);
				verticalAdjust += item.getBounds(0).height + 1;
			}
	
			if(horizontalAdjust < scrollableComponentWidth)
				greatGrandParent.getHorizontalBar().setVisible(false);
			else
			{
				greatGrandParent.getHorizontalBar().setVisible(true);
				
				if(currentHorizontalMaximum < 119)
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust - scrollableComponentWidth + 44);
				else
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 12);
			}

			if(verticalAdjust < scrollableComponentHeight)
				greatGrandParent.getVerticalBar().setVisible(false);
			else			
			{
				greatGrandParent.getVerticalBar().setVisible(true);

				if(currentVerticalMaximum < 119)
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - scrollableComponentHeight + 58);
				else
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - 12);
			}

		}
		catch(Exception e)
		{
		}
	}*/

	/*
	 * Resets the scroll bar upon changes in control size.  The boolean 
	 * indicates if the resize request is from the container of the table or 
	 * from an internal table element.
	 *
	private void resetScrollBarSize(boolean container)
	{
		try
		{
			Composite greatGrandParent = table.getParent().getParent().getParent();	
			int horizontalMax = greatGrandParent.getHorizontalBar().getMaximum();
			int verticalMax = greatGrandParent.getVerticalBar().getMaximum();
	
			int scrollableComponentHeight = greatGrandParent.getBounds().height;
			int scrollableComponentWidth = greatGrandParent.getBounds().width;
	
			int horizontalAdjust = 0;
			int verticalAdjust = 0;
			for(int i = 0; i < table.getColumnCount(); i++)
				horizontalAdjust += table.getColumn(i).getWidth();
			for(int i = 0; i < table.getItemCount(); i++)
			{
				TableItem item = table.getItem(i);
				verticalAdjust += item.getBounds(0).height + 1;
			}
	
			if(horizontalAdjust + 14 < scrollableComponentWidth)
				horizontalAdjust = 0;
			if(verticalAdjust < scrollableComponentHeight)
				verticalAdjust = 0;
	
			if(container)
			{
				greatGrandParent.getHorizontalBar().setMaximum(horizontalMax + horizontalAdjust - MAX_TABLE_WIDTH - 26);
				greatGrandParent.getVerticalBar().setMaximum(verticalMax + verticalAdjust - MAX_TABLE_HEIGHT - 4);
			}
			else
			{
				if(horizontalMax < 119)
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust - scrollableComponentWidth + 44);
				else
					greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 12);

				if(verticalMax < 119)
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - scrollableComponentHeight + 58);
				else
					greatGrandParent.getVerticalBar().setMaximum(verticalAdjust - 12);

//				greatGrandParent.getHorizontalBar().setMaximum(horizontalAdjust + 14);
//				greatGrandParent.getVerticalBar().setMaximum(verticalAdjust);				
			}
		}
		catch(Exception e)
		{
		}
	}*/

	// SelectionListener
	
	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) 
	{
		int selectedRowIndex = table.getSelectionIndex();
		if(cursor != null && !cursor.isDisposed())
		{
			int selectedColumnIndex = cursor.getColumn();
			if(selectedColumnIndex >= 0)
				cursor.setSelection(selectedRowIndex, selectedColumnIndex);
		}		
	}

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) 
	{
	}
	
		
	// IMenuListener
		
	/**
 	* see menuAboutToShow(org.eclipse.jface.action.IMenuManager) in org.eclipse.jface.action.IMenuListener
	*/
	public void menuAboutToShow(IMenuManager imenuMgr)
	{
		datapoolMenuManager.addActions(imenuMgr);
		setMenuMode(datapoolMenuManager);
	}
	
	// Action handlers
	
	/**
	 * @param dpMenuManager
	 */
	private void setMenuMode(DatapoolMenuManager dpMenuManager) 
	{
		if(dpMenuManager == null)
			return;

		int menuMode = 0;
		if(datapool == null)
		{
			menuMode = 5;
		}
		else
		{
			if(cursor != null && !cursor.isDisposed())
				if(cursor.getColumn() == 0)
					menuMode = 0;
				else
					menuMode = 1;
			if(table.getSelectionIndex() == -1)
				menuMode = 2;
			if(datapool.getEquivalenceClassCount() == 0 && datapool.getVariableCount() > 0)
				menuMode = 3;		
			if(datapool.getEquivalenceClassCount() == 0 && datapool.getVariableCount() == 0)
				menuMode = 4;
			if(datapoolPart.isReadOnly())
				menuMode = 5;
		}

		boolean isCutCopyPasteable = false;
		if(table.getSelectionCount() > 0 && tableUtil.getColumnCount() > 1)
			isCutCopyPasteable = true;
		dpMenuManager.setDisplayMode(menuMode, isCutCopyPasteable);
	}

	/*
	 * Inserts a new equivalence class/row group to the table.  The newly
	 * created equivalence class will have one record.  Brings up a dialog to 
	 * gather information about the equivalence class/row group.  Adds the 
	 * equivalence class to the datapool object.  Inserts a row in the table.
	 */
	public void insertRowGroup()
	{
		if(showEquivlenceClasses == false)
			return;
		TableItem previousTableItem = null;
		if(cursor != null && !cursor.isDisposed())
			previousTableItem = cursor.getRow();
		IDatapoolEquivalenceClass previousEquivalenceClass = null;
		if(previousTableItem != null)
			previousEquivalenceClass = (IDatapoolEquivalenceClass)previousTableItem.getData(TAG_EQUIVALENCE_CLASS); 

		DatapoolRowGroupDialog dialog = new DatapoolRowGroupDialog(Display.getCurrent().getActiveShell(), datapool, table, null, previousEquivalenceClass, DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_TITLE_INS"));  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();			
			String name = dialog.getName();
			if(name != null && name.length() != 0)
			{
				IDatapoolEquivalenceClass equivalenceClass = null;
				if(datapool.getEquivalenceClassCount() == 1 && datapool.getEquivalenceClass(0).getRecordCount() == 0)
					equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(0);
				else
				{
					equivalenceClass = datapool.constructEquivalenceClass();
					int insertionIndex = dialog.getInsertionIndex();
					datapool.insertEquivalenceClass(equivalenceClass, insertionIndex + 1);
				}
				equivalenceClass.setName(name);	
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, 0);
			}
			unsetWaitCursor();
		}
	}

	/*
	 * Deletes the currently selected equivalence class/row group.  Removes 
	 * the equivalence class and all assocaited records from the datapool 
	 * object.  Removes the rows from the table.
	 */
	public void deleteRowGroup()
	{
		if(showEquivlenceClasses == false)
			return;
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
			
		setWaitCursor();
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		int equivalenceClassIndex = datapool.getEquivalenceClassIndexById(equivalenceClass.getId());
		datapool.removeEquivalenceClass(equivalenceClassIndex);
		unsetWaitCursor();
	}

	/*
	 * Edits the currently selected equivalence class/row group.  Renames 
	 * and/or moves the equivalence class in the datapool object.  Updates the
	 * table to reflect the edit and/or move.
	 */
	public void editRowGroup()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
		editRowGroupAux(tableItem);
	}

	/*
	 * Function that actually does the editing of the equivalence class/row group.
	 */
	private void editRowGroupAux(TableItem tableItem)
	{
		if(showEquivlenceClasses == false)
			return;
		IDatapoolEquivalenceClass selectedEquivalenceClass = null;
		if(tableItem != null)
			selectedEquivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		int selectedEquivalenceClassIndex = -1;
		if(selectedEquivalenceClass != null)
			selectedEquivalenceClassIndex = datapool.getEquivalenceClassIndexById(selectedEquivalenceClass.getId());
		IDatapoolEquivalenceClass previousEquivalenceClass = null;
		if(selectedEquivalenceClassIndex != -1 && selectedEquivalenceClassIndex != 0)
			previousEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(selectedEquivalenceClassIndex - 1);

		DatapoolRowGroupDialog dialog = new DatapoolRowGroupDialog(Display.getCurrent().getActiveShell(), datapool, table,  selectedEquivalenceClass, previousEquivalenceClass, DatapoolPlugin.getResourceString("DATA_ROW_GRP_DLG_TITLE_EDIT")); //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			String name = dialog.getName();
			if(!selectedEquivalenceClass.getName().equals(name))
			{
				selectedEquivalenceClass.setName(name);	
			}
			int insertionIndex = dialog.getInsertionIndex();
			if(insertionIndex == selectedEquivalenceClassIndex - 1)
			{
				unsetWaitCursor();
				return;
			}			
			datapool.moveEquivalenceClass(selectedEquivalenceClassIndex, insertionIndex);
			unsetWaitCursor();				
		}
	}

	/*
	 * Inserts a new record/row to the table.  Brings up a dialog to gather 
	 * information about the where to place the row.  Adds the record to the 
	 * datapool object.  Inserts a row in the table.
	 */
	public void insertRow()
	{
		TableItem previousTableItem = null;
		if(cursor != null && !cursor.isDisposed())
			previousTableItem = cursor.getRow();
		
		if(showRecords == false)
			return;
		if(showEquivlenceClasses == false)
		{
			if(datapool.getEquivalenceClassCount() == 0)
			{
				setWaitCursor();
				IDatapoolEquivalenceClass equivalenceClass = datapool.constructEquivalenceClass();
				datapool.insertEquivalenceClass(equivalenceClass, 0);
				equivalenceClass.setName(vendorID);
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, 0);
				unsetWaitCursor();
				return;
			}
			if(datapool.getEquivalenceClassCount() == 1 && datapool.getEquivalenceClass(0).getRecordCount() == 0)
			{
				setWaitCursor();
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(0);
				equivalenceClass.setName(vendorID);
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, 0);
				unsetWaitCursor();
				return;
			}
			if(previousTableItem != null)
			{
				int insertionIndex = table.indexOf(previousTableItem);
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)previousTableItem.getData(TAG_EQUIVALENCE_CLASS);
				IDatapoolRecord record = (IDatapoolRecord)previousTableItem.getData(TAG_RECORD);
				if(equivalenceClass == null || record == null)
					return;
				if(equivalenceClass.getRecordCount() == 0)
					return;
				IDatapoolRecord firstECRecord = (IDatapoolRecord)equivalenceClass.getRecord(0);
				if(firstECRecord.equals(record))
				{
					DatapoolAddDialogSimple dialog = new DatapoolAddDialogSimple(Display.getCurrent().getActiveShell(), DatapoolPlugin.getResourceString("DATA_ADD_ROW_TITLE")); //$NON-NLS-1$
					if (dialog.open() == IDialogConstants.OK_ID)
					{
						insertionIndex = insertionIndex + dialog.getIndexAdjustment();
					}
					else
						return;

				}
				setWaitCursor();
				IDatapoolRecord newRecord = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(newRecord, insertionIndex + 1);
				unsetWaitCursor();
				return;
			}
		}
								
		DatapoolRowDialog dialog = new DatapoolRowDialog(Display.getCurrent().getActiveShell(), datapool, table, null, previousTableItem, DatapoolPlugin.getResourceString("DATA_ROW_DLG_TITLE_INS"), showEquivlenceClasses);  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			int insertionEquivalenceClassIndex = dialog.getInsertionEquivalenceClassIndex();
			int insertionRecordIndex = dialog.getInsertionRecordIndex();
			if(insertionEquivalenceClassIndex != -1 && insertionRecordIndex != -1)
			{
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, insertionRecordIndex + 1);
			}
			else
			{
				IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
				IDatapoolRecord record = equivalenceClass.constructRecord(createCells(datapool));
				equivalenceClass.insertRecord(record, 0);
			}
			unsetWaitCursor();		
		}
	}

	/*
	 * Deletes the currently selected record/row.  Removes the record from the 
	 * datapool object.  Removes the row from the table.
	 */
	public void deleteRow()
	{
		if(showRecords == false)
			return;	
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;

		setWaitCursor();
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		IDatapoolRecord record = (IDatapoolRecord)tableItem.getData(TAG_RECORD);
		int equivalenceClassIndex = datapool.getEquivalenceClassIndexById(equivalenceClass.getId());
		int recordIndex = getRecordIndex(equivalenceClass, record);
		equivalenceClass.removeRecord(recordIndex);
		if(equivalenceClass.getRecordCount() == 0)
		{
			ignoreECNotification = true;
			datapool.removeEquivalenceClass(equivalenceClassIndex);			
		}
		unsetWaitCursor();
	}

	/*
	 * Edits the currently selected record/row.  Moves the record in the 
	 * datapool object.  Updates the table to reflect move.
	 */
	public void editRow()
	{
		TableItem tableItem = null;
		if(cursor != null && !cursor.isDisposed())
			tableItem = cursor.getRow();
		else
			return;
		editRowAux(tableItem);		
	}

	/*
	 * Function that actually does the editing of the record/row.
	 */
	private void editRowAux(TableItem tableItem)
	{
		if(showRecords == false)
			return;	
		int rowIndex = table.getSelectionIndex();
		TableItem previousTableItem = null;
		if(rowIndex != 0)
			previousTableItem = table.getItem(rowIndex - 1);

		DatapoolRowDialog dialog = new DatapoolRowDialog(Display.getCurrent().getActiveShell(), datapool, table,  tableItem, previousTableItem, DatapoolPlugin.getResourceString("DATA_ROW_DLG_TITLE_EDIT"), showEquivlenceClasses);  //$NON-NLS-1$
		if (dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
			int equivalenceClassIndex = datapool.getEquivalenceClassIndexById(equivalenceClass.getId());
			IDatapoolRecord record = (IDatapoolRecord)tableItem.getData(TAG_RECORD);
			int recordIndex = getRecordIndex(equivalenceClass, record);

			int insertionEquivalenceClassIndex = dialog.getInsertionEquivalenceClassIndex();
			int insertionRecordIndex = dialog.getInsertionRecordIndex();
			if(equivalenceClassIndex == insertionEquivalenceClassIndex)
			{
				if(recordIndex != insertionRecordIndex + 1)
				{
					if(insertionRecordIndex > recordIndex)
						equivalenceClass.moveRecord(recordIndex, insertionRecordIndex);
					else
						equivalenceClass.moveRecord(recordIndex, insertionRecordIndex + 1);	
				}
			}
			else
			{
				if(insertionEquivalenceClassIndex != -1 && insertionRecordIndex != -1)
				{
					IDatapoolEquivalenceClass targetEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
					equivalenceClass.removeRecord(recordIndex);
					targetEquivalenceClass.insertRecord(record, insertionRecordIndex + 1);
					if(equivalenceClass.getRecordCount() == 0)
					{
						ignoreECNotification = true;
						datapool.removeEquivalenceClass(equivalenceClassIndex);
					}
				}
				else
				{
					IDatapoolEquivalenceClass targetEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(insertionEquivalenceClassIndex);
					equivalenceClass.removeRecord(recordIndex);
					targetEquivalenceClass.insertRecord(record, 0);
					if(equivalenceClass.getRecordCount() == 0)
					{
						ignoreECNotification = true;
						datapool.removeEquivalenceClass(equivalenceClassIndex);
					}
				}
			}
			unsetWaitCursor();		
		}
	}

	/*
	 * Inserts a new variable/column to the table.  Brings up a dialog to 
	 * gather information about the variable/column.  Adds the variable to the 
	 * datapool object.  Inserts a column in the table.
	 */
	public void insertColumn()
	{
		if(showVariables == false)
			return;
		int selectedColumnIndex = -1;
		if(cursor != null && !cursor.isDisposed())
			selectedColumnIndex = cursor.getColumn();
		TableColumn previousTableColumn = null;
		if(selectedColumnIndex > 0)
			previousTableColumn = table.getColumn(selectedColumnIndex);
		IDatapoolVariable previousVariable = null;
		if(previousTableColumn != null)
			previousVariable = (IDatapoolVariable)previousTableColumn.getData(TAG_VARIABLE);

		DatapoolColumnDialog dialog = new DatapoolColumnDialog(Display.getCurrent().getActiveShell(), datapool, null, previousVariable, DatapoolPlugin.getResourceString("DATA_COL_DLG_TITLE_INS"));  //$NON-NLS-1$
		if ( dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			IDatapoolVariable variable = datapool.constructVariable();
			variable.setName(dialog.getName());
			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
			setVariableType(suggestedType, dialog.getType());
			variable.setSuggestedType(suggestedType);
			int insertionIndex = findColumnIndex(dialog.getInsertionVariableName());
			if(insertionIndex == -1)
				insertionIndex = 0;

			datapool.insertVariable(variable, insertionIndex);
			unsetWaitCursor();		
		}
	}

	/*
	 * Deletes the currently selected variable/column.  Removes the column and 
	 * all assocaited cells from the datapool object.  Removes the column from 
	 * the table.
	 */
	public void deleteColumn()
	{
		if(showVariables == false)
			return;	
		if(cursor == null || cursor.isDisposed())
		{
			if(table.getItemCount() == 0 && tableUtil.getColumnCount() > 0)
			{
				DatapoolDeleteColumnDialog dialog = new DatapoolDeleteColumnDialog(Display.getCurrent().getActiveShell(), datapool);
				if ( dialog.open() == IDialogConstants.OK_ID)
				{
					setWaitCursor();
					String variableID = dialog.getDeletedVariableID();
					int variableIndex = datapool.getVariableIndexById(variableID);
					datapool.removeVariable(variableIndex);
					unsetWaitCursor();		
				}			
			}
			else
				return;
		}
		else
		{
			int selectedColumnIndex = cursor.getColumn();
			int selectedRowIndex = table.getSelectionIndex();
			// Don't delete the row name column.
			if (selectedColumnIndex < 1)	
				return;
					
			setWaitCursor();
			TableColumn tableColumn = table.getColumn(selectedColumnIndex);
			IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(TAG_VARIABLE);
			int variableIndex = datapool.getVariableIndexById(variable.getId());
			datapool.removeVariable(variableIndex);
			unsetWaitCursor();
		}
	}
	
	/*
	 * Edits the currently selected variable/column.  Renames and/or moves the 
	 * variable in the datapool object.  Updates the table to reflect the edit 
	 * and/or move.
	 */
	public void editColumn()
	{
		int selectedColumnIndex = cursor.getColumn();
		TableColumn tableColumn = (TableColumn) table.getColumn(selectedColumnIndex);
		editColumnAux(tableColumn);		
	}

	/*
	 * Function that actually does the editing of the variable/column.
	 */	
	private void editColumnAux(TableColumn tableColumn)
	{
		if(showVariables == false)
			return;	
		int columnIndex = table.indexOf(tableColumn);
		TableColumn previousTableColumn = null;
		if(columnIndex != 0)
			previousTableColumn = table.getColumn(columnIndex-1);
		IDatapoolVariable previousVariable = null;
		if(previousTableColumn != null)
			previousVariable = (IDatapoolVariable)previousTableColumn.getData(TAG_VARIABLE); 
		IDatapoolVariable variable = (IDatapoolVariable)tableColumn.getData(TAG_VARIABLE); 
		DatapoolColumnDialog dialog = new DatapoolColumnDialog(Display.getCurrent().getActiveShell(), datapool, variable, previousVariable, DatapoolPlugin.getResourceString("DATA_COL_DLG_TITLE_EDIT"));  //$NON-NLS-1$
		if ( dialog.open() == IDialogConstants.OK_ID)
		{
			setWaitCursor();
			String name = dialog.getName();
			String type = dialog.getType();
			String insertionVariableID = dialog.getInsertionVariableID();
			String previousVariableID = new String();
			if(previousVariable != null)
				previousVariableID = previousVariable.getId();

			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();			
			if(name.equals(variable.getName()) && 
				type.equals(suggestedType.getSuggestedClassName()) &&
				insertionVariableID.equals(previousVariableID))
			{
				unsetWaitCursor();
				return;	
			}
			variable.setName(dialog.getName());
			setVariableType(suggestedType, dialog.getType());
			variable.setSuggestedType(suggestedType);
			Integer variableIndex = (Integer)tableColumn.getData(TAG_VARIABLE_INDEX); 

			int insertionIndex = findColumnIndex(dialog.getInsertionVariableName());
			if(insertionIndex == columnIndex - 1)
			{
				unsetWaitCursor();
				return;
			}
			if(insertionIndex == -1)
				datapool.moveVariable(columnIndex - 1, 0);
			else
				if(insertionIndex > columnIndex)
					datapool.moveVariable(columnIndex - 1, insertionIndex - 1);
				else
					datapool.moveVariable(columnIndex - 1, insertionIndex);
			unsetWaitCursor();				
		}
	}
	
	/* 
	 * Clears the value in cell.
	 */
	public void clearCell()
	{
		clearCellEditor();
		if (cursor != null && !cursor.isDisposed() && cursor.getColumn() > 0)
		{					
			TableItem selectedTableItem = cursor.getRow();
			int selectedColumnIndex = cursor.getColumn();
			if(selectedTableItem != null && !selectedTableItem.isDisposed())
			{
				IDatapoolCell[] rowData = (IDatapoolCell[])selectedTableItem.getData(TAG_DATA); 
				IDatapoolCell cell = rowData[cursor.getColumn() - 1];
				Object value = cell.getCellValue();
				if(value == null)
					return;
				
				if(value instanceof String)
				{
					if(((String)value).length() == 0)
						return;
					else
						value = new String();
				}
				else
					value = createEmptyCellObject(value.getClass().getName());
				
				cell.setCellValue(value);
				selectedTableItem.setText(selectedColumnIndex, (new ValueObject(value)).getDescription());
				cursor.setSelection(table.getSelectionIndex(), selectedColumnIndex);
				cursor.redraw();
				datapoolPart.markDirty();
			}
		}
	}
	
	// currently only single cell selection is supported
	private IDatapoolCell[] getSelectedCells()
	{
		IDatapoolCell[] cells = new IDatapoolCell[0];
		if (cursor != null && cursor.getColumn() > 0)
		{					
			TableItem selectedTableItem = cursor.getRow();
			int selectedColumnIndex = cursor.getColumn();
			if(selectedTableItem != null && !selectedTableItem.isDisposed())
			{
				IDatapoolCell[] rowData = (IDatapoolCell[])selectedTableItem.getData(TAG_DATA); 
				IDatapoolCell cell = rowData[cursor.getColumn() - 1];
				cells = new IDatapoolCell[]{cell};
			}
		}
		return cells;
	}
	
	// IDatapoolListener
	
	/*
	 * The following methods follow a spec that will eventually correspond to a
	 * listener interface that will make callbacks to the Datapool table.  
	 * Based on the callback, the table will update accordingly.  
	 */

	public void variableAdded(IDatapool datapool, int newVariableIndex) 
	{	
		TableColumn tableColumn = new TableColumn(table, SWT.NONE, newVariableIndex + 1);
		IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(newVariableIndex);
		tableColumn.setResizable(true);
		IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
		String type = new String();
		if(suggestedType != null);
			type = suggestedType.getSuggestedClassName();
		tableColumn.setText(variable.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + type);  //$NON-NLS-1$
		tableColumn.setData(TAG_VARIABLE, variable);
		tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(newVariableIndex));
		tableColumn.setWidth(100);
		tableColumn.addSelectionListener(headerListener);
		tableColumn.addControlListener(resizeColumnListener);		
		tableUtil.insertColumn(tableColumn, newVariableIndex);	

		if(cursor != null && !cursor.isDisposed())
		{
			int selectedRowIndex = table.getSelectionIndex();
			if(selectedRowIndex >= 0)
			{
				table.setSelection(selectedRowIndex);
				cursor.setSelection(selectedRowIndex, newVariableIndex + 1);
				cursor.setFocus();
				//resizeScrollBars();
				//setScrollBars();
			}

		}
		datapoolPart.markDirty();		
	}	
	
	public void variableRemoved(IDatapool datapool, int variableIndex)
	{
		tableUtil.deleteColumn(variableIndex + 1);

		if(cursor != null && !cursor.isDisposed())
		{
			if(variableIndex > 0)
			{
				int selectedRowIndex = table.getSelectionIndex();
				if(selectedRowIndex >= 0)
				{
					table.select(selectedRowIndex);
					cursor.setSelection(selectedRowIndex, variableIndex);
					//resizeScrollBars();
					//setScrollBars();
				}
			}
			else
				table.deselectAll();

		}
		datapoolPart.markDirty();		
	}
	
	public void variableMoved(IDatapool datapool, int sourceVariableIndex, int targetVariableIndex)
	{
		tableUtil.moveColumn(sourceVariableIndex + 1, targetVariableIndex + 1);
		if(cursor != null && !cursor.isDisposed())
			if(targetVariableIndex > 0)
			{
				int selectedIndex = table.getSelectionIndex();
				if(selectedIndex >= 0)
				{
					table.select(selectedIndex);
					cursor.setSelection(selectedIndex, targetVariableIndex);
					//setScrollBars();
				}
			}
		datapoolPart.markDirty();		
	}
	
	public void variableChanged(IDatapool datapool, int variableIndex)
	{
		IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(variableIndex);
		int columnIndex = findColumnIndex(variable.getName());
		TableColumn tableColumn = table.getColumn(columnIndex);
		IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
		String type = new String();
		if(suggestedType != null);
			type = suggestedType.getSuggestedClassName();		
		tableColumn.setText(variable.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + type);  //$NON-NLS-1$
		tableColumn.setData(TAG_VARIABLE, variable);
		tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(variableIndex));
		datapoolPart.markDirty();		
	}
	
	public void variableChanged(IDatapool datapool, int variableIndex, String oldName)
	{
		variableChanged(datapool, variableIndex);
	}
	
	public void equivalenceClassChanged(IDatapool datapool, int equivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			// Assumes rows are synced with datapool
			int tableRowIndex = findRowIndex(equivalenceClassIndex, i);
			TableItem tableItem = table.getItem(tableRowIndex);
			tableItem.setText(0, equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(i)); //$NON-NLS-1$
			if(cursor != null && !cursor.isDisposed())
				cursor.redraw();
			datapoolPart.markDirty();		
		}
	}

	public void equivalenceClassChanged(IDatapool datapool, int equivalenceClassIndex, String oldName)
	{
		equivalenceClassChanged(datapool, equivalenceClassIndex);
	}
	
	public void equivalenceClassAdded(IDatapool datapool, int newEquivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(newEquivalenceClassIndex);
		IDatapoolEquivalenceClass previousEquivalenceClass = null;
		int insertionIndex = 0;
		if(newEquivalenceClassIndex > 0)
		{
			previousEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(newEquivalenceClassIndex - 1);
			insertionIndex = findRowIndex(newEquivalenceClassIndex - 1, previousEquivalenceClass.getRecordCount() - 1) + 1;
		}

		for(int j = 0; j < equivalenceClass.getRecordCount(); j++)
		{
			IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(j);
			String rowContents[] = new String[table.getColumnCount()];
			IDatapoolCell[] rowData = new IDatapoolCell[table.getColumnCount() - 1];
			rowContents[0] = equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(j); //$NON-NLS-1$
			for(int k = 0; k < record.getCellCount(); k++)
			{
				IDatapoolCell cell = (IDatapoolCell)record.getCell(k);
				org.eclipse.hyades.execution.runtime.datapool.IDatapoolVariable cellVariable = cell.getCellVariable();
				int index = findColumnIndex(cellVariable.getName());
				String cellValue = cell.getStringValue();
				ValueObject valueObject = new ValueObject(cell.getCellValue());
				if(valueObject != null)
					cellValue = valueObject.getDescription();
				rowContents[index] = cellValue;	
				rowData[index - 1] = cell;
			}
												
			TableItem item = new TableItem(table, SWT.NULL, insertionIndex + j);
			item.setText(rowContents);
			item.setData(TAG_DATA, rowData);
			item.setData(TAG_RECORD, record); 
			item.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass);
		}
		datapoolPart.markDirty();	
	}

	public void equivalenceClassRemoved(IDatapool datapool, int equivalenceClassIndex)
	{
		if(ignoreECNotification)
		{
			ignoreECNotification = false;
			return;
		}
		// Assumes rows are synced with datapool
		int tableRowIndex = findRowIndex(equivalenceClassIndex, 0);
		TableItem tableItem = table.getItem(tableRowIndex);
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
		boolean removeRow = true;
		while(removeRow)
		{
			table.remove(tableRowIndex);
			if(tableRowIndex < table.getItemCount())
			{
				tableItem = table.getItem(tableRowIndex);
				IDatapoolEquivalenceClass nextEquivalenceClass = (IDatapoolEquivalenceClass)tableItem.getData(TAG_EQUIVALENCE_CLASS);
				if(!nextEquivalenceClass.equals(equivalenceClass))
					removeRow = false;
			}
			else
				removeRow = false;
		}
		datapoolPart.markDirty();		
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			int tableSize = table.getItemCount();
			if(tableSize == 0)
			{
				table.deselectAll();
				cursor.dispose();	
				return;								
			}
			if(tableSize > tableRowIndex)
			{
				table.setSelection(tableRowIndex);
				cursor.setSelection(tableRowIndex, 0);
				return;
			}
			if(tableSize > tableRowIndex - 1)	
			{
				table.setSelection(tableRowIndex - 1);
				cursor.setSelection(tableRowIndex - 1, 0);
			}
		}		
	}
	
	public void equivalenceClassMoved(IDatapool datapool, int sourceEquivalenceClassIndex, int targetEquivalenceClassIndex)
	{
		if(sourceEquivalenceClassIndex == targetEquivalenceClassIndex)
			return;

		int targetIndex = 0;
		if(targetEquivalenceClassIndex != -1)
		{
			IDatapoolEquivalenceClass targetEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(targetEquivalenceClassIndex);
			int targetRecordCount = targetEquivalenceClass.getRecordCount();
			if(sourceEquivalenceClassIndex < targetEquivalenceClassIndex)
				targetRecordCount -= 1;
			targetIndex = findRowIndex(targetEquivalenceClassIndex, targetRecordCount);
		}
		
		IDatapoolEquivalenceClass sourceEquivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(sourceEquivalenceClassIndex);
		int sourceRecordCount = sourceEquivalenceClass.getRecordCount();
		int sourceIndex = findRowIndex(sourceEquivalenceClassIndex, 0);
		for(int i = 0; i < sourceEquivalenceClass.getRecordCount(); i++)
		{
			if(sourceEquivalenceClassIndex > targetEquivalenceClassIndex)
				tableUtil.moveRow(sourceIndex + sourceRecordCount - 1, targetIndex);
			else
				tableUtil.moveRow(sourceIndex, targetIndex);
		}
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			table.setSelection(targetIndex);
			cursor.setSelection(targetIndex, 0);
		}		
		datapoolPart.markDirty();		
	}

	public void recordAdded(IDatapool datapool, int equivalenceClassIndex, int newRecordIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(newRecordIndex);
		String rowContents[] = new String[table.getColumnCount()];
		IDatapoolCell[] rowData = new IDatapoolCell[table.getColumnCount() - 1];
		if(showEquivlenceClasses)
			rowContents[0] = equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(newRecordIndex); //$NON-NLS-1$
		else
			rowContents[0] = String.valueOf(newRecordIndex);
		for(int i = 0; i < record.getCellCount(); i++)
		{
			IDatapoolCell cell = (IDatapoolCell)record.getCell(i);
			org.eclipse.hyades.execution.runtime.datapool.IDatapoolVariable cellVariable = cell.getCellVariable();
			int index = findColumnIndex(cellVariable.getName());
			String cellValue = cell.getStringValue();
			ValueObject valueObject = new ValueObject(cell.getCellValue());
			if(valueObject != null)
				cellValue = valueObject.getDescription();
			rowContents[index] = cellValue;	
			rowData[index - 1] = cell;			
		}
		int insertionIndex = findRowIndex(equivalenceClassIndex, newRecordIndex);
		TableItem tableItem = new TableItem(table, SWT.NULL, insertionIndex);
		tableItem.setText(rowContents);
		tableItem.setData(TAG_DATA, rowData);
		tableItem.setData(TAG_RECORD, record);
		tableItem.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass);
		repopulateRowLabels(equivalenceClassIndex);
		if(table.getItemCount() == 1)
			createCursor();		
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			table.setSelection(insertionIndex);
			cursor.setSelection(insertionIndex, 0);
			cursor.setFocus();
		}		
		datapoolPart.markDirty();		
	}
	
	public void recordRemoved(IDatapool datapool, int equivalenceClassIndex, int recordIndex)
	{
		int tableRowIndex = findRowIndex(equivalenceClassIndex, recordIndex);
		table.remove(tableRowIndex);
		repopulateRowLabels(equivalenceClassIndex);

		datapoolPart.markDirty();				
		if(cursor != null && !cursor.isDisposed() && table != null)
		{
			int tableSize = table.getItemCount();
			if(tableSize == 0)
			{
				table.deselectAll();
				cursor.dispose();	
				return;								
			}
			if(tableSize > tableRowIndex)
			{
				table.setSelection(tableRowIndex);
				cursor.setSelection(tableRowIndex, 0);
				return;
			}
			if(tableSize > tableRowIndex - 1)	
			{
				table.setSelection(tableRowIndex - 1);
				cursor.setSelection(tableRowIndex - 1, 0);
			}
		}		
	}
	
	public void recordMoved(IDatapool datapool, int equivalenceClassIndex, int sourceRecordIndex, int targetRecordIndex)
	{
		int sourceRowIndex = findRowIndex(equivalenceClassIndex, sourceRecordIndex);
		int targetRowIndex = findRowIndex(equivalenceClassIndex, targetRecordIndex);
		tableUtil.moveRow(sourceRowIndex, targetRowIndex);
		repopulateRowLabels(equivalenceClassIndex);
		datapoolPart.markDirty();		
	}
	
	public void cellChanged(IDatapool datapool, int equivalenceClassIndex, int recordIndex, int variableIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(recordIndex);
		IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(variableIndex);

		IDatapoolCell cell = null;		
		for(int i = 0; i < record.getCellCount(); i++)
		{
			cell = (IDatapoolCell)record.getCell(i);
			if(cell.getCellVariable().equals(variable))
				break;
			else
				cell = null;
		}

		if(cell != null)
		{
			String cellValue = cell.getStringValue();
			ValueObject valueObject = new ValueObject(cell.getCellValue());
			if(valueObject != null)
				cellValue = valueObject.getDescription();
			String value = cellValue;
			int tableRowIndex = findRowIndex(equivalenceClassIndex, recordIndex);
			TableItem tableItem = table.getItem(tableRowIndex);
			tableItem.setText(variableIndex + 1, value);
			cursor.setSelection(tableRowIndex, equivalenceClassIndex + 1);
			datapoolPart.markDirty();
		}
		datapoolPart.markDirty();		
	}

	public void equivalenceClassReordered(IDatapool datapool, int equivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(i);
			int newTableRowIndex = findRowIndex(equivalenceClassIndex, i);
			int oldTableRowIndex = findRowIndex(equivalenceClassIndex, record);
			tableUtil.swapRow(oldTableRowIndex, newTableRowIndex);
		}
		datapoolPart.markDirty();		
	}
	
	public void recordChanged(IDatapool datapool, int EquivClassIndex, int recordIndex) 
	{
		return;
	}

	public void save(IDatapool datapool) 
	{
		alreadySaved = true;
	}

	// End IDatapoolListener

	/* 
	 * Finds the index of the row in the table that corresponds to the record 
	 * given by the record object.  Returns -1 if the record is not found.
	 */
	private int findRowIndex(int equivalenceClassIndex, IDatapoolRecord record)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		int startingRowIndex = findRowIndex(equivalenceClassIndex, 0);
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			IDatapoolRecord currentRecord = (IDatapoolRecord)table.getItem(startingRowIndex + i).getData(TAG_RECORD);
			if(currentRecord.equals(record))
				return startingRowIndex + i;
		}
		return -1;
	}

	/*
	 * Finds the index of the row in the table that corresponds to the record
	 * at the given equivalence class and record index in the datapool object.
	 */
	private int findRowIndex(int equivalenceClassIndex, int recordIndex)
	{
		int count = recordIndex;
		for(int i = 0; i < equivalenceClassIndex; i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i);
			count += equivalenceClass.getRecordCount();
		}
		return count;
	}

	/*
	 * Recreates the row labels for a row group when a memeber changes or
	 * moves.
	 */	
	private void repopulateRowLabels(int equivalenceClassIndex)
	{
		IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(equivalenceClassIndex);
		String name = equivalenceClass.getName();
		int tableRowIndex = findRowIndex(equivalenceClassIndex, 0);
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			if(table.getItemCount() <= tableRowIndex + 1)
				continue;
			TableItem tableItem = table.getItem(tableRowIndex + i);
			if(showEquivlenceClasses)
				tableItem.setText(0, name + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(i)); //$NON-NLS-1$
			else
				tableItem.setText(0, String.valueOf(i));
		}
	}
	
	private int getRecordIndex(IDatapoolEquivalenceClass equivalenceClass, IDatapoolRecord record)
	{
		for(int i = 0; i < equivalenceClass.getRecordCount(); i++)
		{
			org.eclipse.hyades.execution.runtime.datapool.IDatapoolRecord compRecord = equivalenceClass.getRecord(i);
			if(compRecord.equals(record))
				return i;
		}
		return -1;
	}
	
	protected Object[] createCells(IDatapool datapool)
	{
	 	Object[] cells = new Object[datapool.getVariableCount()];
	 	for(int i = 0; i < datapool.getVariableCount(); i++)
	 		cells[i] = new String();
	 	return cells;
	}

	private void setVendorConfiguration()
	{
		if(this.vendorID == null)
			return;
		IExtensionPoint extensionPoint = Platform.getPluginRegistry().getExtensionPoint("org.eclipse.hyades.test.ui.datapool.vendorConfigurationExtension"); //$NON-NLS-1$

		if (extensionPoint != null)
		{
			try {
	  
				IConfigurationElement[] extensionPoints = extensionPoint.getConfigurationElements();
				for(int i = 0; i < extensionPoints.length; i++)
				{
					String extVendorID = extensionPoints[i].getAttribute("vendorID"); //$NON-NLS-1$
					if(this.vendorID.equals(extVendorID))
					{
						String strShowEC = extensionPoints[i].getAttribute("showEquivalenceClassCommands"); //$NON-NLS-1$
						Boolean boolShowEC = Boolean.valueOf(strShowEC);
						showEquivlenceClasses = boolShowEC.booleanValue();
						String strShowVar = extensionPoints[i].getAttribute("showVariableCommands"); //$NON-NLS-1$
						Boolean boolShowVar = Boolean.valueOf(strShowVar);
						showVariables = boolShowVar.booleanValue();
						String strShowRec = extensionPoints[i].getAttribute("showRecordCommands"); //$NON-NLS-1$
						Boolean boolShowRec = Boolean.valueOf(strShowRec);
						showRecords = boolShowVar.booleanValue();
					}
				}
			}
			catch(Exception e)
			{
			}
		}	
		
	}	
	
	public TableViewer getViewer()
	{
		return viewer;
	}
	
	private void setVariableType(IDatapoolSuggestedType suggestedType, String type)
	{
		if(suggestedType == null)
			return;
		if(TypeChecker.getInstance().isBoolean(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_BOOLEAN);
		else if(TypeChecker.getInstance().isNumber(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_NUMBER);
		else if(TypeChecker.getInstance().isString(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_STRING);
		else if(TypeChecker.getInstance().isEnumeration(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_ENUMERATION);
		else
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_COMPLEX);
		
		suggestedType.setSuggestedClassName(type);	
	}
	
	public void save()
	{
		datapoolFactory.save(datapool);
		alreadySaved = true;
	}
	
	public void dispose()
	{
		clearCellEditor();
		if(!cursor.isDisposed())
			cursor.dispose();
		if(!contextMenu.isDisposed())
			contextMenu.dispose();
		if(!table.isDisposed())
			table.dispose();
		if(viewer != null)
			viewer = null;
		
		datapool = null;
		tableUtil = null;
		datapoolPart = null;
		datapoolFactory = null;
		vendorID = null;
		controlEditor = null;
		focusListener = null;
		cellEditorListener = null;		
	}
	
	public IDatapool getDatapool()
	{
		return datapool;
	}
	
	/**
	 * Given a data pool part, register cut/copy and paste actions, used for copying or moving cell contents,
	 * for example, copying one cell's contents to another cell.
	 * 
	 * @param part the editor part that actions will be associated with
	 */
	private void connectCutCopyPasteActions(final IDatapoolPart part) {
		if (part instanceof IDatapoolPartExtended) {
			IWorkbenchPartSite site = ((IDatapoolPartExtended) part).getPartSite();
			if (site != null) {
				if (this.cutAction == null) {
					this.cutAction = new CutAction(this.getViewer(), this);
				}
				if (this.copyAction == null) {
					this.copyAction = new CopyAction(this.getViewer(), this);
				}
				if (this.pasteAction == null) {
					this.pasteAction = new PasteAction(this.getViewer(), this);
				}
				IKeyBindingService keyBindingService = site.getKeyBindingService();
				keyBindingService.registerAction(cutAction);
				keyBindingService.registerAction(copyAction);
				keyBindingService.registerAction(pasteAction);
			}
		}
	}
		
	public void cut()
	{
		DatapoolClipboard.getInstance().clear();		
		int columnIndex = -1;
		if(cursor != null && !cursor.isDisposed())
			columnIndex = cursor.getColumn();
		if(columnIndex == -1)
			return;
		if(columnIndex == 0)
			cutRecords();
		else
			cutCell();			
	}
	
	public void copy()
	{
		DatapoolClipboard.getInstance().clear();		
		int columnIndex = -1;
		if(cursor != null && !cursor.isDisposed())
			columnIndex = cursor.getColumn();
		if(columnIndex == -1)
			return;
		if(columnIndex == 0)
			copyRecords();
		else
			copyCell();		
	}
	
	public void paste()
	{
		int columnIndex = -1;
		if(cursor != null && !cursor.isDisposed())
			columnIndex = cursor.getColumn();
		if(columnIndex == -1)
			return;
		if(columnIndex == 0)
			pasteRecords();
		else
			pasteCell();		
	}
		
	private void cutRecords()
	{
		TableItem[] tableItems = table.getSelection();
		if(tableItems == null || table.getSelectionCount() == 0)
			return;
		for(int i = 0; i < tableItems.length; i++)
		{
			cutRecord(tableItems[i]);
		}
		DatapoolClipboard.getInstance().update();
	}
	
	private void cutRecord(TableItem item)
	{
		copyRecord(item);
		clearRecord(item);
	}
	
	private void clearRecord(TableItem item)
	{
		IDatapoolRecord record = (IDatapoolRecord)item.getData(TAG_RECORD);
		if(record == null || record.getCellCount() == 0)
			return;
		int cellCount = record.getCellCount();
		for(int i = 0; i < cellCount; i++)
		{
			IDatapoolCell cell = (IDatapoolCell)record.getCell(i);
			cell.setCellValue(null);
			item.setText(i + 1, new String());
		}
		datapoolPart.markDirty();
	}
	
	private void cutCell()
	{
		copyCell();
		clearCell();
	}
		
	private void copyRecords()
	{
		TableItem[] tableItems = table.getSelection();
		if(tableItems == null || table.getSelectionCount() == 0)
			return;
		for(int i = 0; i < tableItems.length; i++)
		{
			copyRecord(tableItems[i]);
		}
		DatapoolClipboard.getInstance().update();		
	}
	
	private void copyRecord(TableItem item)
	{
		IDatapoolRecord record = (IDatapoolRecord)item.getData(TAG_RECORD);
		if(record == null || record.getCellCount() == 0)
			return;
		DatapoolClipboard.getInstance().addRecordData(record);
	}
	
	private void copyCell()
	{
		int rowIndex = -1;
		TableItem row = null;
		int columnIndex = -1;
		if(cursor == null || cursor.isDisposed())
			return;

		columnIndex = cursor.getColumn();
		row = cursor.getRow();
		if(row != null)
			rowIndex = table.indexOf(row);
		if(columnIndex == -1 || rowIndex == -1 || row == null)
			return;
		
		IDatapoolRecord record = (IDatapoolRecord)row.getData(TAG_RECORD);
		IDatapoolCell cell = (IDatapoolCell)record.getCell(columnIndex - 1);
		DatapoolClipboard.getInstance().addCellData(cell);
		DatapoolClipboard.getInstance().update();		
	}
	
	private void pasteRecords()
	{
		Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_PASTE_OVERWRITE")}; //$NON-NLS-1$
		String message = MessageFormat.format("{0}", messageElements); //$NON-NLS-1$
		MessageDialog overwriteDialog = new MessageDialog(parent.getShell(),
														  DatapoolPlugin.getResourceString("PASTE_TEXT"), //$NON-NLS-1$
														  null,
														  message, 
														  MessageDialog.WARNING, 
														  new String[] {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL}, 
														  0);
		
		if (overwriteDialog.open() != 0)
			return;
	
		TableItem[] tableItems = table.getSelection();
		if(tableItems == null || table.getSelectionCount() == 0)
			return;
		TableItem firstItem = tableItems[0];
		int firstItemIndex = table.indexOf(firstItem);
		Object[] recordDataSet = DatapoolClipboard.getInstance().getRecordDataSet();
		if(recordDataSet == null || recordDataSet.length == 0)
			return;
		for(int i = 0; i < recordDataSet.length; i++)
		{
			if(firstItemIndex + i < table.getItemCount())
			{
				Object[] recordData = (Object[])recordDataSet[i];
				if(recordData != null)
				{
					TableItem tableItem = table.getItem(firstItemIndex + i);
					IDatapoolRecord record = (IDatapoolRecord)tableItem.getData(TAG_RECORD);
					for(int j = 0; j < recordData.length; j++)
					{
						if(j >= record.getCellCount())
							continue;
						IDatapoolCell cell = (IDatapoolCell)record.getCell(j);
						cell.setCellValue(recordData[j]);
						String cellValue = cell.getStringValue();
						ValueObject valueObject = new ValueObject(cell.getCellValue());
						if(valueObject != null)
							cellValue = valueObject.getDescription();
						tableItem.setText(j + 1, cellValue);
						datapoolPart.markDirty();		
					}
				}
			}
			else 
				continue;
		}
	}
	
	private void pasteCell()
	{
		int rowIndex = -1;
		TableItem row = null;
		int columnIndex = -1;
		if(cursor == null || cursor.isDisposed())
			return;

		columnIndex = cursor.getColumn();
		row = cursor.getRow();
		if(row != null)
			rowIndex = table.indexOf(row);
		if(columnIndex == -1 || rowIndex == -1 || row == null)
			return;
		IDatapoolRecord record = (IDatapoolRecord)row.getData(TAG_RECORD);
		IDatapoolCell cell = (IDatapoolCell)record.getCell(columnIndex - 1);
		Object cellValue = DatapoolClipboard.getInstance().getCellData();
		cell.setCellValue(cellValue);
		String cellStringValue = cell.getStringValue();
		ValueObject valueObject = new ValueObject(cell.getCellValue());
		if(valueObject != null)
			cellStringValue = valueObject.getDescription();
		row.setText(columnIndex, cellStringValue);
		cursor.setSelection(table.getSelectionIndex(), columnIndex);
		cursor.redraw();
		datapoolPart.markDirty();
	}
		
	public IDatapoolPart getIDatapoolPart()
	{
		return datapoolPart;
	}
	
	public void refresh()
	{
		this.refresh(datapool);
	}
	
	public void refresh(IDatapool datapool)
	{
		if(this.datapool != null && this.datapool != datapool)
			this.datapool.removeDatapoolListener(this);
		if(datapool != null && datapool != this.datapool)
			datapool.addDatapoolListener(this);

		this.datapool = datapool;		
		if(datapool == null)
		{
			hideTable();
			return;
		}
		
		restoreTable();
		refreshColumns();
		refreshRows();
		//resetCursor();  // to be fixed after 3.0.1.1 bugzilla74984 didn't get CCB approval.
	}
	
	private void hideTable()
	{	
		table.setVisible(false);
		table.setEnabled(false);
		table.setHeaderVisible(false);

	}
	
	private void restoreTable()
	{
		table.setEnabled(true);
		table.setVisible(true);
		table.setHeaderVisible(true);	
	}
	
	private void refreshColumns()
	{
		int newColumnCount = datapool.getVariableCount();
		int oldColumnCount = tableUtil.getColumnCount() - 1;

		if(cursor != null && !cursor.isDisposed())
		{
			int selectedColumnIndex = cursor.getColumn();
			TableItem selectedItem = cursor.getRow();
			if(selectedColumnIndex > newColumnCount)
				cursor.setSelection(selectedItem, newColumnCount);
			else
				cursor.setSelection(selectedItem, selectedColumnIndex);

		}

		if(oldColumnCount > newColumnCount)
		{
			int numberToDelete = oldColumnCount - newColumnCount;
			for(int i = 0; i < numberToDelete; i++)
			{
				tableUtil.deleteColumn(newColumnCount + 1);
			}
		}

		for(int i = 0; i < datapool.getVariableCount(); i++)
		{
			int index = i + 1;
			TableColumn tableColumn = null;
			if(index < tableUtil.getColumnCount())
				tableColumn = table.getColumn(index);
			else
			{
				tableColumn = new TableColumn(table, SWT.NONE, index);
				tableColumn.setWidth(100);
				tableColumn.addSelectionListener(headerListener);	
				tableColumn.addControlListener(resizeColumnListener);
			}
			IDatapoolVariable variable = (IDatapoolVariable)datapool.getVariable(i);
			tableColumn.setResizable(true);
			IDatapoolSuggestedType suggestedType = (IDatapoolSuggestedType)variable.getSuggestedType();
			String type = new String();
			if(suggestedType != null);
				type = suggestedType.getSuggestedClassName();			
			tableColumn.setText(variable.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + type); //$NON-NLS-1$
			tableColumn.setData(TAG_VARIABLE, variable); 
			tableColumn.setData(TAG_VARIABLE_INDEX, new Integer(i)); 
		}		
	}
	
	private void refreshRows()
	{
		int oldRowCount = table.getItemCount();
		int rowIndex = 0;
		for(int i = 0; i < datapool.getEquivalenceClassCount(); i++)
		{
			IDatapoolEquivalenceClass equivalenceClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(i);
			for(int j = 0; j < equivalenceClass.getRecordCount(); j++)
			{
				IDatapoolRecord record = (IDatapoolRecord)equivalenceClass.getRecord(j);
				int cellCount = record.getCellCount();
								
				String rowContents[] = new String[datapool.getVariableCount() + 1];
				IDatapoolCell[] rowData = new IDatapoolCell[datapool.getVariableCount()];
				if(showEquivlenceClasses)
					rowContents[0] = equivalenceClass.getName() + DatapoolPlugin.getResourceString("DATA_EDT_DIVIDER") + String.valueOf(j);  //$NON-NLS-1$
				else
					rowContents[0] = String.valueOf(j);

				for(int k = 0; k < cellCount; k++)
				{
					IDatapoolCell cell = (IDatapoolCell)record.getCell(k);
					IDatapoolVariable cellVariable = (IDatapoolVariable)cell.getCellVariable();
					int index = findColumnIndex(cellVariable.getName());
					String cellValue = cell.getStringValue();
					ValueObject valueObject = new ValueObject(cell.getCellValue());
					if(valueObject != null)
						cellValue = valueObject.getDescription();
					rowContents[index] = cellValue;
					rowData[index - 1] = cell;
				}				
				
				TableItem item = null;
				if(rowIndex < table.getItemCount())
					item = table.getItem(rowIndex);
				else
					item = new TableItem(table, SWT.NULL);
				item.setText(rowContents);
				item.setData(TAG_DATA, rowData);
				item.setData(TAG_RECORD, record); 
				item.setData(TAG_EQUIVALENCE_CLASS, equivalenceClass); 
				rowIndex++;
			}	
		}
		
		try
		{
			if(cursor == null || cursor.isDisposed())
				createCursor();
		}
		catch(Exception e)
		{
		}
		
		int newRowCount = rowIndex;
		if(cursor != null && !cursor.isDisposed())
		{
			int selectedColumnIndex = cursor.getColumn();
			TableItem selectedItem = cursor.getRow();
			int selectedRowIndex = table.indexOf(selectedItem);
			if(selectedRowIndex > newRowCount - 1)
			{
				if(newRowCount - 1 < 0)
				{
					table.deselectAll();
					if(cursor != null && !cursor.isDisposed())
						cursor.dispose();
				}
				else
				{
					table.setSelection(newRowCount - 1);
					cursor.setSelection(newRowCount - 1, selectedColumnIndex);
				}
			}
			else
				cursor.setSelection(selectedRowIndex, selectedColumnIndex);
		}
		
		if(oldRowCount > newRowCount)
		{
			int numberToDelete = oldRowCount - newRowCount;
			for(int i = 0; i < numberToDelete; i++)
			{
				table.remove(newRowCount);
			}
		}
	}	
}
