/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard.exampleproject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import org.eclipse.core.runtime.IConfigurationElement;

import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.util.FileUtil;
import org.eclipse.hyades.ui.internal.util.StringUtil;
 
/**
 * Subclass of the CreationWizard that adjusts the contents of zip files to 
 * reflect the name of the project that was chosen by the user.
 * 
 * <p>The zip file extensions to be changed are supposed to be defined in the 
 * implementation of {@link #addZipExtensions()}.
 * 
 * <p>The zip is supposed to contain only text files that are encoded as defined
 * in {@link #setEncoding(String)}.  The default encoding is "UTF-8".
 * 
 * @author marcelop
 * @since 0.0.1
 */
abstract public class ZipAdjusterCreationWizard 
extends CreationWizard implements IProjectAdjuster
{
	private String encoding = "UTF-8";
	private List pagesToDeleteFile;
	private Set zipExtensionsToChange;
	
	private String[] oldPats;
	private String[] newPats;
	
	/**
	 * Constructor for ComptestExampleCreationWizard.
	 */
	public ZipAdjusterCreationWizard()
	{
		super();
		
		pagesToDeleteFile = new ArrayList();
		zipExtensionsToChange = new HashSet();		
		setAdjuster(this);
		
		addZipExtensions();
	}
	
	/**
	 * Clients should add the extension to be changed in this method
	 * by using the {@link #addZipExtension(String)}.
	 */
	abstract protected void addZipExtensions();

	/**
	 * @see org.eclipse.jface.wizard.IWizard#dispose()
	 */
	public void dispose()
	{
		zipExtensionsToChange.clear();
		pagesToDeleteFile.clear();
		setAdjuster(null);
		super.dispose();
	}
	
	/**
	 * Adds a zip extension to be changed by this wizard.
	 * @param zipExtension
	 * @return
	 */
	public boolean addZipExtension(String zipExtension)
	{
		return zipExtensionsToChange.add(zipExtension);
	}
	
	/**
	 * Removes a zip extension to be changed by this wizard.
	 * @param zipExtension
	 * @return
	 */
	public boolean removeZipExtension(String zipExtension)
	{
		return zipExtensionsToChange.remove(zipExtension);
	}
	
	/**
	 * Removes all zip extension that are changed by this wizard.
	 */	
	public void removeAllZipExtensions()
	{
		zipExtensionsToChange.clear();
	}

	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.IProjectAdjuster#adjust(com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage[], com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage, java.util.zip.ZipFile)
	 */
	public ZipFile adjust(CreationWizardPage[] pages, CreationWizardPage currentPage, ZipFile zipFile)
	{
		if(!needsReplace(pages))
			return null;
		
		pagesToDeleteFile.add(currentPage);
		try
		{
			return copyZip(zipFile.getName(), FileUtil.getTempDir() + File.separator + getPluginId() + FileUtil.getLastSegment(zipFile.getName()), oldPats, newPats);
		}
		catch(Exception e)
		{
			pagesToDeleteFile.remove(currentPage);
			HyadesUIPlugin.logError(e);
		}
		
		return null;
	}
	
	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.IProjectAdjuster#finished(com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage, java.util.zip.ZipFile)
	 */
	public void finished(CreationWizardPage page, ZipFile zipFile)
	{
		if(!pagesToDeleteFile.contains(page))
			return;

		File file = new File(zipFile.getName());
		file.delete();
	}
	
	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.IProjectAdjuster#adjust(com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage[], com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage, java.io.File)
	 */
	public File adjust(CreationWizardPage[] pages, CreationWizardPage currentPage, File file)
	{
		if(!needsReplace(pages))
			return null;
			
		if(file.isDirectory())
		{
			File tempDir = new File(FileUtil.getTempDir(), getPluginId() + file.getName() + "LMAP");
			if(tempDir.exists())
				FileUtil.delete(tempDir);
				
			if(copyDir(file, tempDir, oldPats, newPats))
			{
				pagesToDeleteFile.add(currentPage);
				return tempDir;
			}
				
			FileUtil.delete(tempDir);
		}
		
		return null;
	}

	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.IProjectAdjuster#finished(com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage, java.io.File)
	 */
	public void finished(CreationWizardPage page, File file)
	{
		if(!pagesToDeleteFile.contains(page))
			return;
			
		FileUtil.delete(file);
	}


	/**
	 * Creates a copy of a zip file.
	 * 
	 * If an entry has a pattern oldPat[i] it will be replaced by newPat[i].  If
	 * this feature is not required the arrays should be null or 0 length (but it will be
	 * much faster to perform a simple file copy since this method reads each entry from
	 * the source file).
	 */
	public ZipFile copyZip(String sourceZip, String targetZip, String[] oldPats, String[] newPats)
	throws IOException, FileNotFoundException
	{
		if(oldPats == null)
			oldPats = new String[0];
		if(newPats == null)
			newPats = new String[0];
			
		if(oldPats.length != newPats.length)
			throw new IllegalArgumentException("oldPats.length != newPas.length");
		
		boolean succeed = false;
		File target = new File(targetZip);
		boolean exists = target.exists();
		if(!exists)
			target.createNewFile();
		
		ZipInputStream zipInputStream = new ZipInputStream(new FileInputStream(sourceZip));
		ZipOutputStream zipOutputStream = new ZipOutputStream(new FileOutputStream(target));
		
		try
		{
			while(true)
			{
				ZipEntry entry = zipInputStream.getNextEntry();
				if(entry == null)
					break;

				if(entry.isDirectory())
				{
					zipOutputStream.putNextEntry(entry);
					FileUtil.streamCopy(zipInputStream, zipOutputStream);
				}
				else
				{
					byte[] buffer = null;
					
					String content = FileUtil.readFromInputStream(encoding, zipInputStream, true);
					String newContent = content;
					for (int i = 0; i < oldPats.length; i++)
						newContent = StringUtil.replace(newContent, oldPats[i], newPats[i]);
					
					buffer = StringUtil.encode(encoding, newContent);
					if(buffer == null)
						buffer = newContent.getBytes();
					
					ZipEntry newEntry = new ZipEntry(entry);
					newEntry.setCompressedSize(-1);
					if(!content.equals(newContent))
					{
						newEntry.setSize(buffer.length);
						
						Checksum checksum = new CRC32();
						checksum.update(buffer, 0, buffer.length);
						newEntry.setCrc(checksum.getValue());
					}
					
					zipOutputStream.putNextEntry(newEntry);
					zipOutputStream.write(buffer);					
				}
				
				zipInputStream.closeEntry();
				zipOutputStream.closeEntry();
			}
			
			succeed = true;
		}
		finally
		{
			try{zipInputStream.close();}catch(Exception e){}
			try{zipOutputStream.close();}catch(Exception e){}
		}
		
		if(!succeed)
		{
			if(!exists)
				target.delete();
		}
		
		return new ZipFile(target);
	}
	
	private boolean needsReplace(CreationWizardPage[] pages)
	{
		if(oldPats != null)
			return true;
			
		List pagesWithNewProjectNames = new ArrayList();
		for(int i = 0; i < pages.length; i++)
		{
			IConfigurationElement[] natures = pages[i].getConfigurationElement().getChildren("nature");
			if((natures != null) && (natures.length > 0))
				pagesWithNewProjectNames.add(pages[i]);

			else if(!pages[i].getProjectName().equals(pages[i].getInitialProjectName()))
				pagesWithNewProjectNames.add(pages[i]);
		}
		
		if(pagesWithNewProjectNames.isEmpty())		
			return false;

		oldPats = new String[pagesWithNewProjectNames.size()*2];
		newPats = new String[pagesWithNewProjectNames.size()*2];
		for(int i=0, max=pagesWithNewProjectNames.size(); i<max; i++)
		{
			CreationWizardPage page = (CreationWizardPage)pagesWithNewProjectNames.get(i);
			oldPats[i] = page.getInitialProjectName() + "/";
			oldPats[i+1] = page.getInitialProjectName() + "\\";
			newPats[i] = page.getProjectName() + "/";
			newPats[i+1] = page.getProjectName() + "\\";
		}
		
		return true;
	}
	
	private boolean copyFiles(File source, File target)
	{		
		try
		{
			if(!target.exists())
				target.createNewFile();

			FileInputStream inputStream = null;
			FileOutputStream outputStream = null;

			try
			{
				inputStream = new FileInputStream(source);
				outputStream = new FileOutputStream(target);
				FileUtil.streamCopy(inputStream, outputStream);
			}
			finally
			{
				if(inputStream != null)
					inputStream.close();
				if(outputStream != null)
					outputStream.close();
			}
		}
		catch(Exception e)
		{
			HyadesUIPlugin.logError(e);
			return false;
		}
		
		return true;
	}
		
	private boolean copyDir(File source, File target, String[] oldPats, String[] newPats)
	{
		File[] files = source.listFiles();
		if(files == null)
			return true;
		
		if(!target.exists())
			target.mkdirs();
		
		for(int i = 0; i < files.length; i++)
		{
			if(files[i].isFile())
			{
				File newFile = new File(target, files[i].getName()).getAbsoluteFile();
				if(zipExtensionsToChange.contains(FileUtil.getFileExtension(files[i])))
				{
					try
					{
						copyZip(files[i].getAbsolutePath(), newFile.getAbsolutePath(), oldPats, newPats); 
					}
					catch(Exception e)
					{
						HyadesUIPlugin.logError(e);
					}					
				}
				else if(newFile.getName().equals(".classpath"))
				{
					try
					{
						String content = FileUtil.readFromFile(encoding, files[i], false);
						if(content != null)
						{
							content = replaceClasspathVariables(content);
							FileUtil.writeToFile(encoding, newFile, false, content);
						}
					}
					catch(IOException e)
					{
						HyadesUIPlugin.logError(e);
					}
				}
				else if(!copyFiles(files[i], newFile))
					return false;				
			}
			else if(files[i].isDirectory())
			{
				if(!"CVS".equals(files[i].getName()))
				{
					if(!copyDir(files[i], new File(target, files[i].getName()), oldPats, newPats))
						return false;
				}
			}
		}
		
		return true;	
	}
	
	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.IProjectAdjuster#beforeWorspaceRefresh(com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage[])
	 */
	public void beforeWorspaceRefresh(CreationWizardPage[] pages)
	{
	}

	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.IProjectAdjuster#afterWorspaceRefresh(com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizardPage[])
	 */
	public void afterWorspaceRefresh(CreationWizardPage[] pages)
	{
	}
	
	/**
	 * Returns the current encoding for the contents of the zip files.
	 * @return String
	 */
	public String getEncoding()
	{
		return encoding;
	}

	/**
	 * Sets the current encoding for the contents of the zip files.
	 * @param string
	 */
	public void setEncoding(String encoding)
	{
		this.encoding = encoding;
	}
}
