/**********************************************************************
 * Copyright (c) 2003,2004 Scapa Technologies Limited and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import java.io.*;
import java.util.*;
import java.util.zip.*;

public class ZipStringBuffer {

	int complen = 0;

	private int CHUNK_SIZE = 10240;
	ArrayList chunks = new ArrayList();

	//empty chunk
	Chunk openChunk = new Chunk(null);
	
	StringBuffer tail = new StringBuffer();

	public ZipStringBuffer() {
	}

	public ZipStringBuffer(String val) {
		append(val);
	}
	
	public ZipStringBuffer(boolean useZip) {
		if (!useZip) {
			CHUNK_SIZE = Integer.MAX_VALUE;
		}
	}
	
	public ZipStringBuffer(int chunk_size) {
		CHUNK_SIZE = chunk_size;
		if (CHUNK_SIZE < 1024) CHUNK_SIZE = 1024;
	}

	public int compressedLength() {
		return complen + tail.length();
	}
	
	public int length() {
		return (chunks.size() * CHUNK_SIZE) + tail.length();
	}

	private void updateChunks() {
		while (tail.length() > CHUNK_SIZE) {
			String buf = tail.substring(0,CHUNK_SIZE);
			tail.delete(0,CHUNK_SIZE);
			addChunk(new StringBuffer(buf));
		}
	}
	
	private void addChunk(StringBuffer sb) {
		Chunk chunk = new Chunk(sb);
		chunk.shrink();
		chunks.add(chunk);
		complen += chunk.dat.length;
	}
	
	public ZipStringBuffer append(Object o) {
		tail.append(o);
		updateChunks();	
		return this;
	}
	public ZipStringBuffer append(String o) {
		tail.append(o);
		updateChunks();	
		return this;
	}
	public ZipStringBuffer append(int c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(double c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(float c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(long c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(byte c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(boolean c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(char c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(char[] c) {
		tail.append(c);
		updateChunks();
		return this;
	}
	public ZipStringBuffer append(char[] c, int offset, int len) {
		tail.append(c,offset,len);
		updateChunks();
		return this;
	}


	public char charAt(int n) {
		int index = n / CHUNK_SIZE;
		int offset = n % CHUNK_SIZE;
//EditorPlugin.DBG.info("index="+index+",offset="+offset);		
		if (index >= chunks.size()) {
			
			return tail.charAt(offset);
		} else {
		
			Chunk chunk = (Chunk)chunks.get(index);
			if (chunk != openChunk) {
				openChunk.shrink();
				openChunk = chunk;	
			}
			chunk.grow();
			
			return chunk.sb.charAt(offset);
		}
	}

	public void getChars(int srcBegin, int srcEnd, char[] dst, int dstBegin) {  
		String s = substring(srcBegin,srcEnd);
		s.getChars(0,s.length(),dst,dstBegin);
	}
	
	public String substring(int begin) {
		return substring(begin,length());
	}

	public String substring(int begin, int end) {
		if (begin == end) return "";
		int s_index = begin / CHUNK_SIZE;
		int s_offset = begin % CHUNK_SIZE;
		int f_index = (end-1) / CHUNK_SIZE;
		int f_offset = (end-1) % CHUNK_SIZE;
		
		if (s_index >= chunks.size()) {
			return tail.substring(s_offset,f_offset);	
		}
		StringBuffer sb = new StringBuffer();
		
		//append the first bit
		Chunk chunk = (Chunk)chunks.get(s_index);
		chunk.grow();
		sb.append(chunk.sb.substring(s_offset));
		chunk.shrink();
		
		//append any other complete chunks
		for (int i = s_index+1; i < f_index; i++) {
			chunk = (Chunk)chunks.get(i);
			chunk.grow();
			sb.append(chunk.sb.toString());	
		chunk.shrink();
		}

		if (f_index >= chunks.size()) {
			sb.append(tail.substring(0,f_offset+1));
		} else {		
			chunk = (Chunk)chunks.get(f_index);
			chunk.grow();
			sb.append(chunk.sb.substring(0,f_offset+1));
			chunk.shrink();
		}

		return sb.toString();		
	}

	public String toString() {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < chunks.size(); i++) {
			Chunk chunk = (Chunk)chunks.get(i);
			chunk.grow();
			sb.append(chunk.sb.toString());
		}	
		sb.append(tail.toString());
		return sb.toString();
	}
	
	public void toStream(OutputStream out) throws IOException {
		for (int i = 0; i < chunks.size(); i++) {
			Chunk chunk = (Chunk)chunks.get(i);
			chunk.grow();
			out.write(chunk.sb.toString().getBytes());
		}	
		out.write(tail.toString().getBytes());
	}

	public int indexOf(char c, int index) {
		while (index < length()) {
			if (charAt(index) == c) return index;
			index++;			
		}
		return -1;
	}

	public int lastIndexOf(char c) {
		int index = length()-1;
		while (index > 0) {
			if (charAt(index) == c) return index;
			index--;			
		}
		return -1;
	}

	class Chunk {
		boolean compressed = true;
		StringBuffer sb;
		byte[] dat;	
		
		public Chunk(StringBuffer sb) {
			this.sb = sb;	
		}
		
		void shrink() {
			if (sb == null) return;
			try {
				dat = compress(sb.toString().getBytes());
			} catch (Exception e) {
				e.printStackTrace();
			}	
			sb = null;
		}
		void grow() {
			if (dat == null) return;
			try {
				sb = new StringBuffer(new String(decompress(dat)));
			} catch (Exception e) {
				e.printStackTrace();
			}	
			dat = null;
		}
	}

	private byte[] compress(byte[] dat) throws IOException {
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		GZIPOutputStream gzout = new GZIPOutputStream(bout);
		
		gzout.write(dat,0,dat.length);
		
		gzout.finish();
		gzout.close();
		
		return bout.toByteArray();
	}
	
	private byte[] decompress(byte[] dat) throws IOException {
		ByteArrayInputStream bin = new ByteArrayInputStream(dat);
		GZIPInputStream gzin = new GZIPInputStream(bin);
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
				
		byte[] buf = new byte[1024];
		int n = 0;
		while (n != -1) {
			n = gzin.read(buf,0,1024);
			if (n > 0) {
				bout.write(buf,0,n);	
			}	
		}
		return bout.toByteArray();
	}
}