package org.eclipse.hyades.ui.sample.svg.generator;
/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

import java.io.Serializable;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Date;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
/**
 * <code>SVGPieChart</code> generates a pie chart graphic DOM using Scalable Vector Graphics (SVG). 
 * 
 * @version 1.60.1.17
 */
public class SVGPieChart extends SVGDocumentGenerator implements IGraphicDocumentStyle, IDataInputProcessingExceptionCodes, IDataInputConstants, Serializable {
	private final PieChartDataRetriever dataRetriever;
	
	// constants for determining tick direction
	private final int NINETY_DEG = 1;
	private final int NEGATIVE_NINETY_DEG = 2;    
	private final int ONE_EIGHTY_DEG = 3;		  
	private final int ZERO_DEG = 4;
	
	/**
	 *  Sole constructor.
	 */
	public SVGPieChart() {
		super();
		graphicType = IGraphicTypeConstants.PIE_CHART;		
		dataRetriever = new PieChartDataRetriever();
	}
		
	/**
	 * In addition to those thrown by <code>PieChartDataRetriever</code>, this
	 * method throws <code>DataInputProcessingException</code> for codes <code>MISSING_TOTALS,
	 * MISSING_PERCENTAGES</code>.
	 * 
	 * @see IGraphicDocumentGenerator#generateGraphicDocument(GraphicDocumentProperties)
	 */	
	public Document generateGraphicDocument(GraphicDocumentProperties input) throws DataInputProcessingException {							
		GraphicAttributes graphicAttrs = createGraphicAttributes(input);		
		Document newDataDocument = graphicAttrs.getDataDocument();		
								
		int numberOfDatasets = 0;
		//String [] datasetTotals = null;
		//String [] datasetPercentages = null;
			
		// if there's no data, nothing to do		
		try {		
			numberOfDatasets = dataRetriever.getNumberOfDatasets(newDataDocument);				
		} catch (DataInputProcessingException e) {
			e.setWidgetType(graphicType);
			//System.out.println(e.getWidgetType() + " " + e.getElement() + ": " + e.getMessage());
			throw e;			
		}
		
		DataRange dataRange = null;
		SegmentMarker[] markers = null;
		DataSet[] dataSets = null;
		String [] legendFlyOverLabels = null;
		

		// The data range
		try {
			dataRange = dataRetriever.getDataRange(newDataDocument, "");
		} catch (DataInputProcessingException e) {
			e.setWidgetType(graphicType);
			//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
			throw e;				
		}					

		// The datasets
		dataSets = new DataSet[numberOfDatasets];
		for(int i=0; i < numberOfDatasets; i++) {
            dataSets[i] = dataRetriever.getDatasetWithPosition(newDataDocument, Integer.toString(i));
		}

		//get the Fly over text for the legend item.						
		legendFlyOverLabels = getLegendFlyOverLabels(dataSets);
		
		// start calculation
		double gWidth = graphicAttrs.getGraphicWidth();
		double gHeight = graphicAttrs.getGraphicHeight();
		double cx;
		
		int yOffset = (int)(gHeight * 0.152);	

		if (!graphicAttrs.isLegendSuppressed()) {
		    double maxLengthOfLegendLabel = getMaxLengthOfLegendLabel(graphicAttrs, dataSets);
		    double maxAlloableLegendWidth = graphicAttrs.getGraphicWidth() / 3;
	   		if (maxLengthOfLegendLabel > maxAlloableLegendWidth) {
				maxLengthOfLegendLabel = maxAlloableLegendWidth;
			}
			gWidth -= maxLengthOfLegendLabel + 12;
		}

		cx = gWidth * 0.5;	
		//double radius = gHeight >= gWidth ? gWidth * 0.173 : gHeight * 0.233; 
		double radius = gHeight >= gWidth ? gWidth * 0.245 : gHeight * 0.233; 
		double cy = gHeight * .52;		
		String translateString = "translate(" + cx + "," + cy + ")";		
		
		Document generatedDocument = createSvgDocument(newDataDocument.getImplementation(), Short.toString(graphicAttrs.getGraphicWidth()), Short.toString(graphicAttrs.getGraphicHeight()));
		Element svgRoot = generatedDocument.getDocumentElement();
		registerEventHandler(svgRoot, "onload", "init(evt)");	
		addDefinitions(generatedDocument, graphicAttrs, numberOfDatasets);					
		addJavaScriptFunctions(generatedDocument, graphicAttrs, numberOfDatasets);
 		
 		// graphic outline 
 		if (!graphicAttrs.isOutlineSuppressed()) {
 			addOutline(generatedDocument, graphicAttrs);
 		}	 		
 		 		
		// timestamp
		if (!graphicAttrs.isTimestampSuppressed()) {
			try {
				String timestamp = dataRetriever.getTimestamp(newDataDocument);
				Date date = GraphicAttributes.parseISO8601Date(timestamp);
	        	addTimeStamp(generatedDocument, graphicAttrs, (date == null) ? timestamp : graphicAttrs.formatDate(date));
			} catch (DataInputProcessingException e) {
				e.setWidgetType(graphicType);
				//System.out.println(e.getWidgetType() + " " + e.getElement() + " " + e.getMessage());
				throw e;
			}
		}
			
		// title bar
 		if (!graphicAttrs.isTitlebarSuppressed()) {	 		
	   		addTitleBar(generatedDocument, graphicAttrs);
 		}

		// preference icon
		if(!graphicAttrs.isUserPreferencesSuppressed() && graphicAttrs.getSvgFormatOnly())
		{
			addPreferenceIcon(generatedDocument, graphicAttrs);
		}
		 		
    	// pie labels	
	   	try {
			Hashtable titles = dataRetriever.getTitles(newDataDocument);		
 		    addTitles(generatedDocument, graphicAttrs, svgRoot, titles, cx, yOffset);
		} catch (DataInputProcessingException e) {
			e.setWidgetType(graphicType);
			//System.out.println(e.getWidgetType() + " " + e.getElement() + ": " + e.getMessage());			
			throw e;
		}
       // data lines and area
		Element gTextBoxElement = generatedDocument.createElement("g");   
				
		addPieSlices(generatedDocument, graphicAttrs, svgRoot, dataSets, radius, cx, cy, translateString);  	 
	   		   	
        // legend
        if (!graphicAttrs.isLegendSuppressed()) {      		
	  		try {
	       		String [] legendLabels = dataRetriever.getLegendLabels(newDataDocument);		       
				addLegend(generatedDocument, graphicAttrs, gTextBoxElement, legendLabels, legendFlyOverLabels, (int)(gWidth), yOffset); 
	  		} catch (DataInputProcessingException e) {
	  			e.setWidgetType(graphicType);
	  			//System.out.println(e.getWidgetType() + " " + e.getElement() + ": " + e.getMessage());	  			
	  			throw e;
	  		} 		
    	}	
		svgRoot.appendChild(gTextBoxElement);    	
        return generatedDocument;		
	}	
	
	protected void addJavaScriptFunctions(Document generatedDocument, GraphicAttributes attrs, int numberOfSets) {		
		Element parent = generatedDocument.getDocumentElement();	
		super.addJavaScriptFunctions(generatedDocument, attrs, numberOfSets, 0, 0, 0, 0);
		
		Element newElement = generatedDocument.createElement("script");
		newElement.setAttribute("type", JAVASCRIPT_VERSION);
		parent.appendChild(newElement);
	
		ECMAScriptGenerator jscript = new ECMAScriptGenerator();
		StringBuffer functions = new StringBuffer(jscript.generatePieChartHelperFunctions());							
		CDATASection newCDATASection = generatedDocument.createCDATASection(functions.toString());

		newElement.appendChild(newCDATASection);
	}	
	
	/**
	 * Adds the titles to the SVG graphic. 
	 * 
	 * @param parent element to which the title-related elements are appended
	 * @param input the titles
	 * @param cx centre of the pie
	 * @param yOffset vertical offset
	 */
	private void addTitles(Document generatedDocument, GraphicAttributes attrs, Element parent, Hashtable input, double cx, int yOffset) {				
		int rotation = 0;
		int xOffset = 10;
		Element gElement = generatedDocument.createElement("g");
		String title = null;
		
		// add a font group for the labels and then add the labels
		// pie slice value title
		title = (String)input.get(VALUE_TITLE);
		if (title != null) {			
			gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id", "labelsValues0");
			gElement.setAttribute("class", "labelsValues anchorAtMiddle");
//			gElement.setAttribute("font-family", attrs.getGraphicFont());
			parent.appendChild(gElement);				
			// need the centre point of the circle
			// NumberFormatException not possible here
			addLabel(generatedDocument, attrs, gElement, title, null, cx, (attrs.getGraphicHeight() * 0.9), rotation);
		}
		// total value title
		title = (String)input.get(TOTAL_TITLE);
		if (title != null) {			
			gElement = generatedDocument.createElement("g");
			gElement.setAttribute("id", "labelsValues1");
			gElement.setAttribute("class", "legendTitle anchorAtStart");
//			gElement.setAttribute("font-family", attrs.getGraphicFont());
			parent.appendChild(gElement);	
			// magic numbers for positioning			
			addLabel(generatedDocument, attrs, gElement, title, null, xOffset, yOffset + 10, rotation);
		}
	}	
	
	// add style element with default style sheet rules for the pie chart
	private void addDefaultInternalStylesheet(Document generatedDocument, GraphicAttributes attrs, Element parent, int sets) {
		StringBuffer styleRules = new StringBuffer(COMMON_STYLE);
		styleRules.append(" ");
		styleRules.append(PIE_STYLE);
		
		String [] palette = (String [])(attrs.getGraphicColorPalette()).get(attrs.getGraphicColorPaletteName());		
		// palette has limited number of colours, so loop if more sets
		int paletteLength = palette.length;		  
		for (int i = 0, j = 0; i < sets; i++) {
		  j = (i + 1) % paletteLength;
		  j = (j == 0) ? paletteLength - 1 : j-1;			
		  styleRules.append(" .dataset" + i + "{stroke:black; fill:" + palette[j] + ";}");	
		  if (!attrs.isLegendSuppressed()) {	  
		  	styleRules.append(" .shape" + i + "{stroke:none; fill:" + palette[j] + ";}");
		  }
		}
		
		CDATASection newCDATASection = generatedDocument.createCDATASection(styleRules.toString());
		
		Element newElement = generatedDocument.createElement("style");
		newElement.setAttribute("id", "defaultStyleRules");
		newElement.setAttribute("type", "text/css");
		newElement.appendChild(newCDATASection);
		parent.appendChild(newElement);
	}	
	
	// add defs element with pie chart definitions 
	private void addDefinitions(Document generatedDocument, GraphicAttributes attrs, int numberOfDatasets) {
		Element defsElement = super.addDefinitions(generatedDocument, attrs);
		addDefaultInternalStylesheet(generatedDocument, attrs, defsElement, numberOfDatasets);		
		
		// id: tickline		
		Element gElement_tick = generatedDocument.createElement("g");
		gElement_tick.setAttribute("id", "tickline");
			
		Element tickElement = generatedDocument.createElement("polyline");
		//tickElement.setAttribute("class", "st1, wedgeTick");
		tickElement.setAttribute("class", "grid");
		tickElement.setAttribute("points", "0 0 8 0");
		gElement_tick.appendChild(tickElement);
		defsElement.appendChild(gElement_tick);
		
	}	
	
	// add the slices to the pie	
	private void addPieSlices(Document generatedDocument, GraphicAttributes attrs, Element parent, DataSet[] dataSets, double radius, double cx, double cy, String translateString) {				
		double pieValue = 0;
		double sliceValue = 0;
		double percent = 0;	
		double degree = 0;
		double previousDegree = 0;
		double radian = 0;
		double xPosition = 0;
		double yPosition = 0;		
		Element newElement, gColourElement, gToggleValueElement, gToggleBothValuesElement;
		Element gTogglePercentElement;
		
		// groups for toggling values and percentages visibility		
		// slice value
		gToggleValueElement = generatedDocument.createElement("g");
		gToggleValueElement.setAttribute("id","exactValues");			
		gToggleValueElement.setAttribute("onclick","toggleVisibility(\"exactValues\")");
		gToggleValueElement.setAttribute("visibility", "hidden");
		// add it to the colour group for color changes
		parent.appendChild(gToggleValueElement);	
		
		// percentage value
		gTogglePercentElement = generatedDocument.createElement("g");
		gTogglePercentElement.setAttribute("id", "percentValues");
		gTogglePercentElement.setAttribute("onclick","toggleVisibility(\"percentValues\")");
		gTogglePercentElement.setAttribute("visibility", "hidden");
		parent.appendChild(gTogglePercentElement);
		
		// both values
		gToggleBothValuesElement = generatedDocument.createElement("g");
		gToggleBothValuesElement.setAttribute("id", "bothValues");
		gToggleBothValuesElement.setAttribute("onclick","toggleVisibility(\"bothValues\")");
		parent.appendChild(gToggleBothValuesElement);

									
		// caluclate the total value of the pie
		for (int i = 0; i < dataSets.length; i++) {
			pieValue += dataSets[i].getTotal();
		}	
				
		// for each slice, calculate the slice arc and position from the sliceValue percentage
		for (int i = 0; i < dataSets.length; i++) {	
			// NumberFormatException not possible here								
			sliceValue = dataSets[i].getTotal();								
			percent = (pieValue > 0) ? sliceValue / pieValue : 0;				
			// calculate the degrees of the percentage
			degree = percent * 360;
			radian = degree * 0.01745329252;
			// find the x and y position based on the radius and the degrees
			xPosition = (Math.cos(radian)) * radius;
			yPosition = (Math.sin(radian)) * radius;
			
			// calculate the arc path			
			String path = "M " + radius + " 0 ";
			
			if (degree == 360) {
				path += " A "
					  + radius
					  + " "
					  + radius
					  + " 0 0 0 "
					  + -radius
					  + " 0 "
					  + " A "
					  + radius
					  + " "
					  + radius
					  + " 0 0 0 "
					  + radius
					  + " 0 ";
			}
			else if (degree != 0) {
				path += " A "
					  + radius
					  + " "
					  + radius
					  + " 0 " + ((degree < 180) ? 0 : 1) + " 0 "
					  + (xPosition)
					  + " "
					  + (-yPosition);
			}

			path += " L 0 0  Z";
					
			// set the slice transformation					
			String transformation =(i ==0) ? translateString : translateString + " rotate(-" + previousDegree + ")";
				
			// group element for data colour changes and toggling exact values visibility
			gColourElement = generatedDocument.createElement("g");
			gColourElement.setAttribute("id", "pieColour" + i);
			gColourElement.setAttribute("class", "dataset" + i);				
			gColourElement.setAttribute("onclick", "toggleVisibility(\"pieColour\");");
			parent.appendChild(gColourElement);										
				
			newElement = generatedDocument.createElement("path");						
			newElement.setAttribute("transform", transformation);						
			newElement.setAttribute("d", path);											
			gColourElement.appendChild(newElement);
			
			String percentageLabel = attrs.formatPercentage(percent);
			String sliceValueLabel = attrs.formatNumber(sliceValue);
			
			// add the value and percentage labels for the slice
			addLabels(generatedDocument, attrs, gToggleBothValuesElement, gToggleValueElement, gTogglePercentElement, i, percentageLabel, degree, previousDegree, sliceValueLabel, radius, cx, cy, translateString);
			// set up for loop
			previousDegree += degree;
		}		
	}
	
	// add value and percentage to each pie slice
	private void addLabels(Document generatedDocument, GraphicAttributes attrs, Element bothValuesElement, Element valueElement, Element percentElement, int index, String percentageLabel, double degree, double previous, String sliceValueLabel, double radius, double cx, double cy, String translateString) {		
		Element newElement;
		//double degree = percentage * 360;					
		double rad = (degree / 2 + previous) * 0.01745329252;
		double xpos = (Math.cos(-1 * rad)) * radius;
		double ypos = (Math.sin(-1 * rad)) * radius;		
		
		String rotateString = EMPTY_STRING;
		int rotateFlag = 0;
				
		// adjust the tick rotation value
		if (cy + ypos > cy && cy + ypos >= cy + 7.0/8.0 * radius) {
			// ypos is near the bottom of the pie chart
			rotateString = "rotate(" + (90) + "," + xpos + "," + ypos + ")";
			rotateFlag = NINETY_DEG;	
		} else if (cy + ypos < cy && cy + ypos <= cy - 7.0/8.0 * radius) {  
			 // ypos is near the top of the pie chart
			rotateString = "rotate(" + (-90) + "," + xpos + "," + ypos + ")";
			rotateFlag = NEGATIVE_NINETY_DEG;
		} else if (cx + xpos < cx) {  
			// if the tick is on the left side of the pie chart
			rotateString = "rotate(" + (180) + "," + xpos + "," + ypos + ")";	
			rotateFlag = ONE_EIGHTY_DEG;
		} else {
			rotateFlag = ZERO_DEG;	
		}
				
		// xlink the tick line 
		newElement = generatedDocument.createElement("use");
		newElement.setAttribute("xlink:href", "#tickline");
		newElement.setAttribute("x", Double.toString(xpos));
		newElement.setAttribute("y", Double.toString(ypos));  
		newElement.setAttribute("transform", translateString + rotateString);	  
		valueElement.appendChild(newElement);
		percentElement.appendChild(newElement);   // newly added
		bothValuesElement.appendChild(newElement);
		
		// create the group for toggling visibility
		String stylesheetClass;
		
		// all text labels will be anchored at start if they are located on
		// right side of the circle; otherwise, the labels will be anchored 
		// at end.
		if (cx + xpos >= cx) {
			stylesheetClass = "pieValues anchorAtStart";							
		} else {
			stylesheetClass = "pieValues anchorAtEnd";
		}		
		
		// percentage and value - need to separate these			
		String label = percentageLabel + ",  " + sliceValueLabel;
		
		// value and percentage labels
		String percentLabel = percentageLabel;
		//String sliceValueLabel = sliceValue;
		
		// determine label x and y positions
		Vector labelPosVec = calculateLabelPositions(cx, xpos, ypos, rotateFlag, label);
		// NumberFormatException not possible here
		double labelXPos = Double.parseDouble(labelPosVec.firstElement().toString());
		double labelYPos = Double.parseDouble(labelPosVec.lastElement().toString());
//		String graphicFont = attrs.getGraphicFont();
		
		// draw percentage label
		newElement = generatedDocument.createElement("text");	
		newElement.setAttribute("x", Double.toString(labelXPos));
		newElement.setAttribute("y", Double.toString(-1 * labelYPos));
		newElement.setAttribute("transform", translateString);		
		newElement.setAttribute("class", stylesheetClass + " shape" + index);
//		newElement.setAttribute("font-family", graphicFont);
		newElement.appendChild(generatedDocument.createTextNode(percentLabel));
		percentElement.appendChild(newElement);

		// draw value label
		newElement = generatedDocument.createElement("text");	
		newElement.setAttribute("x", Double.toString(labelXPos));	
		newElement.setAttribute("y", Double.toString(-1 * labelYPos));
		newElement.setAttribute("transform", translateString);		
		newElement.setAttribute("class", stylesheetClass + " shape" + index);
//		newElement.setAttribute("font-family", graphicFont);
		newElement.appendChild(generatedDocument.createTextNode(sliceValueLabel));
		valueElement.appendChild(newElement);							
		
		// draw both values
		newElement = generatedDocument.createElement("text");	
		newElement.setAttribute("x", Double.toString(labelXPos));	
		newElement.setAttribute("y", Double.toString(-1 * labelYPos));
		newElement.setAttribute("transform", translateString);		
		newElement.setAttribute("class", stylesheetClass + " shape" + index);
//		newElement.setAttribute("font-family", graphicFont);
		newElement.appendChild(generatedDocument.createTextNode(label));
		bothValuesElement.appendChild(newElement);							
	}	

	// determine value and percentage label position	
	private Vector calculateLabelPositions(double cx, double xpos, double ypos, int rotateFlag, String label) {
		double labelXPos = 0;
		double labelYPos = 0;		
		
		// store the label x and y positions into a vector
		Vector labelVec = new Vector(2);  
			
		if (cx + xpos < cx) {   // if the label is located on the left side of the circle				
			if (rotateFlag == NEGATIVE_NINETY_DEG) {
				labelXPos = xpos + 2; 
				labelYPos = -ypos + 12;
			} else if (rotateFlag == NINETY_DEG) {
				labelXPos = xpos + 2;
				labelYPos = -ypos - 20;	
			} else if (rotateFlag == ONE_EIGHTY_DEG) {	
				labelXPos = xpos - 10;
				labelYPos = -ypos - 1;
			}
		} else {   // if the label is located on the right side of the circle
			if (rotateFlag == NEGATIVE_NINETY_DEG) {
				labelXPos = xpos - 3;
				labelYPos = -ypos + 12;	
			} else if (rotateFlag == NINETY_DEG) {
				labelXPos = xpos - 3;
				labelYPos = -ypos - 20;	
			} else if (rotateFlag == ZERO_DEG) {
				labelXPos = xpos + 10;
				labelYPos = -ypos - 1 ;
			}
		}
		
		labelVec.add(new Double(labelXPos));
		labelVec.add(new Double(labelYPos));
		
		return labelVec;
	}
	
	/**
	 * Retrieves all configuration attributes from the data retriever
	 */
	private void getConfigAttributes(GraphicAttributes attrs) throws DataInputProcessingException {
		Hashtable configProperties = dataRetriever.getConfigurationAttributes(attrs.getConfigDocument(), attrs.getDataDocument());	
		if (configProperties == null) {
			return;
		}	
		String width = (String)configProperties.get(WIDTH);
		String height = (String)configProperties.get(HEIGHT);
		//String fontFamily = (String)configProperties.get(FONT);
		String title = (String)configProperties.get(TITLE);
		String timestampPrefix = (String)configProperties.get(TIMESTAMP_PREFIX);
		//String suppressLegend = (String)configProperties.get(SUPPRESS_LEGEND);
		String legendTitle = (String)configProperties.get(LEGEND_TITLE);
					
		if (width != null) {
			attrs.setGraphicWidth(width);
		}	
		
		if (height != null) {
			attrs.setGraphicHeight(height);
		}	
		
		//if (fontFamily != null) {
		//	attrs.setGraphicFont(fontFamily);			
		//}			
		
		if (title != null) {
			attrs.setGraphicTitle(title);			
		}
			
		if (timestampPrefix != null) {
			attrs.setGraphicTimestampPrefix(timestampPrefix);			
		}	
		
		//if (suppressLegend != null) {
		//	attrs.setLegendSuppressed(suppressLegend);			
		//}	
		
		if (legendTitle != null) {
			attrs.setGraphicLegendTitle(legendTitle);			
		}	
		
		/* Test 

		System.out.println("From the input documents Graphic Color Palette: " + attrs.getGraphicColorPalette());
		System.out.println("From the input documents Graphic Color Palette Name: " + attrs.getGraphicColorPaletteName());
		System.out.println("From the input documents Graphic Font: " + attrs.getGraphicFont());
		System.out.println("From the input documents Graphic Height: " + attrs.getGraphicHeight());
		System.out.println("From the input documents Graphic Legend Title: " + attrs.getGraphicLegendTitle());
		System.out.println("From the input documents Graphic Timestamp Prefix: " + attrs.getGraphicTimestampPrefix());
		System.out.println("From the input documents Graphic Title: " + attrs.getGraphicTitle());
		System.out.println("From the input documents Graphic Type: " + attrs.getGraphicType());
		System.out.println("From the input documents Graphic Width: " + attrs.getGraphicWidth());
		System.out.println("From the input documents Preferences Icon Text: " + attrs.getPreferencesIconText());
		System.out.println("From the input documents Preferences Page: " + attrs.getPreferencesPage());
		System.out.println("From the input documents Preferences Page Height: " + attrs.getPreferencesPageHeight());
		System.out.println("From the input documents Preferences Page Width: " + attrs.getPreferencesPageWidth());
		/* Test */	
	}
		
	/**
	 * Initialize the pie chart attributes
	 */
	private GraphicAttributes createGraphicAttributes(GraphicDocumentProperties input) throws DataInputProcessingException {
		GraphicAttributes graphicAttrs = new GraphicAttributes();
			
		// set up the defaults
		graphicAttrs.setGraphicWidth(PIE_DEFAULT_WIDTH);
		graphicAttrs.setPreferencesPage(IUserPreferencesConstants.PIE_PREFS);
		graphicAttrs.setPreferencesPageWidth(IUserPreferencesConstants.PREFS_WIDTH);
		graphicAttrs.setPreferencesPageHeight(IUserPreferencesConstants.PIE_PREFS_HEIGHT);
		
		// get the input documents
		graphicAttrs.setConfigDocument(input.getConfigDocument());
		graphicAttrs.setDataDocument(input.getDataDocument());
		
		// (re)set any properties from the input documents		
		getConfigAttributes(graphicAttrs);
		
		// (re)set any properties from the input bean
		graphicAttrs.setAttributesFrom(input);		
				
		return graphicAttrs;
	}	
}

